/*
messagewall.h - MessageWall main declarations
Copyright (C) 2002 Ian Gulliver

This program is free software; you can redistribute it and/or modify
it under the terms of version 2 of the GNU General Public License as
published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#ifndef _MESSAGEWALL_H
#define _MESSAGEWALL_H

#define MESSAGEWALL_VERSION "1.0.8"

#include "firemake.h"
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <openssl/ssl.h>
#include <firestring.h>

#define max(a,b) (a > b ? a : b)
#define min(a,b) (a < b ? a : b)
#define messagewall_malloc(type) (type *) firestring_malloc(sizeof(type))

#define PROGRESS_WAITRMX 1
#define PROGRESS_WAITDNSBL_DOMAIN 2
#define PROGRESS_WAITRDNS 3
#define PROGRESS_WAITDNSBL 4
#define PROGRESS_WAITDNSDCC 5
#define PROGRESS_WORKING 6

#define AUTH_STATUS_NONE 0
#define AUTH_STATUS_LOGIN_USER 1
#define AUTH_STATUS_LOGIN_PASS 2
#define AUTH_STATUS_PLAIN 3

#define HASH_SIZE 65536 /* 256 * 256 (hash values are 2byte) */

struct messagewall_client_t { /* client session control block */
	struct mime_part_t *parts;
	int fd;
	int can_relay;
	struct in_addr ip;
	struct firestring_estr_t username;
	struct firestring_estr_t password;
	struct firestring_estr_t buffer;
	struct firestring_estr_t message;
	struct firestring_estr_t from;
	struct firestring_estr_t fromdomain;
	struct firestring_estr_t *to;
	struct firestring_estr_t *warnings;
	int *points;
	int num_warnings;
	int have_from;
	int num_to;
	int in_data;
	int message_point;
	int send_progress;
	int error;
	int backend;
	int rdns_fd;
	int rmx_fd;
	int rmx_a_fd;
	int *dnsbl_fd;
	struct in_addr *dnsbl_ip;
	int *dnsbl_domain_fd;
	struct in_addr *dnsbl_domain_ip;
	int **dnsdcc_fd;
	int num_parts;
	int auth_status;
	time_t lasttalk;
	time_t signon;
	time_t rmx_request;
	time_t dnsdcc_send;
	int resent_dnsbl;
	int resent_dnsdcc;
	int resent_rdns;
	int resent_dnsbl_domain;
	int resent_rmx;
	int errors;
	int warn;
	int bare_lf;
	int score;
	struct messagewall_profile_t *profile;
	SSL *ssl;
	int ssl_handshake;
};

struct messagewall_local_domain_t { /* linked list of local (inside) domains */
	struct firestring_estr_t domain;
	struct messagewall_local_domain_t *next;
};

struct messagewall_relay_ip_t {
	unsigned char ip[4];
	unsigned char mask[4];
	struct messagewall_relay_ip_t *next;
};

struct messagewall_estr_ll_t {
	struct firestring_estr_t string;
	struct messagewall_estr_ll_t *next;
};

struct messagewall_estr_score_ll_t {
	struct firestring_estr_t string;
	int score;
	struct messagewall_estr_score_ll_t *next;
};

struct messagewall_header_reject_t {
	struct firestring_estr_t header;
	struct firestring_estr_t content;
	int score;
	struct messagewall_header_reject_t *next;
};

struct messagewall_backend_t {
	int client;
	int fd;
	int eightbit;
	int pipelining;
	int tls;
	int auth_plain;
#define BACKEND_STATE_NOTCONNECTED 0
#define BACKEND_STATE_CONNECTING 1
#define BACKEND_STATE_GREETING 2
#define BACKEND_STATE_EHLO_RESPONSE 3
#define BACKEND_STATE_STARTTLS_RESPONSE 4
#define BACKEND_STATE_SSL_HANDSHAKE 5
#define BACKEND_STATE_HELO_RESPONSE 6
#define BACKEND_STATE_IDLE 7
#define BACKEND_STATE_CLEARPIPE 8
#define BACKEND_STATE_SENDING 9
#define BACKEND_STATE_WAITDELIVER 10
#define BACKEND_STATE_RSET 11
	int piped;
	int state;
	struct firestring_estr_t buffer;
	struct firestring_estr_t message;
	int sent;
	SSL *ssl;
};

struct messagewall_profile_t { /* profile control block */
	char *name;
	int reject_score;
	int to_cc_check;
	int to_cc_score;
	int from_check;
	int from_score;
	int realname_check;
	int realname_score;
	int rdns_required;
	int rdns_score;
	int rmx_required;
	int rmx_score;
	int reject;
	struct messagewall_header_reject_t *header_rejecti;
	struct messagewall_header_reject_t *header_reject;
	struct messagewall_estr_score_ll_t *body_reject;
	struct messagewall_estr_score_ll_t *body_rejecti;
	struct messagewall_estr_score_ll_t *filename_reject;
	struct messagewall_estr_score_ll_t *mime_reject;
	struct messagewall_estr_ll_t *mime_strip;
	struct messagewall_estr_ll_t *mime_allow;
	struct messagewall_dnsbl_profile_t *dnsbl;
	struct messagewall_dnsbl_profile_t *dnsbl_domain;
	struct messagewall_dnsdcc_profile_t *dnsdcc;
	struct messagewall_virus_profile_t *virus_scan;
	struct messagewall_profile_t *next;
};

struct messagewall_special_user_t {
	struct firestring_estr_t address;
	struct messagewall_profile_t *profile;
	struct messagewall_special_user_t *next;
};

struct messagewall_dnsbl_global_t {
	struct firestring_estr_t dnsbl;
	int i;
	struct messagewall_dnsbl_global_t *next;
};

struct messagewall_dnsdcc_global_t {
	struct firestring_estr_t dnsdcc;
	char * (*check)(struct firestring_estr_t *);
	int i;
	struct messagewall_dnsdcc_global_t *next;
};

struct messagewall_dnsbl_profile_t {
	struct messagewall_dnsbl_global_t *global;
	int have_response_ip;
	struct in_addr response_ip;
	int score;
	struct messagewall_dnsbl_profile_t *next;
};

struct messagewall_dnsdcc_profile_t {
	struct messagewall_dnsdcc_global_t *global;
	int score;
	struct messagewall_dnsdcc_profile_t *next;
};

struct messagewall_virus_profile_t {
	struct messagewall_virus_database_t *global;
	int score;
	struct messagewall_virus_profile_t *next;
};

struct messagewall_virus_identity_t {
	struct firestring_estr_t name;
	struct firestring_estr_t pattern;
	struct messagewall_virus_identity_t *next;
};

struct messagewall_virus_database_t {
	struct messagewall_virus_identity_t *database[65536];
	struct firestring_estr_t filename;
	struct messagewall_virus_database_t *next;
};

struct messagewall_relay_auth_t {
	struct firestring_estr_t username;
	struct firestring_estr_t hash;
	struct messagewall_relay_auth_t *next;
};

#ifndef _MESSAGEWALL_C
extern const char *domain;
extern const char *path_charset;
extern const char *local_domains;
extern const char *relay_ips;
extern const char *special_users;
extern const char *relay_auth;
extern const char *cert;
extern const char *backend_cert;
extern const char *pid_dir;
extern int max_rcpt;
extern long max_message_size;
extern int sevenbit;
extern int dnsbls;
extern int dnsbls_domain;
extern int dnsdccs;

extern struct firestring_estr_t greeting;

extern struct messagewall_local_domain_t **local_domain_hash;
extern struct messagewall_relay_ip_t **relay_ip_hash;
extern struct messagewall_special_user_t **special_user_hash;
extern struct messagewall_relay_auth_t **relay_auth_hash;

extern int max_clients;
extern struct messagewall_client_t *clients;

extern int max_backends;
extern struct messagewall_backend_t *backends;

extern struct sockaddr_in backendaddr;

extern int max_parts;
extern int max_depth;

extern struct messagewall_profile_t *profile_head;
extern struct messagewall_profile_t *profile_relay;
extern struct messagewall_profile_t *profile_default;
extern struct messagewall_dnsbl_global_t *dnsbl_head;
extern struct messagewall_dnsbl_global_t *dnsbl_domain_head;
extern struct messagewall_dnsdcc_global_t *dnsdcc_head;
extern struct messagewall_virus_database_t *virus_head;

extern int dnsbl_timeout;
extern int dnsdcc_timeout;
extern int rmx_timeout;

extern int max_errors;

extern int process;
extern int *auth_readmap;
extern int *auth_writemap;

extern SSL_CTX *client_ctx;
extern SSL_CTX *backend_ctx;
#endif

void handle_hup(int s);
void handle_usr1(int s);
void handle_term(int s);
int socket_nonblock(int s);
void load_timeout(struct firestring_conf_t *config, const char *name, int *timeout, int *resend);
void load_int(struct firestring_conf_t *config, const char *name, int *value);

#endif
