/*
** Copyright (C) 1995-2001 The University of Melbourne.
** This file may only be copied under the terms of the GNU Library General
** Public License - see the file COPYING.LIB in the Mercury distribution.
*/

/*
** mercury_type_info.h -
**  Definitions for accessing the type_infos, type_layouts, and
**  type_functors tables generated by the Mercury compiler.
**  Also contains definitions for accessing the Mercury `univ' type.
**
**  Changes to the structures of type_infos and pseudo_type_infos
**  may also require changes in:
**
**      compiler/polymorphism.m
**      compiler/pseudo_type_info.m
**      compiler/higher_order.m
**
**  Changes to the structures of type_ctor_infos may require changes in:
**
**      compiler/type_ctor_info.m
**      compiler/rtti.m
**      compiler/rtti_out.m
**      compiler/mlds_to_gcc.m
**          (for updating the compiler-generated RTTI
**          structures)
**
**      library/array.m
**      library/builtin.m
**      library/private_builtin.m
**      library/std_util.m
**      runtime/mercury_bootstrap.c
**          (for updating the hand-written RTTI
**          structures)
**
**  Both kinds of changes will of course also require changes to the code
**  that traverses type_infos and type_ctor_infos:
**
**      runtime/mercury_deep_copy_body.h
**      runtime/mercury_tabling.c
**      runtime/mercury_type_info.c
**      library/std_util.m
*/

#ifndef MERCURY_TYPE_INFO_H
#define MERCURY_TYPE_INFO_H

#include "mercury_std.h"    /* for `MR_STRINGIFY' and `MR_PASTEn' */
#include "mercury_types.h"  /* for `MR_Word' */

/*---------------------------------------------------------------------------*/

/*
** The version of the RTTI data structures -- useful for bootstrapping.
** MR_RTTI_VERSION sets the version number in the handwritten type_ctor_infos.
** If you write runtime code that checks this version number and
** can at least handle the previous version of the data
** structure, it makes it easier to bootstrap changes to the data
** structures used for RTTI.
**
** This number should be kept in sync with type_ctor_info_rtti_version in
** compiler/type_ctor_info.m.
*/

#define MR_RTTI_VERSION                 MR_RTTI_VERSION__CLEAN_LAYOUT
#define MR_RTTI_VERSION__INITIAL        2
#define MR_RTTI_VERSION__USEREQ         3
#define MR_RTTI_VERSION__CLEAN_LAYOUT   4

/*
** Check that the RTTI version is in a sensible range.
** The lower bound should be the lowest currently supported version
** number.  The upper bound is the current version number.
** If you increase the lower bound you should also increase the binary
** compatibility version number in runtime/mercury_grade.h (MR_GRADE_PART_0).
*/

#define MR_TYPE_CTOR_INFO_CHECK_RTTI_VERSION_RANGE(typector)    \
    assert(typector->type_ctor_version == MR_RTTI_VERSION__CLEAN_LAYOUT)

/*---------------------------------------------------------------------------*/

/* Forward declarations */

typedef const struct MR_TypeCtorInfo_Struct		*MR_TypeCtorInfo;
typedef       struct MR_TypeInfo_Almost_Struct		*MR_TypeInfo;
typedef const struct MR_PseudoTypeInfo_Almost_Struct	*MR_PseudoTypeInfo;

/*---------------------------------------------------------------------------*/

/*
** The C structures of typeinfos and pseudotypeinfos are sort of lies,
** for two reasons. First, we want one C type that can describe both first
** order, and higher order and tuple (pseudo-) typeinfos, and they have
** different structures (higher order and tuple (pseudo-) typeinfos have
** an extra word, the arity, between the type_ctor_info and the argument
** (pseudo-) typeinfos). Second, we can't rely on the C compiler having a
** mechanism for the declaration of dynamically sized vectors embedded in
** structures.
**
** Instead, the types MR_TypeInfo and MR_PseudoTypeInfo are designed as
** error-detection devices. Values of these types should be manipulated
** only through the macros defined below; the fields of the structures should
** not be accessed directly, and there should be no casts involving such
** values, except in the interface between code written in Mercury and code
** written in C, in which case casts to MR_(Pseudo)TypeInfo and back to MR_Word
** may be required. If this discipline is followed, the macros should catch
** most errors, such as passing pseudo typeinfos where typeinfos are expected.
**
** A pseudo typeinfo is the same as a typeinfo (see polymorphism.m) but
** may also store free type variables, represented as small integers:
** 1 to 512 represent universally quantified type variables
** and 513 to 1024 represent existentially quantified type variables.
** We do not use zero to represent any type variable, for two reasons.
** First, variable numbering starts at one inside the compiler. Second,
** starting at one allows us to use universally quantified type variable
** numbers to be used directly as the offset into a (non-higher-order)
** typeinfo.
**
** This scheme relies on the bit patterns of these integers corresponding
** to memory that is either inaccessible (due to the first page of virtual
** memory being invalid) or is guaranteed to contains something other than
** type_ctor_info structures (such as the code of the program).
*/

/*
** First define generic macro versions of these struct types;
** these are used in the code that the compiler generates
** for static constant typeinfos and pseudotypeinfos.
*/
#define MR_FIRST_ORDER_TYPEINFO_STRUCT(NAME, ARITY)			\
    struct NAME {							\
	MR_TypeCtorInfo     MR_ti_type_ctor_info;			\
	MR_TypeInfo         MR_ti_first_order_arg_typeinfos[ARITY];	\
    }

/* Tuple types also use the higher-order type-info structure. */
#define MR_HIGHER_ORDER_TYPEINFO_STRUCT(NAME, ARITY)			\
    struct NAME {							\
	MR_TypeCtorInfo     MR_ti_type_ctor_info;			\
	MR_Integer          MR_ti_higher_order_arity;			\
	MR_TypeInfo         MR_ti_higher_order_arg_typeinfos[ARITY];	\
    }

#define MR_FIRST_ORDER_PSEUDOTYPEINFO_STRUCT(NAME, ARITY)		\
    struct NAME {							\
	MR_TypeCtorInfo     MR_pti_type_ctor_info;			\
	MR_PseudoTypeInfo   MR_pti_first_order_arg_pseudo_typeinfos[ARITY]; \
    }

/* Tuple types also use the higher-order pseude-type-info structure. */
#define MR_HIGHER_ORDER_PSEUDOTYPEINFO_STRUCT(NAME, ARITY)		\
    struct NAME {							\
	MR_TypeCtorInfo     MR_pti_type_ctor_info;			\
	MR_Integer          MR_pti_higher_order_arity;			\
	MR_PseudoTypeInfo   MR_pti_higher_order_arg_pseudo_typeinfos[ARITY]; \
    }

/*
** Now define specific versions of these struct types,
** which are used by the MR_TypeInfo and MR_PseudoTypeInfo
** typedefs above.
*/
MR_HIGHER_ORDER_TYPEINFO_STRUCT(MR_TypeInfo_Almost_Struct,
	MR_VARIABLE_SIZED);
MR_HIGHER_ORDER_PSEUDOTYPEINFO_STRUCT(MR_PseudoTypeInfo_Almost_Struct,
	MR_VARIABLE_SIZED);

/*
** When converting a MR_PseudoTypeInfo to a MR_TypeInfo, we need the
** MR_TypeInfos corresponding to the type variables in the MR_PseudoTypeInfo.
** A MR_TypeInfoParams array serves this purpose. Because type variables
** start at one, MR_TypeInfoParams arrays also start at one.
*/
typedef MR_TypeInfo     *MR_TypeInfoParams;

/*
** MR_PSEUDOTYPEINFO_EXIST_VAR_BASE should be kept in sync with
** base_type_layout__pseudo_typeinfo_min_exist_var in base_type_layout.m.
**
** MR_PSEUDOTYPEINFO_MAX_VAR should be kept in sync with
** base_type_layout__pseudo_typeinfo_max_var in base_type_layout.m,
** and with the default value of MR_VARIABLE_SIZED in mercury_conf_params.h.
*/
#define MR_PSEUDOTYPEINFO_EXIST_VAR_BASE    512
#define MR_PSEUDOTYPEINFO_MAX_VAR           1024

/*
** Macros for accessing pseudo_type_infos.
**
** The MR_TYPE_VARIABLE_* macros should only be called if
** MR_PSEUDO_TYPEINFO_IS_VARIABLE() returns TRUE.
*/

#define MR_PSEUDO_TYPEINFO_IS_VARIABLE(T)                           \
    ( MR_CHECK_EXPR_TYPE((T), MR_PseudoTypeInfo),		    \
      (MR_Unsigned) (T) <= MR_PSEUDOTYPEINFO_MAX_VAR )

#define MR_TYPE_VARIABLE_IS_EXIST_QUANT(T)                          \
    ( MR_CHECK_EXPR_TYPE((T), MR_PseudoTypeInfo),		    \
      (MR_Word) (T) > MR_PSEUDOTYPEINFO_EXIST_VAR_BASE )
#define MR_TYPE_VARIABLE_IS_UNIV_QUANT(T)                           \
    ( MR_CHECK_EXPR_TYPE((T), MR_PseudoTypeInfo),		    \
      (MR_Word) (T) <= MR_PSEUDOTYPEINFO_EXIST_VAR_BASE )

/*
** This macro converts a pseudo_type_info to a type_info.
** It should only be called if the pseudo_type_info is ground,
** i.e. contains no type variables.
*/
#define MR_pseudo_type_info_is_ground(pseudo_type_info)             \
    ( MR_CHECK_EXPR_TYPE((pseudo_type_info), MR_PseudoTypeInfo),    \
      (MR_TypeInfo) (pseudo_type_info) )			    \

/*
** Macros for retrieving things from type_infos and pseudo_type_infos.
*/

#define MR_TYPEINFO_GET_TYPE_CTOR_INFO(type_info)                   \
    (((type_info)->MR_ti_type_ctor_info != NULL)                    \
        ? (type_info)->MR_ti_type_ctor_info                         \
        : (MR_TypeCtorInfo) (type_info))

#define MR_PSEUDO_TYPEINFO_GET_TYPE_CTOR_INFO(pseudo_type_info)     \
    (((pseudo_type_info)->MR_pti_type_ctor_info != NULL)            \
        ? (pseudo_type_info)->MR_pti_type_ctor_info                 \
        : (MR_TypeCtorInfo) (pseudo_type_info))

#define MR_TYPEINFO_GET_HIGHER_ORDER_ARITY(type_info)               \
    ((type_info)->MR_ti_higher_order_arity)

#define MR_TYPEINFO_GET_TUPLE_ARITY(type_info)                      \
    MR_TYPEINFO_GET_HIGHER_ORDER_ARITY(type_info)

#define MR_PSEUDO_TYPEINFO_GET_HIGHER_ORDER_ARITY(pseudo_type_info) \
    ((pseudo_type_info)->MR_pti_higher_order_arity)

#define MR_PSEUDO_TYPEINFO_GET_TUPLE_ARITY(pseudo_type_info)        \
    MR_PSEUDO_TYPEINFO_GET_HIGHER_ORDER_ARITY(pseudo_type_info)

#define MR_TYPEINFO_GET_FIRST_ORDER_ARG_VECTOR(type_info)           \
    ((MR_TypeInfoParams) &(type_info)->MR_ti_type_ctor_info)

#define MR_TYPEINFO_GET_HIGHER_ORDER_ARG_VECTOR(type_info)          \
    ((MR_TypeInfoParams)                                            \
        &(type_info)->MR_ti_higher_order_arity)

#define MR_TYPEINFO_GET_TUPLE_ARG_VECTOR(type_info)                 \
    MR_TYPEINFO_GET_HIGHER_ORDER_ARG_VECTOR(type_info)

/*
** Macros for creating type_infos.
*/

#define MR_first_order_type_info_size(arity)                        \
    (1 + (arity))

#define MR_higher_order_type_info_size(arity)                       \
    (2 + (arity))

#define MR_tuple_type_info_size(arity)                              \
    MR_higher_order_type_info_size(arity)

#define MR_fill_in_first_order_type_info(arena, type_ctor_info, vector) \
    do {                                                            \
        MR_TypeInfo new_ti;                                         \
        new_ti = (MR_TypeInfo) (arena);                             \
        new_ti->MR_ti_type_ctor_info = (type_ctor_info);            \
        (vector) = (MR_TypeInfoParams) &new_ti->MR_ti_type_ctor_info; \
    } while (0)

#define MR_fill_in_higher_order_type_info(arena, type_ctor_info, arity, vector)\
    do {                                                            \
        MR_TypeInfo new_ti;                                         \
        new_ti = (MR_TypeInfo) (arena);                             \
        new_ti->MR_ti_type_ctor_info = (type_ctor_info);            \
        new_ti->MR_ti_higher_order_arity = (arity);                 \
        (vector) = (MR_TypeInfoParams) &new_ti->MR_ti_higher_order_arity;\
    } while (0)

#define MR_fill_in_tuple_type_info(arena, type_ctor_info, arity, vector) \
    MR_fill_in_higher_order_type_info(arena, type_ctor_info, arity, vector)

/*
** Used to define MR_TypeCtorInfos for the builtin types in the hlc grades.
** This needs to be exported for use by the array type in the library.
*/
#ifdef MR_HIGHLEVEL_CODE

#define MR_type_ctor_info_name(MODULE, TYPE, ARITY)			      \
	MR_PASTE2(mercury__,						      \
	MR_PASTE2(MODULE,						      \
	MR_PASTE2(__,							      \
	MR_PASTE2(MODULE,						      \
	MR_PASTE2(__type_ctor_info_,					      \
	MR_PASTE2(TYPE,							      \
	MR_PASTE2(_,							      \
	          ARITY)))))))

#define MR_type_ctor_info_func_name(MODULE, TYPE, ARITY, FUNC)		      \
	MR_PASTE2(mercury__,						      \
	MR_PASTE2(MODULE,						      \
	MR_PASTE2(__,							      \
	MR_PASTE2(FUNC,							      \
	MR_PASTE2(__,							      \
	MR_PASTE2(TYPE,							      \
	MR_PASTE2(_,							      \
	MR_PASTE2(ARITY,						      \
	          _0))))))))

#define MR_special_func_type(NAME, ARITY) \
	MR_PASTE2(MR_, MR_PASTE2(NAME, MR_PASTE2(Func_, ARITY)))

#define MR_define_type_ctor_info(module, type, arity, type_rep)		      \
	const struct MR_TypeCtorInfo_Struct				      \
		MR_type_ctor_info_name(module, type, arity) =		      \
	{								      \
		arity,							      \
		(MR_Box) MR_type_ctor_info_func_name(module, type, arity,     \
				do_unify),				      \
		(MR_Box) MR_type_ctor_info_func_name(module, type, arity,     \
				do_unify),				      \
		(MR_Box) MR_type_ctor_info_func_name(module, type, arity,     \
				do_compare),				      \
		type_rep,						      \
		NULL,							      \
		NULL,							      \
		MR_STRINGIFY(module),					      \
		MR_STRINGIFY(type),					      \
		MR_RTTI_VERSION,					      \
		{ 0 },							      \
		{ 0 },							      \
		-1,							      \
		-1							      \
	}

#endif /* MR_HIGHLEVEL_CODE */

/*---------------------------------------------------------------------------*/

/*
** Definitions for handwritten code, mostly for mercury_compare_typeinfo.
*/

#define MR_COMPARE_EQUAL    0
#define MR_COMPARE_LESS     1
#define MR_COMPARE_GREATER  2

/*---------------------------------------------------------------------------*/

/*
** Definitions and macros for type_ctor_layout definition.
**
** See compiler/base_type_layout.m for more information.
**
** If we don't have enough tags, we have to encode layouts
** less densely. The make_typelayout macro does this, and
** is intended for handwritten code. Compiler generated
** code can (and does) just create two rvals instead of one.
**
** XXX This stuff is part of USEREQ type_ctor_infos and is obsolete;
** it is needed now only for bootstrapping.
*/

/*
** Conditionally define USE_TYPE_LAYOUT.
**
** All code using type_layout structures should check to see if
** USE_TYPE_LAYOUT is defined, and give a fatal error otherwise.
** USE_TYPE_LAYOUT can be explicitly turned off with NO_TYPE_LAYOUT.
**
*/
#if !defined(NO_TYPE_LAYOUT)
    #define USE_TYPE_LAYOUT
#else
    #undef USE_TYPE_LAYOUT
#endif

/*
** Declaration for structs.
*/

#define MR_DECLARE_STRUCT(T)                                        \
    extern const struct T##_struct T
#define MR_DECLARE_TYPE_CTOR_INFO_STRUCT(T)                         \
    extern const struct MR_TypeCtorInfo_Struct T

/*---------------------------------------------------------------------------*/


/*
** Offsets for dealing with `univ' types.
**
** `univ' is represented as a two word structure.
** The first word contains the address of a type_info for the type.
** The second word contains the data.
*/

#define MR_UNIV_OFFSET_FOR_TYPEINFO        0
#define MR_UNIV_OFFSET_FOR_DATA            1

#define	MR_unravel_univ(univ, typeinfo, value)                      \
    do {                                                            \
        typeinfo = (MR_TypeInfo) MR_field(MR_mktag(0), (univ),      \
                        MR_UNIV_OFFSET_FOR_TYPEINFO);               \
        value = MR_field(MR_mktag(0), (univ),                       \
                        MR_UNIV_OFFSET_FOR_DATA);                   \
    } while (0)

#define MR_define_univ_fields(univ, typeinfo, value)                \
    do {                                                            \
        MR_field(MR_mktag(0), (univ), MR_UNIV_OFFSET_FOR_TYPEINFO)  \
            = (MR_Word) (typeinfo);                                 \
        MR_field(MR_mktag(0), (univ), MR_UNIV_OFFSET_FOR_DATA)      \
            = (MR_Word) (value);                                    \
    } while (0)

/*---------------------------------------------------------------------------*/

/*
** Definitions for accessing the representation of the
** Mercury typeclass_info.
*/

#define	MR_typeclass_info_num_extra_instance_args(tci)              \
    ((MR_Integer)(*(MR_Word **)(tci))[0])
#define MR_typeclass_info_num_instance_constraints(tci)             \
    ((MR_Integer)(*(MR_Word **)(tci))[1])
#define MR_typeclass_info_num_superclasses(tci)                     \
    ((MR_Integer)(*(MR_Word **)(tci))[2])
#define MR_typeclass_info_num_type_infos(tci)                       \
    ((MR_Integer)(*(MR_Word **)(tci))[3])
#define MR_typeclass_info_num_methods(tci)                          \
    ((MR_Integer)(*(MR_Word **)(tci))[4])
#define MR_typeclass_info_class_method(tci, n)                      \
    ((MR_Code *)(*(MR_Word **)tci)[(n+4)])

/*
** The following have the same definitions. This is because
** the call to MR_typeclass_info_arg_typeclass_info must already
** have the number of unconstrained type variables for the instance
** added to it.
*/
#define MR_typeclass_info_arg_typeclass_info(tci, n)                \
    (((MR_Word *)(tci))[(n)])
#define MR_typeclass_info_unconstrained_type_info(tci, n)           \
    (((MR_Word *)(tci))[(n)])

/*
** The following have the same definitions. This is because
** the call to MR_typeclass_info_type_info must already have the
** number of superclass_infos for the class added to it.
*/

#define	MR_typeclass_info_superclass_info(tci, n)                   \
    (((MR_Word *)(tci))[MR_typeclass_info_num_extra_instance_args(tci) + (n)])
#define	MR_typeclass_info_type_info(tci, n)                         \
    (((MR_Word *)(tci))[MR_typeclass_info_num_extra_instance_args(tci) + (n)])

/*---------------------------------------------------------------------------*/

/*
** Definitions and functions for categorizing data representations.
*/

/*
** For each enumeration constant, we define it using two names;
** firstly we define the unqualified name, and then we define
** another enumeration constant whose name is the unqualified name
** prefixed with `mercury__private_builtin__' and whose value is
** the same as that of the unqualified name.
** The qualified versions are used by the MLDS->C back-end,
** which generates references to them.
*/
#define MR_DEFINE_ENUM_CONST(x)	\
	x,			\
	MR_PASTE2(mercury__private_builtin__,x) = x

/*
** MR_DataRepresentation is the representation for a particular type
** constructor.  For the cases of MR_TYPE_CTOR_REP_DU and
** MR_TYPE_CTOR_REP_DU_USEREQ, the exact representation depends on the tag
** value -- lookup the tag value in type_ctor_layout to find out this
** information.
*/

typedef enum {
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_ENUM),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_ENUM_USEREQ),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_DU),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_DU_USEREQ),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_NOTAG),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_NOTAG_USEREQ),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_EQUIV),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_EQUIV_VAR),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_INT),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_CHAR),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_FLOAT),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_STRING),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_PRED),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_UNIV),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_VOID),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_C_POINTER),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_TYPEINFO),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_TYPECLASSINFO),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_ARRAY),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_SUCCIP),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_HP),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_CURFR),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_MAXFR),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_REDOFR),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_REDOIP),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_TRAIL_PTR),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_TICKET),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_NOTAG_GROUND),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_NOTAG_GROUND_USEREQ),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_EQUIV_GROUND),
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_TUPLE),
    /*
    ** MR_TYPECTOR_REP_UNKNOWN should remain the last alternative;
    ** MR_TYPE_CTOR_STATS depends on this.
    */
    MR_DEFINE_ENUM_CONST(MR_TYPECTOR_REP_UNKNOWN)
} MR_TypeCtorRep;

/*
** This macro is intended to be used for the initialization of an array
** that converts each MR_TypeCtorRep into a string form. Therefore it
** must be kept synchronized with the definition of MR_TypeCtorRep.
*/

#define MR_CTOR_REP_NAMES                       \
    "ENUM",                                     \
    "ENUM_USEREQ",                              \
    "DU",                                       \
    "DU_USEREQ",                                \
    "NOTAG",                                    \
    "NOTAG_USEREQ",                             \
    "EQUIV",                                    \
    "EQUIV_VAR",                                \
    "INT",                                      \
    "CHAR",                                     \
    "FLOAT",                                    \
    "STRING",                                   \
    "PRED",                                     \
    "UNIV",                                     \
    "VOID",                                     \
    "C_POINTER",                                \
    "TYPEINFO",                                 \
    "TYPECLASSINFO",                            \
    "ARRAY",                                    \
    "SUCCIP",                                   \
    "HP",                                       \
    "CURFR",                                    \
    "MAXFR",                                    \
    "REDOFR",                                   \
    "REDOIP",                                   \
    "TRAIL_PTR",                                \
    "TICKET",                                   \
    "NOTAG_GROUND",                             \
    "NOTAG_GROUND_USEREQ",                      \
    "EQUIV_GROUND",                             \
    "TUPLE",                                    \
    "UNKNOWN"

#define MR_type_ctor_rep_is_basically_du(rep)               \
    (  ((rep) == MR_TYPECTOR_REP_ENUM)                      \
    || ((rep) == MR_TYPECTOR_REP_ENUM_USEREQ)               \
    || ((rep) == MR_TYPECTOR_REP_DU)                        \
    || ((rep) == MR_TYPECTOR_REP_DU_USEREQ)                 \
    || ((rep) == MR_TYPECTOR_REP_NOTAG)                     \
    || ((rep) == MR_TYPECTOR_REP_NOTAG_USEREQ)              \
    || ((rep) == MR_TYPECTOR_REP_NOTAG_GROUND)              \
    || ((rep) == MR_TYPECTOR_REP_NOTAG_GROUND_USEREQ))

/*
** Returns TRUE if the type_ctor_info is used to represent
** multiple types of different arities. The arity is stored
** as the first element of the argument type-info vector.
** This is true for higher-order types and tuple types.
*/
#define MR_type_ctor_rep_is_variable_arity(rep)             \
    (  ((rep) == MR_TYPECTOR_REP_PRED)                      \
    || ((rep) == MR_TYPECTOR_REP_TUPLE))

/*---------------------------------------------------------------------------*/

/*
** The argument number gives the offset in the cell (in a form in which
** it can be given to the MR_field macro directly) of either of the typeinfo
** itself or of the typeclassinfo containing the typeinfo. If the former,
** the offset field will be negative; otherwise, it will be an integer
** which can be given as a second argument to the MR_typeclass_info_type_info
** macro.
*/

typedef struct {
    MR_int_least16_t    MR_exist_arg_num;
    MR_int_least16_t    MR_exist_offset_in_tci;
} MR_DuExistLocn;

/*
** This structure contains information about the typeinfos of the
** existentially quantified type variables occurring in the types of some
** of the arguments of a functor in a du type.
** 
** The num_typeinfos_plain gives the number of typeinfos directly inserted
** at the start of the memory cell of the functor, while the num_tcis field
** gives the number of typeclassinfos inserted after them. The arguments
** visible to the programmer start after these two blocks, which means that
** when accessing them, one must add the sum of num_typeinfos_plain and
** num_tcis to the visible argument number in order to arrive at an offset
** in the cell.
**
** It is possible for a typeclassinfo to contain more than one type variable.
** The num_typeinfos_in_tci field contains the total number of typeinfos stored
** inside the typeclassinfos of the cell.
**
** The typeinfo_locns field points to an array of MR_ExistTypeInfoLocns.
** This array has num_typeinfos_plain + num_typeinfos_in_tci elements,
** each one of which describes the location (directly in the cell or indirectly
** inside a typeclassinfo) of the typeinfo for an existentially quantified
** type variable. The typeinfo for type variable N will be at the offset
** N - MR_PSEUDOTYPEINFO_EXIST_VAR_BASE - 1. (The one is subtracted to convert
** from type var numbering, which starts at 1, to array offset numbering).
*/

typedef struct {
    MR_int_least16_t        MR_exist_typeinfos_plain;
    MR_int_least16_t        MR_exist_typeinfos_in_tci;
    MR_int_least16_t        MR_exist_tcis;
    const MR_DuExistLocn    *MR_exist_typeinfo_locns;
} MR_DuExistInfo;

/*
** This type describes the implementation of a function symbol
** from a (proper) discriminated union type, whether it has standard
** or user-defined-equality.
**
** Functor descriptors are reachable from both the layout and functor tables.
** They all the information one may need about the function symbol, even
** though some of this information may be redundant along some access paths.
** 
** The fields that you are likely to be interested in when you arrive at the
** functor descriptor through the functor table are clustered at the front,
** the fields that you are likely to be interested in when you arrive at the
** functor descriptor through the layout table are clustered at the back.
** This is an attempt to optimize cache effectiveness.
**
** The primary and secondary fields give the corresponding tag values, and
** the sectag_locn field gives the location of the secondary tag.
**
** The ordinal field gives the position of the function symbol in the
** list of function symbols of the type; one function symbol compares
** as less than another iff its ordinal number is smaller.
** 
** The orig_arity field records the visible arity of the functor, without
** the typeinfos and/or typeclass_infos added for existentially typed
** arguments.
**
** The arg_types field points to an array of pseudo typeinfos, one for each
** visible argument.
**
** The arg_type_contains_var field contains a bit vector which has one bit
** for each of the first N (currently N=15) arguments, and one bit shared
** between all the other arguments. One of the first N bits is set iff
** the type of the corresponding argument contains a type variable, while
** the last bit is set iff the types of any of the remaining arguments
** contains a type variable.
** This field is meant to be used only via the MR_arg_type_may_contain_var
** and MR_any_arg_type_may_contain_var macros below. In the absence of
** compiler-recorded information, these macros return conservative answers
** for any argument whose type is not represented in this bit vector.
**
** The arg_name field points to an array of field names, one for each
** visible argument. If no argument has a name, this field will be NULL.
**
** If the functor has any arguments whose types include existentially
** quantified type variables, the exist_info field will point to information
** about those type variable; otherwise, the exist_info field will be NULL.
*/

typedef enum {
    MR_DEFINE_ENUM_CONST(MR_SECTAG_NONE),
    MR_DEFINE_ENUM_CONST(MR_SECTAG_LOCAL),
    MR_DEFINE_ENUM_CONST(MR_SECTAG_REMOTE)
} MR_Sectag_Locn;

typedef struct {
    MR_ConstString          MR_du_functor_name;
    MR_int_least16_t        MR_du_functor_orig_arity;
    MR_int_least16_t        MR_du_functor_arg_type_contains_var;
    MR_Sectag_Locn          MR_du_functor_sectag_locn;
    MR_int_least8_t         MR_du_functor_primary;
    MR_int_least32_t        MR_du_functor_secondary;
    MR_int_least32_t        MR_du_functor_ordinal;
    const MR_PseudoTypeInfo *MR_du_functor_arg_types;
    const MR_ConstString    *MR_du_functor_arg_names;
    const MR_DuExistInfo    *MR_du_functor_exist_info;
} MR_DuFunctorDesc;

/*
** This macro represents the number of bits in the
** MR_du_functor_arg_type_contains_var field of a MR_DuFunctorDesc.
** It should be kept in sync with contains_var_bit_vector_size
** in base_type_layout.m.
*/

#define MR_ARG_TYPE_CONTAINS_VAR_BIT_VECTOR_SIZE    16

#define MR_arg_has_own_contain_var_bit(arg_num)                         \
    ((arg_num) < MR_ARG_TYPE_CONTAINS_VAR_BIT_VECTOR_SIZE - 1)

#define MR_initial_arg_type_may_contain_var(functor_desc, arg_num)      \
    (((functor_desc)->MR_du_functor_arg_type_contains_var &             \
        (1 << (arg_num))) != 0)

#define MR_later_arg_type_may_contain_var(functor_desc)                 \
    (((functor_desc)->MR_du_functor_arg_type_contains_var &             \
        (1 << (MR_ARG_TYPE_CONTAINS_VAR_BIT_VECTOR_SIZE - 1))) != 0)

#define MR_arg_type_may_contain_var(functor_desc, arg_num)              \
    ( MR_arg_has_own_contain_var_bit(arg_num)                           \
    ? MR_initial_arg_type_may_contain_var((functor_desc), (arg_num))    \
    : MR_later_arg_type_may_contain_var(functor_desc))

#define MR_some_arg_type_contains_var(functor_desc)                     \
    ((functor_desc)->MR_du_functor_arg_type_contains_var > 0)

/*---------------------------------------------------------------------------*/

typedef struct {
    MR_ConstString      MR_enum_functor_name;
    MR_int_least32_t    MR_enum_functor_ordinal;
} MR_EnumFunctorDesc;

/*---------------------------------------------------------------------------*/

typedef struct {
    MR_ConstString      MR_notag_functor_name;
    MR_PseudoTypeInfo   MR_notag_functor_arg_type;
    MR_ConstString      MR_notag_functor_arg_name;
} MR_NotagFunctorDesc;

/*---------------------------------------------------------------------------*/

/*
** This type describes the function symbols that share the same primary tag.
** The sharers field gives their number, and thus also the size
** of the array of pointers to functor descriptors pointed to by the
** alternatives field.
**
** The intention is that if you have a word in a DU type that you want to
** interpret, you compute its primary tag and find its MR_DuPtagLayout.
** You then look at the locn field. If it is MR_SECTAG_NONE, you index
** the alternatives field with zero; if it is MR_SECTAG_{LOCAL,REMOTE}, you
** compute the secondary tag and index the alternatives field with that.
**
** A value of type MR_DuTypeLayout points to an array of MR_DuPtagLayout
** structures. The element at index k gives information about primary tag
** value k. The size of the array is recorded in the num_ptags field of the
** type_ctor_info.
*/

typedef struct {
    MR_int_least32_t        MR_sectag_sharers;
    MR_Sectag_Locn          MR_sectag_locn;
    const MR_DuFunctorDesc * const * MR_sectag_alternatives;
} MR_DuPtagLayout;

typedef MR_DuPtagLayout     *MR_DuTypeLayout;

/*---------------------------------------------------------------------------*/

/*
** This type describes the function symbols in an enum type.
**
** An MR_EnumLayout points to an array of pointers to functor descriptors.
** There is one pointer for each function symbol, and thus the size of
** the array is given by the num_functors field of the type_ctor_info.
** The array is ordered on the integer value by which the functor is
** represented.
**
** The intention is that if you have a word in an enum type that you want to
** interpret, you index into the array with the word.
*/

typedef MR_EnumFunctorDesc  **MR_EnumTypeLayout;

/*---------------------------------------------------------------------------*/

/*
** This type describes the single function symbol in a notag type.
**
** An MR_NotagLayout points to the one functor descriptor of the type.
**
** The intention is that if you have a word in a notag type that you want to
** interpret, you look at the given functor descriptor.
*/

typedef MR_NotagFunctorDesc *MR_NotagTypeLayout;

/*---------------------------------------------------------------------------*/

/*
** This type describes the identity of the type that an equivalence type
** is equivalent to, and hence its layout.
**
** An MR_NotagLayout gives the pseudo typeinfo of the type that this type
** is equivalent to.
**
** The intention is that if you have a word in an equivalence type that you
** want to interpret, you expand the pseudo typeinfo into a real typeinfo,
** and then use that to interpret the word.
*/

typedef MR_PseudoTypeInfo   MR_EquivLayout;

/*---------------------------------------------------------------------------*/

/*
** Some types are defined differently for the MLDS back-end.
*/

#ifdef MR_HIGHLEVEL_CODE
  /*
  ** XXX This should be `MR_Box', but MR_Box is not visible here
  ** (due to a cyclic dependency problem), so we use `void *' instead.
  */
  typedef	void        *MR_ProcAddr;
#else
  typedef	MR_Code     *MR_ProcAddr;
#endif

/*---------------------------------------------------------------------------*/

/*
** This type describes the layout in any kind of discriminated union
** type: du, enum and notag. In an equivalence type, it gives the identity
** of the equivalent-to type.
** 
** The layout_init alternative is used only for static initializers,
** because ANSI C89 does not allow you to say which member of a union
** you are initializing, and instead forces you to initialize the first member.
** When we can rely on C99 compilers, layout_init should no longer be needed.
*/

typedef union {
    void                *layout_init;
    MR_DuTypeLayout     layout_du;
    MR_EnumTypeLayout   layout_enum;
    MR_NotagTypeLayout  layout_notag;
    MR_EquivLayout      layout_equiv;
} MR_TypeLayout;

/*---------------------------------------------------------------------------*/

/*
** This type describes the function symbols in any kind of discriminated union
** type: du, enum and notag.
**
** The pointer points to an array of pointers to functor descriptors.
** There is one pointer for each function symbol, and thus the size of
** the array is given by the num_alternatives field of the type_ctor_info.
** The array is ordered on the name of the function symbol, and then on arity.
**
** The intention is that if you have a function symbol you want to represent,
** you can do binary search on the array for the symbol name and arity.
** 
** The functors_init alternative is used only for static initializers;
** see the comment for MR_TypeLayout above.
*/

typedef union {
    void                *functors_init;
    MR_DuFunctorDesc    **functors_du;
    MR_EnumFunctorDesc  **functors_enum;
    MR_NotagFunctorDesc *functors_notag;
} MR_TypeFunctors;

/*---------------------------------------------------------------------------*/

    /*
    ** Structs defining the structure of type_ctor_infos.
    ** A type_ctor_info describes the structure of a particular
    ** type constructor.  One of these is generated for every
    ** `:- type' declaration.
    **
    ** The unify_pred field will soon migrate to the slot now occupied by
    ** the new_unify_pred field. In the near future, the two slots will
    ** contain the same data.
    */

struct MR_TypeCtorInfo_Struct {
    MR_Integer          arity;
    MR_ProcAddr         unify_pred;
    MR_ProcAddr         new_unify_pred;
    MR_ProcAddr         compare_pred;
    MR_TypeCtorRep      type_ctor_rep;
    MR_ProcAddr         unused1;	/* spare */
    MR_ProcAddr         unused2;	/* spare */
    MR_ConstString      type_ctor_module_name;
    MR_ConstString      type_ctor_name;
    MR_Integer          type_ctor_version;
    MR_TypeFunctors     type_functors;
    MR_TypeLayout       type_layout;
    MR_int_least32_t    type_ctor_num_functors;
    MR_int_least8_t     type_ctor_num_ptags;    /* if DU */

/*
** The following fields will be added later, once we can exploit them:
**  union MR_TableNode_Union    **type_std_table;
**  MR_ProcAddr         prettyprinter;
*/
};

/*---------------------------------------------------------------------------*/

/*
** Macros to help the runtime and the library create type_ctor_info
** structures for builtin and special types.
*/

#define MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_FULL(m, cm, n, a, cr, u, c)    \
    MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_FULL_A(u, c)                       \
    MR_PASTE6(mercury_data_, cm, __type_ctor_info_, n, _, a) = {        \
    MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_FULL_B(m, n, a, cr, u, c)

    /* MSVC CPP doesn't like having an empty CM field. */
#define MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_NOCM(m, n, a, cr, u, c)        \
    MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_FULL_A(u, c)                       \
    MR_PASTE5(mercury_data_, __type_ctor_info_, n, _, a) = {            \
    MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_FULL_B(m, n, a, cr, u, c)

#define MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_FULL_A(u, c)                   \
    MR_declare_entry(u);                                                \
    MR_declare_entry(c);                                                \
    MR_STATIC_CODE_CONST struct MR_TypeCtorInfo_Struct                  \

#define MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_FULL_B(m, n, a, cr, u, c)      \
        a,                                                              \
        MR_MAYBE_STATIC_CODE(MR_ENTRY(u)),                              \
        MR_MAYBE_STATIC_CODE(MR_ENTRY(u)),                              \
        MR_MAYBE_STATIC_CODE(MR_ENTRY(c)),                              \
        cr,                                                             \
        NULL,                                                           \
        NULL,                                                           \
        MR_string_const(MR_STRINGIFY(m), sizeof(MR_STRINGIFY(m))-1),    \
        MR_string_const(MR_STRINGIFY(n), sizeof(MR_STRINGIFY(n))-1),    \
        MR_RTTI_VERSION,                                                \
        { 0 },                                                          \
        { 0 },                                                          \
        -1,                                                             \
        -1                                                              \
    }

#define MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_PRED(m, n, a, cr, u, c)        \
    MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_FULL(m, m, n, a, cr, u, c)

#define MR_DEFINE_BUILTIN_TYPE_CTOR_INFO(m, n, a, cr)           \
    MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_FULL(m, m, n, a, cr,       \
        MR_PASTE7(mercury____Unify___, m, __, n, _, a, _0),     \
        MR_PASTE7(mercury____Compare___, m, __, n, _, a, _0))

#define MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_UNUSED(n, a, cr)       \
    MR_DEFINE_BUILTIN_TYPE_CTOR_INFO_NOCM(builtin, n, a, cr,	\
        mercury__unused_0_0,                                    \
        mercury__unused_0_0)

/*---------------------------------------------------------------------------*/

/*
** Code for dealing with the static code addresses stored in
** type_ctor_infos.
*/

/*
** Definitions for initialization of type_ctor_infos. If
** MR_STATIC_CODE_ADDRESSES are not available, we need to initialize
** the special predicates in the type_ctor_infos.
*/

/*
** Macros are provided here to initialize type_ctor_infos, both for
** builtin types (such as in library/builtin.m) and user
** defined C types (like library/array.m). Also, the automatically
** generated code uses these initializers.
**
** Examples of use:
**
** MR_INIT_BUILTIN_TYPE_CTOR_INFO(
**  mercury_data__type_ctor_info_string_0, _string_);
**
** note we use _string_ to avoid the redefinition of string via #define
**
** MR_INIT_TYPE_CTOR_INFO(
**  mercury_data_group__type_ctor_info_group_1, group__group_1_0);
**
** MR_INIT_TYPE_CTOR_INFO_WITH_PRED(
**  mercury_date__type_ctor_info_void_0, mercury__unused_0_0);
**
** This will initialize a type_ctor_info with a single code address.
*/

#ifndef MR_STATIC_CODE_ADDRESSES

  #define MR_MAYBE_STATIC_CODE(X)   NULL

  #define MR_STATIC_CODE_CONST

  #define   MR_INIT_BUILTIN_TYPE_CTOR_INFO(B, T)            		\
  do {                                                      		\
    (B).unify_pred = MR_ENTRY(mercury__builtin_unify##T##2_0);      \
    (B).new_unify_pred = MR_ENTRY(mercury__builtin_unify##T##2_0);  \
    (B).compare_pred = MR_ENTRY(mercury__builtin_compare##T##3_0);  \
  } while (0)

  #define   MR_INIT_TYPE_CTOR_INFO_WITH_PRED(B, P)                  \
  do {                                                              \
    (B).unify_pred = MR_ENTRY(P);                                   \
    (B).new_unify_pred = MR_ENTRY(P);                               \
    (B).compare_pred = MR_ENTRY(P);                                 \
  } while (0)

  #define   MR_INIT_TYPE_CTOR_INFO(B, T)                            \
  do {                                                              \
    (B).unify_pred = MR_ENTRY(mercury____##Unify##___##T);          \
    (B).new_unify_pred = MR_ENTRY(mercury____##Unify##___##T);      \
    (B).compare_pred = MR_ENTRY(mercury____##Compare##___##T);      \
  } while (0)

#else   /* MR_STATIC_CODE_ADDRESSES */

  #define MR_MAYBE_STATIC_CODE(X)   (X)

  #define MR_STATIC_CODE_CONST const

  #define MR_INIT_BUILTIN_TYPE_CTOR_INFO(B, T)              \
    do { } while (0)

  #define MR_INIT_TYPE_CTOR_INFO_WITH_PRED(B, P)            \
    do { } while (0)

  #define MR_INIT_TYPE_CTOR_INFO(B, T)                      \
    do { } while (0)

#endif /* MR_STATIC_CODE_ADDRESSES */

/*---------------------------------------------------------------------------*/

#ifdef MR_HIGHLEVEL_CODE
  extern const struct MR_TypeCtorInfo_Struct 
        mercury__builtin__builtin__type_ctor_info_pred_0,
        mercury__builtin__builtin__type_ctor_info_func_0,
        mercury__builtin__builtin__type_ctor_info_tuple_0;
  #define MR_TYPE_CTOR_INFO_HO_PRED                                     \
        (&mercury__builtin__builtin__type_ctor_info_pred_0)
  #define MR_TYPE_CTOR_INFO_HO_FUNC                                     \
        (&mercury__builtin__builtin__type_ctor_info_func_0)
  #define MR_TYPE_CTOR_INFO_TUPLE                                       \
        (&mercury__builtin__builtin__type_ctor_info_tuple_0)
#else
  MR_DECLARE_TYPE_CTOR_INFO_STRUCT(mercury_data___type_ctor_info_pred_0);
  MR_DECLARE_TYPE_CTOR_INFO_STRUCT(mercury_data___type_ctor_info_func_0);
  MR_DECLARE_TYPE_CTOR_INFO_STRUCT(mercury_data___type_ctor_info_tuple_0);
  #define MR_TYPE_CTOR_INFO_HO_PRED                                     \
        ((MR_TypeCtorInfo) &mercury_data___type_ctor_info_pred_0)
  #define MR_TYPE_CTOR_INFO_HO_FUNC                                     \
        ((MR_TypeCtorInfo) &mercury_data___type_ctor_info_func_0)
  #define MR_TYPE_CTOR_INFO_TUPLE                                       \
        ((MR_TypeCtorInfo) &mercury_data___type_ctor_info_tuple_0)
#endif

#define MR_TYPE_CTOR_INFO_IS_HO_PRED(T)                                 \
        (T == MR_TYPE_CTOR_INFO_HO_PRED)
#define MR_TYPE_CTOR_INFO_IS_HO_FUNC(T)                                 \
        (T == MR_TYPE_CTOR_INFO_HO_FUNC)
#define MR_TYPE_CTOR_INFO_IS_HO(T)                                      \
        (MR_TYPE_CTOR_INFO_IS_HO_FUNC(T) || MR_TYPE_CTOR_INFO_IS_HO_PRED(T))
#define MR_TYPE_CTOR_INFO_IS_TUPLE(T)                                   \
        (T == MR_TYPE_CTOR_INFO_TUPLE)

/*---------------------------------------------------------------------------*/

/*
** MR_compare_type_info returns MR_COMPARE_GREATER, MR_COMPARE_EQUAL, or
** MR_COMPARE_LESS, depending on whether t1 is greater than , equal to,
** or less than t2.
*/

extern  int     MR_compare_type_info(MR_TypeInfo t1, MR_TypeInfo t2);

/*
** MR_collapse_equivalences expands out all the top-level equivalences in
** the argument typeinfo. It guarantees that the returned typeinfo's
** type_ctor_info will not have a MR_TYPE_CTOR_REP_EQUIV* representation.
** However, since it only works on the top level type constructor,
** this is not guaranteed for the typeinfos of the type constructor's
** arguments.
*/

extern  MR_TypeInfo MR_collapse_equivalences(MR_TypeInfo type_info);

/* 
** MR_create_type and MR_make_type_info both turn a pseudo typeinfo into
** a typeinfo, looking up the typeinfos associated with the type variables
** in the pseudointo typeinfo in the supplied vector of type parameters.
**
** The two functions differ in how they allocate memory. MR_create_type_info
** allocates memory for a new type_info on the Mercury heap. Since this
** may modify MR_hp, you need to wrap MR_save_transient_hp() and
** MR_restore_transient_hp() around calls to MR_create_type_info.
** MR_make_type_info allocates memory using MR_GC_malloc, and inserts
** the address of the cells allocated into the list of allocations
** represented by its last argument; it is the caller's responsibility
** to call MR_deallocate() on the list after they have finished using
** the returned typeinfo.
**
** MR_create_type_info and MR_make_type_info both assume that all type
** variables inside the given pseudo typeinfo are universally quantified.
** Their maybe_existq variants do not make this assumption; they also work
** if the pseudo typeinfo contains existentially quantified arguments.
** This can happen only when the pseudo typeinfo describes (part of) the type
** of an argument of a function symbol from a MR_TYPE_CTOR_REP_DU* type.
** These functions also take two extra arguments: the address of the cell,
** which (directly or indirectly) contains the typeinfos of the existentially
** quantified type variables, and the descriptor of the function symbol,
** which describes how those typeinfos can be found in the cell. The cell
** address is supposed to point past the remote secondary tag, if any;
** it should point to the first argument, whether it is a user visible argument
** or a typeinfo/typeclass_info inserted into the cell by the compiler.
**
** All these functions guarantee that if the pseudo typeinfo argument refers
** to a type constructor with no arguments, then they return a one-cell
** typeinfo, and do not require any memory allocation.
**
** These functions should only be called if the pseudo typeinfo may have
** some type variables in it. Otherwise, the pseudo typeinfo should be
** cast to a typeinfo directly, using the macro MR_pseudo_type_info_is_ground.
*/

extern  MR_TypeInfo MR_create_type_info(
                const MR_TypeInfoParams type_info_params,
                const MR_PseudoTypeInfo pseudo_type_info);
extern  MR_TypeInfo MR_create_type_info_maybe_existq(
                const MR_TypeInfoParams type_info_params,
                const MR_PseudoTypeInfo pseudo_type_info,
                const MR_Word *data_value,
                const MR_DuFunctorDesc *functor_descriptor);

struct MR_MemoryCellNode {
    void                        *data;
    struct MR_MemoryCellNode    *next;
};

typedef struct MR_MemoryCellNode *MR_MemoryList;

extern  MR_TypeInfo MR_make_type_info(
                        const MR_TypeInfoParams type_info_params,
                        const MR_PseudoTypeInfo pseudo_type_info,
                        MR_MemoryList *allocated);
extern  MR_TypeInfo MR_make_type_info_maybe_existq(
                        const MR_TypeInfoParams type_info_params,
                        const MR_PseudoTypeInfo pseudo_type_info,
                        const MR_Word *data_value,
                        const MR_DuFunctorDesc *functor_descriptor,
                        MR_MemoryList *allocated);
extern  void        MR_deallocate(MR_MemoryList allocated_memory_cells);

/*---------------------------------------------------------------------------*/

#endif /* not MERCURY_TYPE_INFO_H */
