/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include "epg.hh"
#include "application.hh"
#include "mutex_lock.hh"
#include <string.h>

EpgEvent::EpgEvent()
{
	event_id = 0;
	duration = -1;
	start_time = -1;
	channel = NULL;
}

EpgEvent::EpgEvent(const Channel* channel, const Event& event)
{
	event_id		= event.event_id;
	duration		= event.duration;
	start_time		= event.start_time;
	start_time_text = start_time_to_string(start_time);
	duration_text	= start_time_to_string(duration);
	title			= event.title;
	description		= event.description;
	this->channel	= channel;

	fix_data();
}

EpgEvent::EpgEvent(const EpgEvent& event)
{
	event_id		= event.event_id;
	start_time		= event.start_time;
	start_time_text	= event.start_time_text;
	duration		= event.duration;
	duration_text	= event.duration_text;
	title			= event.title;
	description		= event.description;
	channel			= event.channel;
}

EpgEvent::EpgEvent(const XmlNode& node)
{
	load(node);
}

void EpgEvent::load(const XmlNode& node)
{
	event_id		= node.get_int_attribute_value("event_id");
	start_time		= (glong)node.get_int_attribute_value("start_time");
	start_time_text	= start_time_to_string(start_time);
	duration		= node.get_int_attribute_value("duration");
	duration_text	= duration_to_string(duration);
	title			= node.get_attribute_value("title");
	description		= node.get_attribute_value("description");

	XmlNode channel_node(node.get_parent());
	String channel_name = channel_node.get_attribute_value("name");
		
	channel = &Application::get_current().get_channel(channel_name);
	
	fix_data();
}

void EpgEvent::fix_data()
{
	if (title.is_empty())
	{
		title = _("No title available");
	}

	if (description.is_empty())
	{
		description = _("No description available");
	}
}

String EpgEvent::start_time_to_string(int start_time)
{
	String result;
	if (start_time == -1)
	{
		result = UNKNOWN_TEXT;
	}
	else
	{
		DateTime s(start_time, true);
		result = String::format("%d/%d %.2d:%.2d",
			s.get_day_of_month(),
			s.get_month(),
			s.get_hour(),
			s.get_minute());
	}
	return result;
}
	
String EpgEvent::duration_to_string(int duration)
{
	String result;

	if (duration == -1)
	{
		result =  UNKNOWN_TEXT;
	}
	else
	{
		int duration_minutes = duration / 60;
		if (duration_minutes == 0)
		{
			result = "0 m";
		}
		else
		{
			int hours = duration_minutes / 60;
			int minutes = duration_minutes % 60;
			
			if (hours > 0)
			{
				result = Integer::to_string(hours);
				result += "h";
			}
			
			if (hours > 0 && minutes > 0)
			{
				result += " ";
			}
			
			if (minutes > 0)
			{
				result += Integer::to_string(minutes);
				result += "m";
			}		
		}
	}
	
	return result;
}

const Channel& EpgEvent::get_channel() const
{
	if (channel == NULL)
	{
		throw Exception(_("Failed to get channel from EPG event because it was NULL"));
	}
	return *channel;
}

gboolean EpgEvent::operator==(const EpgEvent& e)
{
	return e.channel->name == channel->name && e.event_id == event_id;
}

xmlNodePtr Epg::get_channel_node(const Channel& channel)
{
	return get_channel_node(channel.name);
}

xmlNodePtr Epg::get_channel_node(const String& channel_name)
{
	xmlNodePtr result = NULL;
	
	XPath xpath(document);
	String expression = String::format(
		"/epg/channel[@name=\"%s\"]",
		channel_name.c_str());
	XPathResult channel_result(xpath.evaluate_expression(expression));
	if (channel_result.get_count() > 1)
	{
		throw Exception(_("Failed to get channel because there was more than one result"));
	}
	
	if (channel_result.get_count() == 0)
	{
		Log::write(_("Creating EPG channel '%s'"), channel_name.c_str());
		XmlNode root_node = document.get_root_node();
		XmlNode channel_node = root_node.create_child_node("channel");
		channel_node.set_attribute("name", channel_name);
		result = channel_node.get_node();
	}
	else
	{
		result = channel_result.get_result(0);
	}
	
	return result;
}

void Epg::load(const String& path)
{
	document.load(path);
	
	if (document.get_root_node() == NULL)
	{
		document.create_root_node("epg");
	}
}
	
int Epg::add_event(const EpgEvent& event)
{
	int result = 0;

	const Channel& channel = event.get_channel();
	int event_id = event.get_event_id();
	
	XPath xpath(document);

	String expression = String::format(
		"/epg/channel[@name=\"%s\"]/event[@event_id=\"%d\"]",
		channel.name.c_str(), event_id);

	XPathResult events(xpath.evaluate_expression(expression));
	if (events.get_count() > 1)
	{
		throw Exception(_("Failed to set event"));
	}
	
	if (events.get_count() == 0)
	{
		glong now = DateTime::now_utc();
		glong end_time = event.get_start_time() + event.get_duration();
		
		if (end_time >= now)
		{				
			result++;
			
			XmlNode channel_node = get_channel_node(channel);

			Log::write("Creating event node for '%d'", event_id);
			
			XmlNode event_node = channel_node.create_child_node("event");
			event_node.set_attribute("event_id",	event_id);
			event_node.set_attribute("start_time",	event.get_start_time());
			event_node.set_attribute("duration",	event.get_duration());
			event_node.set_attribute("title",		event.get_title());
			event_node.set_attribute("description",	event.get_description());
		}
	}
	
	return result;
}
	
int Epg::prune()
{
	int deleted = 0;
	XPath xpath(document);
	XPathResult events = xpath.evaluate_expression("/epg/channel/event");

	glong now = DateTime::now_utc();
	int count = events.get_count();
	for (int i = 0; i < count; i++)
	{
		XmlNode node_current = events.get_result(i);
		glong current_end_time = 0;
		
		try
		{
			EpgEvent event(node_current);
			current_end_time = event.get_start_time() + event.get_duration();
		}
		catch(...)
		{
		}

		if (current_end_time < now)
		{
			node_current.unlink();
			deleted++;
		}
	}

	return deleted;
}

xmlNodePtr Epg::get_current_event(const Channel& channel)
{
	return get_event_at(channel, DateTime::now_utc());
}
	
xmlNodePtr Epg::get_event_at(const Channel& channel, glong when)
{
	XPathResult events = get_events(channel);

	int count = events.get_count();
	for (int i = 0; i < count; i++)
	{
		XmlNode node_current = events.get_result(i);
		EpgEvent event(node_current);
		glong current_start_time = event.get_start_time();
		glong current_end_time = current_start_time + event.get_duration();

		if (current_start_time <= when && current_end_time >= when)
		{
			return node_current.get_node();
		}
	}
	
	return NULL;
}

void Epg::save()
{
	document.save();
}

xmlNodePtr Epg::get_event(const Channel& channel, int event_id)
{
	return get_event(channel.name, event_id);
}

xmlNodePtr Epg::get_event(const String& channel_name, int event_id)
{
	XPath xpath(document);
	String expression = String::format(
		"/epg/channel[@name=\"%s\"]/event[@event_id=\"%d\"]",
		channel_name.c_str(), event_id);
	return xpath.evaluate_expression_single(expression);
}

xmlXPathObjectPtr Epg::get_events(const Channel& channel)
{
	XPath xpath(document);
	String expression = String::format(
		"/epg/channel[@name=\"%s\"]/event",
		channel.name.c_str());
	return xpath.evaluate_expression(expression);
}

GList* Epg::search(const String& query)
{
	XPath xpath(document);
	XPathResult all_events = xpath.evaluate_expression("/epg/channel/event");
	GList* list = NULL;
	gsize count = all_events.get_count();
	
	for (guint index = 0; index < count; index++)
	{
		xmlNodePtr current = all_events.get_result(index);
		EpgEvent event(current);
		String ltitle = event.get_title().lowercase();
		String lquery = query.lowercase();
		const gchar* value = ltitle.find(lquery.c_str());
		if (value != NULL)
		{
			list = g_list_insert_sorted(list,
				current,
				(GCompareFunc)EpgEvent::compare_events);
		}
	}
	
	return list;
}
