/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include "configuration.hh"
#include <gnome.h>

void Configuration::set_int_value (const String& key, int value)
{
	set_string_value(key, Integer::to_string(value));
}

void Configuration::set_boolean_value (const String& key, gboolean value)
{
	set_string_value(key, value ? "true" : "false");
}

void Configuration::set_string_value(const String& key, const String& value)
{
	set_dirty();

	XmlNode node(general_node);
	XPath xpath(node);
	String expression = String::format("/application/general/add[@key=\"%s\"]", key.c_str());
	XPathResult result = xpath.evaluate_expression(expression);
	int count = result.get_count();
	if (count == 0)
	{
		XmlNode new_node = node.create_child_node("add");
		new_node.set_attribute("key", key);
		new_node.set_attribute("value", value);				
	}
	else if (count == 1)
	{
		XmlNode new_node = result.get_result(0);
		new_node.set_attribute("key", key);
		new_node.set_attribute("value", value);				
	}
	else
	{
		throw Exception(_("Failed to get a single result for key '%s'"), key.c_str());
	}
}

gboolean Configuration::has_value(const String& key)
{
	XPath xpath(document);
	String expression = String::format("/application/general/add[@key=\"%s\"]", key.c_str());
	XPathResult result = xpath.evaluate_expression(expression);
	int count = result.get_count();

	if (count > 1)
	{
		throw Exception(_("There are 2 entries for '%s' in the configuration file"), key.c_str());
	}
	
	return (count == 1);
}

String Configuration::get_string_value(const String& key)
{
	XPath xpath(document);
	String expression = String::format("/application/general/add[@key=\"%s\"]", key.c_str());
	XmlNode value_node = xpath.evaluate_expression_single(expression);
	String value = value_node.get_attribute_value("value");
	return value;
}

guint Configuration::get_int_value(const String& key)
{
	String value = get_string_value(key);
	return atoi(value.c_str());
}

gboolean Configuration::get_boolean_value(const String& key)
{
	gboolean result = false;
	String value = get_string_value(key);
	if (value == "1" || value == "true")
	{
		result = true;
	}
	else if (value == "0" || value == "false")
	{
		result = false;
	}
	else
	{
		throw Exception(_("'%s' is not a valid boolean value for key '%s'"), value.c_str(), key.c_str());
	}
	return result;
}

void Configuration::set_default_string_value(const String& key, const String& default_value)
{
	if (!has_value(key))
	{
		set_string_value(key, default_value);
	}
}

void Configuration::set_default_int_value(const String& key, int default_value)
{
	if (!has_value(key))
	{
		set_int_value(key, default_value);
	}
}

void Configuration::set_default_boolean_value(const String& key, gboolean default_value)
{
	if (!has_value(key))
	{
		set_boolean_value(key, default_value);
	}
}

Configuration::Configuration()
{
	is_dirty = false;
	general_node = NULL;
}

Configuration::~Configuration()
{	
	if (is_dirty)
	{
		save();
	}
}

XmlDocument& Configuration::get_document()
{
	return document;
}

void Configuration::load(const String& configuration_file)
{
	is_dirty = false;
	document.load(configuration_file);
	xmlNodePtr root_node = NULL;

	XPath xpath(document);
	XPathResult result_root(xpath.evaluate_expression("/application"));
	if (result_root.get_count() == 0)
	{
		root_node = document.create_root_node("application");
	}
	else
	{
		root_node = result_root.get_result(0);
	}
	
	XPathResult result(xpath.evaluate_expression("/application/general"));
	if (result.get_count() == 0)
	{
		XmlNode parent(root_node);
		general_node = parent.create_child_node("general");
	}
	else
	{
		general_node = result.get_result(0);
	}	
}

void Configuration::save()
{
	document.save();
	is_dirty = false;
}

void Configuration::set_dirty()
{
	is_dirty = true;
}
