/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#ifndef __APPLICATION_H__
#define __APPLICATION_H__

#include "main_window.hh"
#include "dvb_tuner.hh"
#include "configuration.hh"
#include "epg.hh"
#include "glade.hh"
#include "channel_manager.hh"
#include "io.hh"
#include "network.hh"
#include "recording_manager.hh"
#include "status_icon.hh"
#include "streamer.hh"

#define APPLICATION_NAME	"Me TV"
#define CHECK_MAIN_THREAD	Application::get_current().check_thread(__PRETTY_FUNCTION__);

class Application
{
private:
	IO::Channel*		log_file;
	Configuration		configuration;
	Glade				glade;
	ChannelManager		channel_manager;
	Epg					epg;
	DvbTuner*			tuner;
	MainWindow*			main_window;
	StatusIcon*			status_icon;
	GtkWidget*			dialog_about;
	GThread*			epg_thread;
	GStaticRecMutex		stream_mutex;
	GStaticRecMutex		configuration_mutex;
	GStaticRecMutex		error_message_mutex;
	GStaticRecMutex		epg_mutex;
	gboolean			terminate_threads;
	guint				timeout_id;
	String				recording_file_name;
	RecordingManager*	recording_manager;	
	GSList*				epg_events;
	GSList*				error_messages;
	GSList*				auto_surf_iterator;
	GThread*			main_thread;
	String				demux_path;
	String				dvr_path;
	String				exe_path;
	String				exe_directory;
	String				video_channel_name;
	guint				epg_event_list_size;
	gboolean			application_quit;
	xmlNodePtr			scheduled_recording_node;
	time_t				last_recording_check_time;
	String				auto_surf_return_channel;
	gboolean			channel_change_complete;
	Streamer*			streamer;
	Engine*				engine;
	String				video_file;

	static gboolean on_init(gpointer data);
	static gboolean on_timer(gpointer data);
	static gboolean on_check_error_messages_timer(gpointer data);
	static gpointer epg_thread_function(Application* application);
	static gpointer stream_thread_function(Application* application);	
	
	void init();
	void stop();
	void start();
	void start_epg_thread();
	const Transponder& get_current_transponder();
	void stream_loop(IO::Channel& input, guint audio_pid, guint video_pid);
	String fix_path(const String& path);
	String get_configuration_path();
	time_t last_auto_surf_time;

	void check_error_messages();
	gboolean push_epg_event(const EpgEvent& event);
	gboolean check_epg_events();
	
public:
	Application (int argc, char** argv);
	~Application();

	static void run(int argc, char** argv);
	static Application& get_current();
	static String get_directory();
	
	void push_error_message(const String& message);

	Configuration& get_configuration() { return configuration; }
	void quit();

	IO::Channel&		get_log_file();
	Glade&				get_glade() { return glade; }
	Epg&				get_epg() { return epg; }
	RecordingManager&	get_recording_manager();
	void				set_channel_change_complete() { channel_change_complete = true; }
	
	ChannelManager& get_channel_manager() { return channel_manager; }
	Channel& get_channel(const String& channel_name);
	Channel& get_channel(int frequency, int service_id);
	void change_channel (const String& channel_name, gboolean keep_auto_surf = false );
	void change_channel (Channel& channel, gboolean keep_auto_surf = false );

	Channel&		get_video_channel();
	const String&	get_video_channel_name() const;
	String			get_video_event_title();

	void stop_record();
	void start_record(const String& title);
	void start_record(Channel& channel, const String& title);
	void start_record_force(Channel& channel, const String& title);
	String get_recording_file_name() const { return recording_file_name; }
	void toggle_record();
	void record(gboolean state);

	void hide_controls();
	void toggle_visibility();
	
	MainWindow& get_main_window();
	void toggle_fullscreen();
	void toggle_epg();

	void toggle_mute();
	void mute(gboolean state);

	void toggle_broadcast();
	void broadcast(gboolean state);

	void toggle_auto_surf();
	void auto_surf(gboolean state);

	void update_epg();
	void restart_stream();
	
	gboolean check_main_thread(const String& caller);
	
	void show_scheduled_recording_dialog();
	void show_program_details_dialog (const String& channel_name, int program_id);
	void show_scheduled_recording_dialog(const String& description);
	void show_scheduled_recordings_dialog();
	String get_video_file() const;
	
	const String& get_exe_path() const { return exe_path; }

	String get_status_message();
	
	static gint show_message_dialog(const String& message, GtkMessageType type = GTK_MESSAGE_INFO, GtkButtonsType buttons = GTK_BUTTONS_CLOSE);
	static gint show_error_message_dialog(const String& message);
	
	void show_help();
	void show_about();
	
	void remove_current_scheduled_recording();
	gboolean is_scheduled_recording();
	void check_scheduled_recordings() { last_recording_check_time = 0; }
	void set_selected_event(xmlNodePtr event);
	gboolean is_shutting_down() { return application_quit; };

	DvbTuner*	get_tuner();
	Engine*		get_engine();
	Streamer&	get_streamer();
	
	void set_dual_language_state(gint state);
};

#endif
