// Author(s): Muck van Weerdenburg
// Copyright: see the accompanying file COPYING or copy at
// https://svn.win.tue.nl/trac/MCRL2/browser/trunk/COPYING
//
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)
//
/// \file ltsconvert.cpp

#define NAME "ltsconvert"
#define AUTHOR "Muck van Weerdenburg"

#include <string>
#include "mcrl2/atermpp/aterm_init.h"
#include "mcrl2/core/detail/struct_core.h"
#include "mcrl2/lts/lts_io.h"
#include "mcrl2/lts/lts_algorithm.h"
#include "mcrl2/core/messaging.h"
#include "mcrl2/utilities/input_output_tool.h"
#include "mcrl2/utilities/squadt_tool.h"
#include "mcrl2/utilities/mcrl2_gui_tool.h"
#include "mcrl2/lps/specification.h"
#include "mcrl2/exception.h"

using namespace mcrl2::lts;
using namespace mcrl2::lts::detail;
using namespace mcrl2::utilities::tools;
using namespace mcrl2::utilities;
using namespace mcrl2::core;

static const std::set<lts_equivalence> &initialise_allowed_eqs()
{
  static std::set<lts_equivalence> s;
  s.insert(lts_eq_bisim);
  s.insert(lts_eq_branching_bisim);
  s.insert(lts_eq_divergence_preserving_branching_bisim);
  s.insert(lts_eq_sim);
  s.insert(lts_eq_trace);
  s.insert(lts_eq_weak_trace);
  return s;
}
static const std::set<lts_equivalence> &allowed_eqs()
{
  static const std::set<lts_equivalence> &s = initialise_allowed_eqs();
  return s;
}

static inline std::string get_base(std::string const& s) {
  return s.substr(0, s.find_last_of('.'));
}

static ATermAppl get_lps(std::string const& filename)
{
  if (!filename.empty()) {
    FILE* file = fopen(filename.c_str(),"rb");

    if ( file ) {
      ATerm t = ATreadFromFile(file);
      fclose(file);

      if ( (t == NULL) || (ATgetType(t) != AT_APPL) || !(mcrl2::core::detail::gsIsLinProcSpec((ATermAppl) t) || !strcmp(ATgetName(ATgetAFun((ATermAppl) t)),"spec2gen")) )
      {
        gsErrorMsg("invalid LPS-file '%s'\n",filename.c_str());
      }
      else {
        return (ATermAppl) t;
      }
    }
    else {
      gsErrorMsg("unable to open LPS-file '%s'\n",filename.c_str());
    }
  }

  return 0;
}

class t_tool_options 
{
  private:
    std::string     infilename;
    std::string     outfilename;

  public:
    std::string     lpsfile;
    lts_type        intype;
    lts_type        outtype;
    lts_equivalence equivalence;
    std::vector<std::string> tau_actions;   // Actions with these labels must be considered equal to tau.
    bool            print_dot_state;
    bool            determinise;
    bool            check_reach;

    inline t_tool_options() : intype(lts_none), outtype(lts_none), equivalence(lts_eq_none),
                       print_dot_state(true), determinise(false), check_reach(true) {
    }

    inline std::string source_string() const {
      return (infilename.empty()) ? std::string("standard input") :
                                    std::string("'" + infilename + "'");
    }

    inline std::string target_string() const {
      return (outfilename.empty()) ? std::string("standard output") :
                                     std::string("'" + outfilename + "'");
    }

    inline lts_extra get_extra(lts_type type, std::string const &base_name = "") const {
      if ( type == lts_dot )
      {
        lts_dot_options opts;
        opts.name = new std::string(base_name); // XXX Ugh!
        opts.print_states = print_dot_state;
        return lts_extra(opts);
      } else {
        if ( !lpsfile.empty() )
        {
          ATermAppl spec = get_lps(lpsfile);

          if ( spec != NULL )
          {
            if ( mcrl2::core::detail::gsIsLinProcSpec(spec) )
            {
              return lts_extra(mcrl2::lps::specification(spec));
            } else {
              return lts_extra((ATerm) spec);
            }
          }
        }
        return lts_extra();
      }
    }

    void read_lts(lts& l) const 
    {
      gsVerboseMsg("reading LTS from %s...\n", source_string().c_str());

      lts_extra extra = get_extra(intype);

      try
      { 
        if (infilename.empty()) 
        {
          lts l_temp(std::cin,intype,extra);
          l_temp.swap(l);
        }
        else 
        {
          mcrl2::lts::detail::read_from(l,infilename,intype,extra);
        }
      }
      catch (mcrl2::runtime_error &e)
      {
        throw mcrl2::runtime_error("cannot read LTS from " + source_string() +
                                   ".\nretry with -v/--verbose for more information.\n" +
                                   e.what());
      }

      if ( check_reach ) {
        gsVerboseMsg("checking reachability of input LTS...\n");

        if ( !reachability_check(l,true) ) {
          gsWarningMsg("not all states of the input LTS are reachable from the initial state; removed unreachable states to ensure correct behaviour in LTS tools (including this one)!\n");
        }
      }
    }


    void set_source(std::string const& filename) {
      infilename = filename;
    }

    void set_target(std::string const& filename) {
      outfilename = filename;

      if ( outtype == lts_none ) {
        gsVerboseMsg("trying to detect output format by extension...\n");

        outtype = mcrl2::lts::detail::guess_format(outfilename);

        if ( outtype == lts_none ) {
          if ( !lpsfile.empty() ) {
            gsWarningMsg("no output format set; using fsm because --lps was used\n");
            outtype = lts_fsm;
          } else {
            gsWarningMsg("no output format set or detected; using default (mcrl2)\n");
            outtype = lts_mcrl2;
          }
        }
      }
    }

    void write_lts(lts& l) const 
    {
      gsVerboseMsg("writing LTS to %s...\n", target_string().c_str());

      try
      {
        if (outfilename.empty()) 
        {
          l.write_to(std::cout,outtype,get_extra(outtype, "stdout"));
        }
        else 
        {
          l.write_to(outfilename, outtype, get_extra(outtype, get_base(outfilename)));
        }
      }
      catch (mcrl2::runtime_error &e)
      {
        throw mcrl2::runtime_error("cannot write LTS to " + target_string() +
                                        "\nretry with -v/--verbose for more information.\n" + 
                                        e.what());
      }
    }
};

using namespace std;

typedef squadt_tool< input_output_tool > ltsconvert_base;
class ltsconvert_tool : public ltsconvert_base
{
  private:
    t_tool_options tool_options;

  public:
    ltsconvert_tool() :
      ltsconvert_base(NAME,AUTHOR,
        "convert and optionally minimise an LTS",
        "Convert the labelled transition system (LTS) from INFILE to OUTFILE in the\n"
        "requested format after applying the selected minimisation method (default is\n"
        "none). If OUTFILE is not supplied, stdout is used. If INFILE is not supplied,\n"
        "stdin is used.\n"
        "\n"
        "The output format is determined by the extension of OUTFILE, whereas the input\n"
        "format is determined by the content of INFILE. Options --in and --out can be\n"
        "used to force the input and output formats. The supported formats are:\n"
        + mcrl2::lts::detail::supported_lts_formats_text(lts_mcrl2)
      )
    {
    }

    bool run()
    {
      lts l;

      tool_options.read_lts(l);
      if (!l.hide_actions(tool_options.tau_actions))
      { throw mcrl2::runtime_error("Cannot hide actions");
      }

      if ( tool_options.equivalence != lts_eq_none )
      {
        gsVerboseMsg("reducing LTS (modulo %s)...\n", name_of_equivalence(tool_options.equivalence).c_str());
        gsVerboseMsg("before reduction: %lu states and %lu transitions \n",l.num_states(),l.num_transitions());
        reduce(l,tool_options.equivalence);
        gsVerboseMsg("after reduction: %lu states and %lu transitions\n",l.num_states(),l.num_transitions());
      }

      if ( tool_options.determinise )
      {
        gsVerboseMsg("determinising LTS...\n");
        gsVerboseMsg("before determinisation: %lu states and %lu transitions\n",l.num_states(),l.num_transitions());
        determinise(l);
        gsVerboseMsg("after determinisation: %lu states and %lu transitions\n",l.num_states(),l.num_transitions());
      }

      tool_options.write_lts(l);

      return true;
    }

  protected:
    void add_options(interface_description &desc)
    {
      ltsconvert_base::add_options(desc);
  
      desc.add_option("no-reach",
          "do not perform a reachability check on the input LTS");
      desc.add_option("no-state",
          "leave out state information when saving in dot format", 'n');
      desc.add_option("determinise", "determinise LTS", 'D');
      desc.add_option("lps", make_mandatory_argument("FILE"),
          "use FILE as the LPS from which the input LTS was generated; this might"
          "be needed to store the correct parameter names of states when saving "
          "in fsm format and to convert non-mCRL2 LTSs to a mCRL2 LTS", 'l');
      desc.add_option("in", make_mandatory_argument("FORMAT"),
          "use FORMAT as the input format", 'i').
        add_option("out", make_mandatory_argument("FORMAT"),
          "use FORMAT as the output format", 'o');
      desc.add_option("equivalence", make_mandatory_argument("NAME"),
          "generate an equivalent LTS, preserving equivalence NAME:\n"
          +supported_lts_equivalences_text(allowed_eqs())
          , 'e');
      desc.add_option("tau", make_mandatory_argument("ACTNAMES"),
          "consider actions with a name in the comma separated list ACTNAMES to "
          "be internal (tau) actions in addition to those defined as such by "
          "the input");
    }

    void set_tau_actions(std::vector <std::string>& tau_actions, std::string const& act_names)
    {
      std::string::size_type lastpos = 0, pos;
      while ( (pos = act_names.find(',',lastpos)) != std::string::npos )
      {
        tau_actions.push_back(act_names.substr(lastpos,pos-lastpos));
        lastpos = pos+1;
      }
      tau_actions.push_back(act_names.substr(lastpos));
    }

    void parse_options(const command_line_parser &parser)
    {
      if (parser.options.count("lps")) {
        if (1 < parser.options.count("lps")) {
          std::cerr << "warning: multiple LPS files specified; can only use one\n";
        }

        tool_options.lpsfile = parser.option_argument("lps");
      }
      if (parser.options.count("in")) {
        if (1 < parser.options.count("in")) {
          std::cerr << "warning: multiple input formats specified; can only use one\n";
        }

        tool_options.intype = mcrl2::lts::detail::parse_format(parser.option_argument("in"));

        if (tool_options.intype == lts_none) {
          std::cerr << "warning: format '" << parser.option_argument("in") <<
                       "' is not recognised; option ignored" << std::endl;
        }
      }
      if (parser.options.count("out")) {
        if (1 < parser.options.count("out")) {
          std::cerr << "warning: multiple output formats specified; can only use one\n";
        }

        tool_options.outtype = mcrl2::lts::detail::parse_format(parser.option_argument("out"));

        if (tool_options.outtype == lts_none) {
          std::cerr << "warning: format '" << parser.option_argument("out") <<
                       "' is not recognised; option ignored" << std::endl;
        }
      }

      if (parser.options.count("equivalence")) {

        tool_options.equivalence = parse_equivalence(
            parser.option_argument("equivalence"));

        if ( allowed_eqs().count(tool_options.equivalence) == 0 )
        {
          parser.error("option -e/--equivalence has illegal argument '" +
              parser.option_argument("equivalence") + "'");
        }
      }

      if (parser.options.count("tau")) 
      {
        set_tau_actions(tool_options.tau_actions, parser.option_argument("tau"));
      }

      tool_options.determinise                       = 0 < parser.options.count("determinise");
      tool_options.check_reach                       = parser.options.count("no-reach") == 0;
      tool_options.print_dot_state                   = parser.options.count("no-state") == 0;

      if ( tool_options.determinise && (tool_options.equivalence != lts_eq_none) ) {
        parser.error("cannot use option -D/--determinise together with LTS reduction options\n");
      }

      if (2 < parser.arguments.size()) {
        parser.error("too many file arguments");
      }
      else {
        if (0 < parser.arguments.size()) {
          tool_options.set_source(parser.arguments[0]);
        }
        else {
          if ( tool_options.intype == lts_none ) {
            gsWarningMsg("cannot detect format from stdin and no input format specified; assuming aut format\n");
            tool_options.intype = lts_aut;
          }
        }
        if (1 < parser.arguments.size()) {
          tool_options.set_target(parser.arguments[1]);
        }
        else {
          if ( tool_options.outtype == lts_none ) {
            if ( !tool_options.lpsfile.empty() ) {
              gsWarningMsg("no output format set; using fsm because --lps was used\n");
              tool_options.outtype = lts_fsm;
            } else {
              gsWarningMsg("no output format set or detected; using default (aut)\n");
              tool_options.outtype = lts_aut;
            }
          }
        }
      }
    }

#ifdef ENABLE_SQUADT_CONNECTIVITY
  public:
    /** \brief configures tool capabilities */
    void set_capabilities(tipi::tool::capabilities&) const;

    /** \brief queries the user via SQuADT if needed to obtain configuration information */
    void user_interactive_configuration(tipi::configuration&);

    /** \brief check an existing configuration object to see if it is usable */
    bool check_configuration(tipi::configuration const&) const;

    /** \brief performs the task specified by a configuration */
    bool perform_task(tipi::configuration&);
#endif
};

// SQuADT protocol interface
#ifdef ENABLE_SQUADT_CONNECTIVITY
#include <mcrl2/utilities/mcrl2_squadt_interface.h>

static const char* lts_file_for_input  = "lts_in";  ///< file containing an LTS that can be imported using the LTS library
static const char* lts_file_for_output = "lts_out"; ///< file used to write the output to
static const char* lps_file_auxiliary  = "lps_aux"; ///< LPS file needed for some conversion operations

static const char* option_selected_transformation            = "selected_transformation";               ///< the selected transformation method
static const char* option_selected_output_format             = "selected_output_format";                ///< the selected output format
static const char* option_no_reachability_check              = "no_reachability_check";                 ///< do not check reachability of input LTS
static const char* option_no_state_information               = "no_state_information";                  ///< dot format output specific option to not save state information
static const char* option_tau_actions                        = "tau_actions";                           ///< the actions that should be recognised as tau

void ltsconvert_tool::set_capabilities(tipi::tool::capabilities& c) const {
  std::set< lts_type > const& input_formats(mcrl2::lts::detail::supported_lts_formats());

  for (std::set< lts_type >::const_iterator i = input_formats.begin(); i != input_formats.end(); ++i) 
  {
    c.add_input_configuration(lts_file_for_input, tipi::mime_type(mcrl2::lts::detail::mime_type_for_type(*i)), tipi::tool::category::conversion);
  }
}

void ltsconvert_tool::user_interactive_configuration(tipi::configuration& c) {
  using namespace tipi;
  using namespace tipi::layout;
  using namespace tipi::layout::elements;

  /* Create display */
  tipi::tool_display d;

  layout::vertical_box& m = d.create< vertical_box >().set_default_margins(margins(0, 5, 0, 5));

  /* Helper for format selection */
  mcrl2::utilities::squadt::radio_button_helper < mcrl2::lts::lts_type > format_selector(d);

  m.append(d.create< horizontal_box >().
                append(d.create< label >().set_text("Output format : ")).
                append(format_selector.associate(mcrl2::lts::lts_aut, "Aldebaran",true)).
                append(format_selector.associate(mcrl2::lts::lts_mcrl, "SVC/mCRL")).
                append(format_selector.associate(mcrl2::lts::lts_mcrl2, "mCRL2")).
                append(format_selector.associate(mcrl2::lts::lts_svc, "SVC")).
#ifdef USE_BCG
                append(format_selector.associate(mcrl2::lts::lts_bcg, "BCG")).
#endif
                append(format_selector.associate(mcrl2::lts::lts_fsm, "FSM")).
                append(format_selector.associate(mcrl2::lts::lts_dot, "dot")),
           margins(0,5,0,5));

  file_control& lps_file_field        = d.create< file_control >();
  checkbox&   check_reachability    = d.create< checkbox >();
  checkbox&   add_state_information = d.create< checkbox >();

  m.append(d.create< label >()).
    append(d.create< horizontal_box >().
                append(d.create< label >().set_text("LPS file name : ")).
                append(lps_file_field)).
    append(check_reachability.set_label("Perform reachability check").set_status(true)).
    append(add_state_information.set_label("Add state information").set_status(true));

//  mcrl2::utilities::squadt::change_visibility_on_toggle(format_selector.get_button(dot), top.get(), add_state_information);
//  format_selector.get_button(dot).on_change(boost::bind(::mcrl2::utilities::squadt::change_visibility_on_toggle, format_selector.get_button(dot), top.get(), add_state_information));

  /* Helper for transformation selection */
  mcrl2::utilities::squadt::radio_button_helper < lts_equivalence > transformation_selector(d);

  m.append(d.create< label >().set_text("LTS transformation:")).
    append(transformation_selector.associate(lts_eq_none, "none", true)).
    append(transformation_selector.associate(lts_eq_bisim, "reduction modulo strong bisimulation equivalence")).
    append(transformation_selector.associate(lts_eq_branching_bisim, "reduction modulo branching bisimulation equivalence")).
    append(transformation_selector.associate(lts_eq_divergence_preserving_branching_bisim, "reduction modulo divergence preserving branching bisimulation equivalence")).
    append(transformation_selector.associate(lts_eq_sim, "reduction modulo strong simulation equivalence")).
    append(transformation_selector.associate(lts_eq_trace, "determinisation and reduction modulo trace equivalence")).
    append(transformation_selector.associate(lts_eq_weak_trace, "determinisation and reduction modulo weak trace equivalence")).
    append(transformation_selector.associate(lts_red_determinisation, "determinisation")); 

  text_field& tau_field                   = d.create< text_field >();

  m.append(d.create< label >()).
    append(d.create< horizontal_box >().
                append(d.create< label >().set_text("Internal (tau) actions : ")).
                append(tau_field.set_text("tau")));

  /* Attach events */
//  transformation_selector.get_button(no_transformation).
//        on_change(boost::bind(mcrl2::utilities::squadt::change_visibility_on_toggle,
//              transformation_selector.get_button(no_transformation), top.get(), tau_field));
//  transformation_selector.get_button(minimisation_modulo_strong_bisimulation).
//        on_change(boost::bind(mcrl2::utilities::squadt::change_visibility_on_toggle,
//              transformation_selector.get_button(minimisation_modulo_strong_bisimulation), top.get(), bisimulation_add_eq_classes));
//  transformation_selector.get_button(minimisation_modulo_branching_bisimulation).
//        on_change(boost::bind(mcrl2::utilities::squadt::change_visibility_on_toggle,
//              transformation_selector.get_button(minimisation_modulo_branching_bisimulation), top.get(), bisimulation_add_eq_classes));

  button& okay_button = d.create< button >().set_label("OK");

  // Add some default values for existing options in the current configuration
  if (c.option_exists(option_selected_output_format)) {
    format_selector.set_selection(c.get_option_argument< mcrl2::lts::lts_type >(option_selected_output_format, 0));
  }
  if (c.option_exists(option_selected_transformation)) {
    transformation_selector.set_selection(c.get_option_argument< lts_equivalence >(option_selected_transformation, 0));
  }
  if (c.input_exists(lps_file_auxiliary)) {
      lps_file_field.set_text(c.get_input(lps_file_auxiliary).location());
  }
  if (c.option_exists(option_no_reachability_check)) {
    check_reachability.set_status(c.get_option_argument< bool >(option_no_reachability_check));
  }
  if (c.option_exists(option_no_state_information)) {
    add_state_information.set_status(c.get_option_argument< bool >(option_no_state_information));
  }
  if (c.option_exists(option_tau_actions)) {
    tau_field.set_text(c.get_option_argument< std::string >(option_tau_actions));
  }

  send_display_layout(d.manager(m.append(okay_button, layout::top)));

  /* Wait until the ok button was pressed */
  okay_button.await_change();

  /* Add output file to the configuration */
  std::string     output_name(c.get_output_name("." +
                    mcrl2::lts::detail::extension_for_type(format_selector.get_selection())));
  tipi::mime_type output_type(tipi::mime_type(mcrl2::lts::detail::mime_type_for_type(format_selector.get_selection())));

  if (c.output_exists(lts_file_for_output)) {
    tipi::configuration::object& output_file = c.get_output(lts_file_for_output);

    output_file.type(output_type);
    output_file.location(output_name);
  }
  else {
    c.add_output(lts_file_for_output, output_type, output_name);
  }

  /* Add lps file when output is FSM format or when the output is mCRL2 and the input is Aldebaran or mCRL */
  if ((format_selector.get_selection() == lts_fsm && (
         c.get_input(lts_file_for_input).type().sub_type() == "svc" ||
         c.get_input(lts_file_for_input).type().sub_type() == "svc+mcrl" ||
         c.get_input(lts_file_for_input).type().sub_type() == "lts"))
   || (format_selector.get_selection() == lts_mcrl2 && (
         c.get_input(lts_file_for_input).type().sub_type() == "aut" ||
         c.get_input(lts_file_for_input).type().sub_type() == "svc" ||
         c.get_input(lts_file_for_input).type().sub_type() == "svc+mcrl"))) {

    if (c.input_exists(lps_file_auxiliary)) {
      c.get_input(lps_file_auxiliary).location(lps_file_field.get_text());
    }
    else {
      c.add_input(lps_file_auxiliary, tipi::mime_type("lps", tipi::mime_type::application), lps_file_field.get_text());
    }
  }
  else {
    c.remove_input(lps_file_auxiliary);
  }

  lts_equivalence selected_transformation = transformation_selector.get_selection();

  if (selected_transformation != lts_eq_none) {
    c.add_option(option_selected_transformation).set_argument_value< 0 >(selected_transformation);
  }
  c.add_option(option_selected_output_format).set_argument_value< 0 >(format_selector.get_selection());

  c.add_option(option_no_reachability_check).set_argument_value< 0 >(check_reachability.get_status());

  if (format_selector.get_selection() == lts_dot) {
    c.add_option(option_no_state_information).
       set_argument_value< 0 >(add_state_information.get_status());
  }
  else {
    c.remove_option(option_no_state_information);
  }

  if (!tau_field.get_text().empty()) {
    c.add_option(option_tau_actions).set_argument_value< 0 >(tau_field.get_text());
  }
  else {
    c.remove_option(option_tau_actions);
  }

  send_clear_display();
}

bool ltsconvert_tool::check_configuration(tipi::configuration const& c) const 
{
  bool result = true;

  result &= c.input_exists(lts_file_for_input);
  result &= c.output_exists(lts_file_for_output);

  return (result);
}

bool ltsconvert_tool::perform_task(tipi::configuration& c) {

  if (c.input_exists(lps_file_auxiliary)) {
    tool_options.lpsfile = c.get_input(lps_file_auxiliary).location();
  }
  if (c.option_exists(option_no_state_information)) {
    tool_options.print_dot_state = !(c.get_option_argument< bool >(option_no_state_information));
  }
  if (c.option_exists(option_no_reachability_check)) {
    tool_options.check_reach = !(c.get_option_argument< bool >(option_no_reachability_check));
  }

  tool_options.intype  = mcrl2::lts::detail::parse_format(c.get_output(lts_file_for_input).type().sub_type());
  tool_options.outtype = mcrl2::lts::detail::parse_format(c.get_output(lts_file_for_output).type().sub_type());
  tool_options.set_source(c.get_input(lts_file_for_input).location());
  tool_options.set_target(c.get_output(lts_file_for_output).location());

  if (c.option_exists(option_selected_transformation)) {
    lts_equivalence method = c.get_option_argument< lts_equivalence >(option_selected_transformation);

    tool_options.equivalence = method;

    if (method == mcrl2::lts::lts_red_determinisation) {
      tool_options.determinise = true;
    }

    if (c.option_exists(option_tau_actions)) {
      set_tau_actions(tool_options.tau_actions, c.get_option_argument< std::string >(option_tau_actions));
    }
  }

  run();

  send_clear_display();

  return true;
}
#endif

class ltsconvert_gui_tool: public mcrl2_gui_tool<ltsconvert_tool> {
public:
	ltsconvert_gui_tool() {

		std::vector<std::string> values;

		m_gui_options["determinise"] = create_checkbox_widget();

		values.clear();
		values.push_back("bisim");
		values.push_back("branching-bisim");
		values.push_back("dpbranching-bisim");
		values.push_back("sim");
		values.push_back("trace");
		values.push_back("weak-trace");
		m_gui_options["equivalence"] = create_radiobox_widget(values);
		m_gui_options["lps"] = create_filepicker_widget();
		m_gui_options["no-state"] = create_checkbox_widget();
		m_gui_options["no-reach"] = create_checkbox_widget();
		m_gui_options["tau"] = create_textctrl_widget();

		//-iFORMAT, --in=FORMAT    use FORMAT as the input format
		//-oFORMAT, --out=FORMAT   use FORMAT as the output format

	}
};





int main(int argc, char **argv)
{
  MCRL2_ATERMPP_INIT(argc, argv)

  return ltsconvert_gui_tool().execute(argc,argv);
}
