// Author(s): Diana Koenraadt, Remco Blewanus, Bram Schoenmakers, Thorstin Crijns, Hans Poppelaars, Bas Luksenburg, Jonathan Nelisse
//
// Distributed under the Boost Software License, Version 1.0.
// (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)
//
/// \file grape_frame.cpp
//
// Implements the main frame of the application class.

#include "wx.hpp" // precompiled headers

#include <wx/cmdproc.h>
#include <wx/filesys.h>
#include <wx/fs_arc.h>
#include <wx/html/helpctrl.h>
#include <wx/statusbr.h>
#include <wx/tglbtn.h>
#include <wx/timer.h>
#include <wx/fontenum.h>

#include "grape_frame.h"
#include "grape_events.h"
#include "grape_menubar.h"
#include "grape_toolbar.h"
#include "grape_glcanvas.h"
#include "grape_logpanel.h"
#include "grape_clipboard.h"
#include "grape_listbox.h"
#include "grape_ids.h"

#include "mcrl2/utilities/basename.h"

// window icon
#include "pics/grape.xpm"

#ifdef _WIN32
  #include <windows.h>
#endif

using namespace grape::grapeapp;
using namespace grape::libgrape;

grape_frame::grape_frame( const wxString &p_filename )
: wxFrame( NULL, wxID_ANY, _T("GraPEmCRL2"), wxDefaultPosition, wxDefaultSize, wxDEFAULT_FRAME_STYLE, _T("GraPEmCRL2") )
, m_modified( false )
, m_specification( 0 )
, m_mode( GRAPE_MODE_NONE )
{
  m_current_diagram = 0;
  // initialize widgets
  m_dataspecbutton = new wxToggleButton( this, GRAPE_DATATYPE_SPEC_BUTTON, _T("Data type specification") );
  m_process_diagram_list = new grape_listbox(this, GRAPE_PROCESS_DIAGRAM_LIST, this);
  m_architecture_diagram_list = new grape_listbox(this, GRAPE_ARCHITECTURE_DIAGRAM_LIST, this);
  m_splitter = new wxSplitterWindow( this, GRAPE_SPLITTER );
  m_clipboard = new grape_clipboard( this );
  m_timer = new wxTimer( this, GRAPE_TIMER );
  m_timer->Start( 2000, true );


  int gl_args[] = {WX_GL_RGBA, WX_GL_DOUBLEBUFFER, WX_GL_DEPTH_SIZE, 16, 0};
  m_datatext = new wxTextCtrl( m_splitter, GRAPE_DATASPEC_TEXT, wxEmptyString, wxDefaultPosition, wxDefaultSize, wxTE_RICH | wxTE_MULTILINE | wxTE_PROCESS_TAB | wxTE_PROCESS_ENTER );
  wxFontEnumerator font_enum;
  font_enum.EnumerateFacenames(wxFONTENCODING_SYSTEM, true);
  wxArrayString font_names = font_enum.GetFacenames();
  font_names.Sort();
  wxString facename;
  for (unsigned int i = 0; i < font_names.GetCount(); ++i)
  {
    if (font_names[i].Find(_T("Courier")) != wxNOT_FOUND)
    {
      facename = font_names[i];
      break;
    }
  }
  if (facename.IsEmpty())
  {
    if (font_names.GetCount() > 0) facename = font_names[0];
  }
  m_datatext_font = wxFont(10, wxFONTFAMILY_TELETYPE, wxFONTSTYLE_NORMAL, wxFONTWEIGHT_NORMAL, false, facename);
  m_glcanvas = new grape_glcanvas(m_splitter, gl_args, this);
  m_logpanel = new grape_logpanel(this);
  m_logpanel->Hide();
  m_splitter->SetSplitMode(wxSPLIT_HORIZONTAL);
  m_splitter->Initialize(m_glcanvas);
  m_statusbar = new wxStatusBar(this);
  m_menubar = new grape_menubar();

  SetBackgroundColour(m_menubar->GetBackgroundColour());

  // place widgets in sizers
  wxBoxSizer *main_box = new wxBoxSizer(wxHORIZONTAL);
  main_box->Add(m_splitter, 1, wxEXPAND | wxBOTTOM, 5);
	
  wxBoxSizer *process_box = new wxBoxSizer(wxVERTICAL);
  wxStaticText *proc_text = new wxStaticText( this, -1, _T("Processes"), wxDefaultPosition, wxDefaultSize, wxALIGN_CENTRE );
  process_box->Add( proc_text, 0, wxALIGN_CENTRE | wxEXPAND );  //Doesn't work in LINUX
  process_box->Add(m_process_diagram_list, 1, wxEXPAND | wxALL | wxALIGN_CENTER, 5);

  wxBoxSizer *architecture_box = new wxBoxSizer(wxVERTICAL);
  wxStaticText *arch_text = new wxStaticText( this, -1, _T("Architectures"), wxDefaultPosition, wxDefaultSize, wxALIGN_CENTRE );
  architecture_box->Add( arch_text, 0, wxALIGN_CENTRE | wxEXPAND );   //Doesn't work in LINUX
  architecture_box->Add(m_architecture_diagram_list, 1, wxEXPAND | wxALL | wxALIGN_CENTER, 5);

  wxBoxSizer *right_box = new wxBoxSizer(wxVERTICAL);
  right_box->Add(m_dataspecbutton, 0, wxEXPAND);
  right_box->Add(architecture_box, 1, wxEXPAND);
  right_box->Add(process_box, 1, wxEXPAND);

  main_box->Add(right_box, 0, wxEXPAND | wxALIGN_RIGHT);

  // initialize frame
  SetSizer(main_box);
  SetSize(wxSize(1000, 800));

  // set window icon
  SetIcon( grape_xpm );

  // attach statusbar, menubar and set the frame status (which sets the correct toolbar)
  SetStatusBar(m_statusbar);
  SetMenuBar(m_menubar);
  set_mode( GRAPE_MODE_NONE );

  // init counter
  m_counter = 0;
  // attach event handler to edit menu
  m_event_handler = new wxCommandProcessor;
  m_event_handler->SetEditMenu(m_menubar->get_menu( grape_menubar::GRAPE_MENU_EDIT ));
  m_event_handler->Initialize();

  // initialize variables
  bool show_messagebox = false;
  if ( p_filename.IsEmpty() ) // no filename supplied, use default one
  {
    wxCommandEvent event;
    event_menu_new( event );
  }
  else  // filename supplied
  {
    wxFileName filename = wxFileName( p_filename );
    // check filename
    if ( filename.FileExists() && ( filename.GetExt().Lower() == _T("gra") ) ) // filename ok
    {
      grape_event_open *event = new grape_event_open( this, p_filename );
      m_event_handler->Submit( event, false );
      update_bars();
    }
    else // filename not ok, open new
    {
      show_messagebox = true;
      wxCommandEvent event;
      event_menu_new( event );
    }
  }

  set_is_modified( false ); // implicitely calls set_title

  wxFileSystem fs;
  wxFileSystem::AddHandler(new wxArchiveFSHandler);
  m_help_controller = new wxHtmlHelpController(wxHF_DEFAULT_STYLE, this);
  wxString filename = wxEmptyString;



 
 //GRAPE_HELP_DIR
 //
  mcrl2::utilities::basename basename;
  std::string path  = basename.get_executable_basename();

    bool found = false;
    // binary in build tree
    wxString wxpath(path.c_str(), wxConvUTF8);
  
    if( fs.FindFileInPath( &filename, wxpath , _T("/grapehelp.zip") ) )
      {
        m_help_controller->AddBook( wxFileName( filename ) );
        found = true;
      }
    // binary in install/distribution
    if( fs.FindFileInPath( &filename, wxpath , _T("/../share/doc/mcrl2/grape/grapehelp.zip") ) )
      {
        m_help_controller->AddBook( wxFileName( filename ) );
        found = true;
      }
  #ifdef _WIN32
    // Multi-Configuration
    if( fs.FindFileInPath( &filename, wxpath , _T("/../grapehelp.zip") ) )
      {
        m_help_controller->AddBook( wxFileName( filename ) );
        found = true;
      }

    std::string install_win32_path = basename.get_win32_install_path();
    wxString wxinstall_path(install_win32_path.c_str(), wxConvUTF8);
    if(!install_win32_path.empty())
    {
      // binary in install/distribution
      if( fs.FindFileInPath( &filename, wxinstall_path , _T("/share/doc/mcrl2/grape/grapehelp.zip") ) )
        {
          m_help_controller->AddBook( wxFileName( filename ) );
          found = true;
        }
      }
  #endif

    if( !found )
    {
      wxString info;
      info << wxT("Help file \"grapehelp.zip\" could not be found in:\n- ");
      info << wxpath << wxT("/grapehelp.zip") ;
      info << wxT("\n- ");
      std::string::size_type t = path.find_last_of("\\");
      wxString wxsubpath(path.substr(0,t).c_str(), wxConvUTF8);
      info << wxsubpath << wxT("/share/mcrl2/grape/grapehelp.zip"); 
  #ifdef _WIN32
    if(!install_win32_path.empty())
    {
      info << wxT("\n- ");
      info << wxsubpath << wxT("/grapehelp.zip"); 
      info << wxT("\n- ");
      info << wxinstall_path << wxT("\\share\\mcrl2\\grape\\grapehelp.zip");
    }
  #endif 
      wxMessageBox(  info  , _T("Warning"), wxOK | wxICON_EXCLAMATION);
    }

  // show frame
  Show();
    
  if ( show_messagebox )
  {
    wxMessageBox( _T("The file \"") + p_filename + _T("\" could not be opened."), _T("Warning"), wxOK | wxICON_EXCLAMATION );
  }
}

grape_frame::~grape_frame( void )
{
  delete m_process_diagram_list;
  m_process_diagram_list = 0;
  delete m_architecture_diagram_list;
  m_architecture_diagram_list = 0;
  delete m_datatext;
  m_datatext = 0;
  delete m_glcanvas;
  m_glcanvas = 0;
  delete m_logpanel;
  m_logpanel = 0;
  delete m_splitter;
  m_splitter = 0;
  delete m_specification;
  m_specification = 0;
  delete m_event_handler;
  m_event_handler = 0;
  delete m_timer;
  m_timer = 0;
  delete m_help_controller;
  m_help_controller = 0;
  delete m_clipboard;
  m_clipboard = 0;
  delete m_dataspecbutton;
  m_dataspecbutton = 0;
  SetStatusBar(0);
  delete m_statusbar;
  SetMenuBar(0);
  delete m_menubar;
}

grape_menubar * grape_frame::get_menubar( void )
{
  return m_menubar;
}

grape_listbox * grape_frame::get_process_diagram_listbox( void )
{
  return m_process_diagram_list;
}

grape_listbox * grape_frame::get_architecture_diagram_listbox( void )
{
  return m_architecture_diagram_list;
}

grape_logpanel * grape_frame::get_logpanel( void )
{
  return m_logpanel;
}

wxStatusBar * grape_frame::get_statusbar( void )
{
  return m_statusbar;
}

grape_glcanvas * grape_frame::get_glcanvas( void )
{
  return m_glcanvas;
}

wxCommandProcessor * grape_frame::get_event_handler( void )
{
  return m_event_handler;
}

grape_clipboard * grape_frame::get_clipboard( void )
{
  return m_clipboard;
}

wxFileName grape_frame::get_filename( void ) const
{
  return m_filename;
}

void grape_frame::set_filename( const wxFileName &p_filename )
{
  m_filename = p_filename;
}

bool grape_frame::get_is_modified( void ) const
{
  return m_modified;
}

void grape_frame::set_is_modified( bool p_modified )
{
  // set modified
  m_modified = p_modified;
  // set title
  set_title();
  update_bars();
}


grape_specification* grape_frame::get_grape_specification( void ) const
{
  return m_specification;
}

void grape_frame::set_grape_specification( grape_specification* p_grape_spec )
{
  m_specification = p_grape_spec;
}

void grape_frame::set_toolbar( grape_mode p_mode )
{
  wxToolBar *current_toolbar = GetToolBar();
  delete current_toolbar;

  grape_toolbar *toolbar = 0;

  switch ( p_mode )
  {
    case GRAPE_MODE_NONE:
      toolbar = new grape_toolbar(this);
      toolbar->set_mode( grape_toolbar::GRAPE_TOOLMODE_NOSPEC );
      m_glcanvas->set_canvas_state( IDLE );
      break;
    case GRAPE_MODE_DATASPEC: // fall through
    case GRAPE_MODE_SPEC:
      toolbar = new grape_toolbar(this);
      toolbar->set_mode( grape_toolbar::GRAPE_TOOLMODE_SPEC );
      m_glcanvas->set_canvas_state( IDLE );
      break;
    case GRAPE_MODE_ARCH:
      toolbar = new grape_arch_toolbar(this);
      toolbar->set_mode( grape_toolbar::GRAPE_TOOLMODE_SPEC | grape_toolbar::GRAPE_TOOLMODE_ARCH );
      m_glcanvas->set_canvas_state( SELECT );
      toolbar->ToggleTool( GRAPE_TOOL_SELECT, true );
      break;
    case GRAPE_MODE_PROC:
      toolbar = new grape_proc_toolbar(this);
      toolbar->set_mode( grape_toolbar::GRAPE_TOOLMODE_SPEC | grape_toolbar::GRAPE_TOOLMODE_PROC );
      m_glcanvas->set_canvas_state( SELECT );
      toolbar->ToggleTool( GRAPE_TOOL_SELECT, true );
      break;
    default: toolbar = 0;
  }

  toolbar->Realize();
  SetToolBar( toolbar );
}

void grape_frame::set_id_counter( long p_counter )
{
  m_counter = p_counter;
}

unsigned int grape_frame::get_new_id( void )
{
  return m_counter++;
}

BEGIN_EVENT_TABLE(grape_frame, wxFrame)
  // file menu
  EVT_MENU(wxID_NEW, grape_frame::event_menu_new)
  EVT_MENU(wxID_OPEN, grape_frame::event_menu_open)
  EVT_MENU(wxID_CLOSE, grape_frame::event_menu_close)
  EVT_MENU(wxID_SAVE, grape_frame::event_menu_save)
  EVT_MENU(wxID_SAVEAS, grape_frame::event_menu_saveas)
  EVT_MENU(wxID_PRINT, grape_frame::event_menu_print)
  EVT_MENU(wxID_EXIT, grape_frame::event_menu_quit)

  // edit menu
  EVT_MENU(wxID_UNDO, grape_frame::event_menu_undo)
  EVT_MENU(wxID_REDO, grape_frame::event_menu_redo)
  EVT_MENU(wxID_CUT, grape_frame::event_menu_cut)
  EVT_MENU(wxID_COPY, grape_frame::event_menu_copy)
  EVT_MENU(wxID_PASTE, grape_frame::event_menu_paste)
  EVT_MENU(wxID_DELETE, grape_frame::event_menu_delete)
  EVT_MENU(GRAPE_MENU_PROPERTIES, grape_frame::event_menu_properties)
  //EVT_MENU(GRAPE_MENU_SELECT_ALL, grape_frame::event_menu_select_all)
  //EVT_MENU(GRAPE_MENU_DESELECT_ALL, grape_frame::event_menu_deselect_all)
  EVT_MENU(GRAPE_MENU_DATATYPESPEC, grape_frame::event_datatype_spec)

  // specification menu
  EVT_MENU(GRAPE_MENU_ADD_ARCHITECTURE_DIAGRAM, grape_frame::event_menu_add_architecture_diagram)
  EVT_MENU(GRAPE_MENU_ADD_PROCESS_DIAGRAM, grape_frame::event_menu_add_process_diagram)
  EVT_MENU(GRAPE_MENU_VALIDATE, grape_frame::event_menu_validate)
  EVT_MENU(GRAPE_MENU_EXPORTMCRL2, grape_frame::event_menu_exportmcrl2)
  EVT_MENU(GRAPE_MENU_EXPORTIMAGE, grape_frame::event_menu_exportimage)
  EVT_MENU(GRAPE_MENU_RENAME_DIAGRAM, grape_frame::event_menu_rename_diagram)
  EVT_MENU(GRAPE_MENU_REMOVE_DIAGRAM, grape_frame::event_menu_remove_diagram)

  // help menu
  EVT_MENU(wxID_HELP, grape_frame::event_menu_help)

  // toolbar + tools menu
  EVT_TOOL(GRAPE_TOOL_SELECT, grape_frame::event_tool_selected)
  EVT_TOOL(GRAPE_TOOL_ADD_INITIAL_DESIGNATOR, grape_frame::event_tool_selected)
  EVT_TOOL(GRAPE_TOOL_ADD_REFERENCE_STATE, grape_frame::event_tool_selected)
  EVT_TOOL(GRAPE_TOOL_ADD_STATE, grape_frame::event_tool_selected)
  EVT_TOOL(GRAPE_TOOL_ADD_TERMINATING_TRANSITION, grape_frame::event_tool_selected)
  EVT_TOOL(GRAPE_TOOL_ADD_NONTERMINATING_TRANSITION, grape_frame::event_tool_selected)
  EVT_TOOL(GRAPE_TOOL_ADD_PROCESS_REFERENCE, grape_frame::event_tool_selected)
  EVT_TOOL(GRAPE_TOOL_ADD_ARCHITECTURE_REFERENCE, grape_frame::event_tool_selected)
  EVT_TOOL(GRAPE_TOOL_ADD_CHANNEL, grape_frame::event_tool_selected)
  EVT_TOOL(GRAPE_TOOL_ADD_CHANNEL_COMMUNICATION, grape_frame::event_tool_selected)
  EVT_TOOL(GRAPE_TOOL_ADD_COMMENT, grape_frame::event_tool_selected)
  EVT_TOOL_ENTER(wxID_ANY, grape_frame::update_statusbar )

  // listbox
  EVT_TOGGLEBUTTON(GRAPE_DATATYPE_SPEC_BUTTON, grape_frame::event_datatype_spec)
  EVT_LISTBOX(GRAPE_ARCHITECTURE_DIAGRAM_LIST, grape_frame::event_select_diagram)
  EVT_LISTBOX(GRAPE_PROCESS_DIAGRAM_LIST, grape_frame::event_select_diagram)
  EVT_LISTBOX_DCLICK(GRAPE_ARCHITECTURE_DIAGRAM_LIST, grape_frame::event_menu_rename_diagram)
  EVT_LISTBOX_DCLICK(GRAPE_PROCESS_DIAGRAM_LIST, grape_frame::event_menu_rename_diagram)

  // frame
  EVT_CLOSE( grape_frame::event_window_close )
  EVT_TIMER(GRAPE_TIMER, grape_frame::grape_event_timer)

  // datatype specification
  EVT_TEXT( GRAPE_DATASPEC_TEXT, grape_frame::dataspec_modified )

END_EVENT_TABLE()

wxHtmlHelpController* grape_frame::get_help_controller( void )
{
  return m_help_controller;
}

void grape_frame::toggle_view( grape_mode old_mode, grape_mode p_mode )
{
  if ( p_mode == GRAPE_MODE_DATASPEC ) // switching to dts
  {
    load_datatype_specification();
    m_dataspecbutton->SetValue( true );
    m_splitter->ReplaceWindow(m_splitter->GetWindow1(), m_datatext);
    dataspec_setstyle();
    m_datatext->Show();
    m_datatext->SetFocus();
    m_glcanvas->Hide();
  }
  else // switching back to canvas
  {
    save_datatype_specification();
    m_splitter->ReplaceWindow(m_splitter->GetWindow1(), m_glcanvas);
    m_glcanvas->Show();
    m_datatext->Hide();
    if ( m_glcanvas->get_diagram() && (m_glcanvas->get_diagram() != m_current_diagram || p_mode == GRAPE_MODE_PROC || p_mode == GRAPE_MODE_ARCH || old_mode == GRAPE_MODE_DATASPEC))
    {
      grape_event_select_diagram *event = new grape_event_select_diagram( this, m_glcanvas->get_diagram()->get_name() );
      m_event_handler->Submit( event, false );
    }
    m_current_diagram = m_glcanvas->get_diagram();
    if (m_current_diagram)
    {
      int pos = m_architecture_diagram_list->FindString(m_glcanvas->get_diagram()->get_name(), true);
      if ( pos != wxNOT_FOUND )
      {
        m_architecture_diagram_list->SetSelection( pos );
      }
      else
      {
        pos = m_process_diagram_list->FindString(m_glcanvas->get_diagram()->get_name(), true);
        m_process_diagram_list->SetSelection( pos );
      }
    }
  }
  wxCommandEvent event;
  update_statusbar(event);
}

void grape_frame::load_datatype_specification()
{
  if ( m_specification && !m_datatext->IsModified() )
  {
    datatype_specification *dts = m_specification->get_datatype_specification();

    if ( dts )
    {
      m_datatext->ChangeValue( dts->get_declarations() );
    }
  }
}

void grape_frame::save_datatype_specification()
{
  if ( m_specification )
  {
    datatype_specification *dts = m_specification->get_datatype_specification();

    if ( dts && m_datatext->IsModified() )
    {
      dts->set_declarations( m_datatext->GetValue() );
      m_datatext->SetModified( false );
    }
  }
}

void grape_frame::set_title()
{
  wxString title = m_filename.GetFullName();

  if ( !title.IsEmpty() )
  {
    if ( m_modified )
    {
      title += _T(" [modified]");
    }

    title += _T(" - ");
  }

  title += _T("GraPEmCRL2");

  SetTitle( title );
}

void grape_frame::set_mode( grape_mode p_mode )
{
  if ( ( p_mode == m_mode ) && ( p_mode != GRAPE_MODE_NONE ) )
  {
    return;
  }

  grape_mode old_mode = m_mode;
  m_mode = p_mode;

  switch( p_mode )
  {
    case GRAPE_MODE_NONE:
    {
      m_process_diagram_list->Select( wxNOT_FOUND );
      m_architecture_diagram_list->Select( wxNOT_FOUND );
      m_menubar->set_mode( grape_menubar::GRAPE_MENUMODE_NOSPEC );
      m_dataspecbutton->Enable( false );
      m_dataspecbutton->SetValue( false );
      break;
    }
    case GRAPE_MODE_SPEC:
    {
      m_process_diagram_list->Select( wxNOT_FOUND );
      m_architecture_diagram_list->Select( wxNOT_FOUND );
      m_menubar->set_mode( grape_menubar::GRAPE_MENUMODE_SPEC );
      m_dataspecbutton->Enable();
      m_dataspecbutton->SetValue( false );
      break;
    }
    case GRAPE_MODE_ARCH:
    {
      m_process_diagram_list->Select( wxNOT_FOUND );
      m_menubar->set_mode( grape_menubar::GRAPE_MENUMODE_SPEC | grape_menubar::GRAPE_MENUMODE_ARCH );
      m_dataspecbutton->Enable();
      m_dataspecbutton->SetValue( false );
      break;
    }
    case GRAPE_MODE_PROC:
    {
      m_architecture_diagram_list->Select( wxNOT_FOUND );
      m_menubar->set_mode( grape_menubar::GRAPE_MENUMODE_SPEC | grape_menubar::GRAPE_MENUMODE_PROC );
      m_dataspecbutton->Enable();
      m_dataspecbutton->SetValue( false );
      break;
    }
    case GRAPE_MODE_DATASPEC:
    {
      m_process_diagram_list->Select( wxNOT_FOUND );
      m_architecture_diagram_list->Select( wxNOT_FOUND );
      m_menubar->set_mode( grape_menubar::GRAPE_MENUMODE_DATASPEC );
      m_dataspecbutton->Enable();
      m_dataspecbutton->SetValue( true );
      break;
    }
    default:
    {
		// cannot be the case
      // assert( false );
    }
  }

  set_toolbar( p_mode );
  toggle_view( old_mode, p_mode );
}

grape_mode grape_frame::get_mode( void )
{
  return m_mode;
}

void grape_frame::update_bars( void )
{
  update_menubar();
  update_toolbar();
  wxCommandEvent event;
  update_statusbar( event );
}

void grape_frame::update_toolbar( void )
{
  // update SAVE for the toolbar
  GetToolBar()->EnableTool( wxID_SAVE, m_modified );

  // update UNDO and REDO right for the toolbar
  GetToolBar()->EnableTool( wxID_UNDO, m_menubar->IsEnabled( wxID_UNDO ) );
  GetToolBar()->SetToolShortHelp( wxID_UNDO, m_menubar->FindItem( wxID_UNDO )->GetLabelFromText( m_menubar->FindItem( wxID_UNDO )->GetLabel() ) );
  GetToolBar()->EnableTool( wxID_REDO, m_menubar->IsEnabled( wxID_REDO ) );
  GetToolBar()->SetToolShortHelp( wxID_REDO, m_menubar->FindItem( wxID_REDO )->GetLabelFromText( m_menubar->FindItem( wxID_REDO )->GetLabel() ) );

  // update DELETE and PROPERTIES right for the toolbar
  GetToolBar()->EnableTool( wxID_DELETE, m_menubar->IsEnabled( wxID_DELETE ) );
  GetToolBar()->EnableTool( GRAPE_MENU_PROPERTIES, m_menubar->IsEnabled( GRAPE_MENU_PROPERTIES ) );

  // update VALIDATE for the toolbar
  GetToolBar()->EnableTool( GRAPE_MENU_VALIDATE, m_menubar->IsEnabled( GRAPE_MENU_VALIDATE ) );
}

void grape_frame::update_menubar( void )
{
  // update SAVE for the menubar
  m_menubar->Enable( wxID_SAVE, m_modified );
  diagram *dia_ptr = m_glcanvas->get_diagram();
  if ( dia_ptr && !(m_mode & GRAPE_MODE_DATASPEC) )
  {
    bool object_selected = dia_ptr->count_selected_objects() > 0;
    wxString dia_name = dia_ptr->get_name();
    m_menubar->SetLabel( GRAPE_MENU_VALIDATE, wxString(_T("&Validate ")).Append(dia_name).Append(wxString(_T("\tF5"))) );
    m_menubar->SetHelpString( GRAPE_MENU_VALIDATE, _T("Validate diagram ") + dia_name );
    m_menubar->SetLabel( GRAPE_MENU_EXPORTMCRL2, wxString(_T("Export ")).Append(dia_name).Append(_T(" to &mCRL2...\tCtrl-E")) );
    m_menubar->SetHelpString( GRAPE_MENU_EXPORTMCRL2, wxString(_T("Export diagram ")).Append(dia_name).Append(_T(" to mCRL2")) );
    m_menubar->SetLabel( GRAPE_MENU_EXPORTIMAGE, wxString(_T("Export ")).Append(dia_name).Append(_T(" to &image...\tCtrl-I")) );
    m_menubar->SetHelpString( GRAPE_MENU_EXPORTIMAGE, wxString(_T("Export ")).Append(dia_name).Append(_T(" to image")) );
    m_menubar->Enable( GRAPE_MENU_VALIDATE, true );
    m_menubar->Enable( GRAPE_MENU_EXPORTMCRL2, true );
    m_menubar->Enable( GRAPE_MENU_EXPORTIMAGE, true );
    m_menubar->Enable( wxID_DELETE, object_selected );
    //m_menubar->Enable( GRAPE_MENU_DESELECT_ALL, object_selected );
    // m_menubar->Enable( GRAPE_MENU_SELECT_ALL, m_glcanvas->count_visual_object() > 1 ); // 1 because the visibility frame or the preamble are also visual objects, wich can not be selected
    m_menubar->Enable( GRAPE_MENU_PROPERTIES, dia_ptr->count_selected_objects() == 1 );
  }
  else if (m_mode & GRAPE_MODE_DATASPEC)
  {
    m_menubar->SetLabel( GRAPE_MENU_VALIDATE, _T("&Validate data type specification\tF5") );
    m_menubar->SetHelpString( GRAPE_MENU_VALIDATE, _T("Validate data type specification") );
    m_menubar->SetLabel( GRAPE_MENU_EXPORTMCRL2, _T("Export data type specification to &mCRL2...\tCtrl-E") );
    m_menubar->SetHelpString( GRAPE_MENU_EXPORTMCRL2, _T("Export data type specification to mCRL2") );
    m_menubar->SetLabel( GRAPE_MENU_EXPORTIMAGE, _T("Export to &image...\tCtrl-I") );
    m_menubar->SetHelpString( GRAPE_MENU_EXPORTIMAGE, _T("Export to image") );
    m_menubar->Enable( GRAPE_MENU_VALIDATE, true );
    m_menubar->Enable( GRAPE_MENU_EXPORTMCRL2, true );
    m_menubar->Enable( GRAPE_MENU_EXPORTIMAGE, false );
    m_menubar->Enable( wxID_DELETE, false );
    //m_menubar->Enable( GRAPE_MENU_DESELECT_ALL, false );
    //m_menubar->Enable( GRAPE_MENU_SELECT_ALL, false );
    m_menubar->Enable( GRAPE_MENU_PROPERTIES, false );
  }
  else if (m_mode & GRAPE_MODE_SPEC)
  {
    m_menubar->SetLabel( GRAPE_MENU_VALIDATE, _T("&Validate\tF5") );
    m_menubar->SetHelpString( GRAPE_MENU_VALIDATE, _T("Validate") );
    m_menubar->SetLabel( GRAPE_MENU_EXPORTMCRL2, _T("Export to &mCRL2...\tCtrl-E") );
    m_menubar->SetHelpString( GRAPE_MENU_EXPORTMCRL2, _T("Export diagram to mCRL2") );
    m_menubar->SetLabel( GRAPE_MENU_EXPORTIMAGE, _T("Export to &image...\tCtrl-I") );
    m_menubar->SetHelpString( GRAPE_MENU_EXPORTIMAGE, _T("Export to image") );
    m_menubar->Enable( GRAPE_MENU_VALIDATE, false );
    m_menubar->Enable( GRAPE_MENU_EXPORTMCRL2, false );
    m_menubar->Enable( GRAPE_MENU_EXPORTIMAGE, false );
    m_menubar->Enable( wxID_DELETE, false );
    m_menubar->Enable( GRAPE_MENU_PROPERTIES, false );
  }
}

void grape_frame::update_statusbar( wxCommandEvent& )
{
  if ( m_mode == GRAPE_MODE_DATASPEC )
  {
    m_statusbar->SetStatusText( _T("In the text field you can enter a data type specification") );
  }
  else
  {
    if ( ( m_mode == GRAPE_MODE_ARCH ) || ( m_mode == GRAPE_MODE_PROC ) )
    {
      canvas_state state = m_glcanvas->get_canvas_state();
      wxString status_text;
      switch( state )
      {
        case SELECT:
          status_text = _T("Click -> select object. Drag -> move object. Drag border -> resize object. Double click -> edit object properties."); break;
        case ADD_COMMENT:
          status_text = _T("Click to add a comment"); break;
        case ADD_TERMINATING_TRANSITION:
          status_text = _T("Drag from a state to add a terminating transition"); break;
        case ADD_NONTERMINATING_TRANSITION:
          status_text = _T("Drag from a beginstate to an endstate to add a transition"); break;
        case ADD_INITIAL_DESIGNATOR:
          status_text = _T("Click a state to add an initial designator"); break;
        case ADD_STATE:
          status_text = _T("Click to add a state"); break;
        case ADD_REFERENCE_STATE:
        case ADD_PROCESS_REFERENCE:
          status_text = _T("Click to add a process reference"); break;
        case ADD_ARCHITECTURE_REFERENCE:
          status_text = _T("Click to add an architecture reference"); break;
        case ADD_CHANNEL:
          status_text = _T("Click on a reference to add a channel"); break;
        case ADD_CHANNEL_COMMUNICATION:
          status_text = _T("Drag from a channel to another channel to add a channel communication"); break;
        case IDLE:
        default: status_text = wxEmptyString; break;
      }
      if ( m_statusbar->GetStatusText() != _T("Click to select. Double click -> Rename current diagram. Press Delete -> Remove current diagram.") )
      {
        m_statusbar->SetStatusText( status_text );
      }
    }
  }
}
