/*
 * semantic.c
 *
 * Copyright 1998, 1999 Michael Elizabeth Chastain, <mailto:mec@shout.net>.
 * Licensed under the Gnu Public License, Version 2.
 *
 * Semantic functions for block, expr, word.
 */

#include <ctype.h>

#include "mconfig.h"



/*
 * Interpret a block down to the command level, passing the commands to
 * a callback function.  My major contribution to value is interpreting
 * verb_IF.
 */
void 
block_walk(const block_type * block, int menu_descend,
	   fn_scb_type * fn, void *param)
{
	const statement_type *statement;

	for (statement = block->first; statement != NULL; statement = statement->next) {
		switch (statement->verb) {
		default:
		case verb_NULL:
			error_enum_bogus();
			break;

		case verb_IF:
			if (expr_is_true(statement->sc_condition)) {
				block_walk(statement->sc_block_left, menu_descend, fn, param);
			} else {
				if (statement->sc_block_right != NULL)
					block_walk(statement->sc_block_right, menu_descend, fn, param);
			}
			break;

		case verb_MENU:
			(*fn) (statement, param);
			if (menu_descend)
				block_walk(statement->sc_block_left, menu_descend, fn, param);
			break;

		case verb_mainmenu_name:
		case verb_comment:
		case verb_text:
		case verb_unset:
		case verb_ask_bool:
		case verb_ask_hex:
		case verb_ask_int:
		case verb_ask_string:
		case verb_ask_tristate:
		case verb_def_bool:
		case verb_def_hex:
		case verb_def_int:
		case verb_def_string:
		case verb_def_tristate:
		case verb_dep_bool:
		case verb_dep_mbool:
		case verb_dep_hex:
		case verb_dep_int:
		case verb_dep_string:
		case verb_dep_tristate:
		case verb_nchoice:
			(*fn) (statement, param);
			break;
		}
	}
}



/*
 * Return whether an expression is true.
 */
int 
expr_is_true(const expr_type * expr)
{
	switch (expr->op) {
	default:
		error_enum_bogus();
		return 0;

	case op_word:
		{
			piece_type      piece = word_eval(expr->word_left, 0);
			return piece.len > 0;
		}

	case op_equal:
	case op_not_equal:
		{
			piece_type      piece_left = word_eval(expr->word_left, 0);
			piece_type      piece_right = word_eval(expr->word_right, 0);
			/* take a deep breath */
			return (piece_left.len == piece_right.len && memcmp(piece_left.ptr, piece_right.ptr, piece_left.len) == 0) == (expr->op == op_equal);
		}

	case op_and:
		return expr_is_true(expr->expr_left) && expr_is_true(expr->expr_right);

	case op_or:
		return expr_is_true(expr->expr_left) || expr_is_true(expr->expr_right);

	case op_not:
		return !expr_is_true(expr->expr_left);
	}
}



/*
 * Return whether a statement allows a given input character.
 * This is for the string-oriented statments (hex, int, string).
 */
int 
statement_allow_char(const statement_type * statement, int c)
{
	switch (statement->verb) {
	default:
		error_enum_bogus();
		return 0;

	case verb_ask_hex:
	case verb_def_hex:
	case verb_dep_hex:
		return isxdigit(c);

	case verb_ask_int:
	case verb_def_int:
	case verb_dep_int:
		if (statement->sb_symbol->value.len == 0) {
			if (c == '0')
				return 0;
			if (c == '-')
				return 1;
		}
		return isdigit(c);

	case verb_ask_string:
	case verb_def_string:
	case verb_dep_string:
		if (c == '"' || c == '\\')
			return 0;
		return isprint(c);
	}
}



/*
 * Return the allowable values for a bool or tristate.
 *
 * The values are a bit mask of the following conditions:
 *
 *   0x01: "n"
 *   0x02: "m"
 *   0x04: "y"
 *
 * The values come from the inherent nature of the verb, the value of
 * CONFIG_MODULES, and the dependency list of this statement.
 *
 * For hex, int, and string, this method still returns a boolean
 * value, which depends on CONFIG_MODULES and the dependency list.
 */
int 
statement_bools_allowed(const statement_type * statement)
{
	int             allowed;

	/*
         * FIXME: verb_define_bool=m is a compatibility hack.
         */

	/* allow all */
	allowed = 0x1 | 0x2 | 0x4;

	/* booleans can't be m */
	if (statement->verb == verb_ask_bool || statement->verb == verb_dep_bool ||
	    statement->verb == verb_dep_mbool) {
		allowed &= ~0x02;
	} else {
		/* check $CONFIG_MODULES */
		symbol_type    *symbol;
		piece_type      name;
		piece_type      value;

		name.ptr = "CONFIG_MODULES";
		name.len = 14;
		symbol = symbol_lookup(name);
		value = symbol ? symbol->value : piece_empty;
		if (value.len == 1 && value.ptr[0] == 'y');
		else
			allowed &= ~0x02;
	}

	/* walk the dependency list */
	if (statement->sb_dep_list != NULL) {
		const word_type *word;
		for (word = statement->sb_dep_list->first; word != NULL; word = word->next) {
			piece_type      value = word_eval(word, 0);
			if (value.len == 1 && value.ptr[0] == 'y');
			else if (value.len == 1 && value.ptr[0] == 'm') {
				if (statement->verb == verb_dep_mbool);
				else
					allowed &= ~0x04;
			} else
				allowed &= ~(0x04 | 0x02);
		}
	}
	/* that's all, folks */
	return allowed;
}



/*
 * Evaluate the string value of a word and return it.
 * Literal words have a precomputed value.
 * More complex words compute the value.
 * Caller specifies whether computed values are grabbed or short-lived.
 */
piece_type 
word_eval(const word_type * word, int grab)
{
	if (word->literal.len >= 0) {
		return word->literal;
	} else {
		static piece_type synthetic[16] =
		{
			{NULL, 0}, {NULL, 0}, {NULL, 0}, {NULL, 0},
			{NULL, 0}, {NULL, 0}, {NULL, 0}, {NULL, 0},
			{NULL, 0}, {NULL, 0}, {NULL, 0}, {NULL, 0},
			{NULL, 0}, {NULL, 0}, {NULL, 0}, {NULL, 0},
		};
		static int      current = 0;

		const atom_type *atom;
		piece_type      piece_ret;
		int             size;

		/* calculate size */
		size = 0;
		for (atom = word->atom_first; atom != NULL; atom = atom->next) {
			if (atom->dollar) {
				symbol_type    *symbol = symbol_lookup(atom->lexeme.piece);
				piece_type      piece = symbol ? symbol->value : piece_empty;
				if (piece.len >= 0)
					size += piece.len;
#if 0
				else
					fprintf(stderr, "word_eval: undefined atom: $%.*s\n",
					  atom->value.len, atom->value.ptr);
#endif
			} else {
				size += atom->lexeme.piece.len;
			}
		}

		/* allocate some memory somewhere */
		if (grab) {
			piece_ret.ptr = grab_memory(size);
			piece_ret.len = size;
		} else {
			current = (current + 1) % (sizeof(synthetic) / sizeof(synthetic[0]));
			if (synthetic[current].len < size) {
				if (synthetic[current].ptr != NULL)
					free((char *) synthetic[current].ptr);
				synthetic[current].ptr = check_malloc(size);
				synthetic[current].len = size;
			}
			piece_ret.ptr = synthetic[current].ptr;
			piece_ret.len = size;
		}

		/* construct value */
		size = 0;
		for (atom = word->atom_first; atom != NULL; atom = atom->next) {
			piece_type      piece = atom->lexeme.piece;

			if (atom->dollar) {
				symbol_type    *symbol = symbol_lookup(atom->lexeme.piece);
				piece = symbol ? symbol->value : piece_empty;
			}
			if (piece.len >= 0)
				memcpy((char *) piece_ret.ptr + size, piece.ptr, piece.len);
			size += piece.len;
		}

		/* that's all, folks */
		return piece_ret;
	}
}
