/*
 *    Copyright (C) 1999-2002 Stijn van Dongen.
*/

#include "ops.h"

#include <ctype.h>
#include <stdlib.h>

#include "util.h"
#include "digest.h"
#include "key.h"
#include "filter.h"
#include "file.h"
#include "segment.h"
#include "parse.h"
#include "ref.h"
#include "read.h"
#include "env.h"
#include "curly.h"
#include "counter.h"
#include "constant.h"

#include "util/txt.h"
#include "util/hash.h"
#include "util/file.h"



#define P_DEF2       "{kd}{cx}         (complain if key exists)"
#define P_SET2       "{kd}{cx}         (do not complain if key exists)"
#define P_APPLY2     "{kd|ak}{v}       (apply key to vararg)"
#define P_WHILE2     "{x}{x}           [eval arg2 while eval arg1 nonzero]"
#define P_TABLE5     "{x}{x}{x}{x}{v}  [row -len -lft -sep -rgt, data]"
#define P_ENV3       "{l}{cx}{cx}      [begin, end]"
#define P_BEGIN1     "{l}              (begin key env)"
#define P_END1       "{l}              (end key env)"

#define P_DIGEST2    "{kn}{x}          (eval x, store in zero-arg key)"
#define P_WRITE3     "{x}{s}{x}        [fname, filter name, expression]"

#define P_DOFILE2    "{x}{aa[!?][+-]}  [fname, mode (existence+output)]"
#define P_STORE2  "{kn}{x}          [zero-arg key, file name] (store content)"

#define P_REF2    "{s}{a[ntlcm]}    [ref, id(Number|Type|Level|Caption|Misc)]"
#define P_REFLOAD6   "{s}{i}{s}{s}{s}{s}   [ref, lev, typ, num, title, misc]"

#define P_QUIT0      "(quit parsing current stack in current file)"
#define P_EXIT0      "(goodbye world)"
#define P_CTRSET2    "{l}{x}           [label, integer]"
#define P_CTRPUT1    "{l}              [label]"
#define P_CTRADD2    "{l}{x}           [label, integer]"
#define P_ISUM1      "{v}              [constituents to be summed]"
#define P_ICLC3      "{a[+-/%*]}{x}{x} [operator, integer, integer]"
#define P_ICMP5      "{x}{x}{x}{x}{x}  [int, int, case lt, case eq, case gt]"

#define P_IFEQ4      "{x}{x}{x}{x}     [op1, op2, case eq, case ne]"
#define P_IFDEF3     "{kd}{x}{x}       [key, case def, case not def]"
#define P_SWITCH2    "{x}{v{<{x}{x}>+{x}?>} [pivot, [case, eval]+ finish]"

#define P_SPECIAL1   "{v<{i}{zc}>+}    [ascii num, translation]+"
#define P_CONSTANT1  "{v<{l}{zc}>+}    [label (used as \\*l*), translation]+"

#define P_DOLLAR2    "{s}{x}           [device, eval if \\device is device]"
#define P_FORMATTED1 "{cx}             [remove ws in x, translate \\%nst%]"
#define P_META2      "{kn}{v{x}+}      [write key call with kn and args]"

#define F_DEVICE     "device filter (customize with \\special#1)"
#define F_TXT        "interprets [\\\\][\\~][\\,][\\|][\\}][\\{]"
#define F_COPY       "identity filter (literal copy)"

#define  READ_INPUT     1           /* uses default file & filter */
#define  READ_IMPORT    2           /* interpretation only        */
#define  READ_READ      4           /* uses default file & filter */
#define  READ_LOAD      8           /* interpretation only        */

const char *legend_g
=
"L e g e n d\n"
"Ab   Meaning              Examples/explanation\n"
"--| |------------------| |---------------------------------------------------|\n"
"kn   key name             \\foo, \\\"foo::bar-zut\"              \n"
"kd   key descriptor       \\foo#2, \\\"+\"#2                     \n"
"ak   anonymous key        \\_#2{foo{\\1}\\bar{\\1}{\\2}}         \n"
"zl   zoem literal         contains only escapes \\\\ \\} \\{ \\, \\| \\~ \\-\n"
"zc   zoem constant        may also contain \\@{..} (at scope), \\*..* (constants)\n"
"x    expression           anything goes, evaluated by primitive  \n"
"cx   constant expression  is not evaluated by the primitive      \n"
"l    label                name spaces for counters, refs, constants, and env\n"
"a    character            presumably a switch of some kind       \n"
"v    vararg               of the form {..} {..} .. {..} (white space allowed)\n"
"cv   constant vararg      is not evaluated by the primitve       \n"
"i    integer              e.g. '123', '-6', no arithmetic expressions allowed\n"
"s    string                                                      \n"
;

static   mcxHash*    yamTable_g        =  NULL;    /* primitives        */
static   mcxTing*    devtxt_g          =  NULL;    /* "\\device"        */
static   int         insertWarn_g      =  1;

const char *strComposites
=
   "\\def{\\input#1}{\\dofile{\\1}{!+}}\n"
   "\\def{\\import#1}{\\dofile{\\1}{!-}}\n"
   "\\def{\\read#1}{\\dofile{\\1}{?+}}\n"
   "\\def{\\load#1}{\\dofile{\\1}{?-}}\n"
   "\\def{\\refcaption#1}{\\ref{\\1}{c}}\n"
   "\\def{\\refnumber#1}{\\ref{\\1}{n}}\n"
   "\\def{\\reflevel#1}{\\ref{\\1}{l}}\n"
   "\\def{\\refmisc#1}{\\ref{\\1}{m}}\n"
   "\\def{\\reftype#1}{\\ref{\\1}{t}}\n"
   "\\def{\\string#1}{\\ifdef{\\1}{\\1}{}}\n"
   "\\def{\\inc#1}{\\ctradd{\\1}{1}}\n"
   "\\def{\\ctr#1}{\\ctrput{\\1}}\n"
   "\\def{\\ctr#2}{\\ctrset{\\1}{\\2}}\n"
   "\\def{\\\"+\"#2}{\\iclc{+}{\\1}{\\2}}\n"
   "\\def{\\\"-\"#2}{\\iclc{-}{\\1}{\\2}}\n"
   "\\def{\\\"/\"#2}{\\iclc{/}{\\1}{\\2}}\n"
   "\\def{\\\"%\"#2}{\\iclc{%}{\\1}{\\2}}\n"
   "\\def{\\\"*\"#2}{\\iclc{*}{\\1}{\\2}}\n"
;


typedef struct
{
   const char*       name
;  const char*       descr
;  yamSeg*           (*yamfunc)(yamSeg* seg)
;
}  cmdHook           ;


static   cmdHook     cmdHookDir[]      =  
{  {  "\\def#2"      ,  P_DEF2         ,  expandDef2        }
,  {  "\\set#2"      ,  P_SET2         ,  expandSet2        }
,  {  "\\dofile#2"   ,  P_DOFILE2      ,  expandDofile2     }
,  {  "\\store#2"    ,  P_STORE2       ,  expandStore2      }
,  {  "\\digest#2"   ,  P_DIGEST2      ,  expandDigest2     }
,  {  "\\write#3"    ,  P_WRITE3       ,  expandWrite3      }
,  {  "\\ifeq#4"     ,  P_IFEQ4        ,  expandIfeq4       }
,  {  "\\switch#2"   ,  P_SWITCH2      ,  expandSwitch2     }
,  {  "\\ifdef#3"    ,  P_IFDEF3       ,  expandIfdef3      }
,  {  "\\special#1"  ,  P_SPECIAL1     ,  expandSpecial1    }
,  {  "\\constant#1" ,  P_CONSTANT1    ,  expandConstant1   }
,  {  "\\$#2"        ,  P_DOLLAR2      ,  expandDollar2     }
,  {  "\\refload#6"  ,  P_REFLOAD6     ,  expandRefload6    }
,  {  "\\ref#2"      ,  P_REF2         ,  expandRef2        }
,  {  "\\ctrset#2"   ,  P_CTRSET2      ,  expandCtrset2     }
,  {  "\\ctrput#1"   ,  P_CTRPUT1      ,  expandCtrput1     }
,  {  "\\ctradd#2"   ,  P_CTRADD2      ,  expandCtradd2     }
,  {  "\\iclc#3"     ,  P_ICLC3        ,  expandIclc3       }
,  {  "\\isum#1"     ,  P_ISUM1        ,  expandIsum1       }
,  {  "\\icmp#5"     ,  P_ICMP5        ,  expandIcmp5       }
,  {  "\\apply#2"    ,  P_APPLY2       ,  expandApply2      }
,  {  "\\while#2"    ,  P_WHILE2       ,  expandWhile2      }
,  {  "\\table#5"    ,  P_TABLE5       ,  expandTable5      }
,  {  "\\env#3"      ,  P_ENV3         ,  expandEnv3        }
,  {  "\\begin#1"    ,  P_BEGIN1       ,  expandBegin1      }
,  {  "\\end#1"      ,  P_END1         ,  expandEnd1        }
,  {  "\\formatted#1",  P_FORMATTED1   ,  expandFormatted1  }
,  {  "\\meta#2"     ,  P_META2        ,  expandMeta2       }
,  {  "\\quit"       ,  P_QUIT0        ,  expandQuit0       }
,  {  "\\exit"       ,  P_EXIT0        ,  expandExit0       }
,  {  NULL           ,  NULL           ,  NULL              }
}  ;


mcxbool  yamOpList
(  const char* mode
)
   {
      cmdHook*    cmdhook     =  cmdHookDir
   ;  mcxbool     listAll     =  strstr(mode, "all") != NULL
   ;  mcxbool     match       =  listAll || 0

   ;  if (listAll || strstr(mode, "zoem"))
      {  while (cmdhook && cmdhook->name)
         {  fprintf(stdout, "%-15s %s\n", cmdhook->name, cmdhook->descr)
         ;  cmdhook++
      ;  }
      ;  if (!strstr(mode, "legend"))
         fprintf(stdout, "Additionally supplying \"-l legend\" prints legend\n")
      ;  match = 1
   ;  }

   ;  if (listAll || strstr(mode, "legend"))
      {  fprintf(stdout, "\n%s", legend_g)
      ;  match = 1
   ;  }

   ;  if (listAll || strstr(mode, "macro"))
      {  fprintf
         (stdout, "\nBuilt-in aliases and macro's\n%s", strComposites)
      ;  match = 1
   ;  }

   ;  return match ? TRUE : FALSE
;  }


/*
 *    if file can not be opened, sets filetxt->str to empty string
 *    and return STATUS_FAIL
*/

mcxstatus readFile
(  mcxTing*  fname
,  mcxTing*  filetxt
)  ;


yamSeg* expandStore2
(  yamSeg*  seg
)
   {
      mcxTing*  key         =  mcxTingNew(arg1_g->str)
   ;  mcxTing*  fname       =  mcxTingNew(arg2_g->str)
   ;  mcxTing*  filetxt     =  mcxTingEnsure(NULL, 100)

   ;  if (checkusrname(key->str, key->len) != key->len)
      yamExit("\\store#2", "<%s> is not a valid key", key->str)

   ;  yamDigest(fname, fname)
   ;  readFile(fname, filetxt)

   ;  yamKeyInsert(key, filetxt->str)
   ;  mcxTingFree(&filetxt)

   ;  return(seg)
;  }


yamSeg* expandDofile2
(  yamSeg*  seg
)
   {
      mcxTing*  fname       =  mcxTingNew(arg1_g->str)
   ;  mcxTing*  opts        =  arg2_g
   ;  mcxTing*  filetxt     =  mcxTingEnsure(NULL, 100)

   ;  int            mode
   ;  yamFilterData*   fd
   ;  fltfnc         flt

   ;  if (opts->len != 2)
      {  expandDofile2die:
         yamExit("\\dofile#2", "Second arg <%s> not in {!?}x{+-}", opts->str)
   ;  }

   ;  if (*(opts->str+0) == '!')
      {  if (*(opts->str+1) == '+')
         mode = READ_INPUT
      ;  else if (*(opts->str+1) == '-')
         mode = READ_IMPORT
      ;  else
         goto expandDofile2die;
   ;  }
      else if (*(opts->str+0) == '?')
      {  if (*(opts->str+1) == '+')
         mode = READ_READ
      ;  else if (*(opts->str+1) == '-')
         mode = READ_LOAD
      ;  else
         goto expandDofile2die;
   ;  }
      else
      goto expandDofile2die
      ;

   ;  fd    =     mode & (READ_INPUT | READ_READ)
               ?  yamFilterGetDefaultFd()
               :  NULL
   ;  flt   =     mode & (READ_INPUT | READ_READ)
               ?  yamFilterGetDefaultFilter()
               :  NULL

   ;  if (!yamInputCanPush())
      yamExit
      (  "\\dofile#2", "maximum file include depth (9) reached\n"
         "___ when presented with file <%s>"
      ,  fname->str
      )

   ;  yamDigest(fname, fname)

   ;  if (readFile(fname, filetxt) != STATUS_OK)
      {
         if (mode & (READ_INPUT | READ_IMPORT))
         yamExit("\\dofile#2","failed to open file argument <%s>\n",fname->str)

      ;  else                             /* READ or LOAD */
         {  mcxTingFree(&filetxt)
         ;  mcxTingFree(&fname)
         ;  return seg
      ;  }
   ;  }

   ;  yamInputPush(fname->str, filetxt)
   ;  mcxTingFree(&fname)
   ;  yamOutput(filetxt, flt, fd)
   ;  mcxTingFree(&filetxt)
   ;  yamInputPop()

   ;  return(seg)
;  }


yamSeg* expandMeta2
(  yamSeg*  seg
)
   {
      mcxTing*  key         =  mcxTingNew(arg1_g->str)
   ;  mcxTing*  args        =  mcxTingNew(arg2_g->str)
   ;  mcxTing*  bs          =  mcxTingNew("\\")
   ;  yamSeg *seg1, *seg2, *seg3

   ;  if (checkusrname(key->str, key->len) != key->len)
      yamExit
      (  "\\meta#2"
      ,  "key <%s> is not of the right \\foo, \\\"foo::foo\", or \\$foo\n"
      ,  key->str
      )

   ;  seg3 = yamSegPush(seg, args)
   ;  seg2 = yamSegPush(seg3, key)
   ;  seg2->offset = 1                 /*  prints the foo part */
   ;  seg1 = yamSegPush(seg2, bs)      /*  prints a single backslash */
   ;  return seg1
;  }


yamSeg* expandWhile2
(  yamSeg*  seg
)
   {  mcxTing*  condition  =  mcxTingNew(arg1_g->str)
   ;  mcxTing*  data       =  mcxTingNew(arg2_g->str)
   ;  mcxTing*  condition_ =  mcxTingEmpty(NULL, 10)
   ;  mcxTing*  data_      =  mcxTingEmpty(NULL, 10)
   ;  mcxTing*  newtxt     =  mcxTingEmpty(NULL, 10)
   ;
      while(1)
      {
         mcxTingWrite(condition_, condition->str)
      ;  yamDigest(condition_, condition_)
      ;
         if (atoi(condition_->str))
         {  mcxTingWrite(data_, data->str)
         ;  yamDigest(data_, data_)
         ;  mcxTingAppend(newtxt, data_->str)  
      ;  }
         else
         {  mcxTingFree(&condition_)
         ;  break
      ;  }
   ;  }

      mcxTingFree(&data)
   ;  mcxTingFree(&data_)
   ;  mcxTingFree(&condition)
   ;  mcxTingFree(&condition_)
   ;  return yamSegPush(seg, newtxt)
;  }


yamSeg* expandApply2
(  yamSeg*  seg
)
   {
      mcxTing *key          =  mcxTingNew(arg1_g->str)
   ;  mcxTing *data         =  mcxTingNew(arg2_g->str)
   ;  mcxTing *newtxt       =  mcxTingEmpty(NULL, 10)
   ;  yamSeg *tblseg

   ;  int   x, k
   ;  int   keylen         =  checkusrkey(key->str, key->len, &k)
   ;  int   namelen        =  checkusrname(key->str, key->len)

   ;  if (k<0 || k > 9)
      yamExit
      (  "\\apply#2"
      ,  "loop number <%d> not in [1,9] for key <%s>"
      ,  k
      ,  key->str
      )
   ;  else if (k ==0)
      yamExit
      (  "\\apply#2"
      ,  "remapping functionality not yet implemented"
      )

   ;  if (namelen == 2 && *(key->str+1) == '_')      /* anonymous key */
      {
         mcxTing *ankey, *anval
      ;  int cc            =  closingcurly(key, keylen, NULL, EXIT_ON_FAIL)

;  if (0) fprintf(stderr, ">>> anonymous key <%s>\n", key->str)

      ;  if (cc+keylen+1 != key->len)
         yamExit("\\apply#2", "anonymous key <%s> not ok", key->str)

      ;  ankey             =  mcxTingNNew(key->str, keylen)
      ;  anval             =  mcxTingNNew(key->str+keylen+1, cc-1)
      ;  mcxTingShrink(key, keylen)

;  if (0)
{  fprintf(stderr, ">>> mapped an key <%s> to <%s>\n", ankey->str, anval->str)
;  fprintf(stderr, ">>> wrote key <%s>\n", key->str)
;
}
      ;  if (yamKeyInsert(ankey, anval->str) != ankey)
         mcxTingFree(&ankey)

      ;  mcxTingFree(&anval)
   ;  }

      else if (keylen != key->len)
      {
         yamExit
         (  "\\apply#2"
         ,  "key <%s> is not of the right \\foo, \\\"foo::foo\", and \\$foo\n"
         ,  key->str
         )
   ;  }

   ;  tblseg = yamSegPush(seg, data)

   /* perhaps this block should be encapsulated by parse.c
    * pity we have expandkey here.
    */
   ;  while ((x = parsescopes(tblseg, k)) == k)
      {
         yamSeg*  rowseg

      ;  mcxTingWrite(key_g, key->str)
      ;  rowseg = expandkey(seg)
      ;  mcxTingAppend(newtxt,rowseg->txt->str)
      ;  yamSegFree(&rowseg)
   ;  }

   ;  mcxTingFree(&key)
   ;  yamSegFree(&tblseg)

   ;  return yamSegPush(seg, newtxt)
;  }



/*
*/

yamSeg* expandTable5
(  yamSeg*  seg
)
   {  mcxTing *txtnum    =  mcxTingNew(arg1_g->str)
   ;  mcxTing *txtlft    =  mcxTingNew(arg2_g->str)
   ;  mcxTing *txtmdl    =  mcxTingNew(arg3_g->str)
   ;  mcxTing *txtrgt    =  mcxTingNew(arg4_g->str)
   ;  mcxTing *data      =  mcxTingNew(arg5_g->str)
   ;  yamSeg *tmpseg

   ;  mcxTing *txtall    =  mcxTingEnsure(NULL, 100)

   ;  int  x, k

   ;  yamDigest(data, data)
   ;  yamDigest(txtnum, txtnum)
   ;  k = atoi(txtnum->str)

   ;  if (k<=0)
      yamExit("\\table#5", "nonpositive loop number <%d>\n", k)

   ;  tmpseg = yamSegPush(seg, data)

   ;  while ((x = parsescopes(tmpseg, k)) == k)
      {
         int i
      ;  mcxTingAppend(txtall, txtlft->str)
      ;  for (i=1;i<k;i++)
         {  mcxTingAppend(txtall, (key_and_args_g+i)->str)
         ;  mcxTingAppend(txtall, txtmdl->str)
      ;  }
      ;  mcxTingAppend(txtall, (key_and_args_g+k)->str)
      ;  mcxTingAppend(txtall, txtrgt->str)
   ;  }
   ;  mcxTingFree(&txtnum)
   ;  mcxTingFree(&txtlft)
   ;  mcxTingFree(&txtmdl)
   ;  mcxTingFree(&txtrgt)

   ;  yamSegFree(&tmpseg)
   ;  return yamSegPush(seg, txtall)
;  }


yamSeg* expandFormatted1
(  yamSeg*  seg
)
   {
      mcxTing*  txt         =  mcxTingNew(arg1_g->str)
   ;  char* o              =  txt->str
   ;  char* p              =  o
   ;  char* q              =  o
   ;  char* z              =  o + txt->len
   ;  mcxbool formatting   =  TRUE

   ;  int   esc            =  0

   ;  while (p < z)
      {
         if (esc)
         {
            if (*p == '@')
            {  int l    =  closingcurly(txt, p+1-o, NULL, EXIT_ON_FAIL)
            ;  *(q++)   =  '\\'
            ;  *(q++)   =  '@'
            ;  while (l-- && ++p)
               *(q++)   =  *p
            ;  *(q++)   =  *++p
         ;  }
            else if (*p == '%')
            {  while (p<z && *(++p) != '%')
               {  switch(*p)
                  {  case 's' : *(q++) = ' ';  break
                  ;  case 'n' : *(q++) = '\n'; break
                  ;  case 't' : *(q++) = '\t'; break
                  ;  case '<' : formatting = FALSE ; break
                  ;  case '>' : formatting = TRUE ; break
                  ;  default  :
                     yamExit("\\formatted1", "illegal character <%c>", *p)
               ;  }
            ;  }
            ;  if (p == z)
               yamExit("\\formatted1", "missing ']'")
         ;  }
            else if (*p == ':')
            {  do { p++; } while (p<z && *p != '\n')
         ;  }
            else
            {  *(q++) = '\\'
            ;  *(q++) = *p
         ;  }

            esc   =  0
      ;  }
         else if (formatting && isspace(*p))
      ;  else if (*p == '\\')
         esc = 1
      ;  else
         *(q++) = *p

      ;  p++
   ;  }
   ;  *q = '\0'
   ;  txt->len = q-o
   ;  return yamSegPush(seg, txt)
;  }


/*
 *    we have to take care what happens when \digest{\blah}{\blah}
*/

yamSeg* expandDigest2
(  yamSeg*  seg
)
   {
      mcxTing*  key         =  mcxTingNew(arg1_g->str)
   ;  mcxTing*  yamtxt      =  mcxTingNew(arg2_g->str)

   ;  if (checkusrname(key->str, key->len) != key->len)
      yamExit("\\digest#2", "<%s> is not a valid key\n", key->str)

   ;  yamDigest(yamtxt, yamtxt)

   ;  if (yamKeyInsert(key, yamtxt->str) != key)     /* overwriting key */
      mcxTingFree(&key)

   ;  mcxTingFree(&yamtxt)
   ;  return(seg)
;  }


yamSeg* expandWrite3
(  yamSeg*  seg
)
   {
      mcxTing*    fname    =  mcxTingNew(arg1_g->str)
   ;  mcxTing*    yamtxt   =  mcxTingNew(arg3_g->str)
   ;  fltfnc      filter   =  yamFilterGet(arg2_g)
   ;  mcxIOstream *xfout

   ;  if (!filter)
      yamExit("\\write#3", "filter <%s> not found\n", arg2_g->str)

   ;  yamDigest(fname, fname)
   ;  xfout =  yamOutputNew(fname->str)
   ;  mcxTingFree(&fname)

   ;  yamOutput(yamtxt, filter, (yamFilterData*) xfout->ufo)

   ;  return(seg)
;  }


yamSeg* expandDollar2
(  yamSeg*  seg
)
   {
      yamSeg*  newseg      =  NULL
   ;  mcxTing*  device     =  yamKeyGet(devtxt_g)

   ;  if (!device)
      yamExit
      (  "\\$"
      ,  "key [\\device] not defined, rendering use of <%s> useless\n"
      ,  key_g->str
      )
   ;  else
      {
         if (!strcmp(device->str, arg1_g->str))      /* skip '\$'  */
         {
            mcxTing* txt    =  mcxTingNew(arg2_g->str)
         ;  newseg          =  yamSegPush(seg, txt)
      ;  }
         else
         {  newseg          =  seg
      ;  }
   ;  }
   ;  return newseg
;  }


yamSeg* expandIfdef3
(  yamSeg*  seg
)
   {
      mcxTing* val
   ;  mcxTing* yamtxt

   ;  if (checkusrkey(arg1_g->str, arg1_g->len, NULL) != arg1_g->len)
      yamExit
      (  "\\ifdef#3"
      ,  "first argument <%s> is not a valid key signature"
      ,  arg1_g->str
      )

   ;  val      =  yamKeyGet(arg1_g)
   ;  yamtxt   =     val
                  ?  mcxTingNew(arg2_g->str)
                  :  mcxTingNew(arg3_g->str)

   ;  return yamSegPush(seg, yamtxt)
;  }


yamSeg* expandIfeq4
(  yamSeg*  seg
)
   {
      mcxTing*  op1         =  mcxTingNew(arg1_g->str)
   ;  mcxTing*  op2         =  mcxTingNew(arg2_g->str)
   ;  mcxTing*  yamtxt      =  NULL

   ;  yamDigest(op1, op1)
   ;  yamDigest(op2, op2)

   ;  if (!strcmp(op1->str, op2->str))
      yamtxt               =  mcxTingNew(arg3_g->str)
   ;  else
      yamtxt               =  mcxTingNew(arg4_g->str)

   ;  mcxTingFree(&op1)
   ;  mcxTingFree(&op2)

   ;  return yamSegPush(seg, yamtxt)
;  }


yamSeg* expandSwitch2
(  yamSeg*  seg
)
   {
      mcxTing*  keytxt      =  mcxTingNew(arg1_g->str)     
   ;  mcxTing*  chktxt      =  mcxTingNew(arg2_g->str)     
   ;  mcxTing*  tsttxt      =  mcxTingEnsure(NULL, 30)
   ;  mcxTing*  yamtxt      =  NULL
   ;  int   x

   ;  yamSeg*  tmpseg      =  yamSegPush(seg, chktxt)
   ;  yamDigest(keytxt, keytxt)

   ;  while ((x = parsescopes(tmpseg, 2)) == 2)
      {
         mcxTingWrite(tsttxt, arg1_g->str)

      ;  yamDigest(tsttxt, tsttxt)

      ;  if (!strcmp(tsttxt->str, keytxt->str))
         {  yamtxt = mcxTingNew(arg2_g->str)
         ;  break
      ;  }
   ;  }
   ;  if (x == 1)
      {  yamtxt = mcxTingNew(arg1_g->str)     /* fall through / else clause */
   ;  }

   ;  mcxTingFree(&keytxt)
   ;  mcxTingFree(&tsttxt)
   ;  yamSegFree(&tmpseg)  /* this frees chktxt */

   ;  return   yamtxt ? yamSegPush(seg, yamtxt) : seg
;  }


yamSeg* expandConstant1
(  yamSeg*  seg
)
   {
      mcxTing*  yamtxt     =  mcxTingNew(arg1_g->str)     
   ;  yamSeg*  newseg      =  yamSegPush(seg, yamtxt)
   ;  int   x

   ;  while ((x = parsescopes(newseg, 2)) == 2)
      {  mcxTing* key      =  mcxTingNew(arg1_g->str)
      ;  if (yamConstantNew(key, arg2_g->str) != key)
         mcxTingFree(&key)
   ;  }

      if (x != 0)
      {  fprintf(stderr, "___ constant & a half\n")
      ;  exit(1)
   ;  }
   ;  yamSegFree(&newseg)
   ;  return seg
;  }


yamSeg* expandSpecial1
(  yamSeg*  seg
)
   {
      mcxTing*  yamtxt     =  mcxTingNew(arg1_g->str)     
   ;  yamSeg*  newseg      =  yamSegPush(seg, yamtxt)
   ;  int      x

   ;  while ((x = parsescopes(newseg,2)) == 2)
      {
         int   c           =  atoi(arg1_g->str)
      ;  yamSpecialSet(c, arg2_g->str)
   ;  }

      if (x != 0)
      {  fprintf(stderr, "___ special & a half\n")
      ;  exit(1)
   ;  }
   ;  yamSegFree(&newseg)
   ;  return seg
;  }


/*
 *    first argument:   anchor
 *    second:           level
 *    third:            type
 *    fourth:           counter
 *    fifth:            caption
 *    sixth:            misc
*/

yamSeg*  expandRefload6
(  yamSeg* seg
)
   {
      mcxbool newhdl =  yamRefNew
                        (  arg1_g->str ,  arg2_g->str ,  arg3_g->str
                        ,  arg4_g->str ,  arg5_g->str ,  arg6_g->str
                        )
   ;  if (!newhdl)
      fprintf(stderr, "[\\refload#6] key <%s> multiply defined\n", arg1_g->str)

   ;  return seg
;  }


yamSeg*  expandCtrset2
(  yamSeg* seg
)
   {
      mcxTing*  label      =  mcxTingNew(arg1_g->str)
   ;  mcxTing*  newval     =  mcxTingNew(arg2_g->str)
   ;  mcxTing*  ctr        =  yamCtrGet(label)

   ;  yamDigest(newval, newval)

   ;  if (ctr)
      mcxTingFree(&label)
   ;  else
      ctr = yamCtrMake(label)

   ;  yamCtrWrite(ctr, newval->str)
   ;  mcxTingFree(&newval)

   ;  return seg
;  }


yamSeg*  expandIcmp5
(  yamSeg* seg
)
   {
      mcxTing*  t1         =  mcxTingNew(arg1_g->str)
   ;  mcxTing*  t2         =  mcxTingNew(arg2_g->str)
   ;  mcxTing*  ycase
   ;  int i1, i2

   ;  yamDigest(t1, t1)
   ;  yamDigest(t2, t2)
   ;  i1    =  atoi(t1->str)
   ;  i2    =  atoi(t2->str)

   ;  ycase =     i1 < i2
               ?  mcxTingNew(arg3_g->str)
               :     i1 == i2
                  ?  mcxTingNew(arg4_g->str)
                  :  mcxTingNew(arg5_g->str)
   ;  mcxTingFree(&t1)
   ;  mcxTingFree(&t2)
   ;  return yamSegPush(seg, ycase)
;  }


/*
*/

yamSeg*  expandCtrput1
(  yamSeg* seg
)
   {
      mcxTing*   ctr        =  yamCtrGet(arg1_g)
   ;  mcxTing*   yamtxt     =  mcxTingNew(ctr ? ctr->str : "0")

   ;  return yamSegPush(seg, yamtxt)
;  }


/*
*/

yamSeg*  expandIsum1
(  yamSeg* seg
)
   {
      mcxTing*    data        =  mcxTingNew(arg1_g->str)
   ;  mcxTing*    sumtxt
   ;  yamSeg*     sumseg
   ;  char        sumstr[20]
   ;  int         sum         =  0
   ;  int         x

   ;  yamDigest(data, data)
   ;  sumseg = yamSegPush(seg, data)

   ;  while ((x = parsescopes(sumseg, 1)) == 1)
      {
         int d = atoi(arg1_g->str)
      ;  sum += d
   ;  }

   ;  mcxTingFree(&data)
   ;  sprintf(sumstr, "%d", sum)
   ;  sumtxt  =  mcxTingNew(sumstr)
   ;  return yamSegPush(seg, sumtxt)
;  }


/*
*/

yamSeg*  expandIclc3
(  yamSeg* seg
)
   {
      int       mode       =  *(arg1_g->str+0)
   ;  mcxTing*  i1txt      =  mcxTingNew(arg2_g->str)
   ;  mcxTing*  i2txt      =  mcxTingNew(arg3_g->str)
   ;  mcxTing*  i3txt
   ;  char      i3str[20]
   ;  int       i1, i2, i3 =  0

   ;  yamDigest(i1txt, i1txt)
   ;  yamDigest(i2txt, i2txt)
   ;  i1  =  atoi(i1txt->str)
   ;  i2  =  atoi(i2txt->str)
   ;  mcxTingFree(&i1txt)
   ;  mcxTingFree(&i2txt)

   ;  switch(mode)
      {
         case '+'
         :  i3 = i1 + i2 ; break
      ;  case '-'
         :  i3 = i1 - i2 ; break
      ;  case '/'
         :  i3 = i1 / (i2 ? i2 : 1) ; break
      ;  case '%'
         :  i3 = i1 % i2 ; break
      ;  case '*'
         :  i3 = i1 * i2 ; break
      ;  default
         :  yamExit("\\iclc#3", "unknown mode <%c>", mode)
   ;  }

   ;  sprintf(i3str, "%d", i3)
   ;  i3txt = mcxTingNew(i3str)

   ;  return yamSegPush(seg, i3txt)
;  }


/*
*/

yamSeg*  expandCtradd2
(  yamSeg* seg
)
   {
      mcxTing* label       =  mcxTingNew(arg1_g->str)
   ;  mcxTing* ctr         =  yamCtrGet(label)
   ;  mcxTing* addtxt      =  mcxTingNew(arg2_g->str)
   ;  int      a           =  0
   ;  int      c           =  0

   ;  yamDigest(addtxt, addtxt)
   ;  a  =  atoi(addtxt->str)
   ;  mcxTingFree(&addtxt)

   ;  if (ctr)
      {  c  =  atoi(ctr->str)
      ;  mcxTingFree(&label)
   ;  }
      else
      ctr = yamCtrMake(label)

   ;  c +=  a
   ;  yamCtrSet(ctr, c)

   ;  return seg
;  }

/*
 * Dependency with yamRefMember. Agreement:
 * it only returns NULL if second arg not in [ntlcm].
*/

yamSeg*  expandRef2
(  yamSeg* seg
)
   {
      const char* member   =  yamRefMember(arg1_g, *(arg2_g->str+0))
   ;  mcxTing* memtxt      =  member ? mcxTingNew(member) : NULL

   ;  if (!memtxt)
      yamExit("\\ref#2", "second argument invalid (not in [ntlcm]")

   ;  return yamSegPush(seg, memtxt)
;  }


yamSeg*  expandExit0
(  yamSeg* seg
)
   {  fprintf(stderr, "___ premature exit enforced\n___ Goodbye world\n")
   ;  exit(1)
;  }


yamSeg*  expandQuit0
(  yamSeg* seg
)
   {
      while (seg)
      {
         yamSeg* prev_seg  =  seg
      ;  seg               =  seg->prev
      ;  yamSegFree(&prev_seg)
   ;  }
   ;  return NULL
;  }


yamSeg* expandSet2
(  yamSeg*   seg
)
   {  yamSeg*  newseg
   ;  insertWarn_g         =  0
   ;  newseg               =  expandDef2(seg)
   ;  return newseg
;  }


yamSeg* expandEnv3
(  yamSeg*   seg
)
   {  yamEnvNew(arg1_g->str, arg2_g->str, arg3_g->str)
   ;  return seg
;  }


yamSeg* expandBegin1
(  yamSeg*   seg
)  {
      const char* b = yamEnvOpenScope(arg1_g, seg)
   ;  mcxTing* val = b ? mcxTingNew(b) : NULL

   ;  if (!val)
      yamExit("\\begin#2", "env <%s> not found", arg1_g->str)
   ;  return yamSegPush(seg, val)
;  }


yamSeg* expandEnd1
(  yamSeg*   seg
)
   {  const char* e = yamEnvCloseScope(arg1_g->str, seg)
   ;  mcxTing* val = e ? mcxTingNew(e) : NULL

   ;  if (!val)
      yamExit("\\end#2", "env <%s> not found\n", arg1_g->str)
   ;  return yamSegPush(seg, val)
;  }


yamSeg* expandDef2
(  yamSeg*   seg
)
   {
      mcxTing*  key        =  mcxTingNew(arg1_g->str)
   ;  const char* me       =  insertWarn_g ? "\\def#2" : "\\set#2"
   ;  int      keylen      =  checkusrkey(key->str, key->len, NULL)

   ;  if (keylen < 0 || keylen != key->len)
      yamExit(me, "not a valid key signature: <%s>\n", key->str)

   ;  if (mcxHashSearch(key, yamTable_g, MCX_DATUM_FIND))
      yamExit(me, "key tagged <%s> is a zoem primitive\n", key->str)

   ;  if (yamKeyInsert(key, arg2_g->str) != key && insertWarn_g)
      {  fprintf(stderr, "___ overwriting key <%s>\n",key->str)
      ;  mcxTingFree(&key)
   ;  }

   ;  insertWarn_g = 1
   ;  return seg
;  }


void yamOpsStats
(  void
)  {  mcxHashStats(yamTable_g)
;  }


mcxstatus readFile
(  mcxTing*  fname
,  mcxTing*  filetxt
)
   {
      mcxIOstream *xf

   ;  if (fname->len > 123)
      yamExit
      ("readFile", "[readFile] input file name expansion too long (>123)\n")

   ;  if (yamInlineFile(fname, filetxt))
      return STATUS_OK

   ;  else
      {  xf =  mcxIOstreamNew(fname->str, "r")

      ;  if (mcxIOstreamOpen(xf, RETURN_ON_FAIL) != STATUS_OK)
         {
            mcxIOstreamFree(&xf)
         ;  mcxTingEmpty(filetxt, 0)
         ;  return STATUS_FAIL
      ;  }
      ;  yamReadFile(xf, filetxt, 0)
      ;  mcxIOstreamFree(&xf)
      ;  return STATUS_OK
   ;  }

   ;  return STATUS_FAIL
;  }


void yamOpsInitialize
(  int   n
)
   {  cmdHook* cmdhook     =  cmdHookDir

   ;  devtxt_g             =  mcxTingNew("\\device")
   ;  yamTable_g           =  mcxHashNew(n, mcxTingCThash, mcxTingCmp)

   ;  while (cmdhook && cmdhook->name)
      {
         mcxTing*  cmdtxt  =  mcxTingNew(cmdhook->name)
      ;  mcxKV*   kv       =  mcxHashSearch(cmdtxt,yamTable_g,MCX_DATUM_INSERT)
      ;  kv->val           =  cmdhook
      ;  cmdhook++
   ;  }
;  }

void yamOpsMakeComposites
(  void
)
   {  mcxTing*  composites  =  mcxTingNew(strComposites)
   ;  yamDigest(composites, composites)  
   ;  mcxTingFree(&composites)
;  }


xpnfnc yamOpGet
(  mcxTing* txt
)
   {
      mcxKV* kv = mcxHashSearch(txt, yamTable_g, MCX_DATUM_FIND)
   ;  if (kv)
      return ((cmdHook*) kv->val)->yamfunc

   ;  return NULL
;  }




