""" The resource plugin definition. """


# Enthought library imports.
from enthought.envisage import ExtensionItem, ExtensionPoint, PluginDefinition
from enthought.traits.api import List, Str


# The plugin's globally unique identifier (also used as the prefix for all
# identifiers defined in this module).
ID = "enthought.envisage.resource"


###############################################################################
# Extension points.
###############################################################################

class ResourceView(ExtensionPoint):
    """ A Traits UI view associated with a specified resource type. """

    # The unique Id of the view (unique within the 'resource type').
    id = Str

    # The name of the Traits UI Handler class that implements the view.
    class_name = Str

    # The resource type the view applies to.
    resource_type = Str

class ResourceType(ExtensionItem):
    """ The definition of a single resource type. """

    # The name of the class that implements the 'ResourceType' interface.
    class_name = Str

    # The name of the factory that creates the resource type (not needed if
    # the class name is specified).
    factory_class_name = Str

    # Names of classes which this resource type should load 'before'.
    precedes = List(Str)

class ResourceManager(ExtensionPoint):
    """ Contributions to the resource manager. """

    # Resource types.
    resource_types = List(ResourceType)

###############################################################################
# Extensions.
###############################################################################

ACTION_RESOURCE_RESOURCE_TYPE   = ID + '.action_resource_resource_type.' + \
    'ActionResourceResourceType'
ARRAY_RESOURCE_TYPE    = ID + '.array_resource_type.ArrayResourceType'
BOOL_RESOURCE_TYPE     = ID + '.bool_resource_type.BoolResourceType'
CLASS_RESOURCE_TYPE    = ID + '.class_resource_type.ClassResourceType'
FILE_RESOURCE_TYPE     = ID + '.file_resource_type.FileResourceType'
FOLDER_RESOURCE_TYPE   = ID + '.folder_resource_type.FolderResourceType'
DICT_RESOURCE_TYPE     = ID + '.dict_resource_type.DictResourceType'
FLOAT_RESOURCE_TYPE    = ID + '.float_resource_type.FloatResourceType'
INSTANCE_RESOURCE_TYPE = ID + '.instance_resource_type.InstanceResourceType'
INT_RESOURCE_TYPE      = ID + '.int_resource_type.IntResourceType'
LIST_RESOURCE_TYPE     = ID + '.list_resource_type.ListResourceType'
NONE_RESOURCE_TYPE     = ID + '.none_resource_type.NoneResourceType'
STRING_RESOURCE_TYPE   = ID + '.string_resource_type.StringResourceType'
TUPLE_RESOURCE_TYPE    = ID + '.tuple_resource_type.TupleResourceType'

resource_manager = ResourceManager(
    resource_types = [
        ResourceType(
            class_name = NONE_RESOURCE_TYPE,
            precedes   = [FILE_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = FILE_RESOURCE_TYPE,
            precedes   = [FOLDER_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = FOLDER_RESOURCE_TYPE,
            precedes   = [BOOL_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = BOOL_RESOURCE_TYPE,
            precedes   = [INT_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = INT_RESOURCE_TYPE,
            precedes   = [FLOAT_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = FLOAT_RESOURCE_TYPE,
            precedes   = [STRING_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = STRING_RESOURCE_TYPE,
            precedes   = [ARRAY_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = ARRAY_RESOURCE_TYPE,
            precedes   = [DICT_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = DICT_RESOURCE_TYPE,
            precedes   = [LIST_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = LIST_RESOURCE_TYPE,
            precedes   = [TUPLE_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = TUPLE_RESOURCE_TYPE,
            precedes   = [CLASS_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = CLASS_RESOURCE_TYPE,
            precedes   = [INSTANCE_RESOURCE_TYPE]
        ),

        ResourceType(
            class_name = INSTANCE_RESOURCE_TYPE,
        ),

        ResourceType(
            class_name = ACTION_RESOURCE_RESOURCE_TYPE,
            precedes   = [INSTANCE_RESOURCE_TYPE]
        ),

    ]
)

###############################################################################
# The plugin definition.
###############################################################################

class ResourcePluginDefinition(PluginDefinition):
    """ The resource plugin definition. """

    # The plugin's globally unique identifier.
    id = ID

    # The name of the class that implements the plugin.
    class_name = ID + ".resource_plugin.ResourcePlugin"

    # General information about the plugin.
    name          = "Resource Plugin"
    version       = "1.0.0"
    provider_name = "Enthought Inc"
    provider_url  = "www.enthought.com"

    # The Id's of the plugins that this plugin requires.
    requires = ["enthought.envisage.core"]

    # The extension points offered by this plugin.
    extension_points = [ResourceManager, ResourceView]

    # The contributions that this plugin makes to extension points offered by
    # either itself or other plugins.
    extensions = [resource_manager]

#### EOF ######################################################################
