""" Tests a simple plugin. """


# Standard library imports.
import os, sys, unittest

# Enthought library imports.
from enthought.envisage import Application

# Local imports.
from plugin_definitions import PLUGIN_DEFINITIONS
from simple_plugin import SimplePlugin


class SimplePluginTestCase(unittest.TestCase):
    """ Tests a simple plugin. """

    ###########################################################################
    # 'TestCase' interface. 
    ###########################################################################

    def setUp(self):
        """ Prepares the test fixture before each test method is called. """

        self.application = Application(
            argv               = sys.argv,
            plugin_definitions = PLUGIN_DEFINITIONS,
            requires_gui       = False
        )

        self.application.start()

        return 

    def tearDown(self):
        """ Called immediately after each test method has been called. """

        self.application.stop()

        return
    
    ###########################################################################
    # Tests.
    ###########################################################################

    def test_start(self):
        """ start the simple test plugin """

        # Make sure the simple plugin was created and started.
        self.assertNotEqual(SimplePlugin.instance, None)
        self.assertEqual(SimplePlugin.instance.started, True)

        return

    # fixme: Commented out as the test runs fine when run locally but fails
    # when run via CC.
##     def test_extension_point(self):
##         """ offer and contribute to an extension point """

##         er = self.application.extension_registry

##         # Plugin definition imports.
##         from simple_plugin_definition import MyExtensionPoint
        
##         # We should now have a single extension.
##         extensions = er.get_extensions(MyExtensionPoint, 'simple')
##         self.assertEqual(len(extensions), 1)

##         return

    def test_start_plugin(self):
        """ start an individual plugin """

        pa = self.application.plugin_activator

        # Try to start a non-existent plugin.
        self.failUnlessRaises(ValueError, pa.start_plugin, 'nonexistent')
        
        return

    def test_preferences(self):
        """ user preferences """

        plugin = SimplePlugin.instance
        self.assert_(SimplePlugin.instance.started)

        # Get the user preferences.
        preferences = plugin.preferences
        
        # Get a user preference setting.
        value = preferences.get('ready_to_ship', 'ignore me please')
        self.assertEqual(value, True)

        # Set a user preference.
        value = preferences.set('ready_to_ship', False)
        value = preferences.get('ready_to_ship')
        self.assertEqual(value, False)

        # Get the default value of the preference.
        value = preferences.get_default('ready_to_ship')
        self.assertEqual(value, True)
        
        # Get a non-existent user preference with a default value.
        value = preferences.get('dummy', 42)
        self.assertEqual(value, 42)

        # Set it back!
        value = preferences.set('ready_to_ship', True)
        
        # Save them.
        plugin.save_preferences()

        return
    
    def test_corruption(self):
        """ tests corrupting the user profiles which should force the defaults """
        plugin = SimplePlugin.instance
        self.assert_(SimplePlugin.instance.started)
        filename = 'corrupt_preferences'

        # Get the user preferences.
        preferences = plugin.preferences

        # Get the default value of the preference.
        default = preferences.get_default('ready_to_ship')
        self.assertEqual(default, True)
        
        # save a user preference setting to the opposite of the default
        preferences.set('ready_to_ship', not default)
        preferences.save(filename)
        
        # overwrite the file with junk
        f = open(filename, 'w')
        f.write("junk")
        f.close()

        preferences.load(filename)
        value = preferences.get('ready_to_ship')
        self.assertEqual(value, True)
        
        import os
        os.unlink(filename)
        
        return
    
#### EOF ######################################################################
