/*!
  @file           RTEThread_ConsoleCommandQueue.hpp
  @author         StefanP
  @special area   Kernel Console Thread
  @brief          Command Queue
  @see            

\if EMIT_LICENCE
  ========== licence begin  GPL
  Copyright (c) 2002-2004 SAP AG

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  ========== licence end
\endif
*/



#ifndef RTETHREAD_CONSOLECOMMANDQUEUE_HPP
#define RTETHREAD_CONSOLECOMMANDQUEUE_HPP



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/
#include    "heo07.h"

#include    "RunTime/RTE_Types.h"
#include    "RunTime/MemoryManagement/RTEMem_Allocator.hpp" 
#include    "Container/Container_List.hpp" 
#include    "RunTime/Threading/RTEThread_ConsoleCommandWorker.hpp" 
#include    "RunTime/Threading/RTEThread_ConsoleRequest.h"

extern SAPDBTrace_Topic Console_Trace;
/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/


/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/


/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

/*!
  @class          RTEThread_ConsoleCommandQueue
   @description    Command Queue (Singleton)


                This class implements a command queue as a Container List of
                console requests. 
                (All requests received by the console thread are put in this queue. 
                On the other end of the queue the worker threads triggered by the 
                console thread get those requests from the queue.)

 */


class RTEThread_ConsoleCommandQueue: protected Container_List<RTEThread_ConsoleRequest>
{
public:

  /*!
     @description    Returns reference to single instance of RTEThread_ConsoleCommandQueue
     @return value   Reference to RTEThread_ConsoleCommandQueue instance

   */


    static  RTEThread_ConsoleCommandQueue &   Instance ();

  /*!
     @brief          Destructor

   */


    ~RTEThread_ConsoleCommandQueue  ();

  /*!
     @description    Puts a console request into the command queue.
     @param          request [in] - request to be send
     @return value   none

   */

    void            PutCommand      (RTEThread_ConsoleRequest const &   request);

  /*!
     @description    If available a request is read from the command queue and the
                state of the given worker thread is set to CONSOLE_WORKER_STATE_ACTIVE.


                  If the command queue is empty the sate is set to 
                  CONSOLE_WORKER_STATE_WAITING.
     @param          threadInfo [in/out] - Info structure of a worker thread
     @param          request [out] - request (if state was set to 
                                                CONSOLE_WORKER_STATE_ACTIVE)
     @return value   none

   */


    void            GetCommand      (ConsoleWorkerInfo &                threadInfo, 
                                     RTEThread_ConsoleRequest &          request);

  /*!
     @description    Lock the command queue

   */
  

    inline  void        Lock        ();

  /*!
     @description    Unlock the command queue

   */
  

    inline  void        Unlock      ();

private:

  /*!
     @brief          Constructor


     The Constructor is privat to this class to avoid instantiation outside 
     of the class.

   */

    RTEThread_ConsoleCommandQueue   (Allocator &                        alloc);


    static  RTEThread_ConsoleCommandQueue  *m_Instance; 
            teo07_Mutex                     m_commandQueueMutex;
    
}; 



/*---------------------------------------------------------------------------*/

inline  void    RTEThread_ConsoleCommandQueue::Lock (void)
{
    SAPDBTRACE_METHOD_DEBUG ("RTEThread_ConsoleCommandQueue::Lock", 
                             Console_Trace, 9);
    sqlbeginmutex(&m_commandQueueMutex);
}

/*---------------------------------------------------------------------------*/

inline  void    RTEThread_ConsoleCommandQueue::Unlock (void)
{
    SAPDBTRACE_METHOD_DEBUG ("RTEThread_ConsoleCommandQueue::Unlock", 
                             Console_Trace, 9);
    sqlendmutex(&m_commandQueueMutex);
}



#endif  /* RTETHREAD_CONSOLECOMMANDQUEUE_HPP */
