/*
 * Math3d - The 3D Computer Graphics Math Library
 * Copyright (C) 1996-2000 by J.E. Hoffmann <je-h@gmx.net>
 * All rights reserved.
 *
 * This program is  free  software;  you can redistribute it and/or modify it
 * under the terms of the  GNU Lesser General Public License  as published by 
 * the  Free Software Foundation;  either version 2.1 of the License,  or (at 
 * your option) any later version.
 *
 * This  program  is  distributed in  the  hope that it will  be useful,  but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or  FITNESS FOR A  PARTICULAR PURPOSE.  See the  GNU Lesser General Public  
 * License for more details.
 *
 * You should  have received  a copy of the GNU Lesser General Public License
 * along with  this program;  if not, write to the  Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: m3frame.cpp,v 1.4 2000/10/09 12:17:54 jeh Exp $
 */
#define _MATH3D_EXPORT
#include <math3d/m3frame.h>
#include <math3d/m4x4.h>
#include <cmath>


/*!
 * Initial values are
 *
 * \arg Position           = (0,0,0)
 * \arg Viewing direction  = (0,1,0)
 * \arg Viewing up         = (0,0,1)
 * \arg Viewing right      = (1,0,0)
 */
Math3d::M3Frame::M3Frame() 
  : d_pos(0,0,0), d_right(1,0,0), d_dir(0,1,0), d_up(0,0,1)
{
}


/*!
 *
 */
Math3d::M3Frame::M3Frame(const M3d& pos, const M3d& right, const M3d& dir, const M3d& up)
  : d_pos(pos), d_right(right), d_dir(dir), d_up(up)
{
  
}


/*!
 *
 */
Math3d::M3Frame::M3Frame(const M3Frame& fr) 
  : d_pos(fr.d_pos), d_right(fr.d_right), d_dir(fr.d_dir), d_up(fr.d_up)
{
}


/*!
 * Does nothing!
 */
Math3d::M3Frame::~M3Frame()
{
}


/*!
 *
 */
const Math3d::M3Frame&
Math3d::M3Frame::operator=(const M3Frame& fr)
{
  d_pos=fr.d_pos;
  d_dir=fr.d_dir;
  d_up=fr.d_up;
  d_right=fr.d_right;
  return(*this);
}


/*!
 *
 */
void 
Math3d::M3Frame::copy(const M3Frame& fr)
{
  d_pos=fr.d_pos;
  d_dir=fr.d_dir;
  d_up=fr.d_up;
  d_right=fr.d_right;
}


/*!
 * Sets the eye-point
 *
 * \param pos New position
 */
void
Math3d::M3Frame::setPos(const M3d& pos)
{
  d_pos=pos;
}


/*!
 *
 */
void
Math3d::M3Frame::setView(const M3d& tgt, double roll)
{
  M4x4 M;
  M3d d;
  double az,ax;

  M.identity();
  
  d.sub(tgt,d_pos);
  d.normalize();
  az=atan2(d[0],d[1]);
  ax=-asin(d[2]);
  M.rotateY(-roll);
  M.rotateX(ax);
  M.rotateZ(az);
  d_right=M.getCol(0); //FIXME: THIS IS BUGGY!
  d_dir=M.getCol(1);
  d_up=M.getCol(2);
}


/*!
 * Sets the Right, Direction and Up vector.
 * All vectors must be normalized.
 *
 * \param right Right vector
 * \param dir Direction vector
 * \param up Up vector
 */
void
Math3d::M3Frame::setRDU(const M3d& right, const M3d& dir, const M3d& up)
{
  d_right=right;
  d_dir=dir;
  d_up=up;
}


/*! 
 * Rotate around the up axis at the trihedrons position.
 *
 * \param angle Angle in radians
 */
void
Math3d::M3Frame::yaw(double angle)
{
  M4x4 M;
  
  M.loadRotate(d_up,angle);
  M.transform(d_dir);
  d_right.cross(d_dir,d_up);

  d_dir.normalize();
  d_up.normalize();
  d_right.normalize();
}


/*! 
 * Rotate around the right axis at the trihedrons position.
 *
 * \param angle Angle in radians
 */
void
Math3d::M3Frame::pitch(double angle)
{
  M4x4 M;
  
  M.loadRotate(d_right,angle);
  M.transform(d_dir);
  d_up.cross(d_right,d_dir);

  d_dir.normalize();
  d_up.normalize();
  d_right.normalize();
}


/*! 
 * Rotates around the direction axis at the trihedrons position.
 *
 * \param angle Angle in radians
 */
void
Math3d::M3Frame::roll(double angle)
{
  M4x4 M;
  
  M.loadRotate(d_dir,angle);
  M.transform(d_up);
  d_right.cross(d_dir,d_up);

  d_dir.normalize();
  d_up.normalize();
  d_right.normalize();
}


/*! 
 *
 */
void
Math3d::M3Frame::rotate(const MQuat& q)
{
  M4x4 M,X;
  X.loadRotate(q);
  getRotMatrix(&M);
  X*=M;
  setRotMatrix(X);
}


/*! 
 * Translates the eye-point in the local coordiante system (dir,up,right)
 *
 * \param delta Translation vector
 */
void
Math3d::M3Frame::pan(const M3d& delta)
{
  d_pos+=d_right*delta[0];
  d_pos+=d_dir*delta[1];
  d_pos+=d_up*delta[2];
}


/*!
 * Calculates rotation matrix.
 *
 * \param matrix Pointer to store result.
 */
void
Math3d::M3Frame::setRotMatrix(const M4x4 &M)
{
  d_right=M.getRow(0);
  d_dir=M.getRow(1);
  d_up=M.getRow(2);
}


/*!
 * \return The trihedron's eye-point
 */
const Math3d::M3d&
Math3d::M3Frame::getPos() const
{
  return(d_pos);
}


/*!
 * \return Vector in viewing direction.
 */
const Math3d::M3d&
Math3d::M3Frame::getDir() const
{
  return(d_dir);
}


/*!
 * \return Up vector
 */
const Math3d::M3d&
Math3d::M3Frame::getUp() const
{
  return(d_up);
}


/*!
 * \return Right vector
 */
const Math3d::M3d&
Math3d::M3Frame::getRight() const
{
  return(d_right);
}


/*!
 * Calculates rotation matrix.
 *
 * \param matrix Pointer to store result.
 */
void
Math3d::M3Frame::getRotMatrix(M4x4 *M) const
{
  M->identity();
  M->setRow(0,d_right);
  M->setRow(1,d_dir);
  M->setRow(2,d_up);
}


/*!
 * Calculates OpenGL GL_MODELVIEW matrix.
 *
 * \param matrix Pointer to store result.
 */
void
Math3d::M3Frame::getViewMatrix(M4x4 *M) const
{
  getRotMatrix(M);
  M->translate(-d_pos[0],-d_pos[1],-d_pos[2]);
}


/*!
 *
 */
bool 
Math3d::M3Frame::operator==(const M3Frame& fr) const
{
  return(cmp(fr));
}


/*!
 *
 */
bool 
Math3d::M3Frame::operator!=(const M3Frame& fr) const
{
  return(!cmp(fr));
}


/*!
 *
 */
bool 
Math3d::M3Frame::cmp(const M3Frame& fr, double epsilon) const
{
  return(
    d_pos.cmp(fr.d_pos, epsilon) &&
    d_right.cmp(fr.d_right, epsilon) &&
    d_dir.cmp(fr.d_dir, epsilon) &&
    d_up.cmp(fr.d_up, epsilon)
  );  
}







