/*
 * xsingleinstance
 *
 * Program to ensure that a single instance of an X window program
 * exists on the screen.
 *
 * Copyright (c) 2000, Merle F. McClelland for CompanionLink
 * Portions of this code, such as TryChildren and ClientWindow, are
 * based upon routines in the Xmu library, which is Copyright 1989, 1998,
 * The Open Group.  All Rights Reserved.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
 * OPEN GROUP BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * See the files COPYRIGHT and LICENSE for distribution information.
*/

#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <signal.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xlibint.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xproto.h>
#include <X11/xpm.h>
#include <X11/Xatom.h>
#include <X11/extensions/shape.h>

#define FALSE	0
#define TRUE	1

Atom wm_state_atom    = 0;

/*
 * Window_With_Name: routine to locate a window with a given name on a display.
 *                   If no window with the given name is found, 0 is returned.
 *                   If more than one window has the given name, the first
 *                   one found will be returned.  Only top and its subwindows
 *                   are looked at.  Normally, top should be the RootWindow.
 */
Window Window_With_Name(dpy, top, name)
     Display *dpy;
     Window top;
     char *name;
{
        Window *children, dummy;
        unsigned int nchildren;
        int i;
        Window w=0;
        char *window_name;

        if (XFetchName(dpy, top, &window_name) && !strcmp(window_name, name))
          return(top);

        if (!XQueryTree(dpy, top, &dummy, &dummy, &children, &nchildren))
          return(0);

        for (i=0; i<nchildren; i++) {
                w = Window_With_Name(dpy, children[i], name);
                if (w)
                  break;
        }
        if (children) XFree ((char *)children);
        return(w);
}

static Window
TryChildren(Display *dpy, Window win, Atom WM_STATE)
{
	Window root, parent;
	Window *children;
	unsigned int nchildren;
	unsigned int i;
	Atom type = None;
	int format;
	unsigned long nitems, after;
	unsigned char *data;
	Window inf = 0;

	if (!XQueryTree(dpy, win, &root, &parent, &children, &nchildren))
		return 0;
	for (i = 0; !inf && (i < nchildren); i++) 
	{
		XGetWindowProperty(dpy, children[i], WM_STATE, 0, 0, False,
	                        AnyPropertyType, &type, &format, &nitems,
	                        &after, &data);
		if (type)
			inf = children[i];
	}
	for (i = 0; !inf && (i < nchildren); i++)
		inf = TryChildren(dpy, children[i], WM_STATE);
	if (children) XFree((char *)children);
	return inf;
}

Window
ClientWindow(Display *dpy, Window win)
{
	 Atom type = None;
	 int format;
	 unsigned long nitems, after;
	 unsigned char *data;
	 Window inf;

	 if (!wm_state_atom)
	     return win;
	 XGetWindowProperty(dpy, win, wm_state_atom, 0, 0,
			    False, AnyPropertyType,
	                    &type, &format, &nitems, &after, &data);
	 if (type)
	     return win;
	 inf = TryChildren(dpy, win, wm_state_atom);
	 if (!inf)
	     inf = win;
	 return inf;
}

//
// Find a window's state
//
int
wm_state(Display *display, Window w)
{
	Atom real_type;
	int real_format, state = 0;
	unsigned long n, extra;
	unsigned char *data;

	if (XGetWindowProperty(display, w, wm_state_atom, 0L, 2L, False, AnyPropertyType,
	   		&real_type, &real_format, &n, &extra, &data) == Success && n) 
	{
		state = *(int *)data;
		XFree(data);
		return state;
	}
	else
		return -1;
}

//
// bring a window to the foreground
//
void raise_window(Display *display, Window w)
{
	int iState;
	iState = wm_state(display, w);
	printf("attempting to map window\n");
	if (iState == NormalState)
	 		XMapRaised(display, w);
  	else if( iState == IconicState )
	 		XMapRaised(display, w);
}


int main(int argc, char *argv[])
{
	// We just use the DISPLAY environment variable
	char *disp = NULL;
	char *shellToUse;
	int xid;
	int debug;
	int kill_dont_icon = 0;
	
	Display *display;
	int screen;
	Atom window_type_atom;
	Atom window_type_dock_atom;
	Window win;
	Pixmap icon, mask;
	GC gc;
	XGCValues *gv = NULL;
	XpmAttributes attrib;	
	XEvent an_event;
	XSizeHints size_hints;
	Atom wm_delete, wm_protos;
	
	debug = 0;

	if (argc < 4)
	{
		fprintf(stderr, "usage: %s xpm window program \n", argv[0]);
		exit(1);
	}

	// Determine which shell to use to fork processes

	shellToUse = getenv("SHELL");
	if (!shellToUse)
		shellToUse = "/bin/sh";

	if ((display = XOpenDisplay(disp)) == NULL)
	{
		fprintf(stderr, "Cannot connect to X server on display %s.", XDisplayName(disp));
		exit(1);
	}

	wm_delete = XInternAtom(display, "WM_DELETE_WINDOW", False);
	wm_protos = XInternAtom(display, "WM_PROTOCOLS", False);
	window_type_atom = XInternAtom(display, "_NET_WM_WINDOW_TYPE" , False);
	window_type_dock_atom = XInternAtom(display,
					    "_NET_WM_WINDOW_TYPE_DOCK",False);
	wm_state_atom = XInternAtom(display, "WM_STATE", True);
	
	if (argv[1][0] == '-' && argv[1][1] == 'k') kill_dont_icon++;
	screen = DefaultScreen(display);

	win = XCreateSimpleWindow(display,
				  RootWindow(display, screen),
				  0, 0,
				  32, 16,
				  0, BlackPixel(display, screen),
				  WhitePixel(display, screen));

	gc = XCreateGC(display, RootWindow(display, screen), 0, gv);
	
	attrib.valuemask = 0;
	if (XpmReadFileToPixmap( display, win, argv[1+kill_dont_icon],
				 &icon, &mask, &attrib) != XpmSuccess )
	{
	   fprintf(stderr, "monolauch: failed loading image '%s'\n",
		   argv[1+kill_dont_icon]);
	   exit(1);
	}

       XResizeWindow(display, win, attrib.width, attrib.height);

       size_hints.flags = PPosition | PSize | PMinSize;
       size_hints.x = 0;
       size_hints.y = 0;
       size_hints.width      =  attrib.width;
       size_hints.height     =  attrib.height;
       size_hints.min_width  =  attrib.width;
       size_hints.min_height =  attrib.height;
       
       XSetStandardProperties(display, win, "mono", 
			      "mono", 0,
			      argv, argc, &size_hints);

	XCopyArea(display, icon, win, gc, 0, 0,
		  attrib.width, attrib.height, 0, 0);
	
	XShapeCombineMask (display, win, ShapeBounding, 0, 0, mask, ShapeSet);
	XChangeProperty(display, win, window_type_atom, XA_ATOM, 32, 
			PropModeReplace, (unsigned char *)
			&window_type_dock_atom, 1);

	XMapWindow(display, win);
	XSelectInput(display, win, ExposureMask|ButtonPressMask ); 
	signal(SIGCHLD, SIG_IGN);
	while (1) {
	   XNextEvent(display, &an_event);
	   switch (an_event.type) {
	      case Expose:

		 XCopyArea(display, icon, win, gc, 0, 0,
			   attrib.width, attrib.height, 0, 0);
		 XShapeCombineMask (display, win, ShapeBounding,
				    0, 0, mask, ShapeSet);
	      break;
	      case ButtonPress:

		 xid = Window_With_Name(display, RootWindow(display, screen),
					argv[2+kill_dont_icon]);
		 if (xid == 0)
		 {
		    switch (fork())
		    {
		       case 0: // Child process
			  //execlp(argv[3+kill_dont_icon],
			  //	 argv[3+kill_dont_icon], 0);
			  execvp(argv[3+kill_dont_icon], &argv[3+kill_dont_icon]);
			  
			  fprintf(stderr, "Can't exec \"%s\"\n",
				  argv[3+kill_dont_icon]);
			  exit(1);
		       case -1:
			  fprintf(stderr, "Couldn't fork\n");
			  break;
		    }
		 } else {
		    XWindowAttributes xid_attrib;
		    XGetWindowAttributes(display, xid, &xid_attrib);
		    
		    if (xid_attrib.map_state == IsUnmapped ||
			wm_state(display, xid) != NormalState)
		    {
		       //printf("window is not normal / unmapped\n");
		       raise_window(display, xid);
		    }
		    else
		    {
		       if (kill_dont_icon)
		       {
			  int i, n, found = 0;
			  Atom *protocols;
			  XEvent e;
			  printf("die die die\n");
			  
			  if (XGetWMProtocols(display, xid,
					      &protocols, &n))
			  {
			     for (i=0; i<n; i++)
				if (protocols[i] == wm_delete) found++;
			     XFree(protocols);
			  }
			  
			  if (found)
			  {
			     e.type = ClientMessage;
			     e.xclient.window = xid;
			     e.xclient.message_type = wm_protos;
			     e.xclient.format = 32;
			     e.xclient.data.l[0] = wm_delete;
			     e.xclient.data.l[1] = CurrentTime;
			     XSendEvent(display, xid, False,
					NoEventMask, &e);
			  }
			  else
			  {
			     XKillClient(display, xid);
			  }
			  xid = 0;
		       } else {
			  XIconifyWindow(display, xid, screen);
		       }
		    }		 
		    break;
		 } 
	   }
	   
	}
	   XCloseDisplay(display);
	   exit(0);

}
