/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include "toolbar_client.h"

Client*
toolbar_client_new(Wm *w, Window win)
{
   client *c = base_client_new(w, win); 
   c->type = toolbar;
   
   c->configure    = &toolbar_client_configure;
   c->reparent     = &toolbar_client_reparent;
   c->button_press = &toolbar_client_button_press;
   c->redraw       = &toolbar_client_redraw;
   c->hide         = &toolbar_client_hide;
   c->show         = &toolbar_client_show;
   c->move_resize  = &toolbar_client_move_resize;
   c->get_coverage = &toolbar_client_get_coverage;
   c->destroy      = &toolbar_client_destroy;

   client_set_state(c,WithdrawnState); 	/* set initially to signal show() */
   
   return c;
}

void
toolbar_client_configure(Client *c)
{
   if (client_get_state(c) == IconicState) {
      ; 
      
   } else {
      c->y = c->wm->dpy_height - wm_total_toolbar_height(c)
	 - c->height - (2*WBW(c));
      c->x = toolbar_win_offset(c);
      c->width = c->wm->dpy_width - toolbar_win_offset(c) - (2*WBW(c));
   }
}

void
toolbar_client_move_resize(Client *c)
{

   int offset = toolbar_win_offset(c);
   base_client_move_resize(c);

   if (client_get_state(c) == NormalState)
   {
      XResizeWindow(c->wm->dpy, c->window, c->width, c->height);
      XMoveResizeWindow(c->wm->dpy, c->frame, c->x - offset,
		  c->y, c->width + offset, c->height );
   } else {
      XMoveResizeWindow(c->wm->dpy, c->frame, c->x,
		  c->y, c->width + offset, offset );
   }      
}

void
toolbar_client_reparent(Client *c)
{
    XSetWindowAttributes pattr;
    Theme *t = c->wm->theme;
    
    pattr.override_redirect = True;
    pattr.border_pixel = (t->bg_col).pixel;
    pattr.event_mask = ChildMask|ButtonPressMask|ExposureMask;
    c->frame = XCreateWindow(c->wm->dpy, c->wm->root,
			     0, c->y,
			     c->wm->dpy_width, c->height,
			     c->wm->theme->win_border_width,
			     DefaultDepth(c->wm->dpy, c->wm->screen),
			     CopyFromParent,
			     DefaultVisual(c->wm->dpy, c->wm->screen),
			     CWOverrideRedirect|CWBorderPixel|CWEventMask,
			     &pattr);
    XAddToSaveSet(c->wm->dpy, c->window);
    XSetWindowBorderWidth(c->wm->dpy, c->window, 0);
    XReparentWindow(c->wm->dpy, c->window, c->frame,
		    toolbar_win_offset(c), 0);
}

void
toolbar_client_show(Client *c)
{

   XFreePixmap(c->wm->dpy, c->backing);
   c->backing = (Pixmap)NULL;

   if (client_get_state(c) == WithdrawnState)
   {
      wm_restack(c->wm, c, - c->height - WBW(c));
   } else if (client_get_state(c) == IconicState) {
      wm_restack(c->wm, c, -(c->height - toolbar_win_offset(c)));
      c->y = c->y - ( c->height - toolbar_win_offset(c));
      c->x = toolbar_win_offset(c);
   } else { return; /* were already shown */ }
   
   client_set_state(c,NormalState);
   toolbar_client_move_resize(c);
   
   XMapRaised(c->wm->dpy, c->frame);
   XMapRaised(c->wm->dpy, c->window); 

   if (client_want_focus(c))
   {
      dbg("client wants focus");
      XSetInputFocus(c->wm->dpy, c->window,
		     RevertToPointerRoot, CurrentTime);
      c->wm->focused_client = c;
   } 

   
}

void
toolbar_client_hide(Client *c)
{
   if (client_get_state(c) == IconicState) return;
   
   client_set_state(c,IconicState);
   c->ignore_unmap++;
   XUnmapSubwindows(c->wm->dpy, c->frame);
   c->x=0;
   c->y = c->y + (c->height - toolbar_win_offset(c));
   if (c->backing) XFreePixmap(c->wm->dpy, c->backing);
   c->backing = (Pixmap)NULL;
   
   toolbar_client_move_resize(c);

   wm_restack(c->wm, c, c->height - toolbar_win_offset(c));
   //wm_restack(c->wm, c,toolbar_win_offset(c) + WBW(c));
}

void
toolbar_client_destroy(Client *c)
{
   dbg("toolbar got destroy");
   if (c->x == toolbar_win_offset(c))
   {
      wm_restack(c->wm, c, c->height);
      dbg("toolbar destroy (normal state) restack by %i state %i", c->height, client_get_state(c));
   } else {
      wm_restack(c->wm, c, +toolbar_win_offset(c)+WBW(c));
      dbg("toolbar destroy (iconic state) restack by %i state %i", toolbar_win_offset(c), client_get_state(c));
   }
   base_client_destroy(c);
}


void
toolbar_client_get_coverage(Client *c, int *x, int *y, int *w, int *h)
{
   int b = c->wm->theme->win_border_width;
   *x = c->x; *y = c->y + b;
   *w = c->width + toolbar_win_offset(c);

   if (client_get_state(c) == NormalState)
   {
       *h = c->height + b;
   } else {
      *h = toolbar_win_offset(c) + b;
   }
}


void
toolbar_client_button_press(Client *c, XButtonEvent *e)
{
   int cx,cy;
   XEvent ev;
   XGCValues gc_vals;
   Theme *t = c->wm->theme;
   
   dbg("got toolbar button press");
   
   if (client_get_state(c) == NormalState && e->x <= toolbar_win_offset(c))
   {
      cx = cy = t->padding;

      if (XGrabPointer(c->wm->dpy, c->wm->root, False,
		       (ButtonPressMask|ButtonReleaseMask),
		       GrabModeAsync,
		       GrabModeAsync, None, c->wm->curs, CurrentTime)
	  != GrabSuccess)
	 return;

      /* paint 'inverted' button */
      gc_vals.function = GXnand;
      XChangeGC(c->wm->dpy, t->mask_gc, GCFunction, &gc_vals);
      theme_paint_pxm(c->wm, c->frame, cx, cy,
		      &t->buttons[BUTTON_TOOLBAR]);
      XSync(c->wm->dpy, False);	    
      for (;;) 				      /* get release */
      {
	 XMaskEvent(c->wm->dpy,
		    (ButtonPressMask|ButtonReleaseMask), &ev);
	 switch (ev.type)
	 {
	    case ButtonRelease:
	       gc_vals.function = GXcopy;
	       XChangeGC(c->wm->dpy, t->mask_gc, GCFunction, &gc_vals);
	       if (ev.xbutton.x < toolbar_win_offset(c)
		   && ev.xbutton.y > c->y)
		  toolbar_client_hide(c);
	       else
		  c->redraw(c, True);
	       XUngrabPointer(c->wm->dpy, CurrentTime);
	       return;
	 }
      }
   }

      cx = c->width + toolbar_win_offset(c) -
	 t->buttons[BUTTON_TOOLBAR_CLOSE].width - t->padding;
   if (e->x > cx )
   { 
      cy = (toolbar_win_offset(c) - t->buttons[BUTTON_TOOLBAR_CLOSE].height)/2;
      if (XGrabPointer(c->wm->dpy, c->wm->root, False,
		       (ButtonPressMask|ButtonReleaseMask),
		       GrabModeAsync,
		       GrabModeAsync, None, c->wm->curs, CurrentTime)
	  != GrabSuccess)
	 return;

      /* paint 'inverted' button */
      gc_vals.function = GXnand;
       XChangeGC(c->wm->dpy, t->mask_gc, GCFunction, &gc_vals);
       theme_paint_pxm(c->wm, c->frame, cx, cy,
		       &t->buttons[BUTTON_TOOLBAR_CLOSE]);
       
       XSync(c->wm->dpy, False);	    
	 
       for (;;) 				      /* get release */
       {
	  XMaskEvent(c->wm->dpy,
		     (ButtonPressMask|ButtonReleaseMask), &ev);
	  switch (ev.type)
	  {
	     case ButtonRelease:
		gc_vals.function = GXcopy;
		XChangeGC(c->wm->dpy, t->mask_gc, GCFunction, &gc_vals);
		if (ev.xbutton.x > cx)
		   client_deliver_delete(c);
		else
		   c->redraw(c, True);
		XUngrabPointer(c->wm->dpy, CurrentTime);
		return;
	  }
       }
   }
   toolbar_client_show(c);
}

int
toolbar_win_offset(Client *c) /* TODO: rename to win_offset */
{
   int i;
   Theme *t = c->wm->theme;
#ifdef USE_XFT   
   i = max( t->toolbar_xftfont->ascent + t->toolbar_xftfont->descent,
	    t->buttons[BUTTON_TOOLBAR].width,
	    t->buttons[BUTTON_TOOLBAR_CLOSE].height,
	    0) + t->padding + t->win_border_width;
#else
   i = max( t->toolbar_font->ascent + t->toolbar_font->descent,
	    t->buttons[BUTTON_TOOLBAR].width,
	    t->buttons[BUTTON_TOOLBAR_CLOSE].height,
	    0) + t->padding + t->win_border_width;
#endif
   return i;
}

void
toolbar_client_redraw(Client *c, Bool use_cache)
{
   int cx,cy;
   Theme *t = c->wm->theme;

   if (client_get_state(c) == IconicState)
   {
      if (use_cache && c->backing != (Pixmap)NULL)
      {
	 
	 XCopyArea(c->wm->dpy, c->backing, c->frame,
		   c->wm->theme->fg_gc, 0, 0, c->wm->dpy_width,
		   toolbar_win_offset(c), 0, 0 );
	 dbg("using pixmap cache for toolbar");
	 return;
      }

      if (c->backing == (Pixmap)NULL)
	 client_init_backing(c, c->wm->dpy_width,
			     toolbar_win_offset(c));

      theme_render_area(c->wm, c->backing, 0,0,
			c->wm->dpy_width, toolbar_win_offset(c));

      if (c->name) {
#ifdef USE_XFT
      XGlyphInfo       extents;
      XftTextExtents8(c->wm->dpy, t->toolbar_xftfont,
		      (unsigned char *) c->name, strlen(c->name),
		      &extents);
      XftDrawString8(c->xftdraw, &(t->xft_fg), t->toolbar_xftfont,
		     t->padding, t->padding + t->toolbar_xftfont->ascent,
		     (unsigned char *) c->name, strlen(c->name));
#else
	 XDrawString(c->wm->dpy, c->backing, t->toolbar_text_gc,
		     t->padding, t->padding + t->toolbar_font->ascent,
		     c->name, strlen(c->name));
#endif
      }

      cx = c->width + toolbar_win_offset(c) -
	 t->buttons[BUTTON_TOOLBAR_CLOSE].width - t->padding;
      cy = (toolbar_win_offset(c) - t->buttons[BUTTON_TOOLBAR_CLOSE].height)/2;

      theme_paint_pxm(c->wm, c->backing, cx, cy,
		      &c->wm->theme->buttons[BUTTON_TOOLBAR_CLOSE]);

      XCopyArea(c->wm->dpy, c->backing, c->frame,
		c->wm->theme->fg_gc, 0, 0, c->wm->dpy_width,
			     toolbar_win_offset(c),0, 0 );

   } else {

      cx = cy = t->padding;

      if (use_cache && c->backing != (Pixmap)NULL)
      {
	 
	 XCopyArea(c->wm->dpy, c->backing, c->frame,
		   c->wm->theme->fg_gc, 0, 0, c->width,
		   c->height, 0, 0 );
	 dbg("using pixmap cache for toolbar normal");
	 return;
      }

      if (c->backing == (Pixmap)NULL)
	 client_init_backing(c, c->width, c->height);
      
      theme_render_area(c->wm, c->backing, 0,
			  0, c->width, c->height);

      theme_paint_pxm(c->wm, c->backing, cx, cy,
		      &c->wm->theme->buttons[BUTTON_TOOLBAR]);
      
      XCopyArea(c->wm->dpy, c->backing, c->frame,
		c->wm->theme->fg_gc, 0, 0, c->width,c->height,0,0);


   }
}





