/*  Protocol compatible masqdialer server written in C
    Copyright (C) 1998 Charles P. Wright 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <syslog.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
#include <sys/shm.h>
#include <sys/sem.h>
#include <sys/ipc.h>
#include <sys/param.h>

#include "mserver.h"

#ifdef   _SEM_SEMUN_UNDEFINED
union semun
{
	int val;			/* <= value for SETVAL */
	struct semid_ds *buf;		/* <= buffer for IPC_STAT & IPC_SET */
	unsigned short int *array;	/* <= array for GETALL & SETALL */
	struct seminfo *__buf;		/* <= buffer for IPC_INFO */
};
#endif

static int shmid;
static struct shared_mem_t *shmem;
static int semid;

void shmem_init(const char * fn)
{
	key_t ipckey;
	union semun semval;

#ifdef DEBUG
	syslog(LOG_DEBUG, "Executable name is %s", fn);
#endif

	if (access(fn, R_OK) != 0) 
	{
		syslog(LOG_ERR, "Filename %s could not be accessed.", fn);	
		mserver_cleanup();
		exit(1);
	}

	ipckey = ftok(fn, 's');

	if(ipckey == -1 || ipckey == 0xffffffff)
	{
		syslog(LOG_ERR, "Could not create an ipckey, make sure you call the server with a pathname (ex: /usr/local/sbin/mserver)");
		syslog(LOG_ERR, "Filename: %s", fn);
		mserver_cleanup();
		exit(1);
	}

	semid = semget(ipckey, 1, 0600 | IPC_CREAT);

	if (semid == -1) 
	{
		syslog(LOG_ERR, "Could not create semaphore!");
		mserver_cleanup();
		exit(1);
	}

	semval.val = 1;
	if (semctl(semid, 0, SETVAL, semval) == -1) 
	{
		syslog(LOG_ERR, "Could not initialize semaphore!");
		mserver_cleanup();
		exit(1);
	}

	shmid = shmget(ipckey, sizeof(struct shared_mem_t), 0600 | IPC_CREAT);

	if (shmid == -1)
	{
		syslog(LOG_ERR, "Could not allocate shared memory segment!");
		exit(1);
	}

	shmem = (struct shared_mem_t *) shmat(shmid, NULL, 0);

	if (shmem == (struct shared_mem_t *) -1)
	{
		syslog(LOG_ERR, "Could not attach shared memory segment!");
		exit(1);
	}

	/* This won't take effect until there are no references, so we can
	   delete it now and it shouldn't have any adverse affects and also
	   solve the problem of stale segments - hopefully */
	shmctl(shmid, IPC_RMID, NULL);

	shmem->clients = 0;
	shmem->clientnames[0] = 0;
	shmem->lock_dial = 0;
	shmem->lock_kill = 0;
	shmem->pending_kill = false;
	shmem->start_in_bytes = 0;
	shmem->start_in_packets = 0;
	shmem->start_out_bytes = 0;
	shmem->start_out_packets = 0;
}


void shmem_free()
{
	int retval;
	
	retval = shmdt((char *)shmem);		

	if (retval != 0)
	{
		syslog (LOG_ERR, "Could not detach shared memory segment!");
	}		
}

static void shmem_lock()
{
	struct sembuf sops;

	sops.sem_num = 0;
	sops.sem_op = -1;
	sops.sem_flg = SEM_UNDO;
	semop(semid, &sops, 1);
}

static void shmem_unlock()
{
	struct sembuf sops;

	sops.sem_num = 0;
	sops.sem_op = 1;
	sops.sem_flg = SEM_UNDO;
	semop(semid, &sops, 1);
}

void shmem_set_clients(int nclients)
{
	shmem_lock();
	shmem->clients = nclients;
	shmem_unlock();
}

int shmem_get_clients()
{
	int nclients;

	shmem_lock();
	nclients = shmem->clients;
	shmem_unlock();

	return nclients;
}

void shmem_set_dialing(pid_t dclient)
{
	shmem_lock();
	shmem->now_dialing = dclient;
	shmem_unlock();
}

pid_t shmem_get_dialing()
{
	pid_t dclient;

	shmem_lock();
	dclient = shmem->now_dialing;
	shmem_unlock();

	return dclient;
}

void shmem_set_cspeed(int cspeed)
{
	shmem_lock();
	shmem->cspeed = cspeed;
	shmem_unlock();
}

int shmem_get_cspeed()
{
	int cspeed;

	shmem_lock();
	cspeed = shmem->cspeed;
	shmem_unlock();

	return cspeed;
}

void shmem_set_ctime(time_t ctime)
{
	shmem_lock();
	shmem->ctime = ctime;
	shmem_unlock();
}

time_t shmem_get_ctime()
{
	time_t ctime;

	shmem_lock();
	ctime = shmem->ctime;
	shmem_unlock();

	return ctime;
}

void shmem_set_who(const char *clients)
{
	shmem_lock();
	strcpy(shmem->clientnames, clients);
	shmem_unlock();
}

void shmem_get_who(char *who)
{
	shmem_lock();
	strcpy(who, shmem->clientnames);
	shmem_unlock();
}

void shmem_set_cname(const char *cname)
{
	shmem_lock();
	strcpy(shmem->cname, cname);
	shmem_unlock();
}

void shmem_get_cname(char *cname)
{
	shmem_lock();
	strcpy(cname, shmem->cname);
	shmem_unlock();
}

void shmem_set_capname(const char *capname)
{
	shmem_lock();
	strcpy(shmem->capname, capname);
	shmem_unlock();
}

void shmem_get_capname(char *capname)
{
	shmem_lock();
	strcpy(capname, shmem->capname);
	shmem_unlock();
}

void shmem_set_dial_lock(int lock_count)
{
	shmem_lock();
	shmem->lock_dial += lock_count;
	if (shmem->lock_dial < 0)
	{
		shmem->lock_dial = 0;
	}
	shmem_unlock();
}

int shmem_get_dial_lock(void)
{
	int locks;

	shmem_lock();
	locks = shmem->lock_dial;
	shmem_unlock();

	return locks;
}

void shmem_set_kill_lock(int lock_count)
{
	shmem_lock();
	shmem->lock_kill += lock_count;
	if (shmem->lock_kill < 0)
	{
		shmem->lock_kill = 0;
	}
	shmem_unlock();
}

int shmem_get_kill_lock(void)
{
	int locks;

	shmem_lock();
	locks = shmem->lock_kill;
	shmem_unlock();

	return locks;
}

void shmem_set_pending_kill(bool pending_kill)
{
	shmem_lock();
	shmem->pending_kill = pending_kill;
	shmem_unlock();
}

bool shmem_get_pending_kill(void)
{
	int pending_kill;

	shmem_lock();
	pending_kill = shmem->pending_kill;
	shmem_unlock();

	return pending_kill;
}

void shmem_set_captured(bool captured)
{
	shmem_lock();
	shmem->captured = captured;
	shmem_unlock();
}

bool shmem_get_captured(void)
{
	bool captured;

	shmem_lock();
	captured = shmem->captured;
	shmem_unlock();

	return captured;
}

void shmem_set_start_netload(unsigned long start_inb, unsigned long start_inp,
			unsigned long start_outb, unsigned long start_outp)
{
	shmem_lock();
	shmem->start_in_bytes = start_inb;
	shmem->start_in_packets = start_inp;
	shmem->start_out_bytes = start_outb;
	shmem->start_out_packets = start_outp;
	shmem_unlock();
}

void shmem_get_start_netload(unsigned long *start_inb, unsigned long *start_inp,
			unsigned long *start_outb, unsigned long *start_outp)
{
	shmem_lock();
	*start_inb = shmem->start_in_bytes;
	*start_inp = shmem->start_in_packets;
	*start_outb = shmem->start_out_bytes;
	*start_outp = shmem->start_out_packets;
	shmem_unlock();
}

void shmem_set_pppdpid(pid_t pppdpid)
{
	shmem_lock();
	shmem->pppdpid = pppdpid;
	shmem_unlock();
}

pid_t shmem_get_pppdpid(void)
{
	pid_t pppdpid;

	shmem_lock();
	pppdpid = shmem->pppdpid;
	shmem_unlock();

	return pppdpid;
}
