/*  Protocol compatible masqdialer server written in C
    Copyright (C) 1998 Charles P. Wright 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <syslog.h>
#include <pwd.h>
#include <shadow.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "mserver.h"

#ifndef NO_GLIBC
#include <crypt.h> /* libc5 has the crypt function in unistd.h */
#endif

#ifdef HAVE_PAM
#include <security/pam_appl.h>

pam_handle_t *pamh;

#endif /* HAVE_PAM */

extern int csfd;

char authuser[100];
char authpass[100];

bool authenticated = false;

void auth_upscript()
{
	char temp[1024];
	char command[1024];
	char hostname[1024];

	int retval;

	struct hostent *hostp;
	char addr_str[1024];

	if (sock_host(csfd, addr_str, 1024) != 0)
	{
		hostname[0] = '\0';
	}
	else
	{	
		hostp = gethostbyaddr((const char *) addr_str, strlen(addr_str), AF_INET);

		if (hostp)
		{
			strncpy(hostname, hostp->h_name, 1024);
		}
		else
		{		
			strncpy(hostname, addr_str, 1024);
		}
	}

	config_getvalue("authup", temp, 1024);

	if (strncmp(temp, "", 1024))
	{
		snprintf(command, 1024, "%s \"%s\" \"%s\"", temp, hostname, authuser);

		syslog(LOG_DEBUG, "Executing: %s", command);
		retval = util_system_wait(command);
		syslog(LOG_DEBUG, "Return code: %d", retval);
	}
}

void auth_downscript(void)
{
	char temp[1024];
	char command[1024];
	char hostname[1024];

	int retval;

	struct hostent *hostp;
	char addr_str[1024];

	if (sock_host(csfd, addr_str, 1024) != 0)
	{
		hostname[0] = '\0';
	}
	else
	{	
		hostp = gethostbyaddr((const char *) addr_str, strlen(addr_str), AF_INET);

		if (hostp)
		{
			strncpy(hostname, hostp->h_name, 1024);
		}
		else
		{		
			strncpy(hostname, addr_str, 1024);
		}
	}

	config_getvalue("authdown", temp, 1024);

	if (strncmp(temp, "", 1024))
	{
		snprintf(command, 1024, "%s \"%s\" \"%s\"", temp, hostname, authuser);

		syslog(LOG_DEBUG, "Executing: %s", command);
		retval = util_system_wait(command);
		syslog(LOG_DEBUG, "Return code: %d", retval);
	}
}

void auth_setverified(bool nval)
{
	if (nval)
	{
		auth_upscript();
	}
	else
	{
		if (authenticated)
		{
			auth_downscript();
		}
	}
	authenticated = nval;
	syslog(LOG_ERR, "Set Authorized: %d", authenticated);
}

bool auth_getverified(void)
{
	syslog(LOG_ERR, "Get Authorized: %d", authenticated);
	return authenticated;
}

bool auth_check(const char *cname, int auth_extra)
{
	if (!auth_checkip(cname, auth_extra))
	{
		syslog(LOG_ERR, "Invalid IP address for: %s", cname);
		return false;
	}
	
	if (!auth_checkuser(cname, auth_extra))
	{
		syslog(LOG_ERR, "Invalid user for: %s", cname);
		return false;
	}
	return true;
}

bool auth_checkip(const char *cname, int auth_extra)
{
	char addr_str[1024];

	if (sock_host(csfd, addr_str, 1024) != 0)
	{
		syslog (LOG_NOTICE, "IP:UNKNOWN\n");
	}
	else
	{	
		char ipallow[1024];
		char temp[1024];
	
		config_getvalue_cname_specified("ipallow", ipallow, cname, 1024);
					
		if (auth_extra & AUTH_CONNECT)
		{
			config_getvalue_cname_specified("ipallow_connect", temp, cname, 1024);
			if (temp[0] != '\0')
			{
				strncat(ipallow, ":", 1024);
				strncat(ipallow, temp, 1024);
			}
		}
		if (auth_extra & AUTH_DISCONNECT)
		{
			config_getvalue_cname_specified("ipallow_disconnect", temp, cname, 1024);
			if (temp[0] != '\0')
			{
				strncat(ipallow, ":", 1024);
				strncat(ipallow, temp, 1024);
			}
		}
		if (auth_extra & AUTH_CONFIG)
		{
			config_getvalue_cname_specified("ipallow_config", temp, cname, 1024);
			if (temp[0] != '\0')
			{
				strncat(ipallow, ":", 1024);
				strncat(ipallow, temp, 1024);
			}
		}

		syslog (LOG_INFO, "connection name: %s", cname);
		syslog (LOG_INFO, "allowed ips: %s", ipallow);

		if (auth_checkip_backend(addr_str, ipallow))
		{	
			return true;
		}
		else
		{
			return false;
		}
	}
	return false;
}

bool auth_checkuser(const char *cname, int auth_extra)
{
	char userallow[1024];
	char temp[1024];

	char *tok;

	bool found;

	if (!auth_getverified())
	{
		syslog (LOG_INFO, "User not verified!", userallow);
		if (config_getvalue_bool("reqauth", false))
		{
			return false;
		}
	}

	config_getvalue_cname_specified("userallow", userallow, cname, 1024);

	if (auth_extra & AUTH_CONNECT)
	{
		config_getvalue_cname_specified("userallow_connect", temp, cname, 1024);
		if (temp[0] != '\0')
		{
			strncat(userallow, ",", 1024);
			strncat(userallow, temp, 1024);
		}
	}
	if (auth_extra & AUTH_DISCONNECT)
	{
		config_getvalue_cname_specified("userallow_disconnect", temp, cname, 1024);
		if (temp[0] != '\0')
		{
			strncat(userallow, ",", 1024);
			strncat(userallow, temp, 1024);
		}
	}
	if (auth_extra & AUTH_CONFIG)
	{
		config_getvalue_cname_specified("userallow_config", temp, cname, 1024);
		if (temp[0] != '\0')
		{
			strncat(userallow, ",", 1024);
			strncat(userallow, temp, 1024);
		}
	}

	syslog (LOG_INFO, "connection name: %s", cname);
	syslog (LOG_INFO, "allowed users: %s", userallow);
				
	if (!strncmp("", userallow, 1024) || !strncmp("*", userallow, 1024))
	{
		syslog (LOG_DEBUG, "All users allowed to dial out connection.");
		return true;
	}
	
	found = false;

	tok = strtok(userallow, ",");
	do
	{
		if (tok)
		{
			syslog(LOG_INFO, "Token = \"%s\"", tok);
			if (!strncmp(tok, authuser, 1024))
			{	
				if (auth_getverified())	
				{
					found = true;
					break;
				}
			}
			else if (!strncmp(tok, "", 1024))
			{
				found = true;
				break;
			}
			else if (!strncmp(tok, "*", 1024))
			{
				found = true;
				break;
			}
		}
		if (found == true)
		{
			syslog(LOG_INFO, "True Token = \"%s\"", tok);
		}
	}
	while ((tok = strtok(NULL, ",")) != NULL);

	return found;
}

void auth_setuser(const char *username)
{
	auth_setverified(false);
	strncpy(authuser, username, 100);
}

#ifdef HAVE_PAM

/* PAM conversation function */

static int 
auth_pam_conv (int num_msg, const struct pam_message **msg,
	       struct pam_response **resp,
	       void *appdata_ptr)
{
    int replies = 0;
    struct pam_response *reply = NULL;
    
    reply = malloc(num_msg*sizeof(struct pam_response));
    if (!reply)
        return PAM_CONV_ERR;
    
    for (replies = 0; replies < num_msg; replies++) {
	switch (msg[replies]->msg_style) {
	case PAM_PROMPT_ECHO_OFF:
	    reply[replies].resp_retcode = PAM_SUCCESS;
	    reply[replies].resp = strdup(authpass);
	    break;
	case PAM_TEXT_INFO:
	    reply[replies].resp_retcode = PAM_SUCCESS;
	    reply[replies].resp = strdup("XXXX");
	    break;
	case PAM_PROMPT_ECHO_ON:
	default:
	    free (reply);
	    return PAM_CONV_ERR;
	}
    }
    *resp = reply;
    return(PAM_SUCCESS);
}


static struct pam_conv pamc = {
    &auth_pam_conv,
    NULL
};

#endif /* HAVE_PAM */

void auth_cleanup(void)
{
#ifdef HAVE_PAM
        if(config_getvalue_bool("pamauth", true)) {
#ifdef DEBUG
	  syslog(LOG_DEBUG, "Closing PAM session %d", pamh);
#endif /* DEBUG */

	  if(pamh) {
	    pam_close_session(pamh, 0);
	    pam_end(pamh, PAM_SUCCESS);
	    pamh=NULL;
	  }

#ifdef DEBUG
	  /* Workaround to avoid mserver messages being logged as PAM_pwdb */
	  closelog();
	  openlog ("mserver", LOG_NDELAY | LOG_PERROR | LOG_PID, LOG_DAEMON);
#endif /* DEBUG */
	}
#endif /* HAVE_PAM */
}

void auth_setpass(const char *password)
{
	struct passwd *pw_entry;
	char salt[4];

	int i;

#ifdef HAVE_PAM
	int pamerr;
#endif /* HAVE_PAM */
	
	auth_setverified(false);
	strncpy(authpass, password, 100);

#ifdef HAVE_PAM
        if(config_getvalue_bool("pamauth", true)) {

#ifdef DEBUG
	  syslog(LOG_DEBUG, "Authenticating using PAM");
#endif /* DEBUG */

	  if((pamerr=pam_start("mserver", authuser, &pamc, &pamh)) != PAM_SUCCESS) {
	    pam_end(pamh, pamerr);
	    pamh=NULL;
	    syslog(LOG_ERR, "Can't find /etc/pam.d/mserver!");
	  }
    
	  if(pamerr == PAM_SUCCESS && (pamerr=pam_authenticate(pamh, 0)) != PAM_SUCCESS) {
	    pam_end(pamh, pamerr);
	    pamh=NULL;
	    syslog(LOG_ERR, "Couldn't authenticate %s", authuser);
	  }

	  if(pamerr == PAM_SUCCESS && (pamerr=pam_acct_mgmt(pamh, 0)) != PAM_SUCCESS) {
	    pam_end(pamh, pamerr);
	    pamh=NULL;
	    syslog(LOG_ERR, "Couldn't set acct. mgmt for %s", authuser);
	  }

	  if(pamerr == PAM_SUCCESS && (pamerr=pam_setcred(pamh, 0)) != PAM_SUCCESS) {
	    pam_end(pamh, pamerr);
	    pamh=NULL;
	    syslog(LOG_ERR, "Couldn't set credentials for %s", authuser);
	  }

	  if(pamerr == PAM_SUCCESS && (pamerr=pam_open_session(pamh, 0)) != PAM_SUCCESS) {
	    pam_end(pamh, pamerr);
	    pamh=NULL;
	    syslog(LOG_ERR, "Couldn't open session for %s", authuser);
	  }

	  if(pamerr == PAM_SUCCESS) {
	    auth_setverified(true);
	  }

	  for (i = 0; i < 100; i++)
	    {
	      authpass[i] = '\0';
	    }

#ifdef DEBUG
	  /* Workaround to avoid mserver messages being logged as PAM_pwdb */
	  closelog();
	  openlog ("mserver", LOG_NDELAY | LOG_PERROR | LOG_PID, LOG_DAEMON);

	  syslog(LOG_DEBUG, "PAM session %d initiated", pamh);
#endif /* DEBUG */
	}
	else {
#endif /* HAVE_PAM */

#ifdef DEBUG
	  syslog(LOG_DEBUG, "Authenticating using standard authentication");
#endif /* DEBUG */

	  pw_entry = auth_getpwent(authuser);

	  if (pw_entry == NULL)
	    {
#ifdef DEBUG
	      syslog(LOG_DEBUG, "Null password entry for %s!", authuser);
#endif	
	      return;
	    }

	  strncpy(salt, pw_entry->pw_passwd, 2);
	  salt[3] = '\0';
#ifdef DEBUG
	  syslog(LOG_DEBUG, "Salt for %s: %s!", authuser, salt);
#endif

	  if (!strncmp(crypt(authpass, salt), pw_entry->pw_passwd, 1024))
	    {
	      auth_setverified(true);
	      // I don't think we want this kind of stuff hanging around
	      // in memory so nasty people can try and get at it

	      for (i = 0; i < 100; i++)
		{
		  authpass[i] = '\0';
		}
	    }

#ifdef HAVE_PAM
	}
#endif /* HAVE_PAM */
}

struct passwd *auth_getpwent(const char *authuser)
{
	FILE *afile;
	
	struct spwd *sp_local;
	struct passwd *pw_ent;

	char authfile[1024];

	bool shadow;
	bool found;

	config_getvalue("authfile", authfile, 1024);
	shadow = config_getvalue_bool("shadow", false);

	if (shadow)
	{
#ifdef DEBUG
		syslog (LOG_DEBUG, "Using shadow passwords!");	
#endif

		pw_ent = getpwnam(authuser);		
		sp_local = getspnam(authuser);		
		
		if (sp_local == NULL || pw_ent == NULL)
		{
			if (sp_local == NULL)
			{
				syslog (LOG_DEBUG, "Shadow Password entry (sp_local) null");
			}

			if (pw_ent == NULL)
			{
				syslog (LOG_DEBUG, "Password entry (pw_ent) null");
			}
			return NULL;
		}

		pw_ent->pw_passwd = sp_local->sp_pwdp;
	}
	else
	{
		if (!strncmp(authfile, "", 1024))
		{
			syslog (LOG_ERR, "Authorization file not set!");
			return NULL;
		}

		afile = fopen(authfile, "r");

		if (afile == NULL)
		{
			syslog (LOG_ERR, "Authorization file could not be opened!");
		}
		
		found = false;

		while (found == false)
		{
			pw_ent = fgetpwent(afile);
		
			if (!strncmp(pw_ent->pw_name, authuser, 1024))
			{
				found = true;
			}
		}

		fclose(afile);
	}

	return pw_ent;
}

bool auth_checkip_backend(const char *ip, const char *allowed)
{
	/* Allow up to 10 different IPs to try and match */
	char allowed_ips[10][17];
	char checkquads[4][4];
	char quads[4][4];

	bool match, badquad;
	int masks;

	int i, n, j, k;
	size_t len;

	n = masks = 0;

	len = strlen(allowed);

	if (len == 0)
	{
		return true;
	}

	for (i = 0; i < len + 1; i++)
	{
		if (allowed[i] == ':' || allowed[i] == '\0')
		{
			allowed_ips[masks][n] = '\0';
			n = 0;
			masks++;
			if (masks >= 10)
			{
				syslog (LOG_ERR, "You are only allowed 10 IP masks.  Previous masks will be overwritten!");
				masks--;
			}
			allowed_ips[masks][0] = '\0';
		}
		else if (!isdigit(allowed[i]) && (allowed[i] != '.') && (allowed[i] != '*'))
		{
			syslog(LOG_INFO, "Invalid character (%c) in an ipallow variable: '%s'!", allowed[i], allowed);
		}
		else
		{
			if (n <= 17)
			{
				allowed_ips[masks][n++] = allowed[i];
			}
			else
			{
				syslog(LOG_ERR, "Invalid IP address, exceeds 17 characters: %s", allowed);
			}
		}
	}

	n = j = k = 0;

	len = strlen(ip);

	for (i = 0; i <= len; i++)
	{
		if (ip[i] == '.' || ip[i] == '\0')
		{
			checkquads[n][j] = '\0';
			n++;
			j = 0;
		}
		else
		{
			if (j < 4)
			{
				checkquads[n][j] = ip[i];
			}
			j++;
		}
	}
#ifdef DEBUG
	syslog (LOG_DEBUG, "Quadded Check IP: %s.%s.%s.%s", checkquads[0], checkquads[1], checkquads[2], checkquads[3]);	
#endif

	i = 0;
	
	match = false;
	
	while ((match == false) && (i < masks))
	{
		badquad = false;

		n = j = k = 0;

		len = strlen(allowed_ips[i]);
		for (k = 0; k <= len; k++)
		{
			if (allowed_ips[i][k] == '.' || allowed_ips[i][k] == '\0')
			{
				quads[n][j] = '\0';
				n++;
				j = 0;
			}
			else
			{
				if (j < 4)
				{
					quads[n][j] = allowed_ips[i][k];
				}
				j++;
			}
		}
	
#ifdef 	DEBUG
		syslog (LOG_DEBUG, "Quadded IP: %s.%s.%s.%s", quads[0], quads[1], quads[2], quads[3]);	
#endif

		badquad = false;

		for (n = 0; n < 4; n++)
		{
			if (!strncmp(quads[n], "*", 4))
			{
#ifdef DEBUG
				syslog (LOG_DEBUG, "Good quad [%d]: %s:%s", n, quads[n], checkquads[n]);
#endif
			}
			else if (atoi(quads[n]) == atoi(checkquads[n]))
			{
#ifdef DEBUG
				syslog (LOG_DEBUG, "Good quad [%d]: %s:%s", n, quads[n], checkquads[i]);
#endif
			}
			else
			{	
#ifdef DEBUG
				syslog (LOG_DEBUG, "Bad quad [%d]: %s:%s", n, quads[n], checkquads[n]);
#endif
				badquad = true;
			}
		}

		if (badquad == false)
		{
#ifdef DEBUG
			syslog(LOG_DEBUG, "Match!");
#endif
			match = true;
		}

		i++;
#ifdef DEBUG
		syslog (LOG_DEBUG, "i: %d masks: %d", i, masks);
#endif
	}

	return match;	
}
