// Preferences.cc - source file for the mailfilter program
// Copyright (c) 2000 - 2002  Andreas Bauer <baueran@in.tum.de>
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
// USA.

#include <string>
#include <list>
#include <cstdlib>
#include <iostream>
#include <strstream>
extern "C" {
#include <ctype.h>
}
#include "Preferences.hh"
#include "Account.hh"
#include "RFC822.hh"
#include "i18n.hh"
#include "mailfilter.hh"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

using namespace std;

namespace pref {

  Preferences::Preferences(const string& rcfileName) {
    // Set the program's default parameters
    file = rcfileName;
    maxsize = 0;
    maxsizeFriends = 0;
    icase = true;
    normal = false;
    maxLineLength = 0;    // Maximum line length of any header field, normally 998 bytes
    timeOut = 30;         // Number of seconds to wait for the server after a command was issued, normally 30 seconds
    regType = 0;
    regNewline = 0;
    verboseLevel = 3;     // Verbose default level, if none is given in the preferences file
    test = false;
    showHeaders = false;
    delDuplicates = false;
    compiled = false;     // Flag, indicating whether the filters are already pre-compiled (or not)
  }
  
  
  Preferences::~Preferences() {
    // Now clean up the memory used by Regular Expressions
    vector<filterInfo>::iterator curFilter = getFilters()->begin();
    vector<friendInfo>::iterator curFriend = getFriends()->begin();

    // Only try to free memory of filters, if they were pre-compiled before (otherwise we're having a nice crash here...)
    if (compiled) {
      while (curFilter != getFilters()->end()) {
	regExp.free(&curFilter->cFilter);
	curFilter++;
      }

      while (curFriend != getFriends()->end()) {
	regExp.free(&curFriend->cFilter);
	curFriend++;
      }
    }

    // Clean up the rest...
    servers.clear();
    filters.clear();
    friends.clear();
  }
  
  
  // Taken from Bjarne Stroustrup's "The C++ Programming Language",
  // Second Edition, Section 20.3.8
  int Preferences::cmpNoCase(const string& s, const string& s2) {
    string::const_iterator p = s.begin();
    string::const_iterator p2 = s2.begin();

    while (p != s.end() && p2 != s2.end()) {
      if (toupper(*p) != toupper(*p2))
	return (toupper(*p) < toupper(*p2))? -1 : 1;
      ++p;
      ++p2;
    }
    
    return (s2.size() == s.size())? 0 : (s.size() < s2.size())? -1 : 1;
  }


  void Preferences::keyword(const string keyw, const string currOp, const string ident) {
    filterInfo newFilter;
    friendInfo newFriend;
    
    if ( (cmpNoCase(keyw, "SERVER") == 0) && currOp == "=" )
      popServer.name = ident;
    else if ( (cmpNoCase(keyw, "USER") == 0) && currOp == "=" )
      popServer.user = ident;
    else if ( (cmpNoCase(keyw, "PASS") == 0) && currOp == "=" )
      popServer.pass = ident;
    else if ( (cmpNoCase(keyw, "PROTOCOL") == 0) && currOp == "=" ) {
      if (cmpNoCase(ident, "pop3") == 0)
	popServer.protocol = POP3;
      else if (cmpNoCase(ident, "imap") == 0)
	popServer.protocol = IMAP;
      else {
	showErrorParameter(keyw, currOp, ident);
	throw MalformedPrefsFile();
      }
    }
    else if ( (cmpNoCase(keyw, "LOGFILE") == 0) && currOp == "=" )
      setLogfile(ident);
    else if ( (cmpNoCase(keyw, "REG_CASE") == 0) && currOp == "=" ) {
      if (cmpNoCase(ident, "no") == 0)
	icase = false;
      else if (cmpNoCase(ident, "yes") == 0)
	icase = true;
      else {
	showErrorParameter(keyw, currOp, ident);
	throw MalformedPrefsFile();
      }
    }
    else if ( (cmpNoCase(keyw, "NORMAL") == 0) && currOp == "=" ) {
      if (cmpNoCase(ident, "no") == 0)
	normal = false;
      else if (cmpNoCase(ident, "yes") == 0)
	normal = true;
      else {
	showErrorParameter(keyw, currOp, ident);
	throw MalformedPrefsFile();	      
      }
    }
    else if ( (cmpNoCase(keyw, "ICASE") == 0) || (cmpNoCase(keyw, "MAXSIZE") == 0) || (cmpNoCase(keyw, "MODE") == 0) && currOp == "=" ) {
      showErrorDeprecated(keyw);  // The deprecated keywords have to be responded to by an error handler
      throw MalformedPrefsFile();
    }
    else if ( (cmpNoCase(keyw, "REG_NEWLINE") == 0) && currOp == "=" ) {
      if (cmpNoCase(ident, "no") == 0)
	regNewline = 0;
      else if (cmpNoCase(ident, "yes") == 0)
	regNewline = REG_NEWLINE;
      else {
	showErrorParameter(keyw, currOp, ident);
	throw MalformedPrefsFile();	      
      }
    }
    else if ( (cmpNoCase(keyw, "REG_TYPE") == 0) && currOp == "=" ) {
      if (cmpNoCase(ident, "basic") == 0)
	regType = 0;
      else if (cmpNoCase(ident, "extended") == 0)
	regType = REG_EXTENDED;
      else {
	showErrorParameter(keyw, currOp, ident);
	throw MalformedPrefsFile();
      }
    }
    else if (cmpNoCase(keyw, "DENY") == 0) {
      if (currOp == "<>") {
	newFilter.filter = ident;
	newFilter.case_sensitive = CASE_DEFAULT;
	newFilter.negative = true;
	newFilter.matched = false;
	filters.push_back(newFilter);
      }
      else {
	newFilter.filter = ident;
	newFilter.case_sensitive = CASE_DEFAULT;
	newFilter.negative = false;
	newFilter.matched = false;
	filters.push_back(newFilter);
      }
    }
    else if (cmpNoCase(keyw, "DENY_NOCASE") == 0) {
      if (currOp == "<>") {
	newFilter.filter = ident;
	newFilter.case_sensitive = CASE_INSENSITIVE;
	newFilter.negative = true;
	newFilter.matched = false;
	filters.push_back(newFilter);
      }
      else {
	newFilter.filter = ident;
	newFilter.case_sensitive = CASE_INSENSITIVE;
	newFilter.negative = false;
	newFilter.matched = false;
	filters.push_back(newFilter);
      }
    }
    else if (cmpNoCase(keyw, "DENY_CASE") == 0) {
      if (currOp == "<>") {
	newFilter.filter = ident;
	newFilter.case_sensitive = CASE_SENSITIVE;
	newFilter.negative = true;
	newFilter.matched = false;
	filters.push_back(newFilter);
      }
      else {
	newFilter.filter = ident;
	newFilter.case_sensitive = CASE_SENSITIVE;
	newFilter.negative = false;
	newFilter.matched = false;
	filters.push_back(newFilter);
      }
    }
    else if ( (cmpNoCase(keyw, "ALLOW") == 0) && currOp == "=" ) {
      newFriend.filter = ident;
      friends.push_back(newFriend);
    }
    else if ( (cmpNoCase(keyw, "SHOW_HEADERS") == 0) && currOp == "=" ) {
      if (cmpNoCase(ident, "no") == 0)
	showHeaders = false;
      else if (cmpNoCase(ident, "yes") == 0)
	showHeaders = true;
      else {
	showErrorParameter(keyw, currOp, ident);
	throw MalformedPrefsFile();	      
      }      
    }
    else if ( (cmpNoCase(keyw, "DEL_DUPLICATES") == 0) && currOp == "=" ) {
      if (cmpNoCase(ident, "no") == 0)
	delDuplicates = false;
      else if (cmpNoCase(ident, "yes") == 0)
	delDuplicates = true;
      else {
	showErrorParameter(keyw, currOp, ident);
	throw MalformedPrefsFile();	      
      }     
    }
    else if ( (cmpNoCase(keyw, "TEST") == 0) && currOp == "=" ) {
      if (cmpNoCase(ident, "no") == 0)
	test = false;
      else if (cmpNoCase(ident, "yes") == 0)
	test = true;
      else {
	showErrorParameter(keyw, currOp, ident);
	throw MalformedPrefsFile();	      
      }
    }
    else {
      showErrorParameter(keyw, currOp, ident);
      throw MalformedPrefsFile();
    }
  }

  
  void Preferences::keyword(const string keyw, const string currOp, int ident) {
    if (currOp == "=") {
      if (cmpNoCase(keyw, "PORT") == 0) {
	popServer.port = ident;
	servers.push_back(popServer);              // Always push here, cause after this either another server gets configured, or we're done
      }
      else if (cmpNoCase(keyw, "MAXSIZE_DENY") == 0)
	maxsize = ident;
      else if (cmpNoCase(keyw, "MAXSIZE_ALLOW") == 0)
	maxsizeFriends = ident;
      else if (cmpNoCase(keyw, "MAXLENGTH") == 0)
	maxLineLength = ident;
      else if (cmpNoCase(keyw, "TIMEOUT") == 0)
	timeOut = ident;
      else if (cmpNoCase(keyw, "VERBOSE") == 0) {
	if (ident < MIN_VERBOSE || ident > MAX_VERBOSE) {
	  showErrorParameter(keyw, currOp, ident);
	  throw MalformedPrefsFile();
	}
	else
	  setVerboseLevel(ident);
      }
    }
    else {
      showErrorParameter(keyw, currOp, ident);
      throw MalformedPrefsFile();
    }
  }
 

  // Precompile all DENY and ALLOW Regular Expressions
  int Preferences::precompileExpressions(void) {
    vector<filterInfo>::iterator curFilter = getFilters()->begin();
    vector<friendInfo>::iterator curFriend = getFriends()->begin();
    int error = 0;

    try {
      // Compile spam filters
      while ( curFilter != getFilters()->end() ) {
	error = 0;

	if ( ((getIcase() == false) && (curFilter->case_sensitive == CASE_DEFAULT)) || (curFilter->case_sensitive == CASE_INSENSITIVE) ) {
	  if ((error = regExp.comp(&curFilter->cFilter, curFilter->filter, REG_ICASE | getRegType() | getRegNewline())) != 0) {
	    size_t errLength = regExp.error(error, &curFilter->cFilter, (char*)NULL, (size_t)0);
	    char errBuf[errLength + 1];
	    regExp.error(error, &curFilter->cFilter, errBuf, sizeof errBuf);
	    cerr << PACKAGE << _(": Could not pre-compile this Regular Expression '") << curFilter->filter << "'." << endl;
	    cerr << PACKAGE << _(": Internal error code from the Regular Expression library: ") << error << endl;
	    cerr << PACKAGE << ": " << errBuf << endl;
	    return -1;
	  }
	}
	else if ( ((getIcase() == true) && (curFilter->case_sensitive == CASE_DEFAULT)) || (curFilter->case_sensitive == CASE_SENSITIVE) ) {
	  if ((error = regExp.comp(&curFilter->cFilter, curFilter->filter, getRegType() | getRegNewline())) != 0) {
	    size_t errLength = regExp.error(error, &curFilter->cFilter, (char*)NULL, (size_t)0);
	    char errBuf[errLength + 1];
	    regExp.error(error, &curFilter->cFilter, errBuf, sizeof errBuf);
	    cerr << PACKAGE << _(": Could not pre-compile this Regular Expression '") << curFilter->filter << "'." << endl;
	    cerr << PACKAGE << _(": Internal error code from the Regular Expression library: ") << error << endl;
	    cerr << PACKAGE << ": " << errBuf << endl;
	    return -1;
	  }
	}
	
	curFilter++;
      }
      
      // Compile 'friendly' filters
      while ( curFriend != getFriends()->end() ) {
	error = 0;
	
	if (getIcase() == false) {
	  if ((error = regExp.comp(&curFriend->cFilter, curFriend->filter, REG_ICASE | getRegType() | getRegNewline())) != 0) {
	    size_t errLength = regExp.error(error, &curFriend->cFilter, (char*)NULL, (size_t)0);
	    char errBuf[errLength + 1];
	    regExp.error(error, &curFriend->cFilter, errBuf, sizeof errBuf);
	    cerr << PACKAGE << _(": Could not pre-compile this Regular Expression '") << curFriend->filter << "'." << endl;
	    cerr << PACKAGE << _(": Internal error code from the Regular Expression library: ") << error << endl;
	    cerr << PACKAGE << ": " << errBuf << endl;
	    return -1;
	  }
	}      
	else {
	  if ((error = regExp.comp(&curFriend->cFilter, curFriend->filter, getRegType() | getRegNewline())) != 0) {
	    size_t errLength = regExp.error(error, &curFriend->cFilter, (char*)NULL, (size_t)0);
	    char errBuf[errLength + 1];
	    regExp.error(error, &curFriend->cFilter, errBuf, sizeof errBuf);
	    cerr << PACKAGE << _(": Could not pre-compile this Regular Expression '") << curFriend->filter << "'." << endl;
	    cerr << PACKAGE << _(": Internal error code from the Regular Expression library: ") << error << endl;
	    cerr << PACKAGE << ": " << errBuf << endl;
	    return -1;
	  }
	}
	
	curFriend++;
      }
      
      return 0;
    }
    catch (...) {
      throw;
    };
  }
  

  const vector<serverInfo>& Preferences::getServers(void) {
    return servers;
  }
  
  
  int Preferences::getMaxsize(void) {
    return maxsize;
  }


  int Preferences::getMaxsizeFriends(void) {
    return maxsizeFriends;
  }


  bool Preferences::getIcase(void) {
    return icase;
  }


  bool Preferences::isNormal(void) {
    return normal;
  }


  const string& Preferences::getLogfile(void) {
    return logfile;
  }


  void Preferences::setLogfile(const string& newLogfile) {
    // Only set the log file to a new value if none was previously specified
    // This is important to override the rcfile with a -L command line switch
    if (!logfile.length())
      logfile = newLogfile;
  }

  
  void Preferences::setCompiled(bool flag) {
    compiled = flag;
  }


  bool Preferences::getCompiled(void) {
    return compiled;
  }


  int Preferences::getVerboseLevel(void) {
    return verboseLevel;
  }


  void Preferences::setVerboseLevel(int newMode) {
    if (newMode != verboseLevel)
      verboseLevel = newMode;
  }


  int Preferences::getRegType(void) {
    return regType;
  }


  int Preferences::getRegNewline(void) {
    return regNewline;
  }


  bool Preferences::getShowHeaders(void) {
    return showHeaders;
  }


  bool Preferences::getTestMode(void) {
    return test;
  }


  int Preferences::getMaxLineLength(void) {
    return maxLineLength;
  }


  // Return vector with all compiled filters
  vector<filterInfo>* Preferences::getFilters(void) {
    return &filters;
  }


  // Return vector with all compiled friend rules
  vector<friendInfo>* Preferences::getFriends(void) {
    return &friends;
  }


  // Shows an error if a keyword is deprecated
  void Preferences::showErrorDeprecated(const string& dKeyword) {
    cerr << PACKAGE << _(": Deprecated keyword '") << dKeyword << _("' in '") << file << "'." << endl;;
    
    if (dKeyword == "ICASE")
      cerr << PACKAGE << _(": Instead use the keyword 'REG_CASE' from now on.") << endl;
    else if (dKeyword == "MODE")
      cerr << PACKAGE << _(": Instead use the keyword 'VERBOSE' from now on.") << endl;

    cerr << PACKAGE << _(": Consult the mailfilterrc(5) man page for further details.") << endl;
  }


  // Shows an error if a keyword got wrong arguments
  void Preferences::showErrorParameter(const string& keyword, const string& currOp, const string& parameter) {
    cerr << PACKAGE << _(": A keyword in '") << file << _("' contains invalid parameters.") << endl;
    cerr << PACKAGE << ": '" <<  keyword << currOp << parameter << "'" << endl;
  }
  void Preferences::showErrorParameter(const string& keyword, const string& currOp, int value) {
    cerr << PACKAGE << _(": A keyword in '") << file << _("' contains invalid parameters.") << endl;
    cerr << PACKAGE << ": '" <<  keyword << currOp << value << "'" << endl;
  }

  // Returns true if Mailfilter should delete duplicates, false otherwise
  bool Preferences::getDelDubs(void) {
    return delDuplicates;
  }

  
  // Set time out for server response to issued commands, e. g. OK or ERR after an issued STAT command
  void Preferences::setTimeOut(int newTimeOut) {
    timeOut = newTimeOut;
  }

  
  int Preferences::getTimeOut(void) {
    return timeOut;
  }

}
