%{
/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/****
This parser can read most Debian package database files.
****/
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <ctype.h>
#include "magpie.h"

#define YYERROR_VERBOSE

extern FILE *yyin;

extern char *yytext;
extern int yyleng;
extern int lineno;

struct package_info *current = NULL;

struct package_list *tokenize (char *name, char *restriction);

#define PING() { fprintf (stderr, "%s, line %d\n", __FILE__, __LINE__); }

static char * strlwr (char *s);

/*+
This is a list of the top-level sections in Debian.  The elements
of the array must match the parser logic.
+*/
const char *sections[] = {
	"main", 
	"contrib", 
	"non-US.main", 
	"non-US.contrib", 
	"non-US.non-free", 
	"non-US",
	"non-free", 
	"local",
	"affinity",		/* used by the author, recommended for  */
	"alien",		/* used by g2 player */
	"unknown"
};

/*+
This is a list of the categories in Debian.  The elements of the
array must match the parser logic.
+*/
const char *categories[] =
	{
	"unspecified",
	"admin",
	"base",
	"comm",
	"devel",
	"doc",
	"editors",
	"electronics",
	"games",
	"graphics",
	"hamradio",
	"interpreters",
	"libs",
	"mail",
	"math",
	"misc",
	"net",
	"news",
	"oldlibs",
	"otherosfs",
	"shells",
	"sound",
	"tex",
	"text",
	"utils",
	"web",
	"x11",

	/* this section appears to be a mistake since it's used by 
	 * only one package */
	"science",

	/* these sections are introduced by Stormix */
	"stormbas",
	"storm-sas",

	"unknown"
	};

/*+
This is a list of the priorities in Debian.  The elements of the
array must match the parser logic.
+*/
const char *priorities[] = 
	{ "required", "important", "standard", "optional", "extra", "unknown" };

/*+
This is a list of architectures in Debian.  The elements of the
array must match the parser logic.
+*/
const char *architectures[] =
	{ "all", "alpha", "arm", "hurd-i386", "i386", "m68k", "powerpc", "sparc" };


%}

%token	T_PACKAGE T_VERSION T_PRIORITY T_SECTION T_MAINTAINER
%token	T_DEPENDS T_RECOMMENDS T_SUGGESTS T_ARCHITECTURE
%token	T_FILENAME T_SIZE T_MD5SUM T_DESCRIPTION 
%token	T_PROVIDES T_REPLACES T_CONFLICTS T_PREDEPENDS T_ESSENTIAL
%token	T_SOURCE T_INSTALLED_SIZE T_STATUS T_CONFFILES
%token	T_CONFIG_VERSION 
%token	T_EOM T_EOF

%token	T_DESCRIPTION_TITLE T_DESCRIPTION_BODY
%token	T_DEC T_HEX

%token	T_MAIN T_CONTRIB T_NON_US T_NON_FREE T_LOCAL T_AFFINITY T_ALIEN
%token	T_UNKNOWN

%token	T_ADMIN T_BASE T_COMM T_DEVEL T_DOC T_EDITORS T_ELECTRONICS
%token	T_GAMES T_GRAPHICS T_HAMRADIO T_INTERPRETERS T_LIBS
%token	T_MAIL T_MATH T_MISC T_NET T_NEWS T_OLDLIBS T_OTHEROSFS
%token	T_SHELLS T_SOUND T_TEX T_TEXT T_UTILS T_WEB T_X11 T_SCIENCE
%token	T_STORMBASE T_STORMSAS

%token	T_REQUIRED T_IMPORTANT T_STANDARD T_OPTIONAL T_EXTRA

%token	T_ALL T_ALPHA T_ARM T_HURD_I386 T_I386 T_M68K T_POWERPC T_SPARC

%token	T_TRUE T_FALSE

%token	T_TOKEN T_SLASH
%token	T_LT_RESTRICTION T_GT_RESTRICTION T_EQ_RESTRICTION

%left	T_AND
%left	T_OR
%token	T_SLASH

%%

input
	: package 
		{
		return (int) current;
		}
	  eom
		{
		current = 0;
		}
	| T_EOF
		{
		return 0;
		}
	;

package
	: package_item
	| package package_item
	;

package_item
	: T_PACKAGE token
		{
		current = (struct package_info *) malloc (sizeof (struct package_info));
		memset (current, 0, sizeof (struct package_info));
		current->name = strlwr ((char *) $2);
		current->flag = T_DESCRIPTION;
		}
	| T_VERSION token
		{
		current->version = (char *) $2;
		}
	| T_PRIORITY priority
		{
		current->priority = $2;
		}
	| T_SECTION category
		{
		current->section = 0;
		current->category = $2;
		}
	| T_SECTION section T_SLASH category
		{
		current->section = $2;
		current->category = $4;
		}
	| T_SECTION T_LOCAL
		{
		current->section = 4;		/* local, unknown */
		current->category = 0;		/* unspecified */
		}
	| T_SECTION T_ALIEN
		{
		current->section = 6;		/* local, unknown */
		current->category = 0;		/* unspecified */
		}
	| T_SECTION section				/* handles some non-US items */
		{
		current->section = $2;
		current->category = 0;
		}
	| T_SECTION T_NON_US T_SLASH category
		{
		current->section = 5;		/* non-US, unadorned by subsection */
		current->category = $4;
		}
	| T_MAINTAINER token
		{
		current->maintainer = (char *) $2;
		}
	| T_DEPENDS list
		{
		current->depends = (struct package_list *) $2;
		}
	| T_RECOMMENDS list
		{
		current->recommends = (struct package_list *) $2;
		}
	| T_SUGGESTS list
		{
		current->suggests = (struct package_list *) $2;
		}
	| T_ARCHITECTURE architecture
		{
		current->architecture = $2;
		}
	| T_FILENAME filename
		{
		current->filename = (char *) $2;
		}
	| T_SIZE dec
		{
		current->size = $2;
		}
	| T_MD5SUM hex
		{
		current->md5sum = (char *) $2;
		}
	| T_DESCRIPTION summary
		{
		current->flag = $1;	/* just in case... */
		current->summary = (char *) $2;
		}
	| T_DESCRIPTION summary { current->flag = $1; } description
		{
		current->summary = (char *) $2;
		/* current->description is automatically set */
		}
	| T_INSTALLED_SIZE dec
		{
		current->installed_size = $2;
		}
	| T_SOURCE list
		{
		current->source = (struct package_list *) $2;
		}
	| T_PROVIDES list
		{
		current->provides = (struct package_list *) $2;
		}
	| T_CONFLICTS list
		{
		current->conflicts = (struct package_list *) $2;
		}
	| T_REPLACES list
		{
		current->replaces = (struct package_list *) $2;
		}
	| T_ESSENTIAL boolean
		{
		current->essential = $2;
		}
	| T_PREDEPENDS list
		{
		current->predepends = (struct package_list *) $2;
		}
	| T_STATUS token token token
		{
		current->status[0] = (char *) $2;
		current->status[1] = (char *) $3;
		current->status[2] = (char *) $4;
		}
	| T_CONFFILES { current->flag = T_CONFFILES; } description
		{
		current->flag = T_DESCRIPTION;
		}
	| T_CONFIG_VERSION token
		{
		current->config_version = (char *) $2;
		}
	;

token_leg
	: token
		{
		$$ = (int) tokenize (strlwr ((char *) $1), NULL);
		}
	| token restriction
		{
		$$ = (int) tokenize (strlwr ((char *) $1), (char *) $2);
		}
	;

list
	: dj_list
		{
		$$ = $1;
		}
	| dj_list T_AND list 
		{
		struct package_list *d = (struct package_list *) $1;
		d->next = (struct package_list *) $3;
		$$ = (int) d;
		}
	;

dj_list
	: token_leg
		{
		$$ = $1;
		}
	| token_leg T_OR dj_list
		{
		struct package_list *d = (struct package_list *) $1;
		d->down = (struct package_list *) $3;
		$$ = (int) d;
		}
	;

filename : token { $$ = $1; } ;

restriction
	: T_LT_RESTRICTION
		{
		$$ = (int) strdup (yytext);
		}
	| T_GT_RESTRICTION
		{
		$$ = (int) strdup (yytext);
		}
	| T_EQ_RESTRICTION
		{
		$$ = (int) strdup (yytext);
		}
	;

architecture
	: T_ALL				{ $$ = 0; }
	| T_ALPHA			{ $$ = 1; }
	| T_ARM				{ $$ = 2; }
	| T_HURD_I386		{ $$ = 3; }
	| T_I386			{ $$ = 4; }
	| T_M68K			{ $$ = 5; }
	| T_POWERPC			{ $$ = 6; }
	| T_SPARC			{ $$ = 7; }
	;

priority
	: T_REQUIRED		{ $$ = 0; }
	| T_IMPORTANT		{ $$ = 1; }
	| T_STANDARD		{ $$ = 2; }
	| T_OPTIONAL		{ $$ = 3; }
	| T_EXTRA			{ $$ = 4; }
	;

section
	: T_MAIN						{ $$ = 0; }
	| T_CONTRIB						{ $$ = 1; }
	| T_NON_US T_SLASH T_MAIN		{ $$ = 2; }
	| T_NON_US T_SLASH T_CONTRIB	{ $$ = 3; }
	| T_NON_US T_SLASH T_NON_FREE	{ $$ = 4; }	/* 5 handled elsewhere */
	| T_NON_FREE					{ $$ = 6; }
	| T_LOCAL						{ $$ = 7; }
	| T_AFFINITY					{ $$ = 8; }
	| T_ALIEN						{ $$ = 9; }
	| T_UNKNOWN						{ $$ = 10; }
	;

category
	: T_ADMIN			{ $$ = 1;	}
	| T_BASE			{ $$ = 2;	}
	| T_COMM			{ $$ = 3;	}
	| T_DEVEL			{ $$ = 4;	}
	| T_DOC				{ $$ = 5;	}
	| T_EDITORS			{ $$ = 6;	}
	| T_ELECTRONICS		{ $$ = 7;	}
	| T_GAMES			{ $$ = 8;	}
	| T_GRAPHICS		{ $$ = 9;	}
	| T_HAMRADIO		{ $$ = 10;	}
	| T_INTERPRETERS	{ $$ = 11;	}
	| T_LIBS			{ $$ = 12;	}
	| T_MAIL			{ $$ = 13;	}
	| T_MATH			{ $$ = 14;	}
	| T_MISC			{ $$ = 15;	}
	| T_NET				{ $$ = 16;	}
	| T_NEWS			{ $$ = 17;	}
	| T_OLDLIBS			{ $$ = 18;	}
	| T_OTHEROSFS		{ $$ = 19;	}
	| T_SHELLS			{ $$ = 20;	}
	| T_SOUND			{ $$ = 21;	}
	| T_TEX				{ $$ = 22;	}
	| T_TEXT			{ $$ = 23;	}
	| T_UTILS			{ $$ = 24;	}
	| T_WEB				{ $$ = 25;	}
	| T_X11				{ $$ = 26;	}
	| T_SCIENCE			{ $$ = 27;	 /* }

	| T_MAIN			{ $$ = 28;	}
	| T_CONTRIB			{ $$ = 29; }
	| T_NON_FREE		{ $$ = 30;  */ }

	/* Stormix */
	| T_STORMBASE		{ $$ = 28;	}
	| T_STORMSAS		{ $$ = 29;	}
	;

token
	: T_TOKEN
		{
		int i;

		$$ = 0;
		for (i = 0; i < yyleng && $$ == 0; i++) 
			if (!isspace (yytext[i]))
				$$ = (int) strdup (&yytext[i]);
		}
	;

summary 
	: T_DESCRIPTION_TITLE
		{
		int i;

		$$ = 0;
		for (i = 0; i < yyleng && $$ == 0; i++) 
			if (!isspace (yytext[i]))
				$$ = (int) strdup (&yytext[i]);
		}
	;

body
	: T_DESCRIPTION_BODY
		{
		if (yytext[1] == '.' && yytext[2] == '\0')
			yytext[1] = '\0';
		$$ = (int) strdup (&yytext[1]);
		}
	;

description 
	: body
		{
		switch (current->flag) {
		case T_CONFFILES:
			if (current->cfcnt < MAX_DESCCNT)
				current->conffiles[current->cfcnt++] = (char *) $1;
			else {
				fprintf (stderr, "truncating long description for %s\n", 
					current->name);
				free ((char *) $1);
			}
			break;
		case T_DESCRIPTION:
		default:
			if (current->desccnt < MAX_DESCCNT)
				current->description[current->desccnt++] = (char *) $1;
			else {
				fprintf (stderr, "truncating long description for %s\n", 
					current->name);
				free ((char *) $1);
			}
			break;
		 }
		}
	| description body
		{
		switch (current->flag) {
		case T_CONFFILES:
			if (current->cfcnt < MAX_DESCCNT)
				current->conffiles[current->cfcnt++] = (char *) $2;
			else {
				fprintf (stderr, "truncating long conffiles for %s\n", 
					current->name);
				free ((char *) $2);
			}
			break;
		case T_DESCRIPTION:
		default:
			if (current->desccnt < MAX_DESCCNT)
				current->description[current->desccnt++] = (char *) $2;
			else {
				fprintf (stderr, "truncating long conffiles for %s\n", 
					current->name);
				free ((char *) $2);
			}
			break;
		 }
		}
	;

boolean 
	: T_TRUE	{ $$ = 1; }
	| T_FALSE 	{ $$ = 0; }
	;

dec : T_DEC { $$ =  atol (yytext); } ;

hex : T_HEX 
		{
		$$ = (int) strdup (yytext);
		}
	;

eom : T_EOM | T_EOF ;

%%

struct package_list *tokenize (char *name, char *restriction)
{
	struct package_list *d;
	d = (struct package_list *) malloc (sizeof (struct package_list));
	d->name = name;
	d->restriction = restriction;
	d->next = 0;
	d->down = 0;
	return d;
}

int yywrap (void) { return 1; }

int yyerror (const char *msg)
{ 
	fprintf (stderr, "line %d: %s near \"%s\"\n", lineno, msg, yytext);
	return 0;
}


/*+
Convert a string lower case
+*/
static char *strlwr (char *s)
{
	char *p;

	assert (s);
	for (p = s; *p; p++) {
		*p = tolower (*p);
	}
	return s;
}


/*+
Release a package list
+*/
void free_list (struct package_list *d)
{
	struct package_list *q, *qq;

	while (d) {
		if (d->name)		{ free (d->name); d->name = 0; }
		if (d->restriction)	{ free (d->restriction); d->restriction = 0; }

		q = d->down;
		while (q) {
			if (q->name)		{ free (q->name); q->name = 0; }
			if (q->restriction)	{ free (q->restriction); q->restriction = 0; }

			qq = q->down;
			q->down = 0;
			free (q);
			q = qq;
		}
		d->down = 0;

		qq = d->next;
		d->next = 0;
		free (d);
		d = qq;
	}
}


/*+
Release the memory associated with a package_info object.
We do this so we can check for memory leaks via profiling.
We also clear the pointers to avoid following freed structures
in a debugger.
+*/
void free_info (struct package_info *p)
{
	int i;

	assert (p);

	if (p->name)		{ free (p->name); p->name = 0; }
	if (p->version)		{ free (p->version); p->version = 0; }

	if (p->filename)	{ free (p->filename); p->filename = 0; }
	if (p->maintainer)	{ free (p->maintainer); p->maintainer = 0; }
	if (p->md5sum)		{ free (p->md5sum); p->md5sum = 0; }
	if (p->summary)		{ free (p->summary); p->summary = 0; }
	if (p->status[0])	{ free (p->status[0]); p->status[0] = 0; }
	if (p->status[1])	{ free (p->status[1]); p->status[1] = 0; }
	if (p->status[2])	{ free (p->status[2]); p->status[2] = 0; }
	if (p->config_version)
						{ free (p->config_version); p->config_version = 0; }
	if (p->installed_version)
					{ free (p->installed_version); p->installed_version = 0; }

	for (i = 0; i < MAX_DESCCNT; i++) {
		if (p->description[i]) {
			free (p->description[i]);
			p->description[i] = 0;
		}
	}

	for (i = 0; i < MAX_DESCCNT; i++) {
		if (p->conffiles[i]) {
			free (p->conffiles[i]);
			p->conffiles[i] = 0;
		}
	}

	if (p->source)		{ free_list (p->source); p->source = 0; }
	if (p->depends)		{ free_list (p->depends); p->depends = 0; }
	if (p->predepends)	{ free_list (p->predepends); p->predepends = 0; }
	if (p->recommends)	{ free_list (p->recommends); p->recommends = 0; }
	if (p->suggests)	{ free_list (p->suggests); p->suggests = 0; }
	if (p->conflicts)	{ free_list (p->conflicts); p->conflicts = 0; }
	if (p->replaces)	{ free_list (p->replaces); p->replaces = 0; }
	if (p->provides)	{ free_list (p->provides); p->provides = 0; }

	if (p->r_depends)	{ free_list (p->r_depends); p->r_depends = 0; }
	if (p->r_suggests)	{ free_list (p->r_suggests); p->r_suggests = 0; }
	if (p->r_recommends)
						{ free_list (p->r_recommends); p->r_recommends = 0; }

	free (p);
}
