/*
Magpie - reference librarian for Debian systems
Copyright (C) 2000  Bear Giles <bgiles@coyotesong.com>

This program is free software; you may redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the license, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

static const char rcsid[] = "$Id$";

/*****
This module kicks out the reference stacks in docbook format.
*****/
#include <assert.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/wait.h>
#include "magpie.h"

#define	OUTPUT_DIR	"docbook"
#define	OUTPUT_FILE	"docbook/all.xml"

extern int mkdir (const char *, mode_t);

extern char timestamp[];

/*
 * This is the material written at the head of the document.
 */
static const char header[] = "\
<?xml version='1.0'?>\n\
<!DOCTYPE article PUBLIC '-//Norman Walsh//DTD DocBk XML V3.1.4//EN'\n\
                         'http://nwalsh.com/docbook/xml/3.1.4/docbookx.dtd'>\n\
<article>\n\
<artheader>\n\
<title>(title goes here)</title>\n\
</artheader>\n\
\n";

/*
 * This is the material written at the foot of the document.
 */
static const char footer[] = "\
</article>\n";

#if 0
<refentry>\n\
<refsect1>\n\
<title>Copyright</title>\n\
<para>
This file was automatically created by magpie at %1$s\n\
Author: Bear Giles <email>bgiles@coyotesong.com</email>\n\
</para>\n\
</refsect1>\n\
</refentry>\n\

#endif


/*+
Set up our environment.
+*/
static void setup (void)
{
	char pathname[256], ch;

	mkdir (OUTPUT_DIR, 0755);

	/* weird syntax just in case we're non-ASCII */
	for (ch = 0; ch < 0x7F; ch++) {
		if (islower (ch) || isdigit (ch)) {
			sprintf (pathname, "%s/%c", OUTPUT_DIR, ch);
			mkdir (pathname, 0755);
		}
	}
}


/*+
Write the specified string, quoting any characters that will 
confuse docbook.
+*/
static void docbook_quote (FILE *fp, const char *s)
{
	char ch;
	while ((ch = *s++)) {
		switch (ch) {
		case '<': fputs ("&lt;", fp);  break;
		case '>': fputs ("&gt;", fp);  break;
		case '&': fputs ("&amp;", fp); break;
		default:  fputc (ch, fp);
		}
	}
}

/*+
Write a quick reference.
+*/
static void docbook_quickref (FILE *fp, const char *name, const char *value)
{
	fputs ("\n", fp);
	fputs ("<refsect1>\n", fp);
	fputs ("<title>\n", fp);
	docbook_quote (fp, name);
	fputs ("</title>\n", fp);
	fputs ("<para>\n", fp);
	docbook_quote (fp, value);
	fputs ("</para>\n", fp);
	fputs ("</refsect1>\n", fp);
}

/*+
Write a quick reference.
+*/
static void docbook_quickref_d (FILE *fp, const char *name, long value)
{
	fputs ("\n", fp);
	fputs ("<refsect1>\n", fp);
	fputs ("<title>\n", fp);
	docbook_quote (fp, name);
	fputs ("</title>\n", fp);
	fputs ("<para>\n", fp);
	fprintf (fp, "%ld", value);
	fputs ("</para>\n", fp);
	fputs ("</refsect1>\n", fp);
}

/*+
Write a list of packages
+*/
static void put_list (FILE *fp, const char *title, const struct package_list *d)
{
	struct package_info *p;
	struct package_list *q;

	fputs ("\n", fp);
	fputs ("<refsect1>\n", fp);
	fprintf (fp, "<title>%s</title>", title);
	fputs ("<itemizedlist>\n", fp);
	while (d) {
		q = d->down;
		if (q) {
			fputs ("<listitem>\n", fp);
			fputs ("<para>\n", fp);
			fputs ("<itemizedlist>\n", fp);
			fputs ("<listitem><para>", fp);
			docbook_quote (fp, d->name);
			if (d->restriction) {
				fputc (' ', fp);
				docbook_quote (fp, d->restriction);
				}
			p = mp_lookup (d->name);
			if (p && p->summary) {
				fputs (" - ", fp);
				docbook_quote (fp, p->summary);
			}
			fputs ("</para></listitem>\n", fp);
			while (q) {
				fputs ("<listitem><para>", fp);
				docbook_quote (fp, q->name);
				if (q->restriction) {
					fputc (' ', fp);
					docbook_quote (fp, q->restriction);
				}
				p = mp_lookup (q->name);
				if (p && p->summary) {
					fputs (" - ", fp);
					docbook_quote (fp, p->summary);
				}
				fputs ("</para></listitem>\n", fp);
				q = q->down;
				}
			fputs ("</itemizedlist>\n", fp);
			fputs ("</para>\n", fp);
			fputs ("</listitem>\n", fp);
			}
		else {
			fputs ("<listitem><para>", fp);
			docbook_quote (fp, d->name);
			if (d->restriction) {
				fputc (' ', fp);
				docbook_quote (fp, d->restriction);
			}
			p = mp_lookup (d->name);
			if (p && p->summary) {
				fputs (" - ", fp);
				docbook_quote (fp, p->summary);
			}
			fputs ("</para></listitem>\n", fp);
		}
		d = d->next;
	}

	fputs ("</itemizedlist>\n", fp);
	fputs ("</refsect1>\n", fp);
}


/*+
Write the information about the package in DocBook format.
+*/
static int docbook_dump_package (FILE *fp, const struct package_info *p)
{
	int i;

	fputs ("\n", fp);
	fputs ("<refentry>\n", fp);
	fputs ("<refmeta>\n", fp);
	fputs ("<refentrytitle>Package Information</refentrytitle>\n" ,fp);
	fputs ("<manvolnum>9</manvolnum>\n", fp);
	fputs ("</refmeta>\n", fp);

	/* write package name and synopsis */
	fputs ("<refnamediv>\n", fp);
	fprintf (fp, "<refname>%s</refname>", p->name);
	fputs ("<refpurpose>", fp);
	docbook_quote (fp, p->summary);
	fputs ("</refpurpose>\n", fp);
	fputs ("</refnamediv>\n", fp);
	fputs ("\n", fp);

	/* write package description */
	fputs ("<refsect1>\n", fp);
	fputs ("<title>Description</title>\n", fp);
	fputs ("<para>\n", fp);
	for (i = 0; i < p->desccnt; i++) {
		if (*p->description[i] == '\0')
			fputs ("</para>\n<para>\n", fp);
		else {
			docbook_quote (fp, p->description[i]);
			fputs ("\n", fp);
		}
	}
	fputs ("</para>\n", fp);
	fputs ("</refsect1>\n", fp);
	fputs ("\n", fp);

	/* write package version */
	docbook_quickref (fp, "Version", p->version);

	docbook_quickref (fp, "Maintainer", p->maintainer);

	if (p->predepends)
		put_list (fp, "Predepends", p->predepends);

	if (p->depends)
		put_list (fp, "Depends", p->depends);

	if (p->r_depends)
		put_list (fp, "Required by", p->r_depends);

	if (p->recommends)
		put_list (fp, "Recommends", p->recommends);

	if (p->recommends)
		put_list (fp, "Required by", p->r_recommends);

	if (p->suggests)
		put_list (fp, "Suggests", p->suggests);

	if (p->r_suggests)
		put_list (fp, "Suggested by", p->r_suggests);

	if (p->provides)
		put_list (fp, "Provides", p->provides);

	docbook_quickref (fp, "Architecture", architectures[p->architecture]);

	if (p->filename)
		docbook_quickref (fp, "Filename", p->filename);

	if (p->size)
		docbook_quickref_d (fp, "Size", p->size);

	if (p->installed_size)
		docbook_quickref_d (fp, "Size", p->installed_size);

#if 0
	if (p->status[0])
		fprintf (fp, " status='%s %s %s'", 
			p->status[0], p->status[1], p->status[2]);

	xml_name_value (fp, "md5sum", p->md5sum);
#endif

	/* we don't show source version info */
	if (p->source)
		docbook_quickref (fp, "Source", p->source->name);

	fputs ("\n", fp);
	fputs ("</refentry>\n", fp);
	fputs ("\n", fp);

	return 0;
}


/*+
Write the information about all selected packages in DocBook format.
+*/
static int docbook_dump (const char *pathname)
{
	FILE *fp;
	int i;
	struct package_info *p;

	fp = fopen (pathname, "w");
	fprintf (fp, header);

	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (p->selected)
			docbook_dump_package (fp, p);
	}

	fprintf (fp, footer, timestamp);
	fclose (fp);

	return 0;
}


/*+
+*/
static int docbook_keyword (const char *keyword, int keylen)
{
	FILE *fp;
	int i, j, n;
	struct package_info *p;
	char ch, *s, *last;
	char buffer[8*1024];
	char pathname[256];
	int first;

	sprintf (pathname, "%s/%s.xml", OUTPUT_DIR, keyword);
	fp = fopen (pathname, "w");
	fprintf (fp, header);
	fputs ("<chapter>\n", fp);
	fputs ("  <title>Packages containing '", fp);
	docbook_quote (fp, keyword);
	fputs ("'</title>\n",fp);
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		p->selected = 0;

		/* force description into a single buffer */
		n = 0;
		for (j = 0; j < p->desccnt; j++) {
			n += snprintf (&buffer[n], sizeof buffer - n,
				"%s\n", p->description[j]);
			
		}
		first = 1;
		s = &buffer[1];
		last = buffer;
		while ((s = strstr (s, keyword)) != NULL) {
			ch = s[keylen];
			if (isalpha (s[-1])) {
				s++;
				continue;
			}
			else if (isalpha (ch) && ch != 's' && ch != 'e') {
				s++;
				continue;
			}
			p->selected = 1;
			if (first) {
				first = 0;
				fputs ("<section>\n", fp);
			}
			fprintf (fp, "%*.*s", s - last, s - last, last);
			fputs ("<emphasis>", fp);
			docbook_quote (fp, keyword);
			fputs ("</emphasis>", fp);
			s += keylen;
			last = s;
		}
		if (!first) {
			fprintf (fp, "%s", last);
			fputs ("</section>\n", fp);
		}
	}
	fputs ("</chapter>\n", fp);
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		if (p->selected)
			docbook_dump_package (fp, p);
	}
	fprintf (fp, footer, timestamp);
	fclose (fp);

	return 0;
}

/*+
+*/
static int docbook_init (void)
{
	int i;
	struct package_info *p;
	FILE *fp;
	char keyword[256];
	int  keylen;

	mkdir (OUTPUT_DIR, 0755);

	/* select all packages */
	for (i = 0; i < cachecnt; i++)
		cache[i]->selected = 1;
	docbook_dump ("docbook/all.xml");

	/* select installed packages */
	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		p->selected = p->installed || p->unpacked;
	}
	docbook_dump ("docbook/installed.xml");

	for (i = 0; i < cachecnt; i++) {
		p = cache[i];
		p->selected = (strcmp (p->name, "libpam0g") == 0);
	}
	docbook_dump ("docbook/single.xml");

	/* generate documents for matching keywords */
	fp = fopen ("/etc/magpie/keywords", "r");
	if (fp == 0)
		return 0; 
	while (fgets (keyword, sizeof keyword, fp)) {
		if (keyword[0] == '\0' || keyword[0] == '#')
			continue;
		
		keylen = strlen (keyword);
		keyword[--keylen] = '\0';

		docbook_keyword (keyword, keylen);
		}
	fclose (fp);

	return 0;
}

struct magpie_module mod_docbook = { 
	version           : MAGPIE_VERSION,
	description       : "DocBook module",
	init              : docbook_init
};
