#GPL
#GPL  libwhisker copyright 2000,2001,2002 by rfp.labs
#GPL
#GPL  This program is free software; you can redistribute it and/or
#GPL  modify it under the terms of the GNU General Public License
#GPL  as published by the Free Software Foundation; either version 2
#GPL  of the License, or (at your option) any later version.
#GPL
#GPL  This program is distributed in the hope that it will be useful,
#GPL  but WITHOUT ANY WARRANTY; without even the implied warranty of
#GPL  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#GPL  GNU General Public License for more details.
#GPL

=pod

=head1 ++ Sub package: http

The http package is the core package of libwhisker.  It is responsible
for making the HTTP requests, and parsing the responses.  It can handle
HTTP 0.9, 1.0, and 1.1 requests, and allows pretty much every aspect of
the request to be configured and controlled.  The HTTP functions use a
HTTP in/out hash, which is a normal perl hash.  For outgoing HTTP requests
('hin' hashes), the keys/values represent outgoing HTTP headers.  For HTTP
responses ('hout' hashes), the keys/values represent incoming HTTP
headers.  For both, however, there is a special key, 'whisker', whose
value is a hash reference.  The whisker control hash contains more
configuration variables, which include host, port, and uri of the desired
request.  To access the whisker control hash, use the following
notation: $hash{'whisker'}->{'key'}='value';

You should view LW::http_init_request() for a list of core whisker control
hash values.

=cut

##################################################################

=pod

=head1 - Function: LW::http_init_request
   
Params: \%input_hash_to_initialize
Return: Nothing (modifies input hash)

Sets default values to the input hash for use.  Sets the host to
'localhost', port 80, request URI '/', using HTTP 1.1 with GET
method.  The timeout is set to 10 seconds, no proxies are defined, and all
URI formatting is set to standard HTTP syntax.  It also sets the
Connection (Keep-Alive) and User-Agent headers.

NOTICE!!  It's important to use http_init_request before calling 
http_do_request, or http_do_request might puke.  Thus, a special magic 
value is placed in the hash to let http_do_request know that the hash has 
been properly initialized.  If you really must 'roll your own' and not use 
http_init_request before you call http_do_request, you will at least need 
to set the INITIAL_MAGIC value (amongst other things).

=cut

sub http_init_request { # doesn't return anything
 my ($hin)=shift;

 return if(!(defined $hin && ref($hin)));
 %$hin=(); # clear control hash

# control values
 $$hin{'whisker'}={
	req_spacer		=>	' ',
	req_spacer2		=>	' ',
	http_ver		=>	'1.1',
	method			=>	'GET',
	method_postfix		=>	'',
	port			=>	80,
	uri			=>	'/',
	uri_prefix		=>	'',
	uri_postfix		=>	'',
	uri_param_sep		=>	'?',
	host			=>	'localhost',
	http_req_trailer    	=>	'',
	timeout			=>	10,
	include_host_in_uri 	=>	0,
	ignore_duplicate_headers=> 	1,
	normalize_incoming_headers =>	1,
	lowercase_incoming_headers =>	0,
	ssl			=>	0,
	http_eol		=>	"\x0d\x0a",
	force_close		=>	0,
	force_open		=>	0,
	retry			=>	1,
	trailing_slurp		=>	0,
	force_bodysnatch	=>	0,
	INITIAL_MAGIC		=>	31337
};

 
# default header values
 $$hin{'Connection'}='Keep-Alive'; # notice it is now default!
 $$hin{'User-Agent'}="libwhisker/$LW::VERSION"; # heh
}


##################################################################

=pod

=head1 - Function: LW::http_do_request
   
Params: \%hin, \%hout [, \%configs]
Return: >=1 if error; 0 if no error (also modifies hout hash)

*THE* core function of libwhisker.  LW::http_do_request actually performs
the HTTP request, using the values submitted in %hin, and placing result
values in %hout.  This allows you to resubmit %hin is subsequent requests
(%hout is automatically cleared upon execution).  You can submit 'runtime'
config directives as %configs, which will be spliced into
$hin{'whisker'}->{} before anything else.  That means you can do:

LW::http_do_request(\%hin,\%hout,{'uri'=>'/cgi-bin/'});

This will set $hin{'whisker'}->{'uri'}='/cgi-bin/' before execution, and
provides a simple shortcut (note: it does modify %hin).

Please also take note of the 'hin' and 'hout'.  They are named due to 
what's passed (h)in to the function, and what comes (h)out of the 
function.  It really has nothing to do with the direction of requests 
(which is technically opposite).

This function will also retry any requests that bomb out during the 
transaction (but not during the connecting phase).  This is controlled
by the {whisker}->{retry} value.  Also note that the returned error
message in hout is the *last* error received.  All retry errors are
put into {whisker}->{retry_errors}, which is an anonymous array.

=cut

sub http_do_request {
 my @params = @_;
 my $retry_count = ${$params[0]}{'whisker'}->{'retry'} || 0;
 my $ret;
 my @retry_errors;

 return 1 if(!(defined $params[0] && ref($params[0])));
 return 1 if(!(defined $params[1] && ref($params[1])));

 if(defined $params[2]){
	foreach (keys %{$params[2]}){
		${$params[0]}{'whisker'}->{$_}=${$params[2]}{$_};}}

 do {
 	$ret=LW::http_do_request_ex($params[0],$params[1]);
	push @{${$params[1]}{'whisker'}->{'retry_errors'}},
		@retry_errors if scalar(@retry_errors);
	return $ret if($ret==0 || $ret==2);
	push @retry_errors, ${$params[1]}{'whisker'}->{'error'};
	$retry_count--;
 } while( $retry_count >= 0);

 # if we get here, we still had errors, but no more retries
 return 1;
}

##################################################################

=pod

=head1 - Function: LW::http_do_request_ex
   
Params: \%hin, \%hout, \%configs
Return: >=1 if error; 0 if no error

NOTE: you should go through http_do_request(), which calls this function.

This function actually does all the request work.  It is called by
http_do_request(), which has a 'retry wrapper' built into it to catch
errors.

=cut

sub http_do_request_ex {
 my ($hin, $hout, $hashref)=@_;
 my ($temp,$vin,$resp,$S,$a,$b,$vout,@c,$c,$res)=(1,'');
 my $W; # shorthand alias for the {'whisker'} hash

 return 1 if(!(defined $hin  && ref($hin) ));
 return 1 if(!(defined $hout && ref($hout)));

 %$hout=(); # clear output hash
 $$hout{whisker}->{uri}=$$hin{whisker}->{uri}; # for tracking purposes

 if($LW::LW_HAS_SOCKET==0){
	$$hout{'whisker'}->{'error'}='Socket support not available';
	return 2;}

 if(!defined $$hin{'whisker'} || 
    !defined $$hin{'whisker'}->{'INITIAL_MAGIC'} ||
    $$hin{'whisker'}->{'INITIAL_MAGIC'}!=31337 ){
	$$hout{'whisker'}->{'error'}='Input hash not initialized';
	return 2;
 }

 if(defined $hashref){
	foreach (keys %$hashref){
		$$hin{'whisker'}->{$_}=$$hashref{$_};}}

 # if we want anti-IDS, make a copy and setup new values
 if(defined $$hin{'whisker'}->{'anti_ids'}){
	my %copy=%{$hin};
	anti_ids(\%copy,$$hin{'whisker'}->{'anti_ids'});
	$W = $copy{'whisker'};
 } else {
	$W = $$hin{'whisker'};
 }

 if($$W{'ssl'}>0 && $LW::LW_HAS_SSL!=1){
	$$hout{'whisker'}->{'error'}='SSL not available';
	return 2;}

 $TIMEOUT=$$W{'timeout'}||10;

 my $cache_key = defined $$W{'proxy_host'} ?
	join(':',$$W{'proxy_host'},$$W{'proxy_port'}) :
	join(':',$$W{'host'},$$W{'port'});

 if(!defined $http_host_cache{$cache_key}){
	# make new entry
	push(@{$http_host_cache{$cache_key}},
		undef, 	# SOCKET		$$Z[0]
		0,	# $SOCKSTATE		$$Z[1]
		undef,	# INET_ATON		$$Z[2]
		undef,	# $SSL_CTX		$$Z[3]
		undef,	# $SSL_THINGY		$$Z[4]
		'',	# $OUTGOING_QUEUE	$$Z[5]
		'',	# $INCOMING_QUEUE	$$Z[6]
		0,	# $STATS_SYNS		$$Z[7]
		0 )	# $STATS_REQS		$$Z[8]
 }

 # this works, but is it 'legal'?  notice it's not declared 'my'
 $Z = $http_host_cache{$cache_key};

 # use $chost/$cport for actual server we are connecting to
 my ($chost,$cport,$cwhat,$PROXY)=('',80,'',0);

 if(defined $$W{'proxy_host'}){
    $chost=$$W{'proxy_host'};
    $cport=$$W{'proxy_port'}||80;
    $cwhat='proxy';
    $PROXY=1;

    if($$W{'ssl'}>0 && $LW::LW_SSL_LIB==2){
	$ENV{HTTPS_PROXY} ="$$W{'proxy_host'}:";
	$ENV{HTTPS_PROXY}.=$$W{'proxy_port'}||80; }

 } else {
    $chost=$$W{'host'};
    $cport=$$W{'port'};
    $cwhat='host';
 }

 if($$Z[1]>0){ # check to see if socket is still alive
	if(! sock_valid($Z,$hin,$hout) ){
		$$Z[1]=0;
		sock_close($$Z[0],$$Z[4]);
 }	}
 # technically we have a race condition: socket can go
 # bad before we send request, below.  But that's ok,
 # we handle the errors down there.

 if($$Z[1]==0){

	if(defined $$W{'UDP'} && $$W{'UDP'}>0){
		if(!socket(SOCK,PF_INET,SOCK_DGRAM,getprotobyname('udp')||0)){
			$$hout{'whisker'}->{'error'}='Socket() problems (UDP)'; 
			return 2;}
	} else {
		if(!socket(SOCK,PF_INET,SOCK_STREAM,getprotobyname('tcp')||0)){
			$$hout{'whisker'}->{'error'}='Socket() problems'; 
			return 2;}
	}

	$$Z[0]=SOCK; # lame hack to get perl to take variable for socket

	$$Z[5]=$$Z[6]=''; # flush in/out queues

	if($$W{'ssl'}>0){ # ssl setup stuff

	    if($LW::LW_SSL_LIB==1){
		if(! ($$Z[3] = Net::SSLeay::CTX_new()) ){
			$$hout{'whisker'}->{'error'}="SSL_CTX error: $!";
			return 2;}
		if(defined $$W{'ssl_rsacertfile'}){
			if(! (Net::SSLeay::CTX_use_RSAPrivateKey_file($$Z[3], 
					$$W{'ssl_rsacertfile'},
					&Net::SSLeay::FILETYPE_PEM))){
				$$hout{'whisker'}->{'error'}="SSL_CTX_use_rsacert error: $!";
				return 2;}
		}
		if(defined $$W{'ssl_certfile'}){
			if(! (Net::SSLeay::CTX_use_certificate_file($$Z[3], 
					$$W{'ssl_certfile'},
					&Net::SSLeay::FILETYPE_PEM))){
				$$hout{'whisker'}->{'error'}="SSL_CTX_use_cert error: $!";
				return 2;}
		}
		if(! ($$Z[4] = Net::SSLeay::new($$Z[3])) ){
			$$hout{'whisker'}->{'error'}="SSL_new error: $!";
			return 2;}
		if(defined $$W{'ssl_ciphers'}){
			if(!(Net::SSLeay::set_cipher_list($$Z[4], 
					$$W{'ssl_ciphers'}))){
				$$hout{'whisker'}->{'error'}="SSL_set_ciphers error: $!";
				return 2;}
		}
	    }
	}

	$$Z[2]=inet_aton($chost) if(!defined $$Z[2]);
	if(!defined $$Z[2]){ # can't find hostname
		$$hout{'whisker'}->{'error'}="Can't resolve hostname";
		return 2;
	}

	if($$W{'ssl'}>0 && $LW::LW_SSL_LIB==2){
		# proxy set in ENV; we always connect to host
		$$Z[4]= Net::SSL->new(
			PeerAddr => $$hin{'whisker'}->{'host'},
			PeerPort => $$hin{'whisker'}->{'port'},
			Timeout => $TIMEOUT );
		if($@){ $$hout{'whisker'}->{'error'}="Can't connect via SSL: $@[0]";
			return 2;}
		$$Z[4]->autoflush(1);
	} else {
		if($LW::LW_NONBLOCK_CONNECT){
			my $flags=fcntl($$Z[0],F_GETFL,0);
			$flags |= O_NONBLOCK; # set nonblock flag
			if(!(fcntl($$Z[0],F_SETFL,$flags))){ # error setting flag
				$LW::LW_NONBLOCK_CONNECT=0; # revert to normal
			} else {
				my $R=connect($$Z[0],sockaddr_in($cport,$$Z[2]));
				if(!$R){ # we didn't connect...
					if($! != EINPROGRESS){
						close($$Z[0]);
						$$Z[0]=undef; # this is a bad socket
						$$hout{'whisker'}->{'error'}="Can't connect to $cwhat";
						return 2;}
					vec($vin,fileno($$Z[0]),1)=1;
					if(!select(undef,$vin,undef,$TIMEOUT) || !getpeername($$Z[0])){
						close($$Z[0]);
						$$Z[0]=undef; # this is a bad socket
						$$hout{'whisker'}->{'error'}="Can't connect to $cwhat";
						return 2;
				}	}
				$flags &= ~O_NONBLOCK; # clear nonblock flag
				if(!(fcntl($$Z[0],F_SETFL,$flags))){ # not good!
					close($$Z[0]);
					$LW::LW_NONBLOCK_CONNECT=0;
					$$Z[0]=undef;
					$$hout{'whisker'}->{'error'}="Error setting socket to block";
					return 2;
			}	}	
		}	

		if(!defined $$Z[0]){ # this is a safety catch
			$$hout{'whisker'}->{'error'}="Error creating valid socket connection";
			return 2; }

		if($LW::LW_NONBLOCK_CONNECT==0){ # attempt to do a timeout alarm...
			eval {
				local $SIG{ALRM} = sub { die "timeout\n" };
				eval {alarm($TIMEOUT)};
				if(!connect($$Z[0],sockaddr_in($cport,$$Z[2]))){
					alarm(0);
					die("no_connect\n"); }
				eval {alarm(0)};
			};
			if($@ || !(defined $$Z[0])){
				$$hout{'whisker'}->{'error'}="Can't connect to $cwhat";
				return 2;
		}	}

		binmode($$Z[0]); # stupid Windows
		# same as IO::Handle->autoflush(1), without importing 1000+ lines
		my $S=select($$Z[0]); 
		$|++; select($S);
	}

	$$Z[1]=1; $$Z[7]++;

	if($$W{'ssl'}>0){

	    if($LW::LW_SSL_LIB==1){

	        if($PROXY){ # handle the proxy CONNECT stuff...
		    my $SSL_CONNECT = "CONNECT $$W{'host'}".
			":$$W{'port'}/ HTTP/1.0\n\n";
		    syswrite($$Z[0],$SSL_CONNECT, length($SSL_CONNECT)); }

		Net::SSLeay::set_fd($$Z[4], fileno($$Z[0]));
		if(! (Net::SSLeay::connect($$Z[4])) ){
			$$hout{'whisker'}->{'error'}="SSL_connect error: $!";
			sock_close($$Z[0],$$Z[4]); return 2;}

		if(defined $$W{'save_ssl_info'} && 
				$$W{'save_ssl_info'}>0){
			ssl_save_info($hout,$$Z[4]); }
	    }

	} else {
		$$Z[4]=undef;
	}
 }

 if(defined $$W{'ids_session_splice'} &&
            $$W{'ids_session_splice'}>0 &&
		$$W{'ssl'}==0){ # no session_spice over ssl
	setsockopt($$Z[0],SOL_SOCKET,SO_SNDLOWAT,1);
	@c=split(//, &http_req2line($hin));
	# notice we bypass queueing here, in order to trickle the packets
	my $ss;
	foreach $c (@c){ 
		$ss=syswrite($$Z[0],$c,1); # char size assumed to be 1
		if(!defined $ss || $ss==0){
			$$hout{'whisker'}->{'error'}="Error sending session splice request to server";
			sock_close($$Z[0],$$Z[4]); return 1;
		}
		select(undef,undef,undef,.1);
	}
 } else {
	 http_queue(http_req2line($hin)); }

 $$Z[8]++;

 if($$W{'http_ver'} ne '0.9'){

    foreach (keys %$hin){
	next if($_ eq '' || $_ eq 'whisker');
	if(ref($$hin{$_})){ # header with multiple values
		my $key=$_;
		foreach (@{$$hin{$key}}){
		  http_queue("$key: $_$$W{'http_eol'}");}
	} else { # normal header
		http_queue("$_: $$hin{$_}$$W{'http_eol'}");
	}
    }

    if(defined $$W{'raw_header_data'}){
	http_queue($$W{'raw_header_data'});}

    http_queue($$W{'http_eol'});

    if(defined $$W{'data'}){ 
	http_queue($$W{'data'});}

 } # http 0.9 support

 # take a MD5 of queue, if wanted
 if(defined $$W{'queue_md5'}){
	$$hout{'whisker'}->{'queue_md5'}= LW::md5($$Z[5]);
 }


 # all data is wrangled...actually send it now
 if($res=http_queue_send($$Z[0],$$Z[4])){
	$$hout{'whisker'}->{'error'}="Error sending request to server: $res";
	sock_close($$Z[0],$$Z[4]); $$Z[1]=0; return 1;}

 if(defined $$Z[4]){
	if($LW::LW_SSL_LIB==1){ # Net::SSLeay
 		shutdown $$Z[0], 1; 
	} else { # Net::SSL
		shutdown $$Z[4], 1;
	}
 }

 vec($vin,fileno($$Z[0]),1)=1; # wait only so long to read...
 if(!select($vin,undef,undef,$TIMEOUT)){
	$$hout{'whisker'}->{'error'}="Server read timed out";
	sock_close($$Z[0],$$Z[4]); $$Z[1]=0; return 1;}

my ($LC,$CL,$TE,$CO)=('',-1,'',''); # extra header stuff

$$hout{'whisker'}->{'lowercase_incoming_headers'} = 
	$$W{'lowercase_incoming_headers'};

if($$W{'http_ver'} ne '0.9'){

 do { # catch '100 Continue' responses
  ($resp=sock_getline($$Z[0],$$Z[4]))=~tr/\r\n//d;

  if(!defined $resp){
	$$hout{'whisker'}->{'error'}='Error reading HTTP response';
	if($!){ # this should be left over from sysread via sock_getline
		$$hout{'whisker'}->{'error'}.=": $!"; }
	$$hout{'whisker'}->{'data'}=$$Z[6];
	sock_close($$Z[0],$$Z[4]); $$Z[1]=0; # otherwise bad crap lingers
	return 1;}

  if($resp!~/^HTTP\/([0-9.]{3})[ \t]+(\d+)[ \t]{0,1}(.*)/){
	$$hout{'whisker'}->{'error'}="Invalid HTTP response: $resp";
	# let's save the incoming data...we might want it
	$$hout{'whisker'}->{'data'}=$resp;
	while($_=sock_getline($$Z[0],$$Z[4])){ 
		$$hout{'whisker'}->{'data'}.=$_;}
	# normally we'd check the results to see if socket is closed, but
	# we close it anyway, so it doesn't matter
	sock_close($$Z[0],$$Z[4]); $$Z[1]=0; # otherwise bad crap lingers
	return 1;}

  $$hout{'whisker'}->{'http_ver'}	= $1;
  $$hout{'whisker'}->{'http_resp'}	= $2;
  $$hout{'whisker'}->{'http_resp_message'}= $3;
  $$hout{'whisker'}->{'code'}		= $2;

  $$hout{'whisker'}->{'100_continue'}++ if($2 == 100);

  while($_=sock_getline($$Z[0],$$Z[4])){ # check pertinent headers

	$_=~s/[\r]{0,1}\n$//; # anchored regex, so it's fast
	last if ($_ eq ''); # acceptable assumption case?

	my $l2=index($_,':'); # this is faster than regex
	$a=substr($_,0,$l2); 
	$b=substr($_,$l2+1);
	$b=~s/^([ \t]*)//; # anchored regex, so it's fast

	$hout{'whisker'}->{'abnormal_header_spacing'}++ if($1 ne ' ');

	$LC = lc($a);
	next         if($LC eq 'whisker');
	$TE = lc($b) if($LC eq 'transfer-encoding');
	$CL = $b     if($LC eq 'content-length');
	$CO = lc($b) if($LC eq 'connection');

	if($$W{'lowercase_incoming_headers'}>0){
		$a=$LC;
	} elsif($$W{'normalize_incoming_headers'}>0){ 
                $a=~s/(-[a-z])/uc($1)/eg;
 	}

	# save the received header order, in case we're curious
	push(@{$$hout{'whisker'}->{'recv_header_order'}},$a);

	if(defined $$hout{$a} && $$W{'ignore_duplicate_headers'}!=1){
	  if(!ref($$hout{$a})){
	    my $temp=$$hout{$a};
	    delete $$hout{$a};
	    push(@{$$hout{$a}},$temp);
	  }
	  push(@{$$hout{$a}},$b);
	} else {
	  $$hout{$a}=$b;
  }	}

  # did we have a socket error?
  if($!){
	$hout{'whisker'}->{'error'}='Error in reading response/headers';
	sock_close($$Z[0],$$Z[4]); $$Z[1]=0; return 1; }

  if( $CO eq '' ){ # do whatever the client wanted
	$CO = (defined $$hin{'Connection'}) ? lc($$hin{'Connection'}) : 
		'close'; }

 } while($$hout{'whisker'}->{'http_resp'}==100);

} else { # http ver 0.9, we need to fake it
 # Keep in mind lame broken servers, like IIS, still send headers for 
 # 0.9 requests; the headers are treated as data.  Also keep in mind
 # that if the server doesn't support HTTP 0.9 requests, it will spit
 # back an HTTP 1.0 response header.  User is responsible for figuring
 # this out himself.
 $$hout{'whisker'}->{'http_ver'}='0.9';
 $$hout{'whisker'}->{'http_resp'}='200';
 $$hout{'whisker'}->{'http_resp_message'}='';
}

 if($$W{'force_bodysnatch'} || ( $$W{'method'} ne 'HEAD' && 
	$$hout{'whisker'}->{'http_resp'}!=206 &&
	$$hout{'whisker'}->{'http_resp'}!=102)){
  if ($TE eq 'chunked') { 
	if(!defined ($a=sock_getline($$Z[0],$$Z[4]))){
		$$hout{'whisker'}->{'error'}='Error reading chunked data length';
		sock_close($$Z[0],$$Z[4]); $$Z[1]=0; return 1;}
	$a=~tr/a-fA-F0-9//cd; $CL=hex($a); 
	$$hout{'whisker'}->{'data'}='';
	while($CL!=0) { # chunked sucks
		if(!defined ($temp=sock_get($$Z[0],$$Z[4],$CL))){
			$$hout{'whisker'}->{'error'}="Error reading chunked data: $!";
			sock_close($$Z[0],$$Z[4]); $$Z[1]=0; return 1;}
		$$hout{'whisker'}->{'data'}=$$hout{'whisker'}->{'data'} . $temp;
		$temp=sock_getline($$Z[0], $$Z[4]);
		($temp=sock_getline($$Z[0], $$Z[4])) if(defined $temp &&
			$temp=~/^[\r\n]*$/);
		if(!defined $temp){ # this will catch errors in either sock_getline
			$$hout{'whisker'}->{'error'}="Error reading chunked data: $!";
			sock_close($$Z[0],$$Z[4]); $$Z[1]=0; return 1;}
		$temp=~tr/a-fA-F0-9//cd; $CL=hex($temp);}
	# read in trailer headers
	while($_=sock_getline($$Z[0],$$Z[4])){ tr/\r\n//d; last if($_ eq ''); }
	# Hmmmm...error, but we should have full body.  Don't return error
	if($!){ $$Z[1]=0; sock_close($$Z[0],$$Z[4]); }
  } else {
 	if ($CL != -1) {
		if(!defined ($temp=sock_get($$Z[0],$$Z[4],$CL))){
			$$hout{'whisker'}->{'error'}="Error reading data: $!";
			sock_close($$Z[0],$$Z[4]); $$Z[1]=0; return 1;}
	} else {  # Yuck...read until server stops sending....
		$temp=sock_getall($$Z[0],$$Z[4]);
		# we go until we puke, so close socket and don't return error
		sock_close($$Z[0],$$Z[4]); $$Z[1]=0;
	}
	$$hout{'whisker'}->{'data'}=$temp; 
  }
 } # /method ne HEAD && http_resp ne 206 or 102/

 if(($CO ne 'keep-alive' || ( defined $$hin{'Connection'} &&
		lc($$hin{'Connection'}) eq 'close')) && $$W{'force_open'}!=1){
	$$Z[1]=0; sock_close($$Z[0],$$Z[4]); 
 }	 

 # this way we know what the state *would* have been...
 $$hout{'whisker'}->{'sockstate'}=$$Z[1];
 if($$W{'force_close'}>0) {
	$$Z[1]=0; sock_close($$Z[0],$$Z[4]); } 

 $$hout{'whisker'}->{'stats_reqs'}=$$Z[8];
 $$hout{'whisker'}->{'stats_syns'}=$$Z[7];
 $$hout{'whisker'}->{'error'}=''; # no errors
 return 0;
}


##################################################################

=pod

=head1 - Function: LW::http_req2line (INTERNAL)
  
Params: \%hin, $switch
Return: $request

req2line is used internally by LW::http_do_request, as well as provides a
convienient way to turn a %hin configuration into an actual HTTP request
line.  If $switch is set to 1, then the returned $request will be the URI
only ('/requested/page.html'), versus the entire HTTP request ('GET
/requested/page.html HTTP/1.0\n\n').  Also, if the 'full_request_override'
whisker config variable is set in %hin, then it will be returned instead
of the constructed URI.

=cut

sub http_req2line {
 my ($S,$hin,$UO)=('',@_);
 $UO||=0; # shut up -w warning

 # notice: full_request_override can play havoc with proxy settings
 if(defined $$hin{'whisker'}->{'full_request_override'}){
	return $$hin{'whisker'}->{'full_request_override'};

 } else { # notice the components of a request--this is for flexibility

	if($UO!=1){$S.= 	$$hin{'whisker'}->{'method'}.
				$$hin{'whisker'}->{'method_postfix'}.
				$$hin{'whisker'}->{'req_spacer'};
	
		if($$hin{'whisker'}->{'include_host_in_uri'}>0){
			$S.=	'http://';

			if(defined $$hin{'whisker'}->{'uri_user'}){
			$S.=	$$hin{'whisker'}->{'uri_user'};
			if(defined $$hin{'whisker'}->{'uri_password'}){
				$S.=	':'.$$hin{'whisker'}->{'uri_user'};
			}
			$S.=	'@';
			}

			$S.=	$$hin{'whisker'}->{'host'}.
				':'.$$hin{'whisker'}->{'port'};}}

	$S.=	$$hin{'whisker'}->{'uri_prefix'}.
		$$hin{'whisker'}->{'uri'}.
		$$hin{'whisker'}->{'uri_postfix'};

	if(defined $$hin{'whisker'}->{'uri_param'}){
		$S.= 	$$hin{'whisker'}->{'uri_param_sep'}.
			$$hin{'whisker'}->{'uri_param'};}

	if($UO!=1){  if($$hin{'whisker'}->{'http_ver'} ne '0.9'){
			$S.= 	$$hin{'whisker'}->{'req_spacer2'}.'HTTP/'.
				$$hin{'whisker'}->{'http_ver'}.
				$$hin{'whisker'}->{'http_req_trailer'};}
			$S.=	$$hin{'whisker'}->{'http_eol'};}}
 return $S;}



##################################################################

=pod

=head1 - Function LW::sock_close (INTERNAL)
   
Params: $socket_file_descriptor, $SSL_THINGY
Return: nothing

This function will close the indicated socket and SSL connection (if 
necessary).  They are wrapped in eval()s to make sure if the functions 
puke, it doesn't kill the entire program.

=cut

sub sock_close {
	my ($fd,$ssl)=@_;

	eval { close($fd); };
	if(defined $ssl){
	    if($LW::LW_SSL_LIB==1){ # Net::SSLeay
		eval "&Net::SSLeay::free($ssl)";
		eval "&Net::SSLeay::CTX_free($$Z[3])";
	    } else { # Net::SSL
		eval { close($ssl) }; # is that right for Net::SSL?
	    }
	}

	$$Z[4]=undef;
}

##################################################################

=pod

=head1 - Function LW::sock_valid (INTERNAL)
   
Params: $Z reference, \%hin, \%hout
Return: 1 if socket valid, 0 if socket disconnected

This is an internal function used to determine if a socket is
still good (i.e. the other END hasn't closed the connection).
This really only applies to persistent (Keep-Alive) connections.

This function is not intended for external use.

=cut

sub sock_valid {
	my ($z,$Hin,$Hout)=@_;

	my $slurp=$$Hin{'whisker'}->{'trailing_slurp'};
	my ($o,$vin)=(undef,'');

	# closed socket sets read flag (and so does waiting data)
 	vec($vin,fileno($$z[0]),1)=1;
 	if(select(($o=$vin),undef,undef,.01)){ # we have data to read
		my ($hold, $res);

		do {
			$res = sysread($$z[0], $hold, 4096);
			$$z[6].=$hold if($slurp==0); # save to queue
			$$Hout{'whisker'}->{'slurped'}.="$hold\0"
				if($slurp==1); # save to hout hash
			# fall through value of 2 doesn't do anything
		} while ($res && select(($o=$vin),undef,undef,.01));

		if(!defined $res || $res==0){ # error or EOF
			return 0;
		}
	}
    
	return 1;
}

##################################################################

=pod

=head1 - Function: LW::sock_getline (INTERNAL)
   
Params: $socket_file_descriptor, $SSL_THINGY
Return: $string, undef on error (timeout)

This function is used internally to read a line of input (up to a '\n')
from the given socket file descriptor (regular or SSL).

This function is not intended for external use.

=cut

sub sock_getline { # read from socket w/ timeouts
        my ($fd,$ssl) = @_;
        my ($str,$t)=('','');

        $t = index($$Z[6],"\n",0);

        while($t < 0){
                return undef if &http_queue_read($fd,$ssl);
                $t=index($$Z[6],"\n",0);
        }

	# MEMLEAK: use following lines; comment out SPEEDUP and LEGACY lines
	# my $r;
	# ($r,$$Z[6])=unpack('A'.($t+1).'A*',$$Z[6]);
	# return $r;

	# SPEEDUP: use following line; comment out MEMLEAK and LEGACY lines
	# return substr($$Z[6],0,$t+1,'');

	# LEGACY: use following lines; comment out MEMLEAK and SPEEDUP lines
	my $r = substr($$Z[6],0,$t+1);
	substr($$Z[6],0,$t+1)='';
	return $r;
}

##################################################################

=pod

=head1 - Function: LW::sock_get (INTERNAL)
   
Params: $socket_file_descriptor, $SSL_THINGY, required $amount
Return: $string, undef on error

This function is used internally to read input from the given socket 
file descriptor (regular or SSL).  Will abort/error if $amount is not
available.

This function is not intended for external use.

=cut

sub sock_get { # read from socket w/ timeouts
        my ($fd,$ssl,$amount) = @_;
        my ($str,$t)=('','');

	while($amount > length($$Z[6])){
                return undef if &http_queue_read($fd,$ssl);
	}

	# MEMLEAK: use following lines; comment out SPEEDUP and LEGACY lines
	# my $r;
	# ($r,$$Z[6])=unpack('A'.$amount.'A*',$$Z[6]);
	# return $r;

	# SPEEDUP: use following line; comment out MEMLEAK and LEGACY lines
	# return substr($$Z[6],0,$amount,'');

	# LEGACY: use following lines; comment out MEMLEAK and SPEEDUP lines
	my $r = substr($$Z[6],0,$amount);
	substr($$Z[6],0,$amount)='';
	return $r;
}

##################################################################

=pod

=head1 - Function: LW::sock_getall (INTERNAL)
   
Params: $socket_file_descriptor, $SSL_THINGY
Return: $string

This function is used internally to read input from the given socket 
file descriptor (regular or SSL).  It will return everything received
until an error (no data or real error) occurs.

This function is not intended for external use.

=cut

sub sock_getall {
        my ($fd,$ssl) = @_;
        1 while( !(&http_queue_read($fd,$ssl)) );
        return $$Z[6];
}

##################################################################

=pod

=head1 - Function: LW::http_queue_read (INTERNAL)
   
Params: $fd, $ssl
Return: $character, undef on error (timeout)

http_queue_read() will put incoming data from the server into 
the incoming queue for reading.  If there's no more data (or
on error), it will return 1.  Otherwise it returns 0.

This function is really for internal use only.

=cut

sub http_queue_read {
	my ($fd,$ssl)=@_;
	my ($vin, $t)=('','');

	if(defined $ssl){
	    if($LW::LW_SSL_LIB==1){ # Net::SSLeay
		local $SIG{ALRM} = sub { die "timeout\n" };
		eval {
			eval { alarm($TIMEOUT); };
			$t=Net::SSLeay::read($ssl);
			eval { alarm(0); };
		};
        	if($@ || !defined $t || $t eq ''){
			return 1;}
		$$Z[6].=$t;
	    } else { # Net::SSL
		if(!$ssl->read($t,1024)){ return 1;
		} else { $$Z[6].=$t;}
	    }
	} else {
		vec($vin,fileno($fd),1)=1; # wait only so long to read...
		if(!select($vin,undef,undef,$TIMEOUT)){
			return 1;}
               	if(!sysread($fd,$t,4096)){	return 1; # EOF or error
		} else {			$$Z[6].=$t;}
	}

	return 0;
}

##################################################################

=pod

=head1 - Function: LW::http_queue_send (INTERNAL)
   
Params: $sock, $ssl
Return: $status_result (undef=ok, else error message)

This functions sends the current queue (made with LW::http_queue) to the 
server via the specified SSL or socket connection.

=cut

sub http_queue_send { # write to socket
	my ($fd,$ssl)=@_;
	my ($v,$wrote,$err)=('');

	my $len = length($$Z[5]);
	if(defined $ssl){
	    if($LW::LW_SSL_LIB==1){ # Net::SSLeay
		($wrote,$err)=Net::SSLeay::ssl_write_all($ssl,$$Z[5]);
		return 'Could not send entire data queue' if ($wrote!=$len);
		return "SSL_write error: $err" unless $wrote;
	    } else { # Net::SSL
		$ssl->print($$Z[5]);
	    }
	} else {
        	vec($v,fileno($fd),1)=1;
 		if(!select(undef,$v,undef,.01)){ 
			return 'Socket write test failed'; }
		$wrote=syswrite($fd,$$Z[5],length($$Z[5]));
		return "Error sending data queue: $!" if(!defined $wrote);
		return 'Could not send entire data queue' if ($wrote != $len);
	}
	$$Z[5]=''; return undef;
}


##################################################################

=pod

=head1 - Function: LW::http_queue (INTERNAL)
   
Params: $data
Return: nothing

This function will buffer the output to be sent to the server.  Output is 
buffered for various reasons (particularlly because of SSL, but also 
allowing the chance to 'go back' and modify the final output before it's 
actually sent (after header constructions, etc).

=cut

sub http_queue {
	$$Z[5].= shift;
}


##################################################################

=pod

=head1 - Function: LW::http_fixup_request
   
Params: $hash_ref
Return: Nothing

This function takes a %hin hash reference and makes sure the proper 
headers exist (for example, it will add the Host: header, calculate the 
Content-Length: header for POST requests, etc).  For standard requests 
(i.e. you want the request to be HTTP RFC-compliant), you should call this 
function right before you call LW::http_do_request.

=cut

sub http_fixup_request {
 my $hin=shift;

 return if(!(defined $hin && ref($hin)));

 if($$hin{'whisker'}->{'http_ver'} eq '1.1'){
 	$$hin{'Host'}=$$hin{'whisker'}->{'host'} if(!defined $$hin{'Host'});
	$$hin{'Connection'}='Keep-Alive' if(!defined $$hin{'Connection'});
 }

 if(defined $$hin{'whisker'}->{'data'}){ 
 	if(!defined $$hin{'Content-Length'}){
		$$hin{'Content-Length'}=length($$hin{'whisker'}->{'data'});}
#	if(!defined $$hin{'Content-Encoding'}){
#		$$hin{'Content-Encoding'}='application/x-www-form-urlencoded';}
 }

 if(defined $$hin{'whisker'}->{'proxy_host'}){
	$$hin{'whisker'}->{'include_host_in_uri'}=1;}

}

##################################################################

=pod

=head1 - Function: LW::http_reset
     
Params: Nothing
Return: Nothing

The LW::http_reset function will walk through the %http_host_cache, 
closing all open sockets and freeing SSL resources.  It also clears
out the host cache in case you need to rerun everything fresh.

=cut

sub http_reset {
 my $key;

 foreach $key (keys %http_host_cache){
 	# *Z=$http_host_cache{$key};
	sock_close($http_host_cache{$key}->[0],
			$http_host_cache{$key}->[4]);
	delete $http_host_cache{$key};
 }
}

##################################################################

=pod

=head1 - Function: LW::ssl_save_info (INTERNAL)
     
Params: \%hout, $ssl_connection
Return: Nothing

This is an internal function used to save various Net::SSLeay
information into the given hash.  Triggered by setting
{'whisker'}->{'save_ssl_info'}=1.

=cut

sub ssl_save_info {
	my ($hr,$SSL)=@_;
	my $cert;

	return if($LW::LW_SSL_LIB!=1); # only Net::SSLeay used
	$$hr{'whisker'}->{'ssl_cipher'}=Net::SSLeay::get_cipher($SSL);		

	if( $cert = Net::SSLeay::get_peer_certificate($SSL)){
		$$hr{'whisker'}->{'ssl_cert_subject'} = 
			Net::SSLeay::X509_NAME_oneline(
                    	Net::SSLeay::X509_get_subject_name($cert) );

		$$hr{'whisker'}->{'ssl_cert_issuer'} = 
			Net::SSLeay::X509_NAME_oneline(
                    	Net::SSLeay::X509_get_issuer_name($cert) );
	}
}

##################################################################
