/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeGeometryAlgorithms.h
    This file contains Algorithms for Topological Operations.
*/

/**
  *@author Gilberto Ribeiro de Queiroz
  */

#ifndef  __TERRALIB_INTERNAL_GEOMETRYALGORITHMS_H
#define  __TERRALIB_INTERNAL_GEOMETRYALGORITHMS_H

#include "TeGeometry.h"

#include "TePrecision.h"
#include "TeProjection.h"

//! Intersection coordinates of two segments.
typedef vector<TeCoord2D> TeIntersCoordsVec;

/** @defgroup GeometryAlgorithms Geometry Algorithms
 *  TerraLib geometry algorithms.
 *  @{
 */
/** @defgroup TopologicalOperators Topological Operators
 *  @ingroup  GeometryAlgorithms
   Functions that test topologival relation between two objects.
   The topological operators are based on the intersection of interior, exterior and boundary of geometries:
 \par  ------------------------------------------------------------------------------------------------------------------
 \par  | TeGeometry   |  INTERIOR                        | BOUNDARY           | EXTERIOR                                |
 \par  ------------------------------------------------------------------------------------------------------------------
 \par  | TePoint      | The point itself                 | Empty              | Everything except interior              |
 \par  | TeLine2D     | All points except the end points | The two end points | Everything except interior and boundary |
 \par  | TeLinearRing | All points                       | Empty              | Everything except interior and boundary |
 \par  | TePolygon    | Everything between the external  | All rings          | Everything except interior and boundary |
 \par  |              | ring and the inner rings         |                    |                                         |
 \par  ------------------------------------------------------------------------------------------------------------------
 
 \par  We use:
		- I: for Interior
		- E: for Exterior
		- B: for Boundary
		- inter: Intersects
		- ^: AND
		- v: OR
		- A = refer to two-dimensional geometries (TePolygon and TePolygonSet)
		- L = refer to one-dimensional geometries (TeLine2D and TeLineSet)
		- P = refer to 0-dimensional geometries   (TePoint and TePointSet)
 *  @{
 */

/** @defgroup TeEquals Equals test
 *  @ingroup TopologicalOperators
 *  Check if one object is equal another object.
 *  Applies to: P/P, L/L and A/A.
 *  TeEquals(x, y) => (x inter y = x) ^ (y inter x = y)
 *					  (B(x) inter I(y) = false) ^ (B(x) inter E(y) = false) 
 *  @{
*/

//! If a specialized function is not used, returns false.
template<class T1, class T2> inline bool TeEquals(const T1& /*o1*/, const T2& /*o2*/)
{
	return false;
}

//! Check if coordinate 1 and coordinate 2 are equal
template<> bool TeEquals(const TeCoord2D& c1, const TeCoord2D& c2);

//!  Check if point 1 and point 2 are equal
template<> bool TeEquals(const TePoint& p1, const TePoint& p2);

//! Check if lineRed and lineBlue are equal
template<> bool TeEquals(const TeLine2D& redLine, const TeLine2D& blueLine);

//! Check if polygon red and polygon blue are equal
template<> bool TeEquals(const TePolygon& redPol, const TePolygon& bluePol);

//! Check if box 1 and box 2 are equal 
template<> bool TeEquals(const TeBox& bx1, const TeBox& bx2);

//! Check if cell 1 and cell 2 are equal
template<> bool TeEquals(const TeCell& cell1, const TeCell& cell2);
/** @} */ 


/** @defgroup TeDisjoint Disjoint test
 *  @ingroup TopologicalOperators
 *  Check if two objects are disjoint.
 *  Applies to all geometries.
 *  TeDisjoint(x, y) => (x inter y = false)
 *                      (I(x) inter I(y) = false) ^ (I(x) inter B(y) = false) ^ (B(x) inter I(y) = false) ^ (B(x) inter B(y) = false)
 *  @{
*/

//! Check if coordinate cl and coordinate c2 are disjoint
bool TeDisjoint(const TeCoord2D& c1, const TeCoord2D& c2);

//! Check if coordinate and box are disjoint
bool TeDisjoint(const TeCoord2D& c, const TeBox& b);

//! Check if box 1 and box 2 are disjoint
bool TeDisjoint(const TeBox& bx1, const TeBox& bx2);

//! Check if coordinate and line are disjoint
bool TeDisjoint(const TeCoord2D& c, const TeLine2D& l);

//! Check if coordinate and polygon are disjoint
bool TeDisjoint(const TeCoord2D& c, const TePolygon& pol);

//! Check if point l and point 2 are disjoint
bool TeDisjoint(const TePoint& p1, const TePoint& p2);

//! Check if point and object are disjoint
bool TeDisjoint(const TePoint& p, const TeLine2D& l);

//! Check if point and object are disjoint
bool TeDisjoint(const TePoint& p, const TePolygon& pol);

//! Check if lines are disjoint
bool TeDisjoint(const TeLine2D& redLine, const TeLine2D& blueLine);

//! Check if line and polygon are disjoint
bool TeDisjoint(const TeLine2D& l, const TePolygon& pol);

//! Check if polygons are disjoint
bool TeDisjoint(const TePolygon& redPol, const TePolygon& bluePol);

//! Check if cell 1 and cell 2 are disjoint
bool TeDisjoint(const TeCell& cell1, const TeCell& cell2);

//! Check if cell and line are disjoint
bool TeDisjoint(const TeCell& cell, const TeLine2D& line);

//! Check if cell and polygon are disjoint
bool TeDisjoint(const TeCell& cell, const TePolygon& pol);

//! Check if cell and point are disjoint
bool TeDisjoint(const TeCell& cell, const TePoint& point);
/** @} */


/** @defgroup TeIntersects Intersects test
 *  @ingroup TopologicalOperators
 *  Check if one object intersects another object.
 *  Applies to all geometries.
 *  TeIntersects(x, y) => !TeDisjoint(x, y)
 *                     => (I(x) inter I(y) = true) v (I(x) inter B(y) = true) v (B(x) inter I(y) = true) v (B(x) inter B(y) = true)
 *  @{
 */

//! If a specialized function is not used, returns the disjoint negation
template<class T1, class T2> inline bool TeIntersects(const T1& o1, const T2& o2)
{
	return !TeDisjoint(o1, o2);
}

//! Check if coordinate and box intersects
template<> bool TeIntersects(const TeCoord2D& c, const TeBox& b);

//! Check if point and box intersects
template<> bool TeIntersects(const TePoint& p, const TeBox& b);

//! Check if boxes intersects
template<> bool TeIntersects(const TeBox& bx1, const TeBox& bx2);

/** @} */


/** @defgroup TeTouches Touch test
 *  @ingroup TopologicalOperators
 *  Check if two objects touche.
 *  Applies to A/A, L/L, L/A, P/A, P/L.
 *  TeTouches(x, y) => (I(x) inter I(y) = false) ^ (x inter y = true)
 *				    => (I(x) inter I(y) = false) ^ ((B(x) inter I(y) = true) v (I(x) inter B(y) = true) v (B(x) inter B(y) = true))
 *  @{
 */

//! Check if coordinate and line touches
bool TeTouches(const TeCoord2D& c, const TeLine2D& l);

//! Check if coordinate and polygon touches
bool TeTouches(const TeCoord2D& c, const TePolygon& pol);

//! Check if point and object touches
bool TeTouches(const TePoint& p, const TeLine2D& l);

//! Check if point and object touches
bool TeTouches(const TePoint& p, const TePolygon& pol);

//! Check if line and line touches
bool TeTouches(const TeLine2D& redLine, const TeLine2D& blueLine);

//! Check if line and polygon
bool TeTouches(const TeLine2D& l, const TePolygon& pol);

//! Check if polygons touches
bool TeTouches(const TePolygon& redPol, const TePolygon& bluePol);

//! Check if boxes touches
bool TeTouches(const TeBox& bx1, const TeBox& bx2);

//! Check if cells touches
bool TeTouches(const TeCell& c1, const TeCell& c2);

//! Check if cell and line touches
bool TeTouches(const TeLine2D& line, const TeCell& cell);

//! Check if cell and polygon touches
bool TeTouches(const TeCell& c1, const TePolygon& poly);

//! Check if cell and point touches
bool TeTouches( const TePoint& point, const TeCell& c1);
/** @} */


/** @defgroup TeCrosses Crosses test
 *  @ingroup TopologicalOperators
 *  Check if one object crosses another object.
 *  TeCrosses(x, y) => dim(I(x) inter I(y)) = (max(dim(I(x)), dim(I(y))) - 1) ^ (x inter y != x) ^ (x inter y != y) 
 *  Case 1: x = TeLine2D and y = TePolygon
 *          => (I(x) inter I(y) = true) ^ (I(x) inter E(y) = true)
 *  Case 2: x = TeLine2D and y = TeLine2D
 *          => dim(I(x) inter I(y)) = 0
 *  @{
 */

//! Check if red line crosses blue line
bool TeCrosses(const TeLine2D& redLine, const TeLine2D& blueLine);

//! Check if line crosses polygon
bool TeCrosses(const TeLine2D& l, const TePolygon& pol);

//! Check if line and cell crosses
bool TeCrosses(const TeLine2D& l, const TeCell& cell);
/** @} */


/** @defgroup TeWithin Within test
 *  @ingroup TopologicalOperators
 *  Check if one object is within another object.
 *  TeWithin(x, y) => (x inter y = x) ^ (I(x) inter I(y) = true)
 *  Case 1: P/P, P/L and P/A
 *          => (I(x) inter I(y) = true)
 *  Case 2: L/L and A/A
 *          => (I(x) inter I(y) = true) ^ (I(x) inter E(y) = false) ^ (B(x) inter E(y) = false) ^ (B(x) inter B(y) = false)
 *  Case 3: L/A
 *          => (I(x) inter I(y) = true) ^ (I(x) inter E(y) = false) ^ (B(x) inter E(y) = false) ^ (B(x) inter B(y) = false) ^ (I(x) inter B(y) = false)
 *  @{
*/

//! Check if coordinate 1 is within coordinate 2
bool TeWithin(const TeCoord2D& c1, const TeCoord2D& c2);

//! Check if coordinate is within a box
bool TeWithin(const TeCoord2D& c, const TeBox& b);

//! Check if a cordinate is within a line
bool TeWithin(const TeCoord2D& c, const TeLine2D& l);

//! Check if a cordinate is within a polygon
bool TeWithin(const TeCoord2D& c, const TePolygon& pol);

//! Check if point 1 is within point 2
bool TeWithin(const TePoint& p1, const TePoint& p2);

//! Check if point is within object
bool TeWithin(const TePoint& p, const TeLine2D& l);

//! Check if point is within object
bool TeWithin(const TePoint& p, const TePolygon& pol);

//! Check if red line is within blue line
bool TeWithin(const TeLine2D& redLine, const TeLine2D& blueLine);

//! Check if line is within polygon
bool TeWithin(const TeLine2D& l, const TePolygon& pol);

//! Check if red polygon is within blue polygon
bool TeWithin(const TePolygon& redPol, const TePolygon& bluePol);

//! Check if box1 is within box2
bool TeWithin(const TeBox& bx1, const TeBox& bx2);

//! Check if cell1 is within cell2
bool TeWithin(const TeCell& cell1, const TeCell& cell2);

//! Check if line is within cell
bool TeWithin(const TeLine2D& line, const TeCell& cell);

//! Check if cell is within polygon
bool TeWithin(const TeCell& cell, const TePolygon& poly);

//! Check if point is within cell
bool TeWithin(const TePoint& point, const TeCell& cell);
/** @} */

/** @defgroup TeContains Contains test
 *  @ingroup TopologicalOperators
 *  Check if one object contains another object.
 *  TeContains(x, y) = TeWithin(y, x)
 *  @{
 */

//! If a specialized function is not used, returns false
template<class T1, class T2> inline bool TeContains(const T1& o1, const T2& o2)
{
	return TeWithin(o2, o1);
}
/** @} */

/** @defgroup TeOverlaps Overlaps test
 *  @ingroup TopologicalOperators
 *  Check if one object overlaps another object.
 *  TeOverlaps(x, y) => (dim(I(x)) = dim(I(y)) = dim(I(x) inter I(y))) ^ (x inter y != x) ^ (x inter y != y)
 *  Case 1: (x = TePolygon  and y = TePolygon)
 *          => (I(x) inter I(y) = true) ^ (I(x) inter E(y) = true) ^ (E(x) inter I(y) = true)
 *  Case 2: x = TeLine2D and y = TeLine2D
 *          => (dim(I(x) inter I(y)) = 1) ^ (I(x) inter E(y) = true) ^ (E(x) inter I(y) = true)
 *  @{
 */

//! Check if red red overlaps blue line
bool TeOverlaps(const TeLine2D& redLine, const TeLine2D& blueLine);

//! Check if red polygon overlaps blue polygon
bool TeOverlaps(const TePolygon& redPol, const TePolygon& bluePol);

//! Check if cell1 overlaps cell2
bool TeOverlaps(const TeCell& cell1, const TeCell& cell2);

//! Check if cell overlaps polygon
bool TeOverlaps(const TeCell& cell, const TePolygon& poly);
/** @} */


/** @defgroup TeCoveredBy Covered by test
 *  @ingroup TopologicalOperators
 *  Check if one object is covered by another object.
 *  TeCoveredBy(x, y) => (x inter y = x) ^ (I(x) inter I(y) = true)
 *  - Case 1: (x = TePolygon and y = TePolygon) or (x = TeLine2D and y = TeLine2D)
 *            => (I(x) inter I(y) = true) ^ (I(x) inter E(y) = false) ^ (B(x) inter E(y) = false) ^ (B(x) inter B(y) = true)
 *  - Case 2: x = TeLine2D and y = TePolygon
 *            => (I(x) inter I(y) = true) ^ (I(x) inter E(y) = false) ^ (B(x) inter E(y) = false) ^ ((B(x) inter B(y) = true) v (I(x) inter B(y) = true))
 *  @{
 */

//! Check if red line is covered by blue line
bool TeCoveredBy(const TeLine2D& redLine, const TeLine2D& blueLine);

//! Check if line is covered by polygon
bool TeCoveredBy(const TeLine2D& l, const TePolygon& pol);

//! Check if red polygon is covered by blue polygon
bool TeCoveredBy(const TePolygon& redPol, const TePolygon& bluePol);

//! Check if cell1 is covered by cell2
bool TeCoveredBy(const TeCell& cell1, const TeCell& cell2);

//! Check if polygon is covered by cell
bool TeCoveredBy(const TePolygon& poly, const TeCell& cell);

//! Check if line is covered by cell
bool TeCoveredBy(const TeLine2D& line, const TeCell& cell);

/** @} */

/** @defgroup TeCovers Covers test
 *  @ingroup TopologicalOperators
 *  Check if one object covers another object.
 *  TeCovers(x, y) => TeCoveredBy(y, x)
 *  @{
 */
template<typename T1, typename T2> inline bool TeCovers(T1& o1, T2& o2)
{
	return TeCoveredBy(o2, o1);
}
/** @} */

/** @defgroup TeRelation Relation test
 *  @ingroup TopologicalOperators
 *  Return the relation between two objects.
 *  @{
 */

/*! \fn short TeRelation(const TeCoord2D& c, const TeLine2D& l)
    \brief Returns the relation between coordinate c and line l.
	\param c  The coordinate.
	\param l  The line.

	This function may returns: INSIDE, OUTSIDE or BOUNDARY.
	Obs: Doesn't do box elimination, just uses from TeIsOnLine(coordinate, line) elimination
 */
short TeRelation(const TeCoord2D& c, const TeLine2D& l);

 
 /*! \fn short TeRelation(const TeCoord2D& c, const TeLinearRing& ring)
    \brief Point in polygon inside/outside/boundary code.
	\param c    The coordinate to test.
	\param ring The simple polygon to test.

	Check if point is INSIDE/OUTSIDE/BOUNDARY of a given ring.
	The ring is treated as a simple polygon (no holes).
	Obs: Do box elimination.
*/
short TeRelation(const TeCoord2D& c, const TeLinearRing& r);

/*! \fn short TeRelation(const TeCoord2D& c, const TePolygon& pol);
    \brief Point in polygon inside/outside/boundary code.
	\param c    The coordinate to test.
	\param pol  The polygon to test.

	Check if point is INSIDE/OUTSIDE/BOUNDARY.
	Obs: Doesn't do box elimination, just uses from TeRelation(coordinate, ring) elimination.
*/
short TeRelation(const TeCoord2D& c, const TePolygon& pol);

short TeRelation(const TePoint& p, const TePolygon& pol); 

/*! \fn short TeRelation(const TeCoord2D& c, const TePolygonSet& pSet)
    \brief Point in polygon set inside/outside/boundary code.
	\param c     The coordinate to test.
	\param pSet  The polygon set to test.

	Check if point is INSIDE/OUTSIDE/BOUNDARY.
	Obs: Do box elimination.
*/
short TeRelation(const TeCoord2D& c, const TePolygonSet& pSet);

/*! \fn short TeRelation(const TeLine2D& lRed, const TeLine2D& lBlue, const short& relation)
    \brief This function returns the relation between two lines.
	\param lRed     The first line.
	\param lBlue    The second line.
	\param relation The relation that stop the search.

	This function returns the relation between lines. May be: TeDISJOINT, TeTOUCHES, TeWITHIN, TeCONTAINS, TeCROSSES, TeOVERLAPS, TeCOVEREDBY, TeCOVERS, TeEQUALS.
	Obs: Doesn't do box elimination. You must implement the test in your own functions.
*/
short TeRelation(const TeLine2D& lRed, const TeLine2D& lBlue, const short& relation);

/*! \fn short TeRelation(const TeLine2D& line, const TePolygon& pol)
    \brief This function returns the relation between a line and a polygon.
	\param line   The line.
	\param pol    The polygon.

	This function returns the relation between a line and a polygon. May be: TeDISJOINT, TeTOUCHES, TeWITHIN (THE LINE IS WITHIN), TeCROSSES, TeCOVEREDBY (THE LINE IS COVERED BY).
	Obs: Doesn't do box elimination. You must implement the test in your own functions.
*/
short TeRelation(const TeLine2D& line, const TePolygon& pol);

/*! \fn short TeRelation(const TePolygon& pRed, const TePolygon& pBlue)
    \brief This function returns the relation between two polygons.
	\param pRed   The first polygon.
	\param pBlue  The second polygon.

	This function returns the relation between a line and a polygon. May be: TeEQUALS, TeDISJOINT, TeTOUCHES, TeWITHIN (pRed IS WITHIN pBlue), TeCONTANS (pBlue CONTAINS pRed), TeOVERLAPS, TeCOVEREDBY (pRed IS COVERED BY pBlue) or TeCOVERS (pRed COVERS pBlue).
	Obs: Doesn't do box elimination. You must implement the test in your own functions.
*/
short TeRelation(const TePolygon& pRed, const TePolygon& pBlue);
/** @}	*/
/** @}	*/

/** @defgroup BoxTests Special box tests
 *  @ingroup GeometryAlgorithms
 *  Box tests.
 *  @{
*/

//! Check if box1 is disjoint or touches box2
bool TeDisjointOrTouches(const TeBox& bx1, const TeBox& bx2);

//! Check if coordinate c is within or touches segments c1 and c2 box
bool TeWithinOrTouches(const TeCoord2D& c, const TeCoord2D& c1, const TeCoord2D& c2);
/** @} */


/** @defgroup GeometryTests Special Geometry tests
 *  @ingroup GeometryAlgorithms
 *  Geometry tests.
 *  @{
*/

//! Check if geom1 is within or is covered by or is equal to geom2
template<class T1, class T2> 
bool TeWithinOrCoveredByOrEquals(const T1& geom1, const T2& geom2)
{
	short rel = TeRelation(geom1, geom2);

	if((rel&TeINSIDE) || (rel&TeBOUNDARY))
	   return true;

	return false;
}

//! Check if box1 is within or is covered by or is equal to box2
template<>
bool TeWithinOrCoveredByOrEquals(const TeBox& bx1, const TeBox& bx2);

//! Check if line1 is within or is covered by or is equal to line2
template<>
bool TeWithinOrCoveredByOrEquals(const TeLine2D& line1, const TeLine2D& line2);

//! Check if line1 is within or is covered by to pol
template<>
bool TeWithinOrCoveredByOrEquals(const TeLine2D& line1, const TePolygon& pol);

//! Check if pol1 is within or is covered by or is equal to pol2
template<>
bool TeWithinOrCoveredByOrEquals(const TePolygon& pol1, const TePolygon& pol2);

/** @} */


/** @defgroup IntersectionOperators Intersection Operators
 *  @ingroup  GeometryAlgorithms
 *  Functions that calculate the intersection among objects or do intersection test.
 *  @{
 */

/*! \fn bool TeIntersection(const TeBox& bx1, const TeBox& bx2, TeBox& bout)
    \brief Performs the intersection between  box b1 and b2, returning the resulting box bout.
	\param bx1  The first box to do the intersection.
	\param bx2  The second box to do the intersection.
	\param bout The box formed by the intersection of bx1 and bx2.
*/
bool TeIntersection(const TeBox& bx1, const TeBox& bx2, TeBox& bout);

/*! \fn TeCoordPairVect	TeGetIntersections(const TePolygon &poly, const double& y);
    \brief Return the segments that intersept the poly polygon in the line y.
	\param poly    A polygon.
	\param y    The ordinate that cuts the polygons edges.
*/
TeCoordPairVect	TeGetIntersections(const TePolygon &poly, const double& y);
/** @} */


/** @defgroup UnionOperators Union Operators
 *  @ingroup  GeometryAlgorithms
 *  Functions that compute the union of objects.
 *  @{
 */

/*! \fn TeBox TeUnion(const TeBox& bx1, const TeBox& bx2)
    \brief Combine two box, make one that includes both.
	\param bx1  The first box to do the union.
	\param bx2  The second box to do the union.
*/
TeBox TeUnion(const TeBox& bx1, const TeBox& bx2);

/** @} */

/** @defgroup TeLocationFunctions Functions that finds the localization of objects.
 *  @ingroup  GeometryAlgorithms
 *  Functions that finds the localization of objects.
 *  @{
 */

/*! \fn bool TePointInPoly(const TeCoord2D& c, const TeLinearRing& r)
    \brief Point in polygon inside/outside/boundary code.
	\param c    The coordinate to test.
	\param r The simple polygon to test.

	Check if point is INSIDE of a given ring.
	The ring is treated as a simple polygon (no holes).
	Adapted from:
	Samosky, Joseph, "SectionView: A system for interactively specifying and
    visualizing sections through three-dimensional medical image data,
    M.S. Thesis, Department of Electrical Engineering and Computer Science,
    Massachusetts Institute of Technology, 1993.
	Obs: Doesn't do box elimination.
*/
bool TePointInPoly(const TeCoord2D& c, const TeLinearRing& r);

/*! \fn bool TeIsOnSegment(const TeCoord2D& c, const TeCoord2D& c1, const TeCoord2D& c2)
    \brief Check if coordinate c is on segment (segment is closed).
	\param c   The coordinate to be tested.
	\param c1  The first segment's coordinate.
	\param c2  The second segment's coordinate.
*/
bool TeIsOnSegment(const TeCoord2D& c, const TeCoord2D& c1, const TeCoord2D& c2);

/*! \fn bool TeIsOnLine(const TeCoord2D& c, const TeLine2D& l)
    \brief Check if coordinate c is on line boundary or on line interior (see explanation above for definition of boundary and interior of a line).
	\param c   The coordinate to be tested.
	\param l   The line used in the test.

	Obs: Do box elimination.
*/
bool TeIsOnLine(const TeCoord2D& c, const TeLine2D& l);

/*! \fn bool TeLocateLineSegment(TeCoord2D& pin, TeLine2D& line, int& segment, double tol = 0.0)
    \brief Locate the nearest line segment of a coordinate.
	\param pin		The coordinate.
	\param line		The line.
	\param segment  The position of the segment in the line  
	\param tol		Tolerance.
*/
bool TeLocateLineSegment(TeCoord2D& pin, TeLine2D& line, int& segment, double tol = 0.0);
/** @} */

/** @defgroup TeConvexHull Functions to compute the Convex Hull
 *  @ingroup  GeometryAlgorithms
 *  Functions that returns the convex hull of a point list.
 *  @{
 */

 /*! \fn template<class T> TePolygon TeConvexHull(const T& coordSet)
    \brief Returns the convexhull of a given list of coords in counterclockwise.
	\param coordSet A list with coordinates without duplicated coordinates.
	
	This algorithm is based on the book Computational Geometry
	by M. de Berg, M. van Kreveld, M. Overmars and O. Schwarzkopf - Springer Verlag - pp. 6.
	It is O(N log N).

*/
template<class T> TePolygon TeConvexHull(const T& coordSet);

//! This is a explicit specialization that returns the convex hull of a TePolygon
template<> TePolygon TeConvexHull(const TePolygon& p);

//! This is a explicit specialization that returns the convex hull of a TePolygonSet
template<> TePolygon TeConvexHull(const TePolygonSet& ps);

//! This is a explicit specialization that returns the convex hull of a TePointSet. Must be defined!
template<> TePolygon TeConvexHull(const TePointSet& ps);

/** @} */

/** @defgroup TeUtils Utilities functions.
 *  @ingroup  GeometryAlgorithms
 *  @{
 */

//! Given a projection "proj" returns a tolerance value in the same unit of projection to be used in geometric operations
double TeGetPrecision(TeProjection* proj);


//! This class implements the Epsilon-tweaking used in calculus.
class TeGeometryAlgorithmsPrecision
{
	protected:

		//! Constructor
		TeGeometryAlgorithmsPrecision();

	public:		

		//! Tells if d1 is greater than d2 according to tolerance factor.
		static bool IsGreater(const double& d1, const double& d2)
		{
			return ((d1 - d2) > TePrecision::instance().precision());
		}

		//! Tells if d1 is greater than or equal to d2 according to tolerance factor.
		static bool IsGreaterEqual(const double& d1, const double& d2)
		{
			return ((d1 - d2) >= (-TePrecision::instance().precision()));
		}

		//! Tells if d1 is smaller than d2 according to a tolerance factor.
		static bool IsSmaller(const double& d1, const double& d2)
		{
			return ((d1 - d2) < -(TePrecision::instance().precision()));
		}

		//! Tells if d1 is smaller than or equals to d2 according to a tolerance factor.
		static bool IsSmallerEqual(const double& d1, const double& d2)
		{
			return ((d1 - d2) <= TePrecision::instance().precision());
		}

		//! Tells if d1 is equals to d2 according to a tolerance factor.
		static bool IsEqual(const double& d1, const double& d2)
		{
			return (fabs(d1 - d2) <= TePrecision::instance().precision());
		}

		//! Tells if d1 is different from d2 according to a tolerance factor.
		static bool IsDifferent(const double& d1, const double& d2)
		{
			return (fabs(d1 - d2) > TePrecision::instance().precision());
		}
};

inline void TeRemoveDuplicatedCoordinates(TeLine2D& l)
{
	for(unsigned int i = 0; i < l.size() - 1; ++i)
		if(TeEquals(l[i], l[i + 1]))
		{
			l.erase(i);
			--i;
		}

	return;
}

inline void TeRemoveDuplicatedCoordinates(TePolygon& p)
{
	for(unsigned int i = 0; i < p.size(); ++i)
		TeRemoveDuplicatedCoordinates(p[i]);		

	return;
}

/*! \fn void TeReverseLine (TeLine2D& lin);
	\brief Reverses the line.
	\param lin  The line to be reversed.
*/
inline void TeReverseLine(TeLine2D& lin)
{
	for(unsigned int i=0,j=lin.size()-1 ; i<lin.size()/2 ; ++i,--j)
	{
		TeCoord2D p = lin[i];
		lin[i] = lin[j];
		lin[j] = p;
	}

	return;
}

/** @} */

/** @defgroup ValidationOperators Validation Operators
 *  @ingroup  GeometryAlgorithms
 *  Functions that do some validation tests.
 *  @{
 */

/*! \fn bool TeIsConvex(const TeLinearRing& ring);
    \brief Tells if a simple polygon defined as a linear ring is convex or not.
	\param ring  The polygon to test convexity.
*/
bool TeIsConvex(const TeLinearRing& ring);

/*! \fn short TeOrientation(const TeLinearRing& r);
	\brief Returns the orientation of the ring (CLOCKWISE or COUNTERCLOCKWISE);
	\param r The ring to be checked.
*/
short TeOrientation(const TeLinearRing& r);

/** @} */

/** @defgroup MetricOperators Metric Operators
 *  @ingroup  GeometryAlgorithms
 *  Functions that do some usefull metric operations.
 *  @{
 */

/*! \fn void TeGetMiddlePoint(const TeCoord2D& first, const TeCoord2D& last, TeCoord2D& middle)
    \brief Returns the middle point of a segment.
	\param first   The first segment's coordinate.
	\param last    The second segment's coordinate.
	\param middle  The middle point.
*/
void TeGetMiddlePoint(const TeCoord2D& first, const TeCoord2D& last, TeCoord2D& middle);


/*! \fn double TeDistance(const TeCoord2D& c1, const TeCoord2D& c2)
    \brief Euclidian distance between two points.
	\param c1 First coordinate;
	\param c2 Second coordinate;
*/
double TeDistance(const TeCoord2D& c1, const TeCoord2D& c2);

/*! \fn double TeLength(const TeLine2D& l)
    \brief Returns the length of a Line 2D.
	\param l  The line to calculate the length.	
*/
double TeLength(const TeLine2D& l);

/*! \fn double TePerpendicularDistance(const TeCoord2D& first, const TeCoord2D& last, const TeCoord2D& pin, TeCoord2D &pinter)
    \brief Perpendicular distance from point to segment.
	\param first  The first segment's coordinate.
	\param last   The second segment's coordinate.
	\param pin    The point to get the distance from the segment.
	\param pinter The point of intersection on the segment.
*/
double TePerpendicularDistance(const TeCoord2D& first, const TeCoord2D& last, const TeCoord2D& pin, TeCoord2D &pinter);

/*! \fn double TeMinimumDistance(const TeCoord2D& first, const TeCoord2D& last, const TeCoord2D& pin, TeCoord2D& pout, double tol = 0.0);
    \brief Minimal distance from point to segment.
	\param first	The first segment's coordinate.
	\param last		The second segment's coordinate.
	\param pin		The point to get the minimal distance from the segment. This point is inside the segment  
	\param pout		The nearest segment point of the pin point.
	\param tol		Numerical tolerance
*/
double TeMinimumDistance (const TeCoord2D& first, const TeCoord2D& last, const TeCoord2D& pin, TeCoord2D& pout, double tol = 0.0);

/** @defgroup TeGeometryArea Functions to compute the area of a geometry
 *  @ingroup  MetricOperators
 *  Functions that returns the area of a geometry.
 *  @{
 */

/*! \fn template<class T> double TeGeometryArea(const T& geom)
    \brief Returns the area of a TePolygon or TePolygonSet, otherwise return zero.
	\param geom The geometry whose area we want to known.

	This algorithm is based on the book Spatial Databases with Application to GIS
	by Philippe Rigaux, Michel O. Scholl and Agnes Voisard.

*/

template<class T> double TeGeometryArea(const T& /* geom */)
{
	return 0.0;
}

//! This is a explicit specialization that returns the area of a TePolygon
template<>  double TeGeometryArea(const TePolygon& p);

//! This is a explicit specialization that returns the area of a TePolygonSet
template<>  double TeGeometryArea(const TePolygonSet& ps);

//! This is a explicit specialization that returns the area of a Box
template<>  double TeGeometryArea(const TeBox& b);
/** @} */

/** @} */



/** @defgroup GeometryFunction Functions that return geometries.
 *  @ingroup  GeometryAlgorithms
 *  Functions that return geometries.
 *  @{
 */

 /*! \fn TePolygon TeMakePolygon(const TeBox& b)
    \brief Given a box return its polygon representation.
	\param b  The box to create a polygon.
*/
TePolygon TeMakePolygon(const TeBox& b);

/*! \fn TeLinearRing TeSimpleClosedPath(const TePointSet& pSet)
    \brief Given N points, finds a path that doesn't self-intersects, visiting all points and returning to the begginning one.
	\param pSet The point set to form a path.

	It is based on the book Algorithms by Robert Sedgewick, Addisson-Wesley, 1988.
*/
TeLinearRing TeSimpleClosedPath(const TePointSet& pSet);
/** @} */


/** @defgroup TeFindCentroid Functions to compute the centroid
 *  @ingroup  GeometryAlgorithms
 *  Functions that return the centroid.
 *  @{
 */
/*! \fn TeCoord2D TeFindCentroid(const TePolygon& p);
    \brief Calculates the centroid of a polygon.
	\param p A TePolygon whose centroid we want to known.
*/
TeCoord2D TeFindCentroid(const TePolygon& p);

/*! \fn TeCoord2D TeFindCentroid(const TeLine2D& l);
    \brief Calculates a reference point.
	\param l A TeLine whose centroid we want to known.
*/
TeCoord2D TeFindCentroid(const TeLine2D& l);  

/*! \fn TeCoord2D TeFindCentroid(const TeCell& c );
    \brief Calculates the centroid of a cell.
	\param c A TeCell whose centroid we want to known.
*/
TeCoord2D TeFindCentroid(const TeCell& c);

/*! \fn TeCoord2D TeFindCentroid(const TePoint& p );
    \brief Calculates the centroid of a point.
	\param p A TePoint whose centroid we want to known.
*/
TeCoord2D TeFindCentroid(const TePoint& p);

/*! \fn TeCoord2D TeFindCentroid(const TePolygonSet& s );
    \brief Calculates the centroid of a polygon set.
	\param s A TePolygon set whose centroid we want to known.
*/
TeCoord2D TeFindCentroid(const TePolygonSet& s); 

/*! \fn TeCoord2D TeFindCentroid(const TeLineSet& s );
    \brief Calculates the centroid of a line set.
	\param s A TeLine set whose centroid we want to known.
*/
TeCoord2D TeFindCentroid(const TeLineSet& s); 

/*! \fn TeCoord2D TeFindCentroid(const TeCellSet& s );
    \brief Calculates the centroid of a cell set.
	\param s A TeCell set whose centroid we want to known.
*/
TeCoord2D TeFindCentroid(const TeCellSet& s);

/*! \fn TeCoord2D TeFindCentroid(TePointSet& ps );
    \brief Calculates the centroid of a point set.
	\param ps A TePointSet set whose centroid we want to known.
*/
TeCoord2D TeFindCentroid(TePointSet& ps);
/** @} */


/** @defgroup TeNearest Functions to compute the nearest part of an object to a point.
  *  @ingroup  GeometryAlgorithms
  *  Auxiliary functions.
  *  @{
  */
//! Nearest node in set from location pt (i = the node index in the nodeset)
bool TeNearest(TeCoord2D& pt, TeNodeSet& ns , int& i, const double& tol = 0.0);

//! Nearest point in set from location pt (i = the point index in the pointset)
bool TeNearest(TeCoord2D& pt, TePointSet& ps , int& i, const double& tol = 0.0);

//! Nearest text in set from location pt (i = the text index in the textset)
bool TeNearest(TeCoord2D& pt, TeTextSet& ts , int& i, const double& tol = 0.0);

//! Nearest line in set from location pt (i = the line index in the lineset and pi = the closest point)
//! The pi point can be outside a line of the line set
bool TeNearest(TeCoord2D& pt, TeLineSet& ls , int& i, TeCoord2D& pi, const double& tol = 0.0);

//! Nearest line in set from location pt (i = the line index in the lineset and pout = the closest point)
//! The pout point must be inside a line of the line set
bool TeNearest (TeCoord2D& pt,TeLineSet& ls, int& lindex, TeCoord2D& pout, double& dmin, const double& tol = 0.0); 

//! Nearest polygon in set from location pt (i = the polygon index in the polygonset).
bool TeNearest(TeCoord2D& pt, TePolygonSet& ps , int& i, const double& tol = 0.0);

//! Nearest line in set from location pt (i = the line index in lineset) calculated by the vertex of the lines
bool TeNearestByPoints(TeCoord2D& pt, TeLineSet& ls , int& i, double& dist, const double& tol = 0.0);
/** @} */


/** @defgroup TINfunction Functions used by TIN.
 *  @ingroup  GeometryAlgorithms
 *  TIN functions.
 *  @{
 */
/*! \fn bool TeFindTriangleCenter(const TeCoord2D& vert0, const TeCoord2D& vert1, const TeCoord2D& vert2, TeCoord2D& pc); 
    \brief Calculates the center of a triangle.
	\param vert0 First triagle's vertex.
	\param vert1 Second triagle's vertex.
	\param vert2 Third triagle's vertex.
	\param pc    The triangle center.
*/
bool TeFindTriangleCenter(const TeCoord2D& vert0, const TeCoord2D& vert1, const TeCoord2D& vert2, TeCoord2D& pc); 

//! Line simplification.
bool TeLineSimplify(TeLine2D& line, double snap, double maxdist);

/** @} */

/** @defgroup Curvefunction Functions used to make curve.
  *  @ingroup  GeometryAlgorithms
  *  Curve functions.
  *  @{
  */

/*! \fn bool TeGetCenter(TePoint p1, TePoint p2, TePoint p3, TePoint &center);
    \brief Given three points of a circumference, returns the center point.
	\param p1      First point.
	\param p2      Second point.
	\param p3      Third point.
	\param center  Circunference center.
	This algorithm is adapted from http://www.delphiforfun.org/Programs/Math_Topics/circle_from_3_points.htmbook.
*/
bool TeGetCenter(TePoint p1, TePoint p2, TePoint p3, TePoint &center);

/*! \fn double TeGetRadius(TePoint& p1, TePoint& p2, TePoint& p3);
    \brief Given three points of a circumference, returns the radius.
	\param p1      First point.
	\param p2      Second point.
	\param p3      Third point.
	This algorithm is adapted from http://www.delphiforfun.org/Programs/Math_Topics/circle_from_3_points.htmbook.
*/
double TeGetRadius(TePoint &p1, TePoint &p2, TePoint &p3);

/*! \fn bool TeGenerateArc(TePoint& p1, TePoint& p2, TePoint& p3, TeLine2D& arcOut, const short& NPoints);
    \brief Given three points of a circle, returns the an arc passing for this points that is formed by a given number of points.
	\param p1      First point.
	\param p2      Second point.
	\param p3      Third point.
	\param arcOut  The return arc.
	\param NPoints Number of arc points.	
	
	This algorithm is adapted from http://mathforum.org/dr.math/
  
*/
bool TeGenerateArc(TePoint& p1, TePoint& p2, TePoint& p3, TeLine2D& arcOut, const short& NPoints);

/*! \fn bool TeGenerateCircle(TePoint& center, const double& radius, TeLine2D& circle, const short& NPoints);
    \brief Given the center and the radius, returns the circle formed by a given number of points.
	\param center  Center point of the circle.
	\param radius  radius of the circle.
	\param circle  The return circle
	\param NPoints Number of circle points.	
*/
bool TeGenerateCircle(TePoint& center, const double& radius, TeLine2D& circle, const short& NPoints);

/** @} */

/** @} */ // end of group  GeometryAlgorithms


#endif	// __TERRALIB_INTERNAL_GEOMETRYALGORITHMS_H
