/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TePostgreSQL.h
    This file contains a database driver to access PostgreSQL server.
 */

/**
  *@author Gilberto Ribeiro de Queiroz - gribeiro@dpi.inpe.br
  */

/**
  *@description This file contains the especifics routines of PostgreSQL.
  */

/*
 * TODO: 01. Implementar o alterTable que no pode simplesmente alterar o tipo da coluna
 *		 02. No mtodo query do portal, determinar os campos que fazem parte da chave primria
 *       03. Insert Blob
 *
 */

#ifndef  __TERRALIB_INTERNAL_POSTGRESQL_H
#define  __TERRALIB_INTERNAL_POSTGRESQL_H

#include "TeDatabase.h"
#include "TePGInterface.h"


class TePostgreSQLPortal;

//! PostgreSQL database access class.
/*!
	This class contains the implementation of common methods for TerraLib access PostgreSQL.    
 */	
class TePostgreSQL : public TeDatabase
{
	friend class TePostgreSQLPortal;

	public:
		
		//! Constructor
		TePostgreSQL();

		//! Destructor
		virtual ~TePostgreSQL();

		//! Returns the error message from the server
		string errorMessage();

		//! Creates a new database based on "template1" and open a connection to the new one
		virtual bool newDatabase(const string& database, const string& user, const string& password, const string& host, const int& port = 0, bool terralibModel=true);

		//! Opens a conection to a database server
		virtual bool connect(const string& host, const string& user, const string& password, const string& database, int port = 0);

		//! Show the server databases (only for MySQL, Oracle and PostgreSQL)
		virtual bool showDatabases(const string& host, const string& user, const string& password, vector<string>& dbNames, int port = 0);

		//! Closes the conection
		void close(void);

		bool listTables(vector<string>& tableList);

		//! Verifies if a table exist
		bool tableExist(const string& table);

		//! Verifies if a table has a column
	    bool columnExist(const string& table, const string& column, TeAttribute& attr);

		//! Creates a generic table
		bool createTable(const string& table, TeAttributeList &attr);

		//! Adds an attribute field
		bool addColumn(const string& table, TeAttributeRep &rep);

		//! Creates a reationship between two tables
		bool createRelation(const string& relName, const string& table, const string& fieldName, const string& relatedTable, const string& relatedField, bool cascadeDeletion);

		//! Checks if a relation exist
		TeDBRelationType existRelation(const string& tableName, const string& relName);

		//! Executes a SQL command that doesnt return a record set tipically a data definition comand
		bool execute(const string &sql);

		//! Returns a portal associated to this database
		virtual TeDatabasePortal* getPortal();

		//! Creates a table to store version database information 
		virtual bool createDatabaseTable();

		//! Creates a table to stores projections
		bool createProjectionTable();

		//! Creates a table to store Layers information
		bool createLayerTable();

		//! Creates a table to store Representations information
		bool createRepresentationTable();

		//! Creates a table to store Views information
		bool createViewTable();

		//! Creates a table to store Themes information
		bool createThemeTable();

		//! Creates a table to store the groupings associated with a theme
	    bool createGroupingTable();

		//! Creates a table to store information about the attribute tables used by a theme
		bool createThemeTablesTable();

		//! Creates a table to store legends
		bool createLegendTable();

		//! Creates a table to store visual definitions
		bool createVisualTable();

		//! Creates a table to store raster visual definitions
		bool createVisualRasterTable();

		//! Creates a table to store information about the non-spatial tables associated to this layer
		bool createLayerTableTable();

		//! Creates a table to store information about the external tables related to non-spatial tables of layers
		bool createTablesRelationTable();

		//! Creates a table for a polygon geometries
		virtual bool createPolygonGeometry(const string& tableName);

		//! Creates a table for line geometries
		virtual bool createLineGeometry(const string& tableName);

		//! Creates a table for point geometries
		virtual bool createPointGeometry(const string& tableName);

		//! Creates a table for cell geometries
		virtual bool createCellGeometry(const string& tableName);

		//! Creates a table for a text geometries
		bool createTextGeometry(const string& tableName);

		//! Creates a table for a arc geometries
		bool createArcGeometry(const string& tableName);

		//! Creates a table for a node geometries
		virtual bool createNodeGeometry(const string& tableName);

		//! Creates a table for raster geometries
		virtual bool createRasterGeometry(const string& tableName);	

		//! Creates a table to store information about raster representation
		bool createRasterMetadataTable(const string& tableName);

		//! Creates a table for raster geometries
	    virtual bool createRasterTable(const string& tableName);

		//! Creates a table to store information about objects in a theme
		bool createCollectionTable(const string& tableName);

		//! Inserts information about a link to an external table
		bool insertRelationInfo(const int tableId, const string& tField, const string& rTable, const string& rField, int& relId);

		//! Insert information about a table related to a layer
	    bool insertTableInfo(int layerId, TeTable& table, const string& user = "");

		//! Saves a table and its contents in the database
		bool insertTable(TeTable &table);

		//! Alter a property of a table 
		bool alterTable(const string& tableName, TeAttributeRep& rep, const string& oldColName="");

		//! Saves a large binary objects (BLOB) in a row table 
	    bool insertBlob (const string& tableName, const string& columnBlob, TeAttributeRep& columnId, const string& valueId, unsigned char* data, int size);

	    //! Saves a large binary objects (BLOB) in a row table 
	    bool insertBlob (const string& tableName, const string& columnBlob, TeAttributeRep& columnId, const string& valueId, const string& fileName);

		//! Insert information about a geographical projection
		bool insertProjection(TeProjection *proj);

		//! Insert information about a layer
		bool insertLayer(TeLayer *layer);

		//! Insert information about a geometrical representation
		bool insertRepresentation(int layerId, TeRepresentation& rep);

		//! Inserts information about a view
		bool insertView(TeView *view);

		//! Inserts view tree information
		bool insertViewTree(TeViewTree *tree);

		//! Inserts a theme group
		bool insertThemeGroup(TeViewTree* tree);

		//! Inserts theme information
		bool insertTheme(TeTheme *theme);

		//! Inserts information about a table used by a theme 
		bool insertThemeTable(int themeId, int tableId, int relationId, int tableOrder);

		//! Generate the label position (x,y) to each object of a theme
		virtual bool generateLabelPositions(TeTheme *theme);

		//! Inserts legend information
		bool insertLegend(TeLegendEntry *legend);		

		//! Inserts a polygon
		virtual bool insertPolygon(const string& table, TePolygon& p);

		//! Updates a polygon
		virtual bool updatePolygon(const string& table, TePolygon& p);

		//! Inserts a line
		virtual bool insertLine(const string& table, TeLine2D& l);

		//! Updates a line
		virtual bool updateLine(const string& table, TeLine2D& l);

		//! Inserts a point
		virtual bool insertPoint(const string& table, TePoint& p);

		//! Inserts text
		bool insertText(const string& table, TeText& t);

		//! Inserts an arc
		bool insertArc(const string& table,TeArc& arc);

		//! Inserts a node
		virtual bool insertNode(const string& table, TeNode& node);

		//! Inserts a cell
		virtual bool insertCell(const string& table, TeCell& c);		

		//! Insert a raster block into the database
		virtual bool insertRasterBlock(const string& table, const string& blockId, const TeCoord2D& ll, const TeCoord2D& ur, unsigned char* buf,unsigned long size, int band = 0, unsigned int res = 1, unsigned int subband = 0);

		//! Creates a lookup table (used for pallete raster representations)
		bool createLUTTable(const string& name);

		//! Return a string SQL to calculate the statistics to each cell through database functions 
		string getSQLStatistics(TeGroupingAttr& attrs);

		//! Return a string SQL to temporal where 
		string getSQLTemporalWhere(TeTimeInterval& timeInterval, TeTemporalRelation timeOperator, const string& initialTime, const string& finalTime);

		//! Return a string SQL to temporal where
		string getSQLTemporalWhere(int time1, int time2, TeChronon chr, TeTemporalRelation rel, const string& initialTime, const string& finalTime);

		//! Return the database function to generate autonumber values
		string getAutoNumberSQL(const string& table);

		//! Concat values in a vector using unionString as the join between each value
	    string concatValues(vector<string>& values, const string& unionString);

	    //! Returns the SQL function for upper case
	    string toUpper(const string& value);

	protected:

		//! Returns a string with insert of a raster block
		virtual void getInsertRasterBlock(const string& table, const string& blockId, const TeCoord2D& ll, const TeCoord2D& ur, const int& band, const unsigned int& res, const unsigned int& subband, char* buf, const unsigned long& size, string& sql);

		//! Returns a string with update of a raster block
		virtual void getUpdateRasterBlock(const string& table, const string& blockId, const TeCoord2D& ll, const TeCoord2D& ur, const int& band, const unsigned int& res, const unsigned int& subband, char* buf, const unsigned long& size, string& sql);
		
		//! Escape special characters in a string to be used in a SQL statement
		string escapeSequence(const string& from);

		//! Opens a conection to a database server
		bool realConnect(const string& host, const string& user, const string& password, const string& database, int port = 0);

		//! Connection to a PostgreSQL server
		TePGConnection tepg_connection_;

};


//! PostgreSQL data navigation functionalities.
/*!
	This class contains the implementation of common methods for TerraLib navigate throw data.    
 */	
class TePostgreSQLPortal : public TeDatabasePortal
{
	public:

		//! Constructor
		TePostgreSQLPortal();

		//! Constructor
		TePostgreSQLPortal(TeDatabase *pDatabase);
		
		//! Virtual Destructor
		virtual ~TePostgreSQLPortal();

		//! Executes a SQL query that opens a record set
		virtual bool query(const string &qry, TeCursorLocation l = TeSERVERSIDE, TeCursorType t = TeUNIDIRECTIONAL, TeCursorEditType e = TeREADONLY, TeCursorDataType dt = TeTEXTCURSOR);

		//! Fetchs the next row in a record set that shouldve been previously opened
		bool fetchRow();

		//! Fetchs a particular row
		bool fetchRow(int i);

		//! Frees the current record set
		void freeResult();

		//! Gets the last error message
		string errorMessage();

		//! Gets the value of the i-th attribute as a literal
		char* getData(int i);

		//! Gets the value of a named attribute as a literal
		char* getData(const string& s);

		//! Gets the value of the i-th attribute a double
		double getDouble(int i);

		//! Gets the value of a named attribute as a double
		double getDouble(const string& s);

		//! Gets the value of the i-th attribute as an integer
		int	getInt(int i);

		//! Gets the value of a named attribute as an integer
		int	getInt(const string& s);

		//! Gets the value of a named attribute as a boolean
		bool getBool(const string& s);

		//! Gets the value of the i-th attribute as a boolean
		bool getBool(int i);

		//! Gets the value of the i-th attribute as a date 
	    TeTime getDate(int i);

	    //! Gets the value of a named attribute as a date 
	    TeTime getDate(const string& s);

		//! Gets the of a date/time attribute as a string formatted as accepted in further SQL statements
		string getDateAsString(int i);

		//! Gets the of a date/time attribute as a string formatted as accepted in further SQL statements
		string getDateAsString(const string& s);

		//! Gets the value of a named BLOB attribute 
		bool getBlob(const string& s, unsigned char*& data, long& size);

		virtual bool getRasterBlock(unsigned long& size, unsigned char* ptData);

		virtual bool fetchGeometry(TePolygon& pol);
		virtual bool fetchGeometry(TeLine2D& line);
		virtual bool fetchGeometry(TeNode& n);
		virtual bool fetchGeometry(TePoint& p);


	protected:

		//! Return a linear ring and the number of holes in case of external ring
		TeLinearRing getLinearRing(int& numberOfHoles);

		//! Escape special characters in a string to be used in a SQL statement
		string escapeSequence(const string& from);

		// Protected data members
		TePGConnection *con_;
		TePGRecordset tepg_recordset_;

		long curRow_;		

};

#endif	// __TERRALIB_INTERNAL_TePOSTGRESQL_H

