/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeMySQL.h"
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>

extern "C" {
extern char *strxmov(char* dest, const char* src, ...);
}

#define MAX(a,b) a>b?a:b
#define MIN(a,b) a<b?a:b


double swappd(double din)
{
	char  stri[8],stro[8];
	double dout;

	memcpy (stri,&din,8);
	stro[0] = stri[7];
	stro[1] = stri[6];
	stro[2] = stri[5];
	stro[3] = stri[4];
	stro[4] = stri[3];
	stro[5] = stri[2];
	stro[6] = stri[1];
	stro[7] = stri[0];
	memcpy (&dout,stro,8);
	return (dout);
}


TeMySQL::TeMySQL()
{
	isConnected_ = false;
	dbmsName_ = "MySQL";
	bufferBlob_ = 0;
	bufferBlobSize_ = 0;
	mysql_init (&mysql_);
}


TeMySQL::~TeMySQL()
{
	if (bufferBlob_)
		delete bufferBlob_;
	if (isConnected_)
		close ();
}

void
TeMySQL::close()
{
	clear();
	if (bufferBlob_)
		delete bufferBlob_;
	bufferBlob_ = 0;
	bufferBlobSize_ = 0;
	mysql_close(&mysql_);
}

bool 
TeMySQL::newDatabase(const string& database, const string& user, const string& password, const string& host, const int & /* port */, bool terralibModel)
{
	if (mysql_real_connect(&mysql_,host.c_str(),user.c_str(),password.c_str(),"",0,NULL,0) == NULL)
	{
		isConnected_ = false;
		return false;
	}

	string create = "CREATE DATABASE " + database;
	if (mysql_query(&mysql_,create.c_str()) != 0)
//	if (mysql_create_db(&mysql_,database.c_str()) != 0)
	{
		isConnected_ = false;
		return false;
	}

	if (mysql_select_db(&mysql_,database.c_str()) != 0)
	{
		isConnected_ = false;
		return false;
	}

	if(terralibModel)
	{
		//create conceptual model
		if(!this->createConceptualModel())
			return false;
	}
	isConnected_ = true;
	user_ = user;
	host_ = host;
	password_ = password;
	database_ = database;
	return true;
}


bool 
TeMySQL::connect (const string& host, const string& user, const string& password, const string& database, int port)
{
	if (mysql_real_connect(&mysql_,host.c_str(),user.c_str(),password.c_str(),database.c_str(),port,NULL,0) == NULL)
	{
		isConnected_ = false;
		return false;
	}

	isConnected_ = true;
	user_ = user;
	host_ = host;
	password_ = password;
	database_ = database;
	return true;
}


bool 
TeMySQL::showDatabases (const string& host, const string& user, const string& password, vector<string>& dbNames, int port)
{
	unsigned long i;
	MYSQL_ROW row;

	if (mysql_real_connect(&mysql_,host.c_str(),user.c_str(),password.c_str(),0,port,NULL,0) == NULL)
	{
		errorMessage_= string(mysql_error(&mysql_));
		return false;
	}

	MYSQL_RES *result = mysql_list_dbs(&mysql_, "%");

	unsigned int numRows = (unsigned int)mysql_num_rows(result);

	for (i = 0; i < numRows; ++i)
	{
		row = mysql_fetch_row(result);
		dbNames.push_back(row[0]);
	}

	close();

	return true;
}


bool 
TeMySQL::execute (const string &q)
{
	if (mysql_real_query(&mysql_, q.c_str(), q.size()) != 0)
	{
		errorMessage_= string(mysql_error(&mysql_));
		return false;
	}

	return true;
}


bool 
TeMySQL::tableExist(const string& table)
{
	string q ="DESC "+ table;

	if (mysql_real_query(&mysql_, q.c_str(), q.size()) != 0)
	{
		errorMessage_ = string(mysql_error(&mysql_));
	}

	MYSQL_RES*	res_ = mysql_use_result(&mysql_);

	if (res_)
	{
		mysql_free_result(res_);
		return true;
	}
	else
		return false;
}

bool 
TeMySQL::columnExist(const string& table, const string&  column ,TeAttribute& attr)
{
	string q = "SHOW COLUMNS FROM " + table + " LIKE \'" + column + "\'";
	if (mysql_query(&mysql_, q.c_str()) != 0)
	{
		return false;
	}
	MYSQL_RES* fieldsres = mysql_use_result(&mysql_);
	if (!fieldsres)
		return false;
	MYSQL_ROW row = mysql_fetch_row(fieldsres);
	if (!row)
		return false;
	attr.rep_.name_ = column;
	
	string ftype(row[1]);
	string fpk(row[3]);
	if (ftype.find("char") != string::npos)
		attr.rep_.type_ = TeSTRING;
	else if (ftype.find("int") != string::npos)
		attr.rep_.type_ = TeINT;
	else if (ftype.find("double") != string::npos)
		attr.rep_.type_ = TeREAL;
	else if (ftype.find("blob") != string::npos)
		attr.rep_.type_ = TeBLOB;
	else if ((ftype.find("date") != string::npos) || (ftype.find("time") != string::npos))
		attr.rep_.type_ = TeDATETIME;

	unsigned int p1 = ftype.find("(");
	unsigned int p2 = ftype.find(")");
	if (p1 != string::npos && p2 != string::npos)
		attr.rep_.numChar_ = atoi(ftype.substr(p1+1,p2-1).c_str());

	if (fpk.find("PRI") != string::npos)
		attr.rep_.isPrimaryKey_ = true;

	mysql_free_result(fieldsres);
	return true;
}

bool TeMySQL::addColumn (const string& table, TeAttributeRep &rep)
{
	string q ="ALTER TABLE " + table +" ADD ";
	string type;
	char	size[8];

	string field = TeGetExtension(rep.name_.c_str());
	if(field.empty())
		field = rep.name_;

	switch (rep.type_)
	{
		case TeSTRING:
			type = " VARCHAR ";
			sprintf (size,"(%d)",rep.numChar_);
			type += string (size);
			break;
		case TeREAL:
			type = " DOUBLE";
			break;
		case TeINT:
			type = " INT";
			break;
		case TeCHARACTER:
			type = " CHAR";
			break;
		case TeDATETIME:
			type = " DATETIME";
			break;
		case TeBLOB:
			type = " LONGBLOB";
			break; 
		default:
		type = " VARCHAR ";
		sprintf (size,"(%d)",rep.numChar_);
		type += string (size);
		break;
	}
	q += field;
	q += type;
	return execute(q);
}

TeDatabasePortal*  
TeMySQL::getPortal ()
{
	TeMySQLPortal* portal = new TeMySQLPortal (this);
	return portal;
}

bool 
TeMySQL::createRasterTable (const string& table)
{
	if (table.empty())
		return false;

	string create = "CREATE TABLE "+table+" (";
	create += "block_id VARCHAR(50) not null, ";
	create += "lower_x double(24,15) not null, ";
	create += "lower_y double(24,15) not null, ";
	create += "upper_x double(24,15) not null, ";
	create += "upper_y double(24,15) not null, ";
	create += "band_id int(10) unsigned not null, ";
	create += "resolution_factor int(10) unsigned , ";
	create += "subband int(10) unsigned,";
	create += "spatial_data LONGBLOB not null NOT NULL, ";
	create += "block_size int(10) unsigned not null, ";
	create += "PRIMARY KEY (block_id),";
	create += "INDEX (band_id),";
	create += "INDEX (subband),";
	create += "INDEX (resolution_factor),";
	create += "INDEX (lower_x,lower_y,upper_x,upper_y))";
	return execute (create);
}


bool 
TeMySQL::createPolygonGeometry (const string&  table)
{
	if (table.empty())
		return false;

	string q ="CREATE TABLE " + table +"(";
	q += "geom_id INT UNSIGNED not null AUTO_INCREMENT,";
	q += "object_id VARCHAR(50) NOT NULL ,";
	q += "num_coords INT UNSIGNED not null ,";
	q += "num_holes INT UNSIGNED not null ,";
	q += "parent_id INT UNSIGNED not null ,";
	q += "lower_x double(24,15) not null ,";
	q += "lower_y double(24,15) not null ,";
	q += "upper_x double(24,15) not null ,";
	q += "upper_y double(24,15) not null ,";
	q += "ext_max double(24,15) not null ,";
	q += "spatial_data LONGBLOB not null ,";
	q += "PRIMARY KEY (geom_id),";
	q += "INDEX (object_id),";
	q += "INDEX (ext_max),";
	q += "INDEX (lower_x,lower_y,upper_x,upper_y))";
	bool status = execute(q);
	return status;
}


bool 
TeMySQL::createLineGeometry (const string& table)
{
	if (table.empty())
		return false;

	string q ="CREATE TABLE " + table +"(";
	q += "geom_id INT UNSIGNED not null AUTO_INCREMENT,";
	q += "object_id VARCHAR(50) not null ,";
	q += "num_coords INT UNSIGNED not null ,";
	q += "lower_x double(24,15) not null ,";
	q += "lower_y double(24,15) not null ,";
	q += "upper_x double(24,15) not null ,";
	q += "upper_y double(24,15) not null ,";
	q += "ext_max double(24,15) not null ,";
	q += "spatial_data LONGBLOB not null ,";
	q += "PRIMARY KEY (geom_id),";
	q += "INDEX (object_id),";
	q += "INDEX (lower_x,lower_y,upper_x,upper_y))";
	return execute(q);
}


bool
TeMySQL::defineIntegrity(void)
{
	return true;		// mysql doesn't have referencial integrity
						// there is nothing to be done
}


TeMySQLPortal::TeMySQLPortal ( TeMySQL  *m)
{
	mysql_ = m->getMySQL();
	numRows_ = 0;
	numFields_ = 0;
	result_ = 0;
	row_ = 0;
	db_ = m;
}


TeMySQLPortal::TeMySQLPortal ( const TeMySQLPortal& p) : TeDatabasePortal()
{
	mysql_ = p.mysql_;
	numRows_ = 0;
	numFields_ = 0;
	result_ = 0;
	row_ = 0;
}


TeMySQLPortal::~TeMySQLPortal ()
{
	freeResult ();
}


string 
TeMySQL::escapeSequence(const string& from)
{
	string ret = "";
	if (!from.empty())
	{
		char	*to = new char[from.size()*2+1];
		mysql_real_escape_string(&mysql_,to,from.c_str(),from.size());
		ret = to;
		delete  to;
	}

	return ret;
}

bool TeMySQLPortal::query (const string &q, TeCursorLocation /* l */, TeCursorType /* t */, TeCursorEditType /* e */, TeCursorDataType /*dt*/)
{
	if (result_)
		freeResult();
	attList_.clear ();

	if (mysql_real_query(mysql_, q.c_str(), q.size()) != 0)
	{
		string erro = string(mysql_error(mysql_));
		return false;
	}

	result_ = mysql_store_result(mysql_);
	numRows_ = (int)mysql_num_rows(result_);
	if (result_)  // there are rows
	{
		numFields_ = mysql_num_fields(result_);

		MYSQL_FIELD *fields = mysql_fetch_fields(result_);

		int i;
		for(i = 0; i < numFields_; i++)
		{
			TeAttribute attribute;
			switch (fields[i].type)
			{
			case FIELD_TYPE_DECIMAL:
			case FIELD_TYPE_TINY:
			case FIELD_TYPE_SHORT:
			case FIELD_TYPE_LONG:
			case FIELD_TYPE_INT24:
				attribute.rep_.type_ = TeINT;
				attribute.rep_.numChar_ = fields[i].length;
				break;

			case FIELD_TYPE_FLOAT:
			case FIELD_TYPE_DOUBLE:
			case FIELD_TYPE_LONGLONG:
				attribute.rep_.type_ = TeREAL;
				attribute.rep_.numChar_ = fields[i].length;
				attribute.rep_.decimals_ = fields[i].decimals;
				break;

			case FIELD_TYPE_DATE:
			case FIELD_TYPE_TIME:
			case FIELD_TYPE_DATETIME:
			case FIELD_TYPE_YEAR:
			case FIELD_TYPE_NEWDATE:
				attribute.rep_.type_ = TeDATETIME;
				attribute.dateTimeFormat_ = "YYYYsMMsDDsHHsmmsSS24";
				attribute.dateChronon_ = TeSECOND;
				break;

			case FIELD_TYPE_TINY_BLOB:
			case FIELD_TYPE_MEDIUM_BLOB:
			case FIELD_TYPE_LONG_BLOB:
			case FIELD_TYPE_BLOB:
				attribute.rep_.type_ = TeBLOB;
				break;

			case FIELD_TYPE_VAR_STRING:
			case FIELD_TYPE_STRING:
				attribute.rep_.type_ = TeSTRING;
				attribute.rep_.numChar_ = fields[i].length;
				break;

			case FIELD_TYPE_ENUM:
			case FIELD_TYPE_SET:
			default :
				attribute.rep_.type_ = TeUNKNOWN;
				break;
			}
			if (fields[i].flags & PRI_KEY_FLAG)
				attribute.rep_.isPrimaryKey_ = true;

			if (fields[i].flags & AUTO_INCREMENT_FLAG)
				attribute.rep_.isAutoNumber_ = true;

			attribute.rep_.name_ = fields[i].name;
			attList_.push_back ( attribute );
		}
    }
    else  // mysql_store_result() returned nothing; should it have?
    {
        if(mysql_field_count(mysql_) == 0)
        {
	        // query does not return data
            // (it was not a SELECT)
			numFields_ = 0;
            numRows_ = (int)mysql_affected_rows(mysql_);
        }
        else // mysql_store_result() should have returned data
        {
            return false;
        }
    }
	return true;
}


bool 
TeMySQLPortal::fetchRow ()
{
	if ((row_ = mysql_fetch_row(result_)))
		return true;
	else
		return false;
}


bool 
TeMySQLPortal::fetchRow (int i)
{
	if (i >= 0 && i < numRows_)
	{
		mysql_data_seek (result_, i);
		return fetchRow ();
	}
	else
		return false;
}


void TeMySQLPortal::freeResult ()
{
	if (result_)
	{
		mysql_free_result(result_);
		result_ = 0;
	}
}


int 
TeMySQLPortal::insertId ()
{
	return (int)mysql_insert_id(mysql_);
}


int 
TeMySQL::insertId ()
{
	return (int)mysql_insert_id(&mysql_);
}



bool 
TeMySQL::insertProjection (TeProjection *proj)
{
	char q[512];
	sprintf (q,"INSERT INTO te_projection VALUES('','%s',%.15f,%.15f,%.15f,%.15f,%.15f,%.15f,'%s',%.15f,%d,'%s',%.15f,%.15f,%.15f,%.15f,%.15f)",
	proj->name().c_str(),
		proj->params().lon0*TeCRD,
		proj->params().lat0*TeCRD,
		proj->params().offx,
		proj->params().offy,
		proj->params().stlat1*TeCRD,
		proj->params().stlat2*TeCRD,
		proj->params().units.c_str(),
		proj->params().scale,
		proj->params().hemisphere,
		proj->datum().name().c_str(),
		proj->datum().radius(),
		proj->datum().flattening(),
		proj->datum().xShift(),
		proj->datum().yShift(),
		proj->datum().zShift());

	if (!execute(q))
		return false;

	proj->id(insertId());
	return true;
}


bool 
TeMySQL::insertView (TeView *view)
{
	TeProjection* proj = view->projection();
	// save its projection
	if (!proj || !insertProjection(proj))
	{
		errorMessage_ = "No  possvel inserir vista sem projeo";
		return false;
	}

	char q[512];
	sprintf (q,"INSERT INTO te_view VALUES('',%d,'%s','%s',%d)",
			proj->id(),view->name().c_str(),view->user().c_str(),view->isVisible());

	if (!execute(q))
		return false;

	view->id(insertId());

	for (unsigned int th=0; th<view->size(); th++)
	{
		TeViewNode* node = view->get (th);
		if (node->type() == TeTHEME)
		{
			TeTheme *theme = (TeTheme*) node;
			insertTheme (theme);
		}
		else
		{
			TeViewTree* tree = (TeViewTree*)node;
			insertViewTree (tree);
		}
	}

	viewMap_[view->id()] = view;
	return true;
}


bool 
TeMySQL::insertViewTree (TeViewTree *tree)
{
	string save;
	save = "INSERT INTO te_theme (name, layer_id, node_type, view_id, priority) VALUES('','";
	save += tree->name() +  "," + Te2String(tree->parentId()) + "," + Te2String((int)tree->type());
	save += "," + Te2String(tree->view()) + "," + Te2String(tree->priority()) + ")";
	if (!execute(save))
		return false;
	tree->id(insertId());
	return true;
}


bool 
TeMySQL::insertTheme (TeTheme *tem)
{
	string save;
	save = "INSERT INTO te_theme ";
	save += "(layer_id, view_id, name, parent_id, priority, node_type, min_scale, max_scale, ";
	save += "generate_attribute_where, generate_spatial_where, generate_temporal_where, ";
	save += "collection_table, visible_rep, enable_visibility) ";
	save += " VALUES(";
	save += Te2String(tem->layerId()) +", ";
	save += Te2String(tem->view()) +", ";
	save += "'" + tem->name() + "', ";
	save += Te2String(tem->parentId()) +", ";
	save += Te2String(tem->priority()) +", ";
	save += Te2String(tem->type()) +", ";
	save += Te2String(tem->minScale()) +", ";
	save += Te2String(tem->maxScale()) +", ";
	save += "'" + escapeSequence(tem->attributeRest()) + "', ";
	save += "'" + escapeSequence(tem->spatialRest()) + "', ";
	save += "'" + escapeSequence(tem->temporalRest()) + "', ";
	save += "'" + escapeSequence(tem->collectionTable()) + "', ";
	save += Te2String(tem->visibleRep()) +", ";
	save += Te2String(tem->visibility()) +" ) ";

	if (!this->execute(save))
		return false;

	tem->id(this->insertId());

	if (tem->collectionTable().empty())
	{
		string colTab = "te_collection_" + Te2String(tem->id());
		tem->collectionTable(colTab);
		save = "UPDATE te_theme SET collection_table='" + colTab + "' WHERE theme_id=" ;
		save += Te2String(tem->id());
		this->execute(save);
	}

	bool status;

	// insert grouping
	int numSlices = 0;
	if(tem->grouping())
	{
		if(!insertGrouping (tem->id(), tem->grouping()))
			return false;
		numSlices = tem->grouping()->groupNumSlices_;
	}
		
	tem->outOfCollectionLegend().group(-1); 
	tem->outOfCollectionLegend().theme(tem->id()); 
	status = insertLegend (&(tem->outOfCollectionLegend())); 
	if (!status)
		return status;

	tem->withoutDataConnectionLegend().group(-2); 
	tem->withoutDataConnectionLegend().theme(tem->id()); 
	status = insertLegend (&(tem->withoutDataConnectionLegend())); 
	if (!status)
		return status;

	tem->defaultLegend().group(-3); 
	tem->defaultLegend().theme(tem->id()); 
	status = insertLegend (&(tem->defaultLegend())); 
	if (!status)
		return status;

	tem->pointingLegend().group(-4); 
	tem->pointingLegend().theme(tem->id()); 
	status = insertLegend (&(tem->pointingLegend())); 
	if (!status)
		return status;

	tem->queryLegend().group(-5); 
	tem->queryLegend().theme(tem->id()); 
	status = insertLegend (&(tem->queryLegend())); 
	if (!status)
		return status;

	tem->queryAndPointingLegend().group(-6); 
	tem->queryAndPointingLegend().theme(tem->id()); 
	status = insertLegend (&(tem->queryAndPointingLegend())); 
	if (!status)
		return status;

	for (int i = 0; i < numSlices; i++)
	{
		tem->legend()[i].group(i);
		tem->legend()[i].theme(tem->id());
		status = insertLegend (&(tem->legend()[i]));
		if (!status)
			return status;
	}
	if (!status)
		return status;

	themeMap_[tem->id()] = tem;
	return updateThemeTable(tem);
}

bool 
TeMySQL::insertThemeGroup (TeViewTree* tree)
{
	string save;
	save = "INSERT INTO te_theme ";
	save += "(name,parent_id,node_type,view_id,priority) ";
	save += " VALUES ('" + tree->name() + "',1,1, ";
	save += Te2String(tree->view()) +", ";
	save += Te2String(tree->priority()) +") ";
	if (!this->execute(save))
		return false;
	tree->id(this->insertId());
	return true;
}


bool 
TeMySQL::generateLabelPositions (TeTheme *theme)
{
	string	geomTable, query;
	string	collTable = theme->collectionTable();

	if((collTable.empty()) || (!tableExist(collTable)))
		return false;

	if (theme->layer()->hasGeometry(TeLINES))
	{	
		geomTable = theme->layer()->tableName(TeLINES);
				
		string extMax = "CREATE TEMPORARY TABLE maxTable";
		extMax += " SELECT object_id, lower_x, lower_y, upper_x, upper_y";
		extMax += " FROM " + geomTable + " ORDER BY object_id, ext_max DESC";
		
		if(!execute(extMax))
			return false;

		query = "UPDATE " + collTable + ",maxTable SET label_x = (lower_x + upper_x)/2,";
		query += "label_y = (lower_y + upper_y)/2 WHERE c_object_id = object_id";

		if (execute(query) == false)
		{
			execute("DROP TABLE maxTable");
			return false;
		}

		execute("DROP TABLE maxTable");
	}

	if (theme->layer()->hasGeometry(TePOINTS))
	{
		geomTable = theme->layer()->tableName(TePOINTS);

		query = "UPDATE " + collTable + "," + geomTable + " SET label_x = x,";
		query += "label_y = y WHERE c_object_id = object_id";

		if(execute(query) == false)
			return false;
	}
	
	if (theme->layer()->hasGeometry(TePOLYGONS))
	{	
		geomTable = theme->layer()->tableName(TePOLYGONS);
				
		string extMax = "CREATE TEMPORARY TABLE maxTable";
		extMax += " SELECT object_id, lower_x, lower_y, upper_x, upper_y";
		extMax += " FROM " + geomTable + " ORDER BY object_id, ext_max DESC";
		
		if(!execute(extMax))
			return false;

		query = "UPDATE " + collTable + ",maxTable SET label_x = (lower_x + upper_x)/2,";
		query += "label_y = (lower_y + upper_y)/2 WHERE c_object_id = object_id";

		if (execute(query) == false)
		{
			execute("DROP TABLE maxTable");
			return false;
		}
		
		execute("DROP TABLE maxTable");
	}
	
	if(theme->layer()->hasGeometry(TeCELLS))
	{
		geomTable = theme->layer()->tableName(TeCELLS);

		query = "UPDATE " + collTable + "," + geomTable + " SET label_x = (lower_x + upper_x)/2,";
		query += "label_y = (lower_y + upper_y)/2 WHERE c_object_id = object_id";

		if (execute(query) == false)
			return false;	
	}




	
	return true;
}


bool
TeMySQL::insertLayer (TeLayer* layer)
{
	TeProjection* proj = layer->projection();
	if (!proj || !insertProjection(proj))
	{
		errorMessage_ = "Layer sem projeo ou falha na insero da projeo";
		return false;
	}
	int pid = proj->id();
	string save = "INSERT INTO te_layer (layer_id,projection_id,name,lower_x,lower_y,upper_x,upper_y) VALUES('',";
	save += Te2String(pid);
	save += ",'" + layer->name() + "', ";
	save += Te2String(layer->box().x1(),15)+ ", ";
	save += Te2String(layer->box().y1(),15)+ ", ";
	save += Te2String(layer->box().x2(),15)+ ", ";
	save += Te2String(layer->box().y2(),15) + ")";

	// if layer insertion fails remove projection
	if (!this->execute (save))
	{
		string sql = "DELETE FROM te_projection WHERE  projection_id = " + Te2String(pid);
		this->execute(sql);	
		return false;
	}

	layer->id(this->insertId());
	layerMap_[layer->id()] = layer;

	return true;
}


bool TeMySQL::insertLegend (TeLegendEntry *legend)
{
	char q[1024];
	string save;
	sprintf (q,"INSERT INTO te_legend VALUES('',%d,%d,%d,'%s','%s','%s')",
	legend->theme(),
	legend->group(),
	legend->count(),
	legend->from().c_str(),
	legend->to().c_str(),
	legend->label().c_str());
	save = q;

	if (!this->execute(save))
		return false;

	int legId = insertId();
	legend->id(legId);
	legendMap_[legend->id()] = legend;
	return insertVisual(legend);
}


bool
TeMySQL::insertRepresentation (int layerId, TeRepresentation& rep)
{
	if (layerId <= 0)
		return false;
	
	string sql;
	sql = "INSERT INTO te_representation (layer_id, geom_type, geom_table, description,";
	sql += "lower_x, lower_y, upper_x, upper_y, res_x, res_y, num_cols, num_rows) VALUES( ";
	sql += Te2String(layerId) + ", ";
	sql += Te2String(static_cast<int>(rep.geomRep_)) + ", ";
	sql += "'" + rep.tableName_ + "', ";
	sql += "'" + rep.description_ + "',";
	sql += Te2String(rep.box_.x1(),15) + ", ";
	sql += Te2String(rep.box_.y1(),15) + ", ";
	sql += Te2String(rep.box_.x2(),15) + ", ";
	sql += Te2String(rep.box_.y2(),15) + ", ";
	sql += Te2String(rep.resX_,15) + ", ";
	sql += Te2String(rep.resY_,15) + ", ";
	sql += Te2String(rep.nCols_,15) + ", ";
	sql += Te2String(rep.nLins_,15) + ") ";
	if (execute(sql))
	{
		rep.id_ = (this->insertId());
		return true;
	}
	return false;
}

bool 
TeMySQL::insertRelationInfo(const int tableId, const string& tField,
						    const string& eTable, const string& eField, int& relId)
{
	// check if relation already exists
	TeMySQLPortal* portal = (TeMySQLPortal*) this->getPortal();
	if (!portal)
		return false;

	relId = -1;
	string sel = "SELECT relation_id FROM te_tables_relation WHERE";
	sel += " related_table_id = " + Te2String(tableId);
	sel += " AND related_attr = '" + tField + "'";
	sel += " AND external_table_name = '" + eTable + "'";
	sel += " AND external_attr = '" + eField + "'";
	if (!portal->query(sel))
	{
		delete portal;
		return false;
	}

	if (portal->fetchRow())
	{
		relId = atoi(portal->getData(0));
		delete portal;
		return true;
	}
	delete portal;

	string sql = "INSERT INTO te_tables_relation(related_table_id, related_attr, external_table_name, external_attr) ";
	sql += " VALUES( ";
	sql += Te2String(tableId);
	sql += ",'" + tField + "'";
	sql += ",'" + eTable + "'";
	sql += ",'" + eField + "')";
	if (!this->execute(sql))
		return false;
	relId = this->insertId(); 
	return true;
}

bool 
TeMySQL::insertTableInfo (int layerId, TeTable &table, const string& user)
{
	string sql;

	sql = "INSERT INTO te_layer_table (unique_id, attr_link,attr_initial_time, attr_final_time, attr_time_unit,attr_table_type, ";
	sql += " user_name, attr_table ";
	if (layerId > 0)
		sql += ",layer_id";
	sql += ") VALUES( ";
	sql += "'" + table.uniqueName() + "', ";
	sql += "'" + table.linkName() + "', ";
	sql += "'" + table.attInitialTime() + "', ";
	sql += "'" + table.attFinalTime() + "', ";
	sql += "" + Te2String(table.attTimeUnit()) + ", ";
	sql += ""+ Te2String(table.tableType()) + ", ";
	sql += "'" + user + "','" + table.name() +"'";
	if (layerId > 0)
		sql += ", " + Te2String(layerId) ;
	sql += ")";

	if (!this->execute(sql))
		return false;

	int id = (this->insertId());
	table.setId(id);
	return true;
}

bool 
TeMySQL::insertThemeTable (int themeId, int tableId, int relationId, int tableOrder)
{
	string relId;
	if (relationId > 0)
		relId = Te2String(relationId);
	else
		relId = "NULL";

	string sql = "INSERT INTO te_theme_table VALUES ('',";
	sql += Te2String(themeId) + "," + Te2String(tableId) + ",";
	sql += relId + "," + Te2String(tableOrder) + ")";


	if (!execute(sql))
		return false;

	return true;
}


bool 
TeMySQL::insertPolygon (const string& table, TePolygon &poly )
{
	double	ext_max;

	int	i,
		ni,
		size;
	unsigned int k;
	double	*points	= NULL;
	char	buf[256], *query, *end;


	int parentId = 0;
	for ( k = 0; k < poly.size(); k++ )
	{
		TeLinearRing ring ( poly[k] );
		TeBox b = ring.box();
		size = ring.size();
		ni = 0;
		ext_max = MAX(b.width(),b.height());
		if (k==0)
			ni = poly.size()-1;
		else
			ni = 0;

		// Create and save the BLOB
		points = new double[2*sizeof(double)*size];

		int iac = 0;
		for (i=0;i<size;i++)
		{
			points[iac++]=swappd(ring[i].x());
			points[iac++]=swappd(ring[i].y());
		}

		sprintf (buf,"INSERT INTO %s values('','%s',%d,%d,%d,%.15f,%.15f,%.15f,%.15f,%.15f,",
			table.c_str(),escapeSequence(poly.objectId()).c_str(),size,ni,parentId,b.lowerLeft().x(),b.lowerLeft().y(),b.upperRight().x(),b.upperRight().y(),ext_max);

		query = new char[size*4*sizeof(double)+strlen(buf)+4];
		end = strxmov(query,buf,(char*)0);
		*end++ = '\'';
		end += mysql_real_escape_string(&mysql_,end,(char*)points,size*2*sizeof(double));
		*end++ = '\'';
		*end++ = ')';
		delete points;

		if (mysql_real_query(&mysql_,query,(unsigned int) (end - query)) != 0)
		{
			errorMessage_ = string(mysql_error(&mysql_));
			delete query;
			return false;
		}
		delete query;
		query = 0;
		ring.geomId (insertId());
		if (k==0)
			parentId = ring.geomId();
	}
	sprintf(buf,"UPDATE %s SET parent_id=%d WHERE geom_id=%d",table.c_str(),parentId,parentId);
	if (mysql_real_query(&mysql_,buf,(unsigned int)strlen(buf)) != 0)
	{
		errorMessage_ = string(mysql_error(&mysql_));
		delete query;
		return false;
	}	
	return true;
}

bool 
TeMySQL::updatePolygon (const string& table, TePolygon &poly )
{
	double	ext_max;
	int ni;
	unsigned int i,k,size;
	double	*points	= NULL;
	char	buf[256], message[256], *query, *end;
	TeBox b = poly.box();

	for ( k = 0; k < poly.size(); k++ )
	{
		TeLinearRing ring ( poly[k] );
		size = ring.size();
		ni = 0;
		if (k==0)
		{
			ext_max = MAX(b.width(),b.height());
			ni = poly.size()-1;
		}
		else
			ni = 0;

// Create and save the BLOB
		points = new double[2*sizeof(double)*size];
		int iac = 0;
		for (i=0;i<size;i++)
		{
			points[iac++]=swappd(ring[i].x());
			points[iac++]=swappd(ring[i].y());
		}

		sprintf (buf,"UPDATE %s SET num_coords=%d, num_holes=%d, parent_id=%d, lower_x=%g, lower_y=%g, upper_x=%g, upper_y=%g, ext_max=%g, spatial_data=",
			table.c_str(),size,ni,k,b.lowerLeft().x(),b.lowerLeft().y(),b.upperRight().x(),b.upperRight().y(),ext_max);
		query = new char[size*4*sizeof(double)+strlen(buf)+50];
		end = strxmov(query,buf,(char*)0);
		*end++ = '\'';
		end += mysql_real_escape_string(&mysql_,end,(char*)points,size*2*sizeof(double));
		*end++ = '\'';
		sprintf (message," WHERE geom_id=%d",ring.geomId());
		end = strxmov(end,message,0);
		delete points;

		if (mysql_real_query(&mysql_,query,(unsigned int) (end - query)) != 0)
		{
			errorMessage_ = string(mysql_error(&mysql_));
			delete query;
			return false;
		}
		delete query;
		query = 0;
	}
	return true;
}


bool 
TeMySQLPortal::fetchGeometry (TePolygon& poly)
{
	int ni,j;
	TeLinearRing ring = this->getLinearRing(ni);
	poly.objectId ( ring.objectId() );
	poly.geomId ( ring.geomId() );
	poly.add ( ring );
	int parentId = poly.geomId();
	while (fetchRow())
	{
		if (atoi(this->getData("parent_id")) == parentId)
		{
			TeLinearRing ring = getLinearRing(j);
			poly.add ( ring );
		}
		else
			return true;
	}
	return false;
}


TeLinearRing 
TeMySQLPortal::getLinearRing (int &ni)
{
	int index = atoi(getData("geom_id"));
	string object_id = getData("object_id");
	int np = atoi (getData("num_coords"));
	ni = atoi (getData("num_holes"));
	TeBox b (getDouble("lower_x"),getDouble("lower_y"),getDouble("upper_x"),getDouble("upper_y"));
	int k;
	TeLine2D line;
	double x,y,*data = (double*)getData("spatial_data");

	for (k=0;k<np;k++)
	{
		x = swappd(*data++);
		y = swappd(*data++);
		TeCoord2D pt(x,y);
		line.add(pt);
	}

	TeLinearRing ring ( line );
	ring.objectId (object_id);
	ring.geomId (index);
	ring.setBox (b);
	return ring;
}


bool 
TeMySQL::insertLine (const string& table, TeLine2D &l)
{
	char	buf[256],
			*query,
			*end;
	double	*points	= NULL,
			ext_max;
	int		size=l.size(),
			i,
			iac;
	TeBox	b=l.box();

	ext_max = MAX(b.width(),b.height());
	points = new double[2*sizeof(double)*size];
	iac = 0;

	for (i=0;i<size;i++)
	{
		points[iac++]=l[i].x();
		points[iac++]=l[i].y();
	}

	sprintf (buf,"INSERT INTO %s values('','%s',%d,%.15f,%.15f,%.15f,%.15f,%.15f,",
	table.c_str(),
	escapeSequence(l.objectId()).c_str(),
	size,
	b.lowerLeft().x(),b.lowerLeft().y(),b.upperRight().x(),b.upperRight().y(),ext_max);

	query = new char[size*4*sizeof(double)+strlen(buf)+4];

	end = strxmov(query,buf,0);
	*end++ = '\'';
	end += mysql_real_escape_string(&mysql_,end,(char*)points,size*2*sizeof(double));
	*end++ = '\'';
	*end++ = ')';
	delete points;

	if (mysql_real_query(&mysql_,query,(unsigned int) (end - query)) != 0)
	{
		errorMessage_ = string(mysql_error(&mysql_));
		delete query;
		return false;
	}

	l.geomId (insertId());
	delete query;
	return true;
}


bool 
TeMySQL::updateLine (const string& table, TeLine2D &l)
{
	char	buf[256],
			message[256],
			*query,
			*end;
	double	*points	= NULL,
			ext_max;
	int		size=l.size(),
			i,
			iac;
	TeBox	b=l.box();

	ext_max = MAX(b.width(),b.height());
	points = new double[2*sizeof(double)*size];
	iac = 0;

	for (i=0;i<size;i++)
	{
		points[iac++]=l[i].x();
		points[iac++]=l[i].y();
	}

	sprintf (buf,"UPDATE %s SET num_coords=%d, lower_x=%g, lower_y=%g, upper_x=%g, upper_y=%g, ext_max=%g, spatial_data=",
	table.c_str(),size,b.lowerLeft().x(),b.lowerLeft().y(),b.upperRight().x(),b.upperRight().y(),ext_max);
	query = new char[size*4*sizeof(double)+strlen(buf)+50];
	end = strxmov(query,buf,(char*)0);
	*end++ = '\'';
	end += mysql_real_escape_string(&mysql_,end,(char*)points,size*2*sizeof(double));
	*end++ = '\'';
	delete points;

	sprintf (message," WHERE geom_id=%d",l.geomId());
	end = strxmov(end,message,(char*)0);

	if (mysql_real_query(&mysql_,query,(unsigned int) (end - query)) != 0)
	{
		errorMessage_ = string(mysql_error(&mysql_));
		delete query;
		return false;
	}
	delete query;
	return true;
}



bool
TeMySQLPortal::fetchGeometry(TeLine2D& line)
{
	int index = atoi (getData ("geom_id"));
	string object_id = getData ("object_id");
	int np = atoi (getData("num_coords"));
	TeBox b (atof(getData("lower_x")),atof(getData("lower_y")),atof(getData("upper_x")),atof(getData("upper_y")));
	int k;
	double x,y,*data = (double*)getData("spatial_data");

	for (k=0;k<np;k++)
	{
		x = *data++;
		y = *data++;
		TeCoord2D pt(x,y);
		line.add(pt);
	}
	line.objectId (object_id);
	line.geomId (index);
	line.setBox (b);
	return (fetchRow());
}


bool 
TeMySQL::insertPoint(const string& table,TePoint &p)
{
	string q = "INSERT INTO " + table + " VALUES('',";
	q += "'" + escapeSequence(p.objectId()) + "',";
	q += Te2String(p.location().x(),15) + ",";
	q += Te2String(p.location().y(),15) + ")";
	
	if (!this->execute(string(q)))
		return false;
	p.geomId(this->insertId());
	return true;
}


bool 
TeMySQLPortal::fetchGeometry(TePoint& p)
{
	TeCoord2D c(atof (getData("x")), atof (getData("y")));
	p.geomId( atoi(getData("geom_id")));
	p.objectId( string(getData("object_id")));
	p.add(c);
	return (fetchRow());
}


bool 
TeMySQL::insertText(const string& table, TeText &t)
{
	char q[256];
	sprintf (q,"INSERT INTO %s values('','%s',%g,%g,'%s',%g, %g, %g, %g)",
		table.c_str(), escapeSequence(t.objectId()).c_str(),
		t.location().x(), t.location().y(),
		t.textValue().c_str(), t.angle(), t.height(),t.alignmentVert(),t.alignmentHoriz());
	if (!this->execute(string(q)))
		return false;
	t.geomId(this->insertId());

	return true;
}


bool 
TeMySQL::insertArc(const string& table, TeArc &arc)
{
	char q[256];
	sprintf (q,"INSERT INTO %s values('','%s',%d,%d)",
		table.c_str(),escapeSequence(arc.objectId()).c_str(),arc.fromNode().geomId(),arc.toNode().geomId());
	if (!this->execute(string(q)))
		return false;
	arc.geomId(this->insertId());
	return true;
}


bool
TeMySQL::insertNode(const string& table, TeNode &node)
{
	char q[256];
	sprintf (q,"INSERT INTO %s values('','%s',%g,%g)",table.c_str(),node.objectId().c_str(),node.location().x(),node.location().y());
	if (!this->execute(string(q)))
		return false;
	node.geomId(this->insertId());
	return true;
}

bool
TeMySQL::insertBlob (const string& tableName, const string& columnBlob, TeAttributeRep& columnId, const string& valueId, const string& fileName)
{
	unsigned char	*cdata = 0;
	int		size;
	FILE	*fp = 0;
	
	struct	stat buf;
	int		result;
	
	result = stat(fileName.c_str(), &buf);
	
	if( result != 0 )
		return false;
	
	size = buf.st_size;

	cdata = new unsigned char[size];
	fp = fopen(fileName.c_str(), "rb");
	fread(cdata, sizeof(unsigned char), size, fp); 

	bool status = insertBlob (tableName, columnBlob, columnId,  valueId, cdata, size);

	if(fp)
		fclose(fp);

	return status;
}

bool 
TeMySQL::insertBlob (const string& tableName, const string& /* columnBlob */, TeAttributeRep& columnId, const string& valueId, unsigned char* data, int size)
{
	TeMySQLPortal* portal = (TeMySQLPortal*)this->getPortal();
	if (!portal)
		return false;

	string q = "SELECT * FROM "+ tableName +" WHERE "+ columnId.name_ +" = ";
	
	switch (columnId.type_ )
	{
		case TeSTRING:
			q += "'"+ valueId + "'";
			break;
		default:
			q += valueId;
			break;
	}

	if ((!portal->query(q)) || (!portal->fetchRow()))
	{
		delete portal;
		return false;
	}
	delete portal;

	char	message[256],
			*query,
			*end;
	sprintf (message,"UPDATE %s SET media_blob= ",tableName.c_str());

	query = new char[2*size+strlen(message)+4];
	end = strxmov(query,message,0);
	*end++ = '\'';
	end += mysql_real_escape_string(&mysql_,end,(char*)data,size);
	*end++ = '\'';

	if (columnId.type_ == TeSTRING)
		sprintf (message,"WHERE %s='%s'",columnId.name_.c_str(),valueId.c_str());
	else
		sprintf (message,"WHERE %s=%s",columnId.name_.c_str(),valueId.c_str());

	end = strxmov(end,message,0);

	if (mysql_real_query(&mysql_,query,(unsigned int) (end - query)))
	{
		errorMessage_ = string(mysql_error(&mysql_));
		delete query;
		return false;
	}
	delete query;	
	return true;
}
	
bool
TeMySQL::insertRasterBlock(const string& table, const string& blockId, const TeCoord2D& ll, const TeCoord2D& ur, 
						   unsigned char *buf,unsigned long size, int band, unsigned int res, unsigned int subband)
{
	if (blockId.empty()) // no block identifier provided
	{
		errorMessage_ = "bloco sem identificador";
		return false;
	}

	TeMySQLPortal* portal = (TeMySQLPortal*) this->getPortal();
	if (!portal)
		return false;

	bool update = false;
	string q ="SELECT * FROM " + table; 
	q += " WHERE block_id='" + blockId + "'";

	if (!portal->query (q))
	{
		delete portal;
		return false;
	}
		// check if this block is alread in the database
	if (portal->fetchRow())
		update = true;
	delete portal;

	char	message[256],
			*end;
	if (!update)
	{
		sprintf (message,"INSERT INTO %s values('%s',%f,%f,%f,%f,%d,%d,%d,",
			table.c_str(),blockId.c_str(),ll.x(),ll.y(),ur.x(),ur.y(),band,res,subband);
	}
	else
	{
		sprintf (message,"UPDATE %s SET spatial_data=",table.c_str());
	}

	if (bufferBlobSize_ < (long)(2*size+strlen(message)+20))
	{
		if (bufferBlob_)
			delete bufferBlob_;
		bufferBlobSize_ = 2*size+strlen(message)+20;
		bufferBlob_ = new char[bufferBlobSize_];
	}
	end = strxmov(bufferBlob_,message,0);
	*end++ = '\'';
	end += mysql_real_escape_string(&mysql_,end,(char*)buf,size);
	*end++ = '\'';

	if (update)
	{
		sprintf (message,", block_size=%ld WHERE block_id='%s'",size,blockId.c_str());
		end = strxmov(end,message,0);
	}
	else
	{
		sprintf (message,",%ld)",size);
		end = strxmov(end,message,0);
	}

	if (mysql_real_query(&mysql_,bufferBlob_,(unsigned int) (end - bufferBlob_)))
	{
		errorMessage_ = string(mysql_error(&mysql_));
		delete bufferBlob_;
		bufferBlobSize_ = 0;
		bufferBlob_ = 0;
		return false;
	}
	return true;
}

bool
TeMySQLPortal::fetchGeometry(TeNode& n)
{
	TeCoord2D c(atof (getData("x")), atof (getData("y")));
	n.geomId( atoi(getData("geom_id")));
	n.objectId( string(getData("object_id")));
	n.add(c);
	return (fetchRow());
}


bool
TeMySQL::insertCell(const string& table, TeCell &c)
{
	TeBox	b=c.box();
	char	q[256];
	sprintf (q,"INSERT INTO %s values('','%s',%.15f,%.15f,%.15f,%.15f,%d,%d)",
		table.c_str(),c.objectId().c_str(),
		b.lowerLeft().x(),b.lowerLeft().y(),b.upperRight().x(),b.upperRight().y(),
		c.column(),c.line());

	if (!this->execute (string(q)))
		return false;
	c.geomId(this->insertId());
	return true;
}


TeTime
TeMySQLPortal::getDate (int i)
{ 
	string s = getData(i);
	TeTime t(s, TeSECOND, "YYYYsMMsDDsHHsmmsSS","-"); 
	return t; 
}

TeTime
TeMySQLPortal::getDate (const string& s)
{ 
	string sv = getData(s);
	TeTime t(sv, TeSECOND, "YYYYsMMsDDsHHsmmsSS","-"); 
	return t; 
}

string 
TeMySQLPortal::getDateAsString(const string& s)
{
	TeTime t = this->getDate(s);
	if (t.isValid())
	{		
		string tval = "\'"+t.getDateTime("YYYYsMMsDDsHHsmmsSS", "-")+"\'";
		return tval;
	}
	else
		return "";
}

string 

TeMySQLPortal::getDateAsString(int i)
{
	TeTime t = this->getDate(i);
	if (t.isValid())
	{		
		string tval = "\'"+t.getDateTime("YYYYsMMsDDsHHsmmsSS", "-")+"\'";
		return tval;
	}
	else
		return "";
}

double 
TeMySQLPortal::getDouble (const string& s)
{
	return atof(getData(s));
}


double 
TeMySQLPortal::getDouble (int i)
{
	if (row_ == 0)
		return 0.;

	if (row_[i])
		return atof(row_[i]);
	else
		return 0.;
}

int 
TeMySQLPortal::getInt (const string& s)
{
	return atoi(getData(s));
}


int 
TeMySQLPortal::getInt (int i)
{
	if (row_ == 0)
		return 0;

	if (row_[i])
		return atoi(row_[i]);
	else
		return 0;
}


char* 
TeMySQLPortal::getData (int i)
{
	if (row_ == 0)
		return "";

	if (row_[i])
		return row_[i];
	else
		return "";
}


char* 
TeMySQLPortal::getData (const string& s)
{
	string fieldName;
	size_t pos = s.find(".", string::npos,1);
	if (pos != string::npos)
		fieldName = s.substr(pos+1);
	else
		fieldName = s;

	if (row_ == 0)
		return "";

	TeAttributeList::iterator it = attList_.begin();

	int j = 0;
	while ( it != attList_.end() )
	{
		if ((TeConvertToUpperCase(fieldName) == TeConvertToUpperCase((*it).rep_.name_)) ||
		    (TeConvertToUpperCase(s) == TeConvertToUpperCase((*it).rep_.name_)))
		{
			if (row_[j])
				return row_[j];
			else
				return "";
		}
		++it;
		j++;
	}
	return "";
}

bool
TeMySQLPortal::getBlob (const string& s, unsigned char* &data, long& size)
{
	string fieldName;
	size_t pos = s.find(".", string::npos,1);
	if (pos != string::npos)
		fieldName = s.substr(pos+1);
	else
		fieldName = s;

	TeAttributeList::iterator it = attList_.begin();
	int j = 0, i=-1;
	while ( it != attList_.end() )
	{
		if (fieldName == (*it).rep_.name_)
		{
			i=j;
			break;
		}
		++it;
		j++;
	}
	if (i<0)
		return false;

	unsigned long *lengths;
	lengths = mysql_fetch_lengths(result_);
	size = lengths[i];
	if (row_[i])
	{
//		data = (unsigned char*) row_[i];
		data = new unsigned char[size];
		memcpy(data, (unsigned char*) row_[i], size * sizeof(char));
		return true;
	}
	else
	{
		data = 0;
		return false;
	}
}


bool 
TeMySQLPortal::getBool (const string& s)
{
	int val = atoi(getData(s));
	return val ? true : false;
}


bool 
TeMySQLPortal::getBool (int i)
{
	if (row_ == 0)
		return false;
	int val = atoi(getData(i));
	return val ? true : false;
}

bool 
TeMySQLPortal::getRasterBlock(unsigned long& size, unsigned char* ptData)
{
	
	// get the actual length of the compressed data
	size = atoi(getData("block_size"));
	if ( (size>0) && row_[8])
	{
		memcpy((void*)ptData,(void*)(row_[8]),size);
		return true;
	}
	else
	{
		ptData = 0;
		return false;
	}
}

bool
TeMySQL::inClauseValues(const string& query, const string& attribute, vector<string>& inClauseVector)
{
	int i = 0, numRows = 0, count = 0, chunk = 200;
	string id, inClause = "(", plic = "'";

	TeDatabasePortal *pt = getPortal();
	if(pt->query(query) == false)
	{
		delete pt;
		return false;
	}

	while (numRows < pt->numRows())
	{
		if(count < chunk && pt->fetchRow())
		{
			id = pt->getData(attribute);
			inClause = inClause + plic + id + "',";
			++count;
			++numRows;
		}
		else
		{
			inClause.erase(inClause.size()-1,1);
			inClause += ")";
			inClauseVector.push_back(inClause);
			count = 0;
			++i;
			inClause = "(";
		}
	}
	if (numRows == 0)
	{
		delete pt;
		return true;
	}
	else
	{
		inClause.erase(inClause.size()-1,1);
		inClause += ")";
		inClauseVector.push_back(inClause);
	}

	delete pt;
	return true;
}

string 
TeMySQL::getSQLTime(TeTime& time)
{
	string result = "'"+ time.getDateTime("YYYYsMMsDDsHHsmmsSS", "-", ":") +"'";
	return result;
}

string
TeMySQL::getSQLTemporalWhere(int time1, int time2, TeChronon chr, TeTemporalRelation rel, 
						   const string& initialTime, const string& finalTime)
{
	
	//In MySQL the first day of the week is monday with index 0
	if(chr==TeDAYOFWEEK)
	{
		time1 = time1-1;
		if(time1<0)
			time1=6;
		time2 = time2-1;
		if(time2<0)
			time2=6;
	}

	return(TeDatabase::getSQLTemporalWhere(time1, time2, chr, rel, initialTime, finalTime));
}
		

string TeMySQL::concatValues(vector<string>& values, const string& unionString)
{
	string concat = "CONCAT(";
	
	for(unsigned int i = 0; i < values.size(); ++i)
	{
		if(i != 0)
		{
			concat += ", ";

			if(!unionString.empty())
			{
				concat += "'";
				concat += unionString;
				concat += "'";
				concat += ", ";
			}
		}

		concat += values[i];
	}

	concat += ")";

	return concat;
}

string TeMySQL::toUpper(const string& value)
{
	string result  = "upper(";
	       result += value;
		   result += ")";

	return result;
}
