#ifndef TEMRSID_H
#define TEMRSID_H

/*
#ifdef TEMRSIDDLL_EXPORTS
#define DLL_IMP __declspec(dllexport)
#else
#define DLL_IMP __declspec(dllimport)
#endif
*/

namespace LizardTech
{
class MrSIDNavigator;
class MrSIDImageFile;
class ImageBuffer;
}

#pragma warning(disable: 4275)

//class DLL_IMP TeMrSIDReader
class TeMrSIDReader
{
	LizardTech::MrSIDNavigator* sidNav_;
	LizardTech::MrSIDImageFile* sidImageFile_;

	unsigned char* imgBuffer_; //!< image buffer

	const char* mrsidFileName_; 

public:
	//! Color Models 
	typedef enum { 
		ColorModelUnknown,  //!< unknown color model
		ColorModelGray,     //!< gray scale, composed by one 8 bit channel 
		ColorModelMap,      //!< indexed , composed by one 8 bit channel and a palette
		ColorModelRGB,      //!< composed by three 8 bit channels
		ColorModelRGBA      //!< composed by four 8 bit channels
	} ColorModel;

	static bool mrsid_initialized; //!< indicates if MrSID has already been initialized

	//! Returns the color space
	TeMrSIDReader::ColorModel getColorModel() const;	
		
	//! Constructor
	TeMrSIDReader(const char* fname);

	//! Destructor
	~TeMrSIDReader();

	//! Releases internal buffer image
	void clear();

	//! Returns the name of the data file
	const char* fileName()
	{	return mrsidFileName_; }

	// { --- These methods are relative to the level 0 (or original) image
	//@  
	int nBands() const;  
	unsigned int bitsPerPixel() const; 
	unsigned int getWidth() const;
	unsigned int getHeight() const;
	void getDimensions(unsigned int& width, unsigned int& height) const;

	bool hasWorldInfo();

	double originX();
	double originY();
	void getOrigin(double& x, double& y);

	double resX();
	double resY();
	void getResolution(double& rx, double& ry);

	void getWorld(double& x0, double& y0, double& x1, double& y1);
	void getBoundingBox(double& xmin, double& ymin, double& xmax, double& ymax);
	void world2Pixel(double wx, double wy, int& ix, int& iy);
	void pixel2World(int ix, int iy, double& wx, double& wy);

	// Returns the number of levels of resolution available on the image
	int nlev();

	// Returns the maxmum resolution level  available on the image
	int getMaxZoomLevel();

	//@}
	//------- }

	// { --- Methods to get ImageBuffer's
	//@{
	/*! Returns an ImageBuffer for loading an image from MrSIDNavigator.  
	   \param size(in) size of the image buffer
	   \param nbands(in) number of output bands
	   \param bands(in) output bands vector indicating the desired output bands and its order
	   \param data(out) image buffer  
	   \note data must have been previously allocated and its content is band interleaved
	*/
	LizardTech::ImageBuffer* getImageBuffer (unsigned int size, 
		                                     unsigned int nbands,
	                                         const unsigned int* bands, 
											 void* data);

	/*! Returns an ImageBuffer for loading an image from MrSIDNavigator.
	   \param size(in) size of the image buffer
	   \param nbands(in) number of output bands
	   \param bands(in) output bands vector indicating the desired output bands and its order
	   \param data(out) bands image buffer vector
	   \note data will be internally allocated and its content is band interleaved
	*/
	LizardTech::ImageBuffer* getImageBuffer2(unsigned int size, 
											 unsigned int nbands,
	                                         const unsigned int* bands, 
											 void* data[]);

	/*! Returns an ImageBuffer for loading an image from MrSIDNavigator.  
	   \param size(in) size of the image buffer
	   \param nbands(in) number of output bands
	   \param data(out) image buffer  
	   \note  
		- data must have been previously allocated and its content is pixel interleaved
		- band order is always set to 0, 1, 2
	*/
	LizardTech::ImageBuffer* getImageBuffer(unsigned int size, unsigned int nbands, void* data);
	//@}

	//@{
	void getData(unsigned int& w, unsigned int& h, 
		         double& x0, double& y0, 
	             double& x1, double& y1, 
				 unsigned int nbands, 
	             const unsigned int* bands, 
				 void *data[]);
	//@}

	// { --- Methods that are relative to the SidNavigator rectangle

	//! Changes resolution level of MrSIDNavigator to the indicated level
	void zoomTo(int level); 

	//! Gets the current resolution level that navigator is bound
	int getCurrentLevel();

	//! Gets the number of lines and columns at the current level
	void getDimensionAtLevel(int level, int& w, int& h);

	//! Gets the dimension of the current Navigator rectangle 
	void getCurrentRectangleDimension(int& w, int& h);

	void getCurrentBoundingBox(double& xmin, double& ymin, double& xmax, double& ymax);
	
	//! Gets the pixel resolution at the current level
	void getCurrentLevelResolution(double& resx, double& resy);

	//! Move an redimension current rectangle navigator
	/*!
		\param leftColum X index of upper left corner of the rectangle
		\param topRow Y index of the upper corner of the rectangle
		\param width width of the rectangle
		\param height height of the rectangle
		\returns TRUE if the rectangle is valid and can be positioned and FALSE otherwise
	*/
	bool selectArea(int leftColumn, int topRow, int width, int height); 

	//! Gets the image defines by the current rectangle navigator
	/*!
		\param pointer to a selected area of memory to hold the content of the
		current rectangle navigator
	*/
	bool getSelectedArea(unsigned char* data);
	// --- }

};

/*
** ---------------------------------------------------------------
** Inline methods:
*/

inline void TeMrSIDReader::getDimensions(unsigned int& width, unsigned int& height) const
{
  width = getWidth();
  height = getHeight();
}

inline void TeMrSIDReader::getResolution(double& rx, double& ry)
{
  rx = resX();
  ry = resY();
}

inline void TeMrSIDReader::getOrigin(double& x, double& y)
{
  x = originX();
  y = originY();
}

inline void TeMrSIDReader::world2Pixel(double wx, double wy, int& ix, int& iy)
{
  double off_x, off_y;
  double sx, sy;

  getOrigin(off_x, off_y);
  getResolution(sx, sy);

  ix = (int) ((wx - off_x)/sx + 0.5);
  iy = (int) ((wy - off_y)/sy + 0.5);
}

inline void TeMrSIDReader::pixel2World(int ix, int iy, double& wx, double& wy)
{
  double off_x, off_y;
  double sx, sy;

  getOrigin(off_x, off_y);
  getResolution(sx, sy);

  wx = off_x + ix * sx;
  wy = off_y + iy * sy;
}

#endif
