/************************************************************************************
TerraView - visualization and exploration of geographical databases
using TerraLib.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.
This file is part of TerraView. TerraView is free software; you can redistribute it 
and/or modify it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

You should have received a copy of the GNU General Public License
along with TerraView.
The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The software provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use of
this program and its documentation.
*************************************************************************************/

#include "TeDatabaseUtils.h"
#include "TeApplicationUtils.h"
#include "TeDatabase.h"
#include "TeTable.h"
#include "TeKernelParams.h"
#include "TeLayerFunctions.h"
#include "TeErrorLog.h"
#include "TeStatistics.h"

#include <algorithm>

using std::string;

bool createColorSchemeTable(TeDatabase* db, const string& tname)
{
	string table;

	if (!db)
		return false;
	if (tname.empty())
		table = "te_color_scheme";
	else
		table = tname;

	TeAttributeList attList;

	TeAttribute at1;
	at1.rep_.type_ = TeSTRING;
	at1.rep_.numChar_ = 50;
	at1.rep_.name_ = "ramp_name";
	at1.rep_.isPrimaryKey_ = true;
	attList.push_back(at1);	
	
	TeAttribute at2;
	at2.rep_.type_ = TeINT;
	at2.rep_.name_ = "color_order";
	at2.rep_.isPrimaryKey_ = true;
	attList.push_back(at2);	
	
	TeAttribute at3;
	at3.rep_.type_ = TeINT;
	at3.rep_.name_ = "red";
	attList.push_back(at3);	
	
	
	TeAttribute at4;
	at4.rep_.type_ = TeINT;
	at4.rep_.name_ = "green";
	attList.push_back(at4);	
	
	TeAttribute at5;
	at5.rep_.type_ = TeINT;
	at5.rep_.name_ = "blue";
	attList.push_back(at5);	
	
	bool status = db->createTable(table,attList);
	attList.clear();
	return status;
}

bool createChartParamsTable(TeDatabase* db, bool integrity)
{
	if (!db)
		return false;
	
	string table = "te_chart_params";
	
	TeAttributeList attList;

	TeAttribute at5;
	at5.rep_.type_ = TeINT;
	at5.rep_.name_ = "theme_id";
	at5.rep_.isPrimaryKey_ = true;
	attList.push_back(at5);	
	
	TeAttribute at1;
	at1.rep_.type_ = TeSTRING;
	at1.rep_.numChar_ = 50;
	at1.rep_.name_ = "attr_name";
	at1.rep_.isPrimaryKey_ = true;
	attList.push_back(at1);	
	
	TeAttribute at2;
	at2.rep_.type_ = TeINT;
	at2.rep_.name_ = "red";
	attList.push_back(at2);	
	
	TeAttribute at3;
	at3.rep_.type_ = TeINT;
	at3.rep_.name_ = "blue";
	attList.push_back(at3);	
	
	TeAttribute at4;
	at4.rep_.type_ = TeINT;
	at4.rep_.name_ = "green";
	attList.push_back(at4);	
	
	bool status = db->createTable(table,attList);
	attList.clear();

	if(status && integrity)
	{
		if(db->tableExist ("te_theme_application"))
		{
			if (!db->createRelation("fk_chartpar_theme_id", "te_chart_params", "theme_id", "te_theme_application", "theme_id", true))
				return false;
		}
		else
			return false;
	}
	return status;
}


bool createThemeApplicationTable(TeDatabase* db, bool integrity)
{
	if (!db)
		return false;
	
	string table = "te_theme_application";
	
	TeAttributeList attList;

	TeAttribute at1;
	at1.rep_.type_ = TeINT;
	at1.rep_.name_ = "theme_id";
	at1.rep_.isPrimaryKey_ = true;
	attList.push_back(at1);
	
	TeAttribute at2;
	at2.rep_.type_ = TeINT;
	at2.rep_.name_ = "visible_out_coll";
	attList.push_back(at2);	
	
	TeAttribute at3;
	at3.rep_.type_ = TeINT;
	at3.rep_.name_ = "visible_without_conn";
	attList.push_back(at3);	

	TeAttribute at31;
	at31.rep_.type_ = TeSTRING;
	at31.rep_.numChar_ = 255;
	at31.rep_.name_ = "refine_attribute_where";
	attList.push_back(at31);	

	TeAttribute at32;
	at32.rep_.type_ = TeSTRING;
	at32.rep_.numChar_ = 255;
	at32.rep_.name_ = "refine_spatial_where";
	attList.push_back(at32);	

	TeAttribute at33;
	at33.rep_.type_ = TeSTRING;
	at33.rep_.numChar_ = 255;
	at33.rep_.name_ = "refine_temporal_where";
	attList.push_back(at33);	

	TeAttribute at4;
	at4.rep_.type_ = TeSTRING;
	at4.rep_.numChar_ = 50;
	at4.rep_.name_ = "grouping_color";
	attList.push_back(at4);	

	TeAttribute at5;
	at5.rep_.type_ = TeINT;
	at5.rep_.name_ = "grouping_color_direction";
	attList.push_back(at5);	

	TeAttribute at6;
	at6.rep_.type_ = TeINT;
	at6.rep_.name_ = "grouping_objects";
	attList.push_back(at6);	

	TeAttribute at7;
	at7.rep_.type_ = TeREAL;
	at7.rep_.name_ = "pie_diameter";
	attList.push_back(at7);	

	TeAttribute at8;
	at8.rep_.type_ = TeREAL;
	at8.rep_.name_ = "pie_max_diameter";
	attList.push_back(at8);	

	TeAttribute at9;
	at9.rep_.type_ = TeREAL;
	at9.rep_.name_ = "pie_min_diameter";
	attList.push_back(at9);	

	TeAttribute at10;
	at10.rep_.type_ = TeSTRING;
	at10.rep_.numChar_ = 255;
	at10.rep_.name_ = "pie_dimension_attr";
	attList.push_back(at10);	

	TeAttribute at11;
	at11.rep_.type_ = TeREAL;
	at11.rep_.name_ = "bar_max_height";
	attList.push_back(at11);	

	TeAttribute at12;
	at12.rep_.type_ = TeREAL;
	at12.rep_.name_ = "bar_min_height";
	attList.push_back(at12);	

	TeAttribute at13;
	at13.rep_.type_ = TeREAL;
	at13.rep_.name_ = "bar_width";
	attList.push_back(at13);	

	TeAttribute at14;
	at14.rep_.type_ = TeREAL;
	at14.rep_.name_ = "chart_max_value";
	attList.push_back(at14);	

	TeAttribute at15;
	at15.rep_.type_ = TeREAL;
	at15.rep_.name_ = "chart_min_value";
	attList.push_back(at15);	

	TeAttribute at16;
	at16.rep_.type_ = TeINT;
	at16.rep_.name_ = "chart_type";
	attList.push_back(at16);	

	TeAttribute at17;
	at17.rep_.type_ = TeINT;
	at17.rep_.name_ = "chart_objects";
	attList.push_back(at17);
	 
	TeAttribute at18;
	at18.rep_.type_ = TeINT;
	at18.rep_.name_ = "keep_dimension";
	attList.push_back(at18);

	TeAttribute at19;
	at19.rep_.type_ = TeINT;
	at19.rep_.name_ = "show_canvas_leg";
	attList.push_back(at19);

	TeAttribute at20;
	at20.rep_.type_ = TeREAL;
	at20.rep_.name_ = "canvas_leg_x";
	attList.push_back(at20);

	TeAttribute at21;
	at21.rep_.type_ = TeREAL;
	at21.rep_.name_ = "canvas_leg_y";
	attList.push_back(at21);

	TeAttribute at22;
	at22.rep_.type_ = TeINT;
	at22.rep_.name_ = "mix_color";
	attList.push_back(at22);

	TeAttribute at23;
	at23.rep_.type_ = TeINT;
	at23.rep_.name_ = "count_obj";
	attList.push_back(at23);

	TeAttribute at24;
	at24.rep_.type_ = TeSTRING;
	at24.rep_.name_ = "text_table";
	at24.rep_.numChar_ = 50;
	attList.push_back(at24);
	
	TeAttribute at26;
	at26.rep_.type_ = TeSTRING;
	at26.rep_.name_ = "chart_function";
	at26.rep_.numChar_ = 10;
	attList.push_back(at26);
	
	bool status = db->createTable(table,attList);
	attList.clear();
	
	if(status && integrity)
	{
		if (!db->createRelation("fk_themeAppl_theme_id", "te_theme_application", "theme_id", "te_theme", "theme_id", true))
			return false;
	}
	return status;
}


bool populateThemeApplicationTable(TeDatabase* db)
{
	string sel, ins; 
	
	if(db->dbmsName()=="MySQL") //MySQL doesnt support NOT EXISTS  
	{
		sel = " SELECT theme_id FROM te_theme ";
	}
	else
	{
		ins  = " INSERT INTO te_theme_application (theme_id) ";
		ins += " SELECT t.theme_id FROM te_theme t ";
		ins += " WHERE NOT EXISTS ";
		ins += " ( SELECT ta.theme_id FROM ";
		ins += " te_theme_application ta ";
		ins += " WHERE ta.theme_id = t.theme_id ) ";

		return (db->execute(ins));
	}
	
	TeDatabasePortal* portal = db->getPortal();
	if(!portal)
		return false;

	if(!portal->query(sel))
	{
		delete portal;
		return false;
	}

	if(!portal->fetchRow())
	{
		delete portal;
		return true;
	}

	do 
	{
		ins  = " INSERT INTO te_theme_application (theme_id) VALUES (";
		ins += string(portal->getData(0));
		ins += " )";
		
		db->execute(ins);

	} while(portal->fetchRow());

	delete portal;
	return true;
}


bool createMediaTable(TeDatabase* db, const string& tname)
{
	if (!db)
		return false;
	
	TeAttributeList attList;

	TeAttribute at1;
	at1.rep_.type_ = TeSTRING;
	at1.rep_.numChar_ = 255;
	at1.rep_.name_ = "media_name";
	at1.rep_.isPrimaryKey_ = true;
	attList.push_back(at1);	

	TeAttribute at2;
	at2.rep_.type_ = TeSTRING;
	at2.rep_.numChar_ = 50;
	at2.rep_.name_ = "media_type";
	attList.push_back(at2);	

	TeAttribute at3;
	at3.rep_.type_ = TeSTRING;
	at3.rep_.numChar_ = 255;
	at3.rep_.name_ = "description";
	attList.push_back(at3);
	
	TeAttribute at4;
	at4.rep_.type_ = TeBLOB;
	at4.rep_.name_ = "media_blob";
	attList.push_back(at4);	
	
	TeAttribute at5;
	at5.rep_.type_ = TeSTRING;
	at5.rep_.numChar_ = 255;
	at5.rep_.name_ = "media_path";
	attList.push_back(at5);
	
	vector<string> pkeys;
	pkeys.push_back("media_name");
	
	bool status = db->createTable(tname,attList);
	attList.clear();
	return status;
}


bool createLayerMediaTable(TeDatabase* db, int layerId, const string& tname)
{
	if (!db)
		return false;
	
	TeAttributeList attList;

	TeAttribute at1;
	at1.rep_.type_ = TeSTRING;
	at1.rep_.numChar_ = 50;
	at1.rep_.name_ = "object_id";
	at1.rep_.isPrimaryKey_ = true;
	attList.push_back(at1);	

	TeAttribute at3;
	at3.rep_.type_ = TeSTRING;
	at3.rep_.numChar_ = 255;
	at3.rep_.name_ = "media_name";
	at3.rep_.isPrimaryKey_ = true;
	attList.push_back(at3);
	
	TeAttribute at2;
	at2.rep_.type_ = TeSTRING;
	at2.rep_.numChar_ = 50;
	at2.rep_.name_ = "media_table";
	at2.rep_.isPrimaryKey_ = true;
	attList.push_back(at2);	

	TeAttribute at4;
	at4.rep_.type_ = TeINT;
	at4.rep_.name_ = "show_default";
	attList.push_back(at4);	
	
	if(!db->createTable(tname,attList))
	{
		attList.clear();
		return false;
	}

	//insert in te_layer_table
	TeTable table(tname, attList,"object_id", "object_id", TeAttrMedia);
	
	bool status = db->insertTableInfo(layerId, table); 
	db->layerMap()[layerId]->addAttributeTable (table);
		
	attList.clear();
	return status;
}

bool createKernelMetadataTable(TeDatabase* db)
{
	if (!db)
		return false;
	
	TeAttributeList attList;

	TeAttribute at1;
	at1.rep_.type_ = TeINT;
	at1.rep_.name_ = "kernel_id";
	at1.rep_.isPrimaryKey_ = true;
	at1.rep_.isAutoNumber_ = true;
	attList.push_back(at1);	

	TeAttribute at2;
	at2.rep_.type_ = TeINT;
	at2.rep_.name_ = "event_theme_id_1";
	attList.push_back(at2);	
	
	TeAttribute at3;
	at3.rep_.type_ = TeSTRING;
	at3.rep_.numChar_ = 255;
	at3.rep_.name_ = "intensity_attr_name_1";
	attList.push_back(at3);
	
	TeAttribute at4;
	at4.rep_.type_ = TeSTRING;
	at4.rep_.numChar_ = 255;
	at4.rep_.name_ = "intensity_attr_table_1";
	attList.push_back(at4);

	TeAttribute at5;
	at5.rep_.type_ = TeINT;
	at5.rep_.name_ = "kernel_function_1";
	attList.push_back(at5);	

	TeAttribute at6;
	at6.rep_.type_ = TeINT;
	at6.rep_.name_ = "compute_type_1";
	attList.push_back(at6);	

	TeAttribute at7;
	at7.rep_.type_ = TeREAL;
	at7.rep_.name_ = "radius_value_1";
	attList.push_back(at7);	

	TeAttribute at8;
	at8.rep_.type_ = TeINT;
	at8.rep_.name_ = "combination_type";
	attList.push_back(at8);	

	TeAttribute at9;
	at9.rep_.type_ = TeINT;
	at9.rep_.name_ = "event_theme_id_2";
	attList.push_back(at9);	
	
	TeAttribute at10;
	at10.rep_.type_ = TeSTRING;
	at10.rep_.numChar_ = 255;
	at10.rep_.name_ = "intensity_attr_name_2";
	attList.push_back(at10);
	
	TeAttribute at11;
	at11.rep_.type_ = TeSTRING;
	at11.rep_.numChar_ = 255;
	at11.rep_.name_ = "intensity_attr_table_2";
	attList.push_back(at11);

	TeAttribute at12;
	at12.rep_.type_ = TeINT;
	at12.rep_.name_ = "kernel_function_2";
	attList.push_back(at12);	

	TeAttribute at13;
	at13.rep_.type_ = TeINT;
	at13.rep_.name_ = "compute_type_2";
	attList.push_back(at13);	

	TeAttribute at14;
	at14.rep_.type_ = TeREAL;
	at14.rep_.name_ = "radius_value_2";
	attList.push_back(at14);
	
	TeAttribute at15;
	at15.rep_.type_ = TeINT;
	at15.rep_.name_ = "support_theme_id";
	attList.push_back(at15);

	TeAttribute at17;
	at17.rep_.type_ = TeSTRING;
	at17.rep_.numChar_ = 255;
	at17.rep_.name_ = "support_theme_table";
	attList.push_back(at17);

	TeAttribute at18;
	at18.rep_.type_ = TeSTRING;
	at18.rep_.numChar_ = 255;
	at18.rep_.name_ = "support_theme_attr";
	attList.push_back(at18);

	TeAttribute at19;
	at19.rep_.type_ = TeINT;
	at19.rep_.name_ = "generated_layer_id";
	attList.push_back(at19);

	if(!db->createTable("te_kernel_metadata",attList))
	{
		attList.clear();
		return false;
	}

	db->createIndex("te_kernel_metadata", "kernel_meta_ind1", "support_theme_id, support_theme_table, support_theme_attr");
	db->createIndex("te_kernel_metadata", "kernel_meta_ind2", "generated_layer_id");

	return true;
}


bool insertMedia(TeDatabase* bd, const string& tableName, string fileName, const string& description, const string& type, bool blob)
{
	string ins;
	if(blob)
	{
		 if(bd->dbmsName()!="OracleSpatial")
		 {
			ins = "  INSERT INTO "+ tableName +" (media_name, media_type, description) ";
			ins += " VALUES (";
			ins += "'"+ fileName + "'";
			ins += ", '"+ type +"'";
			ins += ", '"+ description +"'";
			ins += ")";
		 }
		 else
		 {
			ins = "  INSERT INTO "+ tableName +" (media_name, media_type, description, media_blob) ";
			ins += " VALUES (";
			ins += "'"+ fileName + "'";
			ins += ", '"+ type +"'";
			ins += ", '"+ description +"'";
			ins += ", EMPTY_BLOB()";
			ins += " )";
		 }
	}
	else
	{
		 ins = "  INSERT INTO "+ tableName +" (media_name, media_type, description, media_path) ";
		 ins += " VALUES (";
		 ins += "'"+ fileName + "'";
		 ins += ", '"+ type +"'";
		 ins += ", '"+ description +"'";
		 ins += ", '"+ fileName +"'";
		 ins += ")";
	}

	if(!bd->execute (ins))
		return false;

	//insert blob
	TeAttributeRep rep;
	rep.name_ = "media_name";
	rep.type_ = TeSTRING;

	if(blob)
	{
		if(!bd->insertBlob (tableName, "media_blob", rep, fileName, fileName))
			return false;
	}

	return true;
}

bool insertMediaObjLayer (TeDatabase* db, TeLayer* layer, const string& objectId, const string& mediaName, const string& mediaTable)
{
	string table = layer->mediaTable();
	int show = 0;

	string ins = "  INSERT INTO "+ table +" (object_id, media_name, media_table, show_default) ";
	ins += " VALUES (";
	ins += "'"+  objectId + "'";
	ins += ", '"+ mediaName +"'";
	ins += ", '"+ mediaTable +"'";
	ins += ", "+ Te2String(show);
	ins += ")";

	if(!db->execute (ins))
		return false;

	return true;
}

bool insertKernelMetadata(TeDatabase* db, TeKernelParams& kernelPar)
{
	string ins = " INSERT INTO te_kernel_metadata (event_theme_id_1, intensity_attr_name_1, ";
	ins += " intensity_attr_table_1, kernel_function_1, compute_type_1, radius_value_1, ";
	ins += " combination_type, event_theme_id_2, intensity_attr_name_2, intensity_attr_table_2, ";
	ins += " kernel_function_2, compute_type_2, radius_value_2, support_theme_id,  ";
	ins += " support_theme_table, support_theme_attr, generated_layer_id)";
	ins += " VALUES ( ";
	ins += Te2String(kernelPar.eventThemeId1_) + ",";
	ins += "'"+ kernelPar.intensityAttrName1_ + "',";
	ins += "'"+ kernelPar.intensityAttrTable1_ + "',";
	ins += Te2String((int)kernelPar.kernelFunction1_) + ",";
	ins += Te2String((int)kernelPar.computeType1_) + ",";
	ins += Te2String(kernelPar.radiusValue1_) + ",";
	ins += Te2String((int)kernelPar.combinationType_) + ",";
	ins += Te2String(kernelPar.eventThemeId2_) + ",";
	ins += "'"+ kernelPar.intensityAttrName2_ + "',";
	ins += "'"+ kernelPar.intensityAttrTable2_ + "',";
	ins += Te2String((int)kernelPar.kernelFunction2_) + ",";
	ins += Te2String((int)kernelPar.computeType2_) + ",";
	ins += Te2String(kernelPar.radiusValue2_) + ",";
	ins += Te2String(kernelPar.supportThemeId_)+ ",";
	ins += "'"+ kernelPar.supportThemeTable_+ "',";
	ins += "'"+ kernelPar.supportThemeAttr_+ "',";
	ins += Te2String(kernelPar.generatedLayerId_);
	ins += ")";

	if(!db->execute (ins))
		return false;

	return true;
}


bool updateKernelMetadata(TeDatabase* db, TeKernelParams& kernelPar)
{
  string del = "DELETE FROM te_kernel_metadata WHERE support_theme_id = ";
  del += Te2String(kernelPar.supportThemeId_);

  if (!db->execute(del)) 
  {
    return false;
  }

  return insertKernelMetadata(db,kernelPar);  
}

bool setDefaultMedia(TeDatabase* db, TeLayer* layer, const string& objectId, const string& mediaName, const string& mediaTable)
{
	string table = layer->mediaTable();
	
	string up = "UPDATE " + table + " SET show_default = 0 WHERE ";
	up += "object_id = '" + objectId + "'";
	if(!db->execute(up))
		return false;
	
	up = "UPDATE "+ table +" SET ";
	up += " show_default = 1 ";
	up += "	WHERE 	object_id = '"+ objectId +"'";
	up += " AND		media_name = '"+ mediaName +"'";
	up += " AND		media_table = '"+ mediaTable +"'";

	if(!db->execute (up))
		return false;

	return true;
}


bool updateMediaDescription(TeDatabase* db, const string& mediaName, const string& mediaTable, const string& desc)
{
	string up = "UPDATE " + mediaTable + " SET description = '" + desc.c_str() + "'";
	up += " WHERE media_name = '" + mediaName + "'";
	if(!db->execute(up))
		return false;
	return true;
}


bool populateColorSchemeTable(TeDatabase* db, const string& tname)
{
	string table;

	if (!db)
		return false;
	if (tname.empty())
		table = "te_color_scheme";
	else
		table = tname;
	if (!db->tableExist(table))
		createColorSchemeTable(db,table);
	
	string ins = "INSERT INTO " + table + " (ramp_name, color_order, red, green, blue)";
	string insert;
	insert = ins + " VALUES ('RED', 1, 255, 225, 225)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 2, 255, 195, 195)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 3, 255, 165, 165)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 4, 255, 135, 135)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 5, 255, 112, 112)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 6, 255, 66, 66)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 7, 255, 0, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 8, 225, 1, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 9, 200, 1, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 10, 175, 1, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 11, 150, 1, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 12, 125, 0, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED', 13, 100, 0, 0)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('GREEN', 1, 231, 255, 231)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 2, 206, 255, 206)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 3, 176, 255, 176)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 4, 136, 255, 135)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 5, 92, 255, 91)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 6, 47, 255, 46)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 7, 0, 255, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 8, 3, 225, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 9, 2, 195, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 10, 2, 170, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 11, 2, 140, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 12, 1, 115, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN', 13, 1, 85, 0)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('BLUE', 1, 225, 226, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 2, 195, 196, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 3, 165, 167, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 4, 135, 137, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 5, 112, 114, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 6, 76, 79, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 7, 41, 44, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 8, 0, 0, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 9, 1, 4, 239)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 10, 1, 4, 210)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 11, 1, 4, 185)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 12, 1, 3, 160)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE', 13, 1, 3, 135)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('CYAN', 1, 225, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 2, 195, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 3, 165, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 4, 135, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 5, 106, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 6, 76, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 7, 41, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 8, 0, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 9, 1, 235, 235)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 10, 1, 215, 215)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 11, 1, 195, 195)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 12, 1, 175, 175)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN', 13, 1, 155, 155)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('ORANGE', 1, 255, 238, 225)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 2, 255, 222, 195)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 3, 255, 205, 165)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 4, 255, 189, 135)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 5, 255, 172, 106)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 6, 255, 156, 76)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 7, 255, 136, 41)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 8, 255, 114, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 9, 239, 105, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 10, 210, 94, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 11, 185, 83, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 12, 160, 72, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ORANGE', 13, 135, 60, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('YELLOW', 1, 255, 250, 225)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 2, 255, 245, 195)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 3, 255, 241, 165)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 4, 255, 236, 135)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 5, 255, 232, 106)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 6, 255, 227, 76)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 7, 255, 222, 41)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 8, 255, 215, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 9, 235, 198, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 10, 210, 177, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 11, 185, 156, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 12, 160, 135, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW', 13, 135, 114, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('MAGENTA', 1, 255, 225, 247)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 2, 255, 195, 239)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 3, 255, 165, 231)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 4, 255, 135, 222)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 5, 255, 106, 214)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 6, 255, 76, 206)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 7, 255, 41, 197)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 8, 255, 1, 186)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 9, 235, 1, 171)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 10, 210, 1, 153)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 11, 185, 1, 135)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 12, 160, 1, 116)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('MAGENTA', 13, 135, 1, 98)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('GRAY', 1, 245, 245, 245)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 2, 225, 225, 225)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 3, 205, 205, 205)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 4, 185, 185, 185)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 5, 165, 165, 165)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 6, 145, 145, 145)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 7, 125, 125, 125)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 8, 105, 105, 105)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 9, 85, 85, 85)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 10, 65, 65, 65)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 11, 45, 45, 45)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 12, 25, 25, 25)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY', 13, 5, 5, 5)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('CHARTREUSE', 1, 246, 255, 225)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 2, 238, 255, 195)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 3, 229, 255, 165)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 4, 220, 255, 135)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 5, 211, 255, 106)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 6, 203, 255, 76)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 7, 193, 255, 41)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 8, 41, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 9, 238, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 10, 238, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 11, 132, 185, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 12, 114, 154, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CHARTREUSE', 13, 96, 135, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('PURPLE', 1, 251, 215, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 2, 249, 195, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 3, 248, 165, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 4, 246, 135, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 5, 242, 96, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 6, 235, 55, 250)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 7, 221, 1, 240)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 8, 207, 1, 225)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 9, 184, 1, 205)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 10, 166, 1, 180)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 11, 138, 1, 150)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 12, 112, 0, 125)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE', 13, 92, 0, 100)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('BLUE TO RED', 1, 23, 1, 220)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 2, 21, 1, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 3, 85, 71, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 4, 101, 103, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 5, 165, 167, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 6, 210, 211, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 7, 255, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 8, 255, 210, 210)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 9, 255, 160, 160)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 10, 255, 111, 111)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 11, 255, 71, 71)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 12, 255, 0, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO RED', 13, 220, 1, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('GRAY TO RED', 1, 85, 85, 85)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 2, 115, 115, 115)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 3, 145, 145, 145)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 4, 170, 170, 170)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 5, 195, 195, 195)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 6, 225, 225, 225)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 7, 255, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 8, 255, 210, 210)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 9, 255, 160, 160)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 10, 255, 111, 111)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 11, 255, 71, 71)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 12, 255, 0, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GRAY TO RED', 13, 220, 1, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('BLUE TO ORANGE', 1, 23, 1, 220)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 2, 21, 1, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 3, 85, 71, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 4, 101, 103, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 5, 165, 167, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 6, 210, 211, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 7, 255, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 8, 255, 230, 210)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 9, 255, 205, 165)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 10, 255, 178, 116)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 11, 255, 153, 71)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 12, 255, 114, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('BLUE TO ORANGE', 13, 220, 98, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('PURPLE TO ORANGE', 1, 198, 1, 220)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 2, 229, 1, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 3, 236, 71, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 4, 240, 111, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 5, 246, 165, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 6, 250, 195, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 7, 255, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 8, 255, 230, 210)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 9, 255, 205, 165)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 10, 255, 178, 116)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 11, 255, 153, 71)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 12, 255, 114, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PURPLE TO ORANGE', 13, 220, 98, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('GREEN TO RED', 1, 3, 220, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 2, 0, 255, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 3, 72, 255, 71)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 4, 112, 255, 111)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 5, 166, 255, 165)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 6, 196, 255, 195)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 7, 255, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 8, 255, 210, 210)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 9, 255, 160, 160)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 10, 255, 111, 111)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 11, 255, 71, 71)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 12, 255, 0, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO RED', 13, 220, 1, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('GREEN TO ORANGE', 1, 3, 220, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 2, 0, 255, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 3, 72, 255, 71)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 4, 112, 255, 111)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 5, 166, 255, 165)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 6, 196, 255, 195)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 7, 255, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 8, 255, 230, 210)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 9, 255, 205, 165)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 10, 255, 178, 116)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 11, 255, 153, 71)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 12, 255, 114, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO ORANGE', 13, 220, 98, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('CYAN TO RED', 1, 1, 220, 220)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 2, 0, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 3, 71, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 4, 116, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 5, 165, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 6, 210, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 7, 255, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 8, 255, 210, 210)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 9, 255, 160, 160)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 10, 255, 111, 111)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 11, 255, 71, 71)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 12, 255, 0, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO RED', 13, 220, 1, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('CYAN TO ORANGE', 1, 1, 220, 220)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 2, 0, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 3, 71, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 4, 116, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 5, 165, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 6, 210, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 7, 255, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 8, 255, 230, 210)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 9, 255, 220, 165)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 10, 255, 178, 116)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 11, 255, 153, 71)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 12, 255, 114, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('CYAN TO ORANGE', 13, 220, 98, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 1, 247, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 2, 255, 245, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 3, 255, 227, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 4, 255, 209, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 5, 255, 183, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 6, 255, 174, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 7, 255, 156, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 8, 255, 132, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 9, 255, 108, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 10, 255, 90, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 11, 255, 61, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 12, 255, 37, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO ORANGE TO RED', 13, 255, 0, 0)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 1, 220, 1, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 2, 240, 1, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 3, 255, 1, 49)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 4, 255, 1, 96)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 5, 255, 1, 144)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 6, 255, 1, 204)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 7, 255, 0, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 8, 223, 1, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 9, 187, 1, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 10, 152, 1, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 11, 116, 1, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 12, 56, 1, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('RED TO PURPLE TO BLUE', 13, 0, 255, 0)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 1, 1, 255, 58)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 2, 1, 255, 88)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 3, 1, 255, 118)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 4, 1, 255, 148)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 5, 1, 255, 178)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 6, 1, 255, 207)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 7, 1, 255, 237)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 8, 1, 243, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 9, 1, 213, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 10, 1, 184, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 11, 1, 124, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 12, 1, 94, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('GREEN TO CYAN TO BLUE', 13, 0, 0, 255)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 1, 255, 255, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 2, 229, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 3, 205, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 4, 170, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 5, 116, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 6, 63, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 7, 3, 235, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 8, 1, 222, 103)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 9, 1, 220, 153)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 10, 1, 170, 138)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 11, 1, 135, 166)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 12, 1, 81, 166)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('YELLOW TO GREEN TO DARK BLUE', 13, 14, 1, 130)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('ELEVATION 1', 1, 175, 255, 231)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 2, 151, 255, 150)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 3, 255, 247, 130)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 4, 122, 205, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 5, 2, 130, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 6, 230, 162, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 7, 230, 125, 31)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 8, 220, 104, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 9, 200, 74, 36)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 10, 170, 1, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 11, 88, 52, 20)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 12, 185, 185, 185)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 1', 13, 255, 55, 255)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('ELEVATION 2', 1, 176, 231, 219)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 2, 228, 255, 180)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 3, 255, 251, 195)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 4, 250, 255, 156)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 5, 225, 212, 124)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 6, 219, 166, 32)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 7, 187, 165, 99)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 8, 161, 103, 51)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 9, 156, 123, 96)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 10, 157, 174, 176)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 11, 217, 192, 217)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 12, 198, 220, 221)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('ELEVATION 2', 13, 255, 255, 255)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 1, 210, 255, 218)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 2, 198, 255, 231)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 3, 157, 255, 225)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 4, 116, 242, 248)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 5, 0, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 6, 83, 231, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 7, 127, 192, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 8, 65, 155, 220)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 9, 1, 123, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 10, 0, 0, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 11, 23, 1, 220)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 12, 16, 1, 200)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('SEA FLOOR ELEVATION', 13, 14, 1, 130)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('FULL SPECTRUM', 1, 255, 0, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 2, 255, 90, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 3, 255, 180, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 4, 241, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 5, 152, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 6, 3, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 7, 1, 255, 148)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 8, 1, 255, 237)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 9, 1, 184, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 10, 0, 0, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 11, 176, 1, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 12, 254, 0, 245)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('FULL SPECTRUM', 13, 254, 0, 155)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('PRECIPITATION', 1, 206, 134, 63)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 2, 255, 166, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 3, 255, 216, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 4, 255, 255, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 5, 186, 239, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 6, 111, 255, 39)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 7, 0, 206, 102)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 8, 3, 220, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 9, 2, 187, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 10, 0, 140, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 11, 0, 135, 140)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 12, 44, 102, 130)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('PRECIPITATION', 13, 32, 71, 146)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('TEMPERATURE', 1, 255, 255, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 2, 255, 101, 150)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 3, 176, 1, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 4, 0, 0, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 5, 1, 130, 255)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 6, 1, 210, 220)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 7, 1, 255, 178)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 8, 0, 255, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 9, 185, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 10, 255, 255, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 11, 255, 180, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 12, 204, 72, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('TEMPERATURE', 13, 180, 1, 1)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('LAND COVER 1', 1, 255, 239, 140)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 2, 255, 201, 125)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 3, 239, 202, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 4, 239, 155, 73)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 5, 206, 174, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 6, 206, 191, 112)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 7, 163, 206, 90)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 8, 155, 206, 50)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 9, 0, 206, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 10, 69, 140, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 11, 122, 139, 53)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 12, 2, 130, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 1', 13, 1, 100, 99)";
	if (!db->execute(insert)) return false;

	insert = ins + " VALUES ('LAND COVER 2', 1, 222, 51, 51)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 2, 239, 155, 73)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 3, 239, 202, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 4, 255, 247, 144)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 5, 189, 239, 104)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 6, 182, 255, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 7, 172, 220, 119)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 8, 155, 206, 50)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 9, 0, 206, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 10, 69, 140, 0)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 11, 122, 139, 53)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 12, 2, 130, 1)";
	if (!db->execute(insert)) return false;
	insert = ins + " VALUES ('LAND COVER 2', 13, 1, 100, 0)";
	if (!db->execute(insert)) return false;
	return true;
}

bool locatePieBar (TeDatabase* db, TeThemeApplication* theme, TeCoord2D pt, double delta) 
{
	if (!db || !theme) 
		return false;

	theme->chartSelected_.clear();
	string	TC = theme->collectionTable();
	string xmin = Te2String(pt.x()-delta);
	string xmax = Te2String(pt.x()+delta);
	string ymin = Te2String(pt.y()-delta);
	string ymax = Te2String(pt.y()+delta);

	TeDatabasePortal* portal = db->getPortal();
	if (!portal)
		return false;

	string sel = "SELECT c_object_id FROM " + TC + " WHERE ";
	sel += "label_x > " + xmin + " AND ";
	sel += "label_x < " + xmax + " AND ";
	sel += "label_y > " + ymin + " AND ";
	sel += "label_y < " + ymax;

	if(!portal->query(sel))
	{
		delete portal;
		return false;
	}

	if(portal->fetchRow() == false)
	{
		delete portal;
		return false;
	}

	theme->chartSelected_ = portal->getData("c_object_id");

	delete portal;

	return true;
}


bool loadChartParams (TeDatabase* db, TeThemeApplication *theme)
{
	if (!db || !theme) 
		return false;

	TeDatabasePortal* portal = db->getPortal();

	string q ="SELECT * FROM te_chart_params WHERE theme_id= ";
	q += Te2String (theme->id());

	if (!portal->query(q))
	{	
		delete portal;
		return false;
	}
	while (portal->fetchRow())
	{
		TeColor cor = portal->getColor ();
		string attname = portal->getData("attr_name");
		int f = attname.find("(");
		if(f >= 0)
		{
			string alias = attname;
			attname.erase(f, attname.size()-f);
			alias.erase(0, f+1);
			alias.erase(alias.size()-1, 1);
			map<string, string>& mapAA = db->mapThemeAlias()[theme->id()];
			theme->concatTableName(attname);
			mapAA[attname] = alias;
		}
		theme->chartAttributes_.push_back(attname);
		theme->chartColors_.push_back(cor);
	}
	delete portal;
	return true;
}


bool loadKernelParams (TeDatabase* db, int supThemeId, const string& supTableName, const string& supAttrName, TeKernelParams& KernelPar)
{
	if (!db) 
		return false;

	TeDatabasePortal* portal = db->getPortal();

	string q ="SELECT event_theme_id_1, intensity_attr_name_1, ";
	q += " intensity_attr_table_1, kernel_function_1, compute_type_1, radius_value_1, ";
	q += " combination_type, event_theme_id_2, intensity_attr_name_2, intensity_attr_table_2, ";
	q += " kernel_function_2, compute_type_2, radius_value_2, support_theme_id,  ";
	q += " support_theme_table, support_theme_attr, generated_layer_id ";
	q += " FROM te_kernel_metadata WHERE ";
	q += " support_theme_id = "+ Te2String (supThemeId);
	q += " AND support_theme_table = '"+ supTableName +"'";
	q += " AND support_theme_attr = '"+ supAttrName +"'";

	if (!portal->query(q))
	{	
		delete portal;
		return false;
	}
	
	if (portal->fetchRow())
	{
		KernelPar.eventThemeId1_ = atoi(portal->getData(0));
		KernelPar.intensityAttrName1_ = string(portal->getData(1));
		KernelPar.intensityAttrTable1_ = string(portal->getData(2));
		KernelPar.kernelFunction1_ = TeKernelFunctionType(atoi(portal->getData(3)));
		KernelPar.computeType1_ = TeKernelComputeType(atoi(portal->getData(4)));
		KernelPar.radiusValue1_ = portal->getDouble(5);
		KernelPar.combinationType_ = TeKernelCombinationType(atoi(portal->getData(6)));
		KernelPar.eventThemeId2_ = atoi(portal->getData(7));
		KernelPar.intensityAttrName2_ = string(portal->getData(8));
		KernelPar.intensityAttrTable2_ = string(portal->getData(9));
		KernelPar.kernelFunction2_ = TeKernelFunctionType(atoi(portal->getData(10)));
		KernelPar.computeType2_ = TeKernelComputeType(atoi(portal->getData(11)));
		KernelPar.radiusValue2_ = portal->getDouble(12);
		KernelPar.supportThemeId_ = atoi(portal->getData(13));
		KernelPar.supportThemeTable_ = string(portal->getData(14));
		KernelPar.supportThemeAttr_ = string(portal->getData(15));
		KernelPar.generatedLayerId_ =  atoi(portal->getData(16));

		vector<int> ids;
		if(KernelPar.eventThemeId1_>0)
			ids.push_back (KernelPar.eventThemeId1_);
		if(KernelPar.eventThemeId2_>0)
			ids.push_back (KernelPar.eventThemeId2_);
		if(KernelPar.supportThemeId_>0)
			ids.push_back (KernelPar.supportThemeId_);

		string sqlIds; 
		vector<int>::iterator it = ids.begin();
		
		while (it != ids.end ())
		{
			sqlIds += Te2String((*it)) +",";
			++it;
		}

		sqlIds.replace (sqlIds.size()-1, 1, " ");
		string sql = "SELECT theme_id, name FROM te_theme WHERE theme_id IN ("+ sqlIds +")";

		portal->freeResult();
		if(!portal->query(sql))
		{
			delete portal;
			return false;
		}

		while(portal->fetchRow())
		{
			int themeId = atoi(portal->getData(0));
			string themeName = string(portal->getData(1));

			if(themeId==KernelPar.eventThemeId1_)
				KernelPar.eventThemeName1_ = themeName;
			else if (themeId==KernelPar.eventThemeId2_)
				KernelPar.eventThemeName2_ = themeName;
			else if (themeId==KernelPar.supportThemeId_)
				KernelPar.supportThemeName_ = themeName;
		}

		if(KernelPar.generatedLayerId_>0)
		{
			sql = "SELECT name FROM te_layer WHERE layer_id = "+ KernelPar.generatedLayerId_;
			
			portal->freeResult();
			if(!portal->query(sql) || !portal->fetchRow())
			{
				delete portal;
				return false;
			}
			
			KernelPar.generatedLayerName_= string(portal->getData(0));
		}

		delete portal;
		return true;
	}
	
	delete portal;
	return false;
}



bool loadKernelParams (TeDatabase* db, int layerId, TeKernelParams& KernelPar)
{
	if (!db) 
		return false;

	TeDatabasePortal* portal = db->getPortal();

	string q ="SELECT event_theme_id_1, intensity_attr_name_1, ";
	q += " intensity_attr_table_1, kernel_function_1, compute_type_1, radius_value_1, ";
	q += " combination_type, event_theme_id_2, intensity_attr_name_2, intensity_attr_table_2, ";
	q += " kernel_function_2, compute_type_2, radius_value_2, support_theme_id,  ";
	q += " support_theme_table, support_theme_attr, generated_layer_id ";
	q += " FROM te_kernel_metadata WHERE ";
	q += " generated_layer_id = "+ Te2String (layerId);

	if (!portal->query(q))
	{	
		delete portal;
		return false;
	}
	
	if (portal->fetchRow())
	{
		KernelPar.eventThemeId1_ = atoi(portal->getData(0));
		KernelPar.intensityAttrName1_ = string(portal->getData(1));
		KernelPar.intensityAttrTable1_ = string(portal->getData(2));
		KernelPar.kernelFunction1_ = TeKernelFunctionType(atoi(portal->getData(3)));
		KernelPar.computeType1_ = TeKernelComputeType(atoi(portal->getData(4)));
		KernelPar.radiusValue1_ = portal->getDouble(5);
		KernelPar.combinationType_ = TeKernelCombinationType(atoi(portal->getData(6)));
		KernelPar.eventThemeId2_ = atoi(portal->getData(7));
		KernelPar.intensityAttrName2_ = string(portal->getData(8));
		KernelPar.intensityAttrTable2_ = string(portal->getData(9));
		KernelPar.kernelFunction2_ = TeKernelFunctionType(atoi(portal->getData(10)));
		KernelPar.computeType2_ = TeKernelComputeType(atoi(portal->getData(11)));
		KernelPar.radiusValue2_ = portal->getDouble(12);
		KernelPar.supportThemeId_ = atoi(portal->getData(13));
		KernelPar.supportThemeTable_ = string(portal->getData(14));
		KernelPar.supportThemeAttr_ = string(portal->getData(15));
		KernelPar.generatedLayerId_ =  atoi(portal->getData(16));

		vector<int> ids;
		if(KernelPar.eventThemeId1_>0)
			ids.push_back (KernelPar.eventThemeId1_);
		if(KernelPar.eventThemeId2_>0)
			ids.push_back (KernelPar.eventThemeId2_);
		if(KernelPar.supportThemeId_>0)
			ids.push_back (KernelPar.supportThemeId_);

		string sqlIds; 
		vector<int>::iterator it = ids.begin();
		
		while (it != ids.end ())
		{
			sqlIds += Te2String((*it)) +",";
			++it;
		}

		sqlIds.replace (sqlIds.size()-1, 1, " ");
		string sql = "SELECT theme_id, name FROM te_theme WHERE theme_id IN ("+ sqlIds +")";

		portal->freeResult();
		if(!portal->query(sql))
		{
			delete portal;
			return false;
		}

		while(portal->fetchRow())
		{
			int themeId = atoi(portal->getData(0));
			string themeName = string(portal->getData(1));

			if(themeId==KernelPar.eventThemeId1_)
				KernelPar.eventThemeName1_ = themeName;
			else if (themeId==KernelPar.eventThemeId2_)
				KernelPar.eventThemeName2_ = themeName;
			else if (themeId==KernelPar.supportThemeId_)
				KernelPar.supportThemeName_ = themeName;
		}

		if(KernelPar.generatedLayerId_>0)
		{
      sql = "";
			sql = 
        "SELECT name FROM te_layer WHERE layer_id = "
        + Te2String(KernelPar.generatedLayerId_);
			
			portal->freeResult();
			if(!portal->query(sql) || !portal->fetchRow())
			{
				delete portal;
				return false;
			}
			
			KernelPar.generatedLayerName_= string(portal->getData(0));
		}

		delete portal;
		return true;
	}
	
	delete portal;
	return false;
}


bool updatePieBarLocation (TeDatabase* db, TeThemeApplication* theme, TeCoord2D pt) 
{
	if (!db || !theme) 
		return false;

	if (theme->chartSelected_.empty() == true)
		return false;

	string	x = Te2String(pt.x());
	string	y = Te2String(pt.y());
	string	TC = theme->collectionTable();
	string mover = "UPDATE " + TC + " SET label_x = " + x + ", label_y = " + y;
	mover += " WHERE c_object_id = '" + theme->chartSelected_ + "'";
	return db->execute(mover);
}

bool updatePointLocation (TeDatabase* db, TeThemeApplication* theme, TeCoord2D pt) 
{
	if (!db || !theme) 
		return false;

	string	x = Te2String(pt.x());
	string	y = Te2String(pt.y());
	string	table = theme->layer()->tableName(TePOINTS);
	string mover = "UPDATE " + table + " SET x = " + x + ", y = " + y;
	mover += " WHERE geom_id = " + Te2String(theme->tePoint_.geomId());
	return db->execute(mover);
}


bool insertChartParams(TeDatabase* db, TeThemeApplication *theme)
{
	if (!db || !theme) 
		return false;

	unsigned int n = theme->chartColors_.size();
	int themeId = theme->id();
	TeColor cor;
	string ins, insert;
	for (unsigned int i=0; i<n; i++)
	{
		cor = theme->chartColors_[i];
		ins = "INSERT INTO te_chart_params (theme_id, attr_name, red, green, blue)";
		insert = ins + " VALUES (" + Te2String(themeId) +",'";
		string attname = theme->chartAttributes_[i];
		map<int, map<string, string> > mapTAA = db->mapThemeAlias();
		map<int, map<string, string> >::iterator tit = mapTAA.find(theme->id());
		if(tit != mapTAA.end())
		{
			map<string, string> mapAA = tit->second;
			theme->concatTableName(attname);
			map<string, string>::iterator it = mapAA.find(attname);
			if(it != mapAA.end())
			{
				string alias = it->second;
				attname += "(" + alias + ")";
			}
		}
		insert += attname +"',";
		insert += Te2String(cor.red_) + ",";
		insert += Te2String(cor.green_) + ",";
		insert += Te2String(cor.blue_) + ")";
		if (!db->execute(insert))
			return false;
	}
	return true;
}


bool deleteChartParams (TeDatabase* db, TeThemeApplication* theme)
{
	if (!db || !theme) 
		return false;

	string id = Te2String(theme->id());
	string del = "DELETE FROM te_chart_params WHERE theme_id = " + id;
	return db->execute(del);
}

bool insertRampColor(TeDatabase* db, const string& name, int /* order */, TeColor& color, const string& tableName)
{
	if (!db) 
		return false;

	string ins = "INSERT INTO " + tableName + " (ramp_name, color_order, red, green, blue)";
	ins += " VALUES ('" +  name + "', " + Te2String(color.red_);
	ins += ", " + Te2String(color.green_);
	ins += ", " + Te2String (color.blue_) + ")";
	return db->execute(ins);
}

bool updateGridColor(TeDatabase* db, int themeId, TeObjectSelectionMode sel, TeColor& color)
{
	if (!db || themeId <= 0) 
		return false;

	string ins = "UPDATE te_legend SET red = " + Te2String(color.red_);
	ins += ", green = " +  Te2String(color.green_);
	ins += ", blue = "  + Te2String (color.blue_);
	ins += " WHERE theme_id = " + Te2String(themeId) + " AND ";
	if(sel == TeDefaultSelection)
		ins += "group_id = -3";
	else if(sel == TePointingSelection)
		ins += "group_id = -4";
	else if(sel == TeQuerySelection)
		ins += "group_id = -5";
	else
		ins += "group_id = -6";
	return db->execute(ins);
}

bool loadThemeApplication (TeDatabase* db, TeThemeApplication *theme, const string& tableName)
{
	if (!db || !theme) 
		return false;

	if (!db->loadTheme(theme))
		return false;

	TeDatabasePortal* portal = db->getPortal();
	if (!portal)
		return false;

	string q = "SELECT * FROM " + tableName + " WHERE theme_id= ";
	q += Te2String (theme->id());

	if (!portal->query(q) || !portal->fetchRow())
	{	
		delete portal;
		return false;
	}
	theme->outOfCollectionVisibility(portal->getBool(1));
	theme->withoutDataConnectionVisibility(portal->getBool(2));
	theme->refineAttributeWhere(string(portal->getData (3)));
	theme->refineSpatialWhere(string(portal->getData (4)));
	theme->refineTemporalWhere(string(portal->getData (5)));

	string name = portal->getData(6);
	if (!name.empty())
		theme->groupColor(name);
	theme->groupColorDir(TeColorDirection(atoi(portal->getData(7))));
	theme->groupObjects(TeSelectedObjects(atoi(portal->getData(8))));
	theme->pieDiameter(portal->getDouble(9));
	theme->pieMaxDiameter(portal->getDouble(10));
	theme->pieMinDiameter(portal->getDouble(11));
	name = portal->getData(12);
	if (!name.empty())
		theme->pieDimAttribute(name);
	theme->barMaxHeight(portal->getDouble(13));
	theme->barMinHeight(portal->getDouble(14));
	theme->barWidth(portal->getDouble(15));
	theme->chartMaxVal(portal->getDouble(16));
	theme->chartMinVal(portal->getDouble(17));
	theme->chartType(TeChartType(atoi(portal->getData(18))));
	theme->chartObjects(TeSelectedObjects(atoi(portal->getData(19))));
	theme->keepDimension(atoi(portal->getData(20)));
	theme->showCanvasLegend(portal->getBool(21));
	TeCoord2D p;
	p.x(portal->getDouble(22));
	p.y(portal->getDouble(23));
	theme->canvasLegUpperLeft(p);
	bool n = portal->getBool(24);
	theme->mixColor(n);
	n = portal->getBool(25);
	theme->countObj(n);
	name = portal->getData(26);
	if (!name.empty())
		theme->textTable(name);
	theme->chartFunction(portal->getData(27));
	portal->freeResult();
	delete portal;
	return loadChartParams(db,theme);
}

bool insertThemeApplication (TeDatabase* db, TeThemeApplication *theme, const string& tableName)
{
	if (!db || !theme) 
		return false;
	if (!db->insertTheme((TeTheme*) theme))
		return false;

	string q = "INSERT INTO " + tableName + " (theme_id, visible_out_coll, visible_without_conn, ";
	q += "refine_attribute_where, refine_spatial_where, refine_temporal_where, ";
	q += "grouping_color, grouping_color_direction, grouping_objects, ";
	q += "pie_diameter, pie_max_diameter, pie_min_diameter, pie_dimension_attr, ";
	q += "bar_max_height, bar_min_height, bar_width, ";
	q += "chart_max_value, chart_min_value, chart_type, chart_objects, ";
	q += "keep_dimension, show_canvas_leg, canvas_leg_x, canvas_leg_y, ";
	q += "mix_color, count_obj, text_table, chart_function) VALUES (";

	q += Te2String(theme->id()) + ", ";
	if (theme->outOfCollectionVisibility())
		q += "1, ";
	else
		q += "0, ";
	if (theme->withoutDataConnectionVisibility())
		q += "1, '";
	else
		q += "0, '";
	q += theme->refineAttributeWhere () + "', '" + theme->refineSpatialWhere() + "', '";
	q += theme->refineTemporalWhere () + "', '" + theme->groupColor() + "', ";
	q += Te2String(theme->groupColorDir()) + ", " + Te2String(theme->groupObjects()) + ", ";
	q += Te2String(theme->pieDiameter(),8) + ", " + Te2String(theme->pieMaxDiameter(),8) + ", ";
	q += Te2String(theme->pieMinDiameter(),8) + ", '" + theme->pieDimAttribute() + "', ";
	q += Te2String(theme->barMaxHeight(),8) + ", " + Te2String(theme->barMinHeight(),8) + ", ";
	q += Te2String(theme->barWidth(),8) + ", " + Te2String(theme->chartMaxVal(),8) + ", ";
	q += Te2String(theme->chartMinVal(),8)+ ", " + Te2String(theme->chartType()) + ", ";
	q += Te2String(theme->chartObjects()) + ", "; 
	q += Te2String(theme->keepDimension()) + ", ";
	if (theme->showCanvasLegend())
		q += "1, ";
	else
		q += "0, ";
	q += Te2String(theme->canvasLegWidth(),8) + ", ";
	q += Te2String(theme->canvasLegHeight(),8) + ", ";
	if (theme->mixColor())
		q += "1, ";
	else
		q += "0, ";
	if (theme->countObj())
		q += "1, '";
	else
		q += "0, '";
	q += theme->textTable() + "', '";
	q += theme->chartFunction() + "')";
	if(!db->execute(q))
		return false;

	for (unsigned int i=0; i<theme->chartAttributes_.size(); i++)
	{
		q = "INSERT INTO te_chart_params VALUES (attr_name, red, blue, green, theme_id) VALUES (";
		string attname = theme->chartAttributes_[i];
		map<int, map<string, string> > mapTAA = db->mapThemeAlias();
		map<int, map<string, string> >::iterator tit = mapTAA.find(theme->id());
		if(tit != mapTAA.end())
		{
			map<string, string> mapAA = tit->second;
			theme->concatTableName(attname);
			map<string, string>::iterator it = mapAA.find(attname);
			if(it != mapAA.end())
			{
				string alias = it->second;
				attname += "(" + alias + ")";
			}
		}
		q += "'" + attname + "', " + Te2String(theme->chartColors_[i].red_)+ ", ";
		q += Te2String(theme->chartColors_[i].green_)+ ", " + Te2String(theme->chartColors_[i].blue_) +", ";
		q += Te2String(theme->id()) + ")";
		if (!db->execute(q))
			return false;
	}
	return true; 
}

bool loadViewSet(TeDatabase* db, const string& user)
{
	if (!db)
		return false;

	TeViewMap::iterator it = db->viewMap().begin();
	while (it != db->viewMap().end())
	{
		delete it->second;
		++it;
	}
	db->viewMap().clear();

	
	TeDatabasePortal* portal = db->getPortal();
	if (!portal)
		return false;

	string q = " SELECT * FROM te_view WHERE user_name = '" + user + "'";
	
	if (!portal->query(q))
	{	
		delete portal;
		return false;
	}

	TeDatabasePortal* themePortal = db->getPortal();
	if (!themePortal)
	{	
		delete portal;
		return false;
	}

	while (portal->fetchRow())
	{
		TeView *view = new TeView();
		view->id (atoi(portal->getData("view_id")));
		int projId = atoi(portal->getData("projection_id"));
		TeProjection* proj = db->loadProjection(projId);
		if (proj != 0)
			view->projection(proj);
		view->name(portal->getData("name"));
		view->user(portal->getData("user_name"));
		view->isVisible (portal->getBool("visibility"));
		
		loadViewTree(db, view, 0);

		vector<TeViewNode*>& themes = view->themes();
		for (unsigned int i = 0; i < themes.size(); i++)
		{
			TeTheme *theme = (TeTheme*)themes[i];
			int id = theme->layerId();
			TeLayerMap::iterator it = db->layerMap().find(id);
			if (it == db->layerMap().end())
			{
				db->layerMap().clear();
				db->loadLayerSet();
			}
			theme->layer(db->layerMap()[id]);
		}
		db->viewMap()[view->id()] = view;
		themePortal->freeResult();
	}

	delete themePortal;
	delete portal;
	return true;
}

bool 
updateView(TeDatabase* db, TeView *view)
{
	if(!db) return false;
	if(!view) return false;

	TeProjection* proj = view->projection();
	if (proj)
	{
		if (proj->id() <= 0)
			db->insertProjection(view->projection());
		else
			db->updateProjection(view->projection());
	}
	else
		return false;

	string sql = "UPDATE te_view SET projection_id=" + Te2String(proj->id());
	sql+= ", name='" + view->name() + "'";
	sql+= ", user_name='" + view->user() + "'";
	sql+= ", visibility= " + Te2String((int)view->isVisible());
	sql +=" WHERE view_id = " + Te2String(view->id());

	if (!db->execute (sql))
		return false;

	TeViewTree* tree = view->root();
	if(tree) 
		updateViewTree(db, tree);

	return true;
}

bool updateThemeApplication (TeDatabase* db, TeThemeApplication *theme)
{
	if (!db || !theme) 
		return false;
	if (!db->updateTheme((TeTheme*) theme))
		return false;

	string sql;

	if (theme->id() <= 0 )  // theme doesnt exist in the database yet
	{
		return insertThemeApplication (db, theme);
	}
	
	// update theme metadata
	sql = "UPDATE te_theme_application SET ";
	sql += "  visible_out_coll=";
	if (theme->outOfCollectionVisibility())
		sql += "1";
	else 
		sql += "0";
	sql += ", visible_without_conn=";
	if (theme->withoutDataConnectionVisibility())
		sql += "1";
	else 
		sql += "0";

	sql += ", refine_attribute_where='" + theme->refineAttributeWhere()+"'";
	sql += ", refine_spatial_where='" + theme->refineSpatialWhere()+"'";
	sql += ", refine_temporal_where='" + theme->refineTemporalWhere()+"'";
	sql += ", grouping_color='" + theme->groupColor() + "'";
	sql += ", grouping_color_direction=" + Te2String(theme->groupColorDir());
	sql += ", grouping_objects=" + Te2String (theme->groupObjects());
	sql += ", pie_diameter=" + Te2String(theme->pieDiameter());
	sql += ", pie_max_diameter=" + Te2String(theme->pieMaxDiameter());
	sql += ", pie_min_diameter=" + Te2String(theme->pieMinDiameter());
	sql += ", pie_dimension_attr='" + theme->pieDimAttribute()+"'";
	sql += ", bar_max_height=" + Te2String(theme->barMaxHeight());
	sql += ", bar_min_height=" + Te2String(theme->barMinHeight());
	sql += ", bar_width=" + Te2String(theme->barWidth());
	sql += ", chart_max_value=" + Te2String(theme->chartMaxVal());
	sql += ", chart_min_value=" + Te2String(theme->chartMinVal());
	sql += ", chart_type=" + Te2String (theme->chartType());
	sql += ", chart_objects=" + Te2String (theme->chartObjects());
	sql += ", keep_dimension=" + Te2String(theme->keepDimension());
	sql += ", show_canvas_leg=";
	if (theme->showCanvasLegend())
		sql += "1";
	else 
		sql += "0";
	sql += ", canvas_leg_x=" + Te2String(theme->canvasLegUpperLeft().x());
	sql += ", canvas_leg_y=" + Te2String(theme->canvasLegUpperLeft().y());
	sql += ", mix_color=";
	if (theme->mixColor())
		sql += "1";
	else 
		sql += "0";
	sql += ", count_obj=";
	if (theme->countObj())
		sql += "1";
	else 
		sql += "0";
	sql += ", text_table='" + theme->textTable () + "'";
	sql += ", chart_function='" + theme->chartFunction () + "'";
	sql += " WHERE theme_id=" + Te2String (theme->id());

	if (!db->execute (sql))
		return false;
         
	// remove its chart parameters and reinsert them
	deleteChartParams(db, theme);
	if(!insertChartParams (db, theme))
		return false;

	return true;

//	return theme->mountSqlJoin (false);
}


bool deleteThemeApplication (TeDatabase* db, int themeId)
{
	if (!db || themeId < 1) 
		return false;
	if (!db->deleteTheme(themeId))
		return false;

	string sql = "DELETE FROM te_kernel_metadata WHERE event_theme_id_1 = "+ Te2String(themeId);
	sql += " OR event_theme_id_2 = "+ Te2String(themeId);
	db->execute(sql);

	if (db->existRelation("te_theme_application", "fk_themeAppl_theme_id") != TeRICascadeDeletion)
	{
		// Delete all chart parameters associated to the theme
		string sql = "DELETE FROM te_theme_application WHERE theme_id =" + Te2String(themeId);
		if (!db->execute (sql))
			return false;
	}

	if (db->existRelation("te_chart_params", "fk_chartpar_theme_id") != TeRICascadeDeletion)
	{
		// Delete all chart parameters associated to the theme
		string sql = "DELETE FROM te_chart_params WHERE theme_id =" + Te2String(themeId);
		if (!db->execute (sql))
			return false;
	}
	return true;
}

TeViewTree* 
loadViewTree(TeDatabase* db, TeView* view, int id)
{
	if(db == NULL) return NULL;
	if(view == NULL) return NULL;
	TeDatabasePortal* portal = db->getPortal();
	string q;
	TeViewTree *node = NULL;

	if(id != 0)
	{
		q = "SELECT * FROM te_theme";
		q += " WHERE view_id = " + Te2String (view->id());
		q += " AND theme_id = " + Te2String(id);
		if (!portal->query(q) || portal->fetchRow())
		{
			delete portal;
			return NULL;
		}

		TeViewNodeType type = (TeViewNodeType)atoi(portal->getData("node_type"));
		if(type != TeTREE)
		{
			portal->freeResult();
			delete portal;
			return NULL;
		}

		node = portal->getViewTree();
		portal->freeResult();
	}

	q ="SELECT * FROM te_theme";
	q += " WHERE view_id = " + Te2String (view->id());
	q += " AND parent_id = " + Te2String(id);
	q += " ORDER BY priority ASC";

	if (!portal->query(q))
	{
		delete portal;
		return node;
	}

	while (portal->fetchRow())
	{
		TeViewNodeType childType = (TeViewNodeType)atoi(portal->getData("node_type"));
		TeViewNode *childNode;
		if (childType == TeTHEME)
		{
			childNode = new TeThemeApplication();
			childNode->id(atoi(portal->getData("theme_id")));
			if(loadThemeApplication(db,(TeThemeApplication*) childNode))
				db->themeMap()[childNode->id()] = (TeTheme*)childNode;
			else
				childNode = 0;
		} 
		else
		{
			childNode = loadViewTree(db, view, atoi(portal->getData("theme_id")));
		}

		if((id==0) && (childNode))
		{
			view->add(childNode);
		} 
		else if (childNode)
		{
//			view->addTheme(childNode);
			node->add(childNode);
		}
	}
	
	delete portal;
	return node;
}

bool updateViewTree (TeDatabase* db, TeViewTree *tree)
{
	if(tree->id() > 0)
	{
		string sql;
		sql = "UPDATE te_theme SET ";
		sql += "name='" + tree->name()+"'";
		sql += ", parent_id=" + Te2String (tree->parentId());
		sql += ", node_type=" + Te2String (tree->type());
		sql += " ,priority=" + Te2String (tree->priority());
//		sql += ", visible_rep=" + Te2String (tree->visible());
		sql += " WHERE theme_id = " + Te2String(tree->id());

		if(!db->execute (sql)) return false;
	}

	for (unsigned int th=0; th<tree->size(); th++)
	{
		TeViewNode* node = tree->retrieve(th);
		if (node->type() == TeTHEME)
		{
			TeThemeApplication *theme = (TeThemeApplication*)node;
			if(!updateThemeApplication(db, theme)) return false;
		}
		else
		{
			TeViewTree* tree = (TeViewTree*)node;
			if(!updateViewTree(db, tree)) return false;
		}
	}

	return true;
}

int createTeLayerMetadata(TeDatabase* db)
{
	TeAttributeList attr;
	TeAttribute		at;

	at.rep_.type_ = TeINT;
	at.rep_.name_ = "layer_id";
	at.rep_.isPrimaryKey_ = true;
	attr.push_back(at);

	at.rep_.isPrimaryKey_ = false;
	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "name";
	at.rep_.numChar_ = 128;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "author";
	at.rep_.numChar_ = 128;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "source";
	at.rep_.numChar_ = 255;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "quality";
	at.rep_.numChar_ = 128;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "description";
	at.rep_.numChar_ = 255;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "date_";
	at.rep_.numChar_ = 64;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "hour_";
	at.rep_.numChar_ = 64;
	attr.push_back(at);

	at.rep_.type_ = TeINT;
	at.rep_.name_ = "transf";
	attr.push_back(at);

	if(db->createTable ("te_layer_metadata", attr) == false)
		return 1;

	bool b = db->tableExist("te_layer_metadata");
	while(b == false)
		b = db->tableExist("te_layer_metadata");

	if (db->createRelation("fk_te_layer_met_id", "te_layer_metadata", "layer_id", "te_layer", "layer_id", true) == false)
		return 2;
	return 3;
}

int createTeTableMetadata(TeDatabase* db)
{
	TeAttributeList attr;
	TeAttribute		at;

	at.rep_.type_ = TeINT;
	at.rep_.name_ = "table_id";
	at.rep_.isPrimaryKey_ = true;
	attr.push_back(at);

	at.rep_.isPrimaryKey_ = false;
	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "name";
	at.rep_.numChar_ = 128;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "author";
	at.rep_.numChar_ = 128;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "source";
	at.rep_.numChar_ = 255;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "quality";
	at.rep_.numChar_ = 128;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "description";
	at.rep_.numChar_ = 255;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "date_";
	at.rep_.numChar_ = 64;
	attr.push_back(at);

	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "hour_";
	at.rep_.numChar_ = 64;
	attr.push_back(at);

	at.rep_.type_ = TeINT;
	at.rep_.name_ = "transf";
	attr.push_back(at);

	if(db->createTable ("te_table_metadata", attr) == false)
		return 1;

	bool b = db->tableExist("te_table_metadata");
	while(b == false)
		b = db->tableExist("te_table_metadata");

	if (db->createRelation("fk_te_table_met_id", "te_table_metadata", "table_id", "te_layer_table", "table_id", true) == false)
		return 2;
	return 3;
}


bool deleteTextVisual(TeLayer* layer)
{
	TeDatabase*	db = layer->database();
	TeDatabasePortal*	portal = db->getPortal();

	string sel = "SELECT geom_table FROM te_representation";
	sel += " WHERE layer_id = " + Te2String(layer->id());
	sel += " AND geom_type = " + Te2String(TeTEXT);

	if(portal->query(sel))
	{
		while(portal->fetchRow())
		{
			string s = portal->getData(0);
			string drop = "DROP TABLE " + s + "_txvisual";
			if(db->execute(drop) == false)
			{
				delete portal;
				return false;	
			}
		}
	}
	delete portal;
	return true;
}


bool isTerralibModel(TeDatabase* db)
{
	vector<string>::iterator it;
	vector<string> tables;
	db->listTables(tables);

	for(unsigned int i = 0; i<tables.size(); i++)
	{
		if((it = find(tables.begin(), tables.end(), "te_projection")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_layer")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_layer_table")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_tables_relation")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_representation")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_view")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_theme")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_grouping")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_theme_table")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_legend")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_visual")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_visual_raster")) == tables.end())
			return false;
		if((it = find(tables.begin(), tables.end(), "te_database")) == tables.end())
			return false;
	}
	if(tables.size() == 0)
		return false;

	return true;
}

string insertObjectIntoCollection(TeTheme* theme, TeCoord2D p, string newId, string newTId)
{
	string erro;
	string sel;
	TeLayer* layer = theme->layer();
	TeDatabase* db = layer->database();
	string CT = theme->collectionTable();
	string CAT = theme->collectionAuxTable();

	TeDatabasePortal* portal = db->getPortal();
	if(newId.empty() == false)
	{
		sel = "SELECT * FROM " + CT + " WHERE c_object_id = '" + newId + "'";
		if(portal->query(sel) && portal->fetchRow())
		{
//			erro = "There is already the object identifier:" + newId;
//			erro += " in the collection table:" + CT;
			delete portal;
			return "There is already";
		}
	}

//	if(newTId.empty() == false)
//	{
//		sel = "SELECT * FROM " + CAT + " WHERE aux0 = '" + newTId + "'";
//		portal->freeResult();
//		if(portal->query(sel) && portal->fetchRow())
//		{
//			erro = "There is already the object identifier:" + newTId;
//			erro += " in the collection auxliar table:" + CAT;
//			delete portal;
//			return erro;
//		}
//	}
	delete portal;

	unsigned int i;
	vector<TeTable> staticTableVec, eventTableVec, temporalTableVec;
	TeAttrTableVector layerTablesVector;
	if(layer->getAttrTables(layerTablesVector))
	{
		for(i=0; i<layerTablesVector.size(); i++)
		{
			TeTable t = layerTablesVector[i];
			if(t.tableType() == TeAttrStatic)
				staticTableVec.push_back(t);
			else if(t.tableType() == TeAttrEvent)
				eventTableVec.push_back(t);
			else if(t.tableType() == TeFixedGeomDynAttr)
				temporalTableVec.push_back(t);
		}
	}
	else
	{
		erro = "Fail in the layer method to gets attribute table associated";
		return erro;
	}

	bool ret;
	if(newId.empty() == false)
	{
		string ins = "INSERT INTO " + CT + " (c_object_id, c_legend_id,";
		ins += " label_x, label_y, c_legend_own, c_object_status) VALUES (";
		ins += "'" + newId + "', 0, " + Te2String(p.x()) + ", " + Te2String(p.y());
		ins += ", 0, 1)";
		ret = db->execute(ins);
		if(ret == false)
			return db->errorMessage();

		for(i=0; i<staticTableVec.size(); i++)
		{
			TeTable t = staticTableVec[i];
			ins = "INSERT INTO " + t.name() + " (" + t.linkName() + ")";
			ins += " VALUES ('" + newId + "')";
			ret = db->execute(ins);
			if(ret == false)
				return db->errorMessage();
		}
		for(i=0; i<eventTableVec.size(); i++)
		{
			TeTable t = eventTableVec[i];
			ins = "INSERT INTO " + t.name() + " (" + t.linkName() + ")";
			ins += " VALUES ('" + newId + "')";
			ret = db->execute(ins);
			if(ret == false)
				return db->errorMessage();
		}
	}

	string uniqueId = newId;
	if(newTId.empty() == false)
	{
		uniqueId = newId + newTId;
		string ins = "INSERT INTO " + CAT + " (object_id, aux0, unique_id, grid_status) VALUES (";
		ins += "'" + newId + "', '" + newTId + "', '" + uniqueId + "', 1)";
		ret = db->execute(ins);
		if(ret == false)
			return db->errorMessage();

		for(i=0; i<temporalTableVec.size(); i++)
		{
			TeTable t = temporalTableVec[i];
			ins = "INSERT INTO " + t.name() + " (" + t.linkName() + ", " + t.uniqueName() + ")";
			ins += " VALUES ('" + newId + "', '" + newTId + "')";
			ret = db->execute(ins);
			if(ret == false)
				return db->errorMessage();
		}
	}
	else
	{
		string ins = "INSERT INTO " + CAT + " (object_id, unique_id, grid_status) VALUES (";
		ins += "'" + newId + "', '" + uniqueId + "', 1)";
		ret = db->execute(ins);
		if(ret == false)
			return db->errorMessage();
	}

	return "";
}

string deletePointedObjects(TeTheme* theme, TeBox& box)
{
	unsigned int i;
	TeLayer* layer = theme->layer();		
	TeDatabase* db = layer->database();
	TeThemeMap&	themeMap = db->themeMap();
	string CT = theme->collectionTable();
	string CAT = theme->collectionAuxTable();
	string tsel, sel, del;
	vector<TeTable> table;
	vector<TeTable> ttable;
	vector<string> selVec;
	vector<string> selAuxVec;

	TeAttrTableVector layerTablesVector;
	layer->getAttrTables(layerTablesVector);

// delete temporal instances of TeFixedGeomDynAttr tables
	for(i=0; i<layerTablesVector.size(); i++)
	{
		TeTable t = layerTablesVector[i];
		TeAttrTableType type = t.tableType();

		if(type == TeFixedGeomDynAttr)
			ttable.push_back(t);
	}

	if(ttable.size())
	{
		tsel = "(SELECT aux0 FROM " + CAT;
		tsel += " WHERE grid_status = 1 OR grid_status = 3)";

		for(i=0; i<ttable.size(); i++)
		{
			del = "DELETE FROM " + ttable[i].name() + " WHERE " + ttable[i].uniqueName() + " IN " + tsel;
			if(db->execute(del) == false)
				return db->errorMessage();
		}
	}

// get selected object ids
	TeDatabasePortal* portal = db->getPortal();
	vector<string> oidVec;
	sel = "SELECT DISTINCT object_id FROM " + CAT;
	sel += " WHERE (grid_status = 1 OR grid_status = 3) AND object_id NOT IN";
	sel += "(SELECT DISTINCT object_id FROM " + CAT;
	sel += " WHERE grid_status = 0 OR grid_status = 2)";
	portal->freeResult();
	if(portal->query(sel))
	{
		while(portal->fetchRow())
		{
			string oid = portal->getData(0);
			oidVec.push_back(oid);
		}
	}

	sel = "SELECT * FROM " + CT + " as T LEFT JOIN " + CAT + " as TT";
	sel += " ON T.c_object_id = TT.object_id WHERE TT.grid_status is null";
	sel += " AND (T.c_object_status = 1 OR T.c_object_status = 3)";
	portal->freeResult();
	if(portal->query(sel))
	{
		while(portal->fetchRow())
		{
			string oid = portal->getData(0);
			oidVec.push_back(oid);
		}
	}
	vector<string>::iterator itB = oidVec.begin();
	vector<string>::iterator itE = oidVec.end();
	selVec = generateInClauses(itB,itE, db, true);

// delete attributes of TeAttrStatic tables and TeAttrEvent tables
	for(i=0; i<layerTablesVector.size(); i++)
	{
		TeTable t = layerTablesVector[i];
		TeAttrTableType type = t.tableType();

		if(type == TeAttrStatic || type == TeAttrEvent)
			table.push_back(t);
	}

	for(i=0; i<table.size(); i++)
	{
		for(i=0; i<selVec.size(); i++)
		{
			del = "DELETE FROM " + table[i].name() + " WHERE ";
			del += table[i].linkName() + " IN " + selVec[i];
			if(db->execute(del) == false)
			{
				delete portal;
				return db->errorMessage();
			}
		}
	}

	double minX = TeMAXFLOAT, minY = TeMAXFLOAT, maxX = -TeMAXFLOAT, maxY = -TeMAXFLOAT;
// delete selected geometries
	if(layer->geomRep() & TePOLYGONS)
	{
		string geo = layer->tableName(TePOLYGONS);
		for(i=0; i<selVec.size(); i++)
		{
			string boxsel = "SELECT * FROM " + geo;
			boxsel += " WHERE object_id IN " + selVec[i];

			portal->freeResult();
			if(portal->query(boxsel))
			{
				while(portal->fetchRow())
				{
					double x1 = portal->getDouble("lower_x");
					double y1 = portal->getDouble("lower_y");
					double x2 = portal->getDouble("upper_x");
					double y2 = portal->getDouble("upper_y");
					TeBox b(x1, y1, x2, y2);
					box = TeUnion(box, b);
				}
			}
			del = "DELETE FROM " + geo + " WHERE object_id IN " + selVec[i];
			if(db->execute(del) == false)
			{
				delete portal;
				return db->errorMessage();
			}
		}
		string bb = "SELECT MIN(lower_x), MIN(lower_y), MAX(upper_x), MAX(upper_y) FROM " + geo;
		portal->freeResult();
		if(portal->query(bb) && portal->fetchRow())
		{
			minX = MIN(minX, portal->getDouble(0));
			minY = MIN(minY, portal->getDouble(1));
			maxX = MAX(maxX, portal->getDouble(2));
			maxY = MAX(maxY, portal->getDouble(3));
		}
	}
	if(layer->geomRep() & TePOINTS)
	{
		string geo = layer->tableName(TePOINTS);
		for(i=0; i<selVec.size(); i++)
		{
			string boxsel = "SELECT * FROM " + geo;
			boxsel += " WHERE object_id IN " + selVec[i];

			portal->freeResult();
			if(portal->query(boxsel))
			{
				while(portal->fetchRow())
				{
					double x = portal->getDouble("x");
					double y = portal->getDouble("y");
					TeBox b(x, y, x, y);
					box = TeUnion(box, b);
				}
			}
			del = "DELETE FROM " + geo + " WHERE object_id IN " + selVec[i];
			if(db->execute(del) == false)
			{
				delete portal;
				return db->errorMessage();
			}
		}
		string bb = "SELECT MIN(x), MIN(y), MAX(x), MAX(y) FROM " + geo;
		portal->freeResult();
		if(portal->query(bb) && portal->fetchRow())
		{
			minX = MIN(minX, portal->getDouble(0));
			minY = MIN(minY, portal->getDouble(1));
			maxX = MAX(maxX, portal->getDouble(2));
			maxY = MAX(maxY, portal->getDouble(3));
		}
	}
	if(layer->geomRep() & TeCELLS)
	{
		string geo = layer->tableName(TeCELLS);
		for(i=0; i<selVec.size(); i++)
		{
			string boxsel = "SELECT * FROM " + geo;
			boxsel += " WHERE object_id IN " + selVec[i];

			portal->freeResult();
			if(portal->query(boxsel))
			{
				while(portal->fetchRow())
				{
					double x1 = portal->getDouble("lower_x");
					double y1 = portal->getDouble("lower_y");
					double x2 = portal->getDouble("upper_x");
					double y2 = portal->getDouble("upper_y");
					TeBox b(x1, y1, x2, y2);
					box = TeUnion(box, b);
				}
			}
			del = "DELETE FROM " + geo + " WHERE object_id IN " + selVec[i];
			if(db->execute(del) == false)
			{
				delete portal;
				return db->errorMessage();
			}
		}
		string bb = "SELECT MIN(lower_x), MIN(lower_y), MAX(upper_x), MAX(upper_y) FROM " + geo;
		portal->freeResult();
		if(portal->query(bb) && portal->fetchRow())
		{
			minX = MIN(minX, portal->getDouble(0));
			minY = MIN(minY, portal->getDouble(1));
			maxX = MAX(maxX, portal->getDouble(2));
			maxY = MAX(maxY, portal->getDouble(3));
		}
	}
	if(layer->geomRep() & TeLINES)
	{
		string geo = layer->tableName(TePOINTS);
		for(i=0; i<selVec.size(); i++)
		{
			string boxsel = "SELECT * FROM " + geo;
			boxsel += " WHERE object_id IN " + selVec[i];

			portal->freeResult();
			if(portal->query(boxsel))
			{
				while(portal->fetchRow())
				{
					double x1 = portal->getDouble("lower_x");
					double y1 = portal->getDouble("lower_y");
					double x2 = portal->getDouble("upper_x");
					double y2 = portal->getDouble("upper_y");
					TeBox b(x1, y1, x2, y2);
					box = TeUnion(box, b);
				}
			}
			del = "DELETE FROM " + geo + " WHERE object_id IN " + selVec[i];
			if(db->execute(del) == false)
			{
				delete portal;
				return db->errorMessage();
			}
		}
		string bb = "SELECT MIN(lower_x), MIN(lower_y), MAX(upper_x), MAX(upper_y) FROM " + geo;
		portal->freeResult();
		if(portal->query(bb) && portal->fetchRow())
		{
			minX = MIN(minX, portal->getDouble(0));
			minY = MIN(minY, portal->getDouble(1));
			maxX = MAX(maxX, portal->getDouble(2));
			maxY = MAX(maxY, portal->getDouble(3));
		}
	}

	double dx = maxX - minX;
	double dy = maxY - minY;
	minX -= dx/20.;
	maxX += dx/20.;
	minY -= dy/20.;
	maxY += dy/20.;

	string up = "UPDATE te_layer SET lower_x = " + Te2String(minX);
	up += ", lower_y = " + Te2String(minY);
	up += ", upper_x = " + Te2String(maxX);
	up += ", upper_y = " + Te2String(maxY);
	up += " WHERE layer_id = " + Te2String(layer->id());
	if(db->execute(up) == false)
	{
		delete portal;
		return db->errorMessage();
	}
	TeBox nbox(minX, minY, maxX, maxY);
	layer->setLayerBox(nbox);

// get collection names of all themes that use this layer
	vector<string> CATFVec;
	vector<string> CATVec;
	vector<string> CTVec;
	string ss = "SELECT theme_id FROM te_theme WHERE layer_id = " + Te2String(layer->id());
	portal->freeResult();
	if(portal->query(ss))
	{
		while(portal->fetchRow())
		{
			int id = atoi(portal->getData(0));
			if(id != theme->id())
			{
				TeTheme* t = themeMap[id];
				TeAttrTableVector& tableVec = t->attrTables();
				for(i=0; i<tableVec.size(); i++)
				{
					string s = t->collectionAuxTable();
					if(tableVec[i].tableType() == TeFixedGeomDynAttr)
						CATFVec.push_back(s);
					else
						CATVec.push_back(s);
				}
				string s = t->collectionTable();
				CTVec.push_back(s);
			}
		}
	}
	delete portal;

// delete selected objects of the collections of other themes
	for(i=0; i<CTVec.size(); i++)
	{
		for(i=0; i<selVec.size(); i++)
		{
			del = "DELETE FROM " + CTVec[i] + " WHERE c_object_id IN " + selVec[i];
			if(db->execute(del) == false)
				return db->errorMessage();
		}
	}

// delete selected objects of the collection this theme
	for(i=0; i<selVec.size(); i++)
	{
		del = "DELETE FROM " + CT + " WHERE c_object_id IN " + selVec[i];
		if(db->execute(del) == false)
			return db->errorMessage();
	}

	up = "UPDATE " + CT + " SET c_object_status = c_object_status - 1 WHERE (c_object_status = 1 OR c_object_status = 3)";
	if(db->execute(up) == false)
		return db->errorMessage();

// delete selected instances of the auxiliar collection of other themes (only TeFixedGeomDynAttr)
	if(ttable.size())
	{
		for(i=0; i<CATFVec.size(); i++)
		{
			del = "DELETE FROM " + CATFVec[i] + " WHERE aux0 IN " + tsel;
			if(db->execute(del) == false)
				return db->errorMessage();
		}
	}

// delete selected instances of the auxiliar collection of other themes (TeAttrStatic and TeAttrEvent)
	unsigned int j;
	for(i=0; i<CATVec.size(); i++)
	{
		for(j=0; j<selVec.size(); j++)
		{
			del = "DELETE FROM " + CATVec[i] + " WHERE object_id IN " + selVec[j];
			if(db->execute(del) == false)
				return db->errorMessage();
		}
	}

// delete selected instances of the auxiliar collection this theme
	del = "DELETE FROM " + CAT + " WHERE grid_status = 1 OR grid_status = 3";
	if(db->execute(del) == false)
		return db->errorMessage();

	return "";
}

bool TeCopyDatabase(TeDatabase* dbFrom, TeDatabase* dbTo)
{
	// -- Step 1 : copies the external tables
	TeAttrTableVector externs;
	dbFrom->getAttrTables(externs,TeAttrExternal);
	if (!externs.empty())
	{
		TeDatabasePortal* portal = dbFrom->getPortal();
		if (!portal)
		{
			// log error here
			return false;
		}

		for (unsigned int nt=0; nt < externs.size(); nt++)
		{
			TeTable table = externs[nt];
			table.setId(-1);
			string sql = "select * from " + table.name();
			if (!portal->query(sql) || portal->numRows() == 0)
			{
				portal->freeResult();
				continue;
			}
			if (!dbTo->createTable(table.name(),table.attributeList()))
			{
				portal->freeResult();
				continue;
			}
			unsigned int nr=0;
			while (portal->fetchRow())
			{
				TeTableRow row;
				for(int i = 0; i < portal->numFields(); i++)
					row.push_back(portal->getData(i));
				table.add(row);
				nr++;
				if (nr % 200)
				{
					dbTo->insertTable(table);
					table.clear();
					nr = 0;
				}
			}
			if (table.size() >0)
				dbTo->insertTable(table);
				table.clear();	
			dbTo->insertTableInfo(-1,table);
			portal->freeResult();
		}
		delete portal;
	}

	// -- Step 2: Copies the layers
	if(!dbFrom->loadLayerSet())
	{
		// log error here
		return false;
	}

	TeLayerMap& fromLayerMap = dbFrom->layerMap();
	TeLayerMap::iterator itFrom = fromLayerMap.begin();

	while(itFrom != fromLayerMap.end())
	{
		TeLayer* fromLayer = (*itFrom).second;
		TeLayer* toLayer = new TeLayer(fromLayer->name(), dbTo, fromLayer->projection());
		map<string, string> tables;
		if (!TeCopyLayerToLayer(fromLayer, toLayer, &tables))
		{
			// log error here: layer couldnt be copied
		}
		++itFrom;
	}
	// -- Set 3: Copies the views and themes
	if(!dbFrom->loadViewSet(dbFrom->user()))
	{
		// log error here
		return false;
	}

	if(!dbTo->loadLayerSet())
	{
		// log error here
		return false;
	}

	TeViewMap&	fromViewMap = dbFrom->viewMap();
	TeViewMap::iterator itvFrom = fromViewMap.begin();
	while (itvFrom != fromViewMap.end())
	{
		TeView* fromView = (*itvFrom).second;			
		TeView* toView = new TeView();

		TeProjection* toViewProjection = 0;
		if (fromView->projection())
			 toViewProjection = TeProjectionFactory::make(fromView->projection()->params());
		
		toView->projection(toViewProjection);
		toView->name(fromView->name());
		toView->user(dbTo->user());
		toView->isVisible(fromView->isVisible());

		if (!dbTo->insertView(toView))
		{
			// log error here
			++itvFrom;
			continue;
		}

		TeLayerMap& toLayerMap = dbTo->layerMap();			
		vector<TeViewNode*>& themeVec = fromView->themes();
		for (unsigned int i = 0; i < themeVec.size(); ++i)
		{
			TeTheme* themeFrom = (TeTheme*) themeVec[i];
			string fromLayerName = themeFrom->layer()->name();
			TeLayer* toLayer = 0;
			TeLayerMap::iterator itTo = toLayerMap.begin();
			while(itTo != toLayerMap.end())
			{
				if(itTo->second->name() == fromLayerName)
				{
					toLayer = itTo->second;
					break;
				}
				++itTo;
			}

			if (!toLayer )
			{
				// log error here
				continue;
			}
			
			TeTheme* themeTo = new TeTheme(themeFrom->name(), toLayer);
			toView->add(themeTo);
						
			themeTo->outOfCollectionLegend(themeFrom->outOfCollectionLegend());
			themeTo->withoutDataConnectionLegend(themeFrom->withoutDataConnectionLegend ());
			themeTo->defaultLegend(themeFrom->defaultLegend());
			themeTo->pointingLegend(themeFrom->pointingLegend());
			themeTo->queryLegend(themeFrom->queryLegend());

			TeAttrTableVector tftablevec = themeFrom->attrTables();
			TeAttrTableVector tttablevec;

			for (unsigned int nt=0; nt<tftablevec.size(); nt++)
			{
				TeTable attTable(tftablevec[nt].name());
				dbTo->loadTableInfo(attTable);
				tttablevec.push_back(attTable);
			}
			themeTo->setAttTables(tttablevec);
			themeTo->attributeRest(themeFrom->attributeRest());
			themeTo->temporalRest(themeFrom->temporalRest());
			themeTo->spatialRest(themeFrom->spatialRest());
			themeTo->visibleRep(themeFrom->visibleRep());
			if(!themeTo->save() || !themeTo->buildCollection())
			{
				// log error here
				continue;
			}
			themeTo->generateLabelPositions();
			
			if(themeFrom->grouping())
			{
				TeGrouping* grouping = new TeGrouping();
				(*grouping) = (*themeFrom->grouping());
				
				themeTo->buildGrouping(grouping);
				TeLegendEntryVector& legends = themeFrom->legend();
				for (unsigned int nl=0; nl<legends.size(); nl++)
					themeTo->setGroupingVisual(nl+1,legends[nl].getVisualMap());
				if (!themeTo->saveGrouping())
				{
						// log error here
				}
			}
		} // end for each theme
		++itvFrom;
	} // end for each view
	return true;
}



