/*
 * Core tagged collection handling
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma implementation

#include <tagcoll/TagCollection.h>
#include <tagcoll/stringf.h>

#include <iterator>
#include <algorithm>
#include <vector>

//#define TRACE
//#define TRACE0
//#define TRACE1

#ifdef TRACE
#include <cstdio>
#include "stringf.h"
#endif

using namespace std;
using namespace Tagcoll;

//template class TagCollection<int, string>;
//template class TagCollection<string, string>;
//namespace Debtags { class Package; }
//template class TagCollection<Debtags::Package*, string>;

static string mergeTags(const string& tag1, const string& tag2) throw ()
{
	// In case of strings, comma-separate them
	return tag1 + ", " + tag2;
}

template<class T>
static T mergeTags(const T& tag1, const T& tag2) throw ()
{
	// By default, just return the first tag
	return tag1;
}

template<class ITEM, class TAG>
typename TagCollection<ITEM, TAG>::Change TagCollection<ITEM, TAG>::applyChange(
		const TagCollection<ITEM, TAG>::Change& change) throw ()
{
	Change rev;

	// Build an item set of the items to change, for quicker lookup
	OpSet<ITEM> tochange;
	for (typename Change::const_iterator i = change.begin(); i != change.end(); i++)
	{
		tochange += i->first; 

		// Handle the untagged items
		if (i->second.empty())
			untagged.insert(i->first);
		else if (untagged.contains(i->first))
		{
			untagged.erase(i->first);
			rev.insert(make_pair(i->first, OpSet<TAG>()));
		}
	}
	
	// Handle the tagged items
	list< OpSet<TAG> > deleted;
	for (typename tagsets_t::iterator i = tagsets.begin();
			i != tagsets.end(); i++)
	{
		OpSet<ITEM> involved = i->second ^ tochange;
		if (!involved.empty())
		{
			// Create the reverse patch
			for (typename OpSet<ITEM>::const_iterator j = involved.begin();
					j != involved.end(); j++)
				rev.insert(make_pair(*j, i->first));

			// Remove the items from their original tagsets
			i->second -= involved;

			// Remember the tagsets that have become empty
			if (i->second.empty())
				deleted.push_back(i->first);

			// Decrement the tag cardinalities accordingly
			for (typename OpSet<TAG>::const_iterator j = i->first.begin();
					j != i->first.end(); j++)
				tags.del(*j, 1);
		}
	}

	for (typename list< OpSet<TAG> >::const_iterator i = deleted.begin();
			i != deleted.end(); i++)
		tagsets.erase(*i);

	// Re-add the tags with the new tagsets
	for (typename Change::const_iterator i = change.begin(); i != change.end(); i++)
		if (!i->second.empty())
			add(i->second, i->first);

	return rev;
}

template<class ITEM, class TAG>
TagcollChange<ITEM, TAG> TagCollection<ITEM, TAG>::applyChange(
		const TagcollChange<ITEM, TAG>& change) throw ()
{
	TagcollChange<ITEM, TAG> rev;

	// Build an item set of the items to change, for quicker lookup
	OpSet<ITEM> tochange;
	for (typename TagcollChange<ITEM, TAG>::const_iterator i = change.begin(); i != change.end(); i++)
	{
		tochange += i->first; 

		// Handle the untagged items
		if (i->second.empty())
			untagged.insert(i->first);
		else if (untagged.contains(i->first))
		{
			untagged.erase(i->first);
			rev.insert(make_pair(i->first, OpSet<TAG>()));
		}
	}
	
	// Handle the tagged items
	list< OpSet<TAG> > deleted;
	for (typename tagsets_t::iterator i = tagsets.begin();
			i != tagsets.end(); i++)
	{
		OpSet<ITEM> involved = i->second ^ tochange;
		if (!involved.empty())
		{
			// Create the reverse patch
			for (typename OpSet<ITEM>::const_iterator j = involved.begin();
					j != involved.end(); j++)
				rev.insert(make_pair(*j, i->first));

			// Remove the items from their original tagsets
			i->second -= involved;

			// Remember the tagsets that have become empty
			if (i->second.empty())
				deleted.push_back(i->first);

			// Decrement the tag cardinalities accordingly
			for (typename OpSet<TAG>::const_iterator j = i->first.begin();
					j != i->first.end(); j++)
				tags.del(*j, 1);
		}
	}

	for (typename list< OpSet<TAG> >::const_iterator i = deleted.begin();
			i != deleted.end(); i++)
		tagsets.erase(*i);

	// Re-add the tags with the new tagsets
	for (typename TagcollChange<ITEM, TAG>::const_iterator i = change.begin(); i != change.end(); i++)
		if (!i->second.empty())
			add(i->second, i->first);

	return rev;
}


template<class ITEM, class TAG>
void TagCollection<ITEM, TAG>::TagContainer::add(const TAG& tag, int card) throw ()
{
	typename TagContainer::iterator i = find(tag);
	if (i == TagContainer::end())
		insert(make_pair(tag, card));
	else
		i->second += card;
}

template<class ITEM, class TAG>
void TagCollection<ITEM, TAG>::TagContainer::del(const TAG& tag, int card) throw ()
{
	typename TagContainer::iterator i = find(tag);
	if (i != TagContainer::end())
		if (i->second > card)
			i->second -= card;
		else
			erase(tag);
}

template<class ITEM, class TAG>
void TagCollection<ITEM, TAG>::add(const ITEM& item) throw ()
{
#ifdef TRACE
	fprintf(stderr, "Add untagged: %d\n", item);
#endif
	untagged.insert(item);
}

template<class ITEM, class TAG>
void TagCollection<ITEM, TAG>::add(const OpSet<ITEM>& items) throw ()
{
#ifdef TRACE
	fprintf(stderr, "Add untagged: %d items\n", items.size());
#endif
	if (untagged.size() > 0)
		untagged += items;
	else
		untagged = items;
}

template<class ITEM, class TAG>
void TagCollection<ITEM, TAG>::add(const OpSet<TAG>& tagset, const ITEM& item) throw ()
{
	if (tagset.size() == 0)
		return;

	// Check if tagset is already present
	bool added;
	typename tagsets_t::iterator i = tagsets.find(tagset);
	if (i != tagsets.end())
	{
		// If it already exists, insert the item
		pair<typename OpSet<ITEM>::iterator, bool> r = i->second.insert(item);
		added = r.second;
	} else {
		// Else insert the item as a new set
		OpSet<ITEM> items;
		items += item;
		tagsets.insert(make_pair(tagset, items));
		added = true;
	}

	// Update tags accordingly
	if (added)
		for (typename OpSet<TAG>::const_iterator i = tagset.begin();
				i != tagset.end(); i++)
			tags.add(*i, 1);
}

template<class ITEM, class TAG>
void TagCollection<ITEM, TAG>::add(const OpSet<TAG>& tagset, const OpSet<ITEM>& items) throw ()
{
	if (tagset.size() == 0 || items.size() == 0)
		return;

	// Check if tagset is already present
	int inserted = 0;
	typename tagsets_t::iterator i = tagsets.find(tagset);
	if (i != tagsets.end())
	{
		// If it already exists, merge the items
		for (typename OpSet<ITEM>::const_iterator j = items.begin();
				j != items.end(); j++)
		{
			pair<typename OpSet<ITEM>::iterator, bool> r = i->second.insert(*j);
			if (r.second)
				inserted++;
		}
	} else {
		// Else insert them
		tagsets.insert(make_pair(tagset, items));
		inserted = items.size();
	}

	// Update tags accordingly
	for (typename OpSet<TAG>::const_iterator i = tagset.begin();
			i != tagset.end(); i++)
		tags.add(*i, inserted);
}

template<class ITEM, class TAG>
int TagCollection<ITEM, TAG>::totalCount() const throw ()
{
	int res = 0;

	for (typename tagsets_t::const_iterator t = tagsets.begin();
			t != tagsets.end(); t++)
		res += t->second.size();

	return res;
}

template<class ITEM, class TAG>
OpSet<ITEM> TagCollection<ITEM, TAG>::getItemsForTagset(const OpSet<TAG>& ts) const throw ()
{
	if (ts.empty())
		return getUntaggedItems();

	typename tagsets_t::const_iterator i = tagsets.find(ts);
	if (i == tagsets.end())
		return OpSet<ITEM>();
	else
		return i->second;
}

template<class ITEM, class TAG>
OpSet<TAG> TagCollection<ITEM, TAG>::getTagsetForItem(const ITEM& item) const throw ()
{
	for (typename tagsets_t::const_iterator t = tagsets.begin();
			t != tagsets.end(); t++)
		if (t->second.contains(item))
			return t->first;
	return OpSet<TAG>();
}

template<class ITEM, class TAG>
OpSet<TAG> TagCollection<ITEM, TAG>::getAllTags() const throw ()
{
	OpSet<TAG> res;

	for (typename TagContainer::const_iterator t = tags.begin();
			t != tags.end(); t++)
		res += t->first;

	return res;
}

template<class ITEM, class TAG>
OpSet<ITEM> TagCollection<ITEM, TAG>::getAllItems() const throw ()
{
	OpSet<ITEM> res = untagged;

	for (typename tagsets_t::const_iterator t = tagsets.begin();
			t != tagsets.end(); t++)
		res += t->second;

	return res;
}

template<class ITEM, class TAG>
OpSet<TAG> TagCollection<ITEM, TAG>::getCompanionTags(const OpSet<TAG>& ts) const throw ()
{
	OpSet<TAG> res;

	for (typename tagsets_t::const_iterator t = tagsets.begin();
			t != tagsets.end(); t++)
		if (t->first.contains(ts))
			res += t->first - ts;

	return res;
}

template<class ITEM, class TAG>
OpSet<ITEM> TagCollection<ITEM, TAG>::getCompanionItems(const OpSet<TAG>& ts) const throw ()
{
	OpSet<ITEM> res;

	for (typename tagsets_t::const_iterator t = tagsets.begin();
			t != tagsets.end(); t++)
		if (t->first.contains(ts))
			res += t->second;

	return res;
}

template<class ITEM, class TAG>
map<ITEM, OpSet<TAG> > TagCollection<ITEM, TAG>::getCompanionItemsAndTagsets(const OpSet<TAG>& ts) const throw ()
{
	map<ITEM, OpSet<TAG> > res;

	for (typename tagsets_t::const_iterator t = tagsets.begin();
			t != tagsets.end(); t++)
		if (t->first.contains(ts))
			for (typename set<ITEM>::const_iterator i = t->second.begin();
					i != t->second.end(); i++)
				res.insert(make_pair(*i, t->first));

	return res;
}

template<class ITEM, class TAG>
list< OpSet<TAG> > TagCollection<ITEM, TAG>::getRelatedTagsets(const OpSet<TAG>& ts, int maxdistance) const throw ()
{
	list< OpSet<TAG> > res;

	for (typename tagsets_t::const_iterator t = tagsets.begin();
			t != tagsets.end(); t++)
	{
		int dist = ts.distance(t->first);
		if (dist > 0 && dist <= maxdistance)
			res.push_back(t->first);
	}

	return res;
}
	

template<class ITEM, class TAG>
TagCollection<ITEM, TAG> TagCollection<ITEM, TAG>::getChildCollection(const TAG& tag) const throw ()
{
	TagCollection<ITEM, TAG> res;
	for (typename tagsets_t::const_iterator ts = tagsets.begin();
			ts != tagsets.end(); ts++)
	{
		if (ts->first.find(tag) == ts->first.end())
			continue;
		// for all tagsets ts having tag

		// Insert the tagset, without tag
		OpSet<TAG> newts = ts->first;
		newts.erase(tag);

		// Check if we make orphans, and preserve them
		if (newts.size() == 0 && ts->second.size() > 0)
			res.add(ts->second);
		else
			res.add(newts, ts->second);
	}
	return res;
}

template<class ITEM, class TAG>
OpSet<TAG> TagCollection<ITEM, TAG>::getImplyingOneOf(const OpSet<TAG>& tags) const throw ()
{
	OpSet<TAG> allWith;
	OpSet<TAG> allWithout;

	for (typename tagsets_t::const_iterator ts = tagsets.begin();
			ts != tagsets.end(); ts++)
	{
		OpSet<TAG> intersection = ts->first ^ tags;

		if (intersection.size() == 0)
			allWithout += ts->first;
		else
			allWith += ts->first;
	}

	return allWith - allWithout;
}

template<class ITEM, class TAG>
OpSet<TAG> TagCollection<ITEM, TAG>::getImpliedBy(const TAG& tag) const throw ()
{
	// The tags implied by `tag' are the result of the intersection between all
	// tagsets that contain tag

	typename tagsets_t::const_iterator ts = tagsets.begin();
	// Find the first tagset that contains `tag'
	for ( ; ts != tagsets.end() && ts->first.find(tag) == ts->first.end(); ts++)
		;
	if (ts == tagsets.end())
		return OpSet<TAG>();

	// Initialize the intersection set with it
	OpSet<TAG> res = ts->first;
	// Remove tag
	res -= tag;

	for (++ts; res.size() > 0 && ts != tagsets.end(); ts++)
		if (ts->first.find(tag) == ts->first.end())
			continue;
		else
			// For all tagsets that contain tag
			// Intersect them with `res'
			res = ts->first ^ res;

	return res;
}

template<class ITEM, class TAG>
TagCollection<ITEM, TAG> TagCollection<ITEM, TAG>::getCollectionWithoutTags(const OpSet<TAG>& tags) const throw ()
{
	OpSet<TAG> candidates = getImplyingOneOf(tags);

	// remove all tags contained in `candidates', merging categories that
	// result with the same tagset;

	TagCollection<ITEM, TAG> res;
	for (typename tagsets_t::const_iterator ts = tagsets.begin();
			ts != tagsets.end(); ts++)
	{
		OpSet<TAG> newts = ts->first - candidates;

		if (newts.size() > 0)
		{
			// Insert the newfound tagset in the new collection
			//res.add(newts, ts->second);
			res.add(ts->first, ts->second);
		}
	}
	if (untagged.size() > 0)
		res.add(untagged);
	return res;
}

template<class ITEM, class TAG>
TagCollection<ITEM, TAG> TagCollection<ITEM, TAG>::getCollectionWithoutTagsetsHaving(const TAG& tag) const throw ()
{
	TagCollection<ITEM, TAG> res;
	for (typename tagsets_t::const_iterator ts = tagsets.begin();
			ts != tagsets.end(); ts++)
	{
		if (ts->first.find(tag) != ts->first.end())
			continue;
		// For all tagsets that do not contain tag

		// Insert the selected tagset in the new collection
		res.add(ts->first, ts->second);
	}
	if (untagged.size() > 0)
		res.add(untagged);
	return res;
}

template<class ITEM, class TAG>
TagCollection<ITEM, TAG> TagCollection<ITEM, TAG>::getCollectionWithoutTagsetsHavingAnyOf(const OpSet<TAG>& tags) const throw ()
{
	TagCollection<ITEM, TAG> res;
	for (typename tagsets_t::const_iterator ts = tagsets.begin();
			ts != tagsets.end(); ts++)
	{
		OpSet<TAG> inters = ts->first ^ tags;
		if (!inters.empty())
			continue;
		// For all tagsets that do not contain some of the tags in `tag`

		// Insert the selected tagset in the new collection
		res.add(ts->first, ts->second);
	}
	if (untagged.size() > 0)
		res.add(untagged);
	return res;
}

template<class ITEM, class TAG>
TAG TagCollection<ITEM, TAG>::findTagWithMaxCardinalityNotIn(const OpSet<TAG>& ntags, int* card) const throw ()
{
	TAG candidate;
	int maxcard = 0;

	for (typename TagContainer::const_iterator i = tags.begin();
			i != tags.end(); i++)
		if (!ntags.contains(i->first) && i->second > maxcard)
		{
			maxcard = i->second;
			candidate = i->first;
		}
	
	if (card)
		*card = maxcard;
	return candidate;
}

template<class ITEM, class TAG>
void TagCollection<ITEM, TAG>::mergeEquivalentTags() throw ()
{
//fprintf(stderr, "mergeEquivalentTags\n");
	// Build a card->tags mapping
	map<int, OpSet<TAG> > cards;
	for (typename TagContainer::const_iterator t = tags.begin();
			t != tags.end(); t++)
		cards[t->second] += t->first;

	// Look for tags with the same cardinality
	map<TAG, TAG> renames;
	OpSet<TAG> items_to_rename;
	for (typename map<int, OpSet<TAG> >::const_iterator c = cards.begin();
			c != cards.end(); c++)
		if (c->second.size() > 1)
		{
//fprintf(stderr, "Found a set of %d tags with card %d\n", c->second.size(), c->first);
			
			// Find out equivalent items
			//
			// Intersect every set of tags with the same cardinality
			// with all tagsets in the collection which gives non-null
			// intersection.  The tags in the resulting set are equivalent.
			OpSet<TAG> equivs = c->second;
			for (typename tagsets_t::const_iterator i = tagsets.begin();
					i != tagsets.end() && equivs.size() > 1; i++)
			{
				OpSet<TAG> intersection = c->second ^ i->first;
				if (!intersection.empty())
					equivs = intersection;
			}
//fprintf(stderr, "Final intersection gives %d nodes\n", equivs.size());

			if (equivs.size() > 1)
			{
				// equivs contains a set of equivalent tags
	
				// Schedule the renaming of equivalent tags into their merged
				// tag

				// Compute the merged name
				typename OpSet<TAG>::const_iterator j = equivs.begin();
				TAG merged = *j;
				int card = tags[*j];
				for (j++; j != equivs.end(); j++)
					merged = mergeTags(merged, *j);

//string mname = stringf::fmt(merged);
//fprintf(stderr, "Merged name: %.*s\n", PFSTR(mname));

				// Insert the merged item in `tags'
				tags.insert(make_pair(merged, card));

				// Store the renames to be done
				// Remove the now-merged items from `tags'
				for (typename OpSet<TAG>::const_iterator j = equivs.begin();
						j != equivs.end(); j++)
				{
					renames.insert(make_pair(*j, merged));
					items_to_rename += *j;
					tags.erase(*j);
				}
			}
		}
	
	if (renames.size() > 0)
	{
		// Perform the renames

		// Find the tagsets that need rename
		vector< OpSet<TAG> > to_rename;
		for (typename tagsets_t::const_iterator i = tagsets.begin(); i != tagsets.end(); i++)
		{
			OpSet<TAG> inters = i->first ^ items_to_rename;
			if (!inters.empty())
				to_rename.push_back(i->first);
		}

		// Rename the tags inside the selected tagsets
		for (typename vector< OpSet<TAG> >::const_iterator i = to_rename.begin();
				i != to_rename.end(); i++)
		{
			// Store the old tag data and remove it from tagsets
			typename tagsets_t::iterator t = tagsets.find(*i);
			OpSet<TAG> tags = t->first;
			OpSet<ITEM> items = t->second;
			tagsets.erase(t);
			
			// Compute the renamed tagset
			OpSet<TAG> newtags;
			for (typename OpSet<TAG>::const_iterator r = tags.begin();
					r != tags.end(); r++)
			{
				typename map<TAG, TAG>::const_iterator ren = renames.find(*r);
				if (ren == renames.end())
					newtags += *r;
				else
					newtags += ren->second;
			}

			// Insert the resulting tagset back into tagsets
			tagsets.insert(make_pair(newtags, items));
		}
	}
}

template<class ITEM, class TAG>
void TagCollection<ITEM, TAG>::removeTagsWithCardinalityLessThan(int card) throw ()
{
	// Build the list of tags to remove
	OpSet<TAG> to_remove;
	for (typename TagContainer::const_iterator i = tags.begin();
			i != tags.end(); i++)
		if (i->second < card)
			to_remove += i->first;

	// Remove the tags from `tags'
	for (typename OpSet<TAG>::const_iterator i = to_remove.begin();
			i != to_remove.end(); i++)
		tags.erase(*i);

	// Select the tagsets to be changed from `tagsets'
	vector< OpSet<TAG> > to_change;
	for (typename tagsets_t::const_iterator i = tagsets.begin();
			i != tagsets.end(); i++)
	{
		OpSet<TAG> inters = i->first ^ to_remove;
		if (! inters.empty())
			to_change.push_back(i->first);
	}

	// Remove the tags from `tagsets'
	for (typename vector< OpSet<TAG> >::const_iterator i = to_change.begin();
			i != to_change.end(); i++)
	{
		// Remove the old mapping
		typename tagsets_t::iterator t = tagsets.find(*i);
		OpSet<ITEM> items = t->second;
		tagsets.erase(t);

		// Compute the new tagset
		OpSet<TAG> newts = *i - to_remove;

		// Insert the new mapping
		t = tagsets.find(newts);
		if (t == tagsets.end())
			tagsets.insert(make_pair(newts, items));
		else
			t->second += items;
	}
}

template<class ITEM, class TAG>
void TagCollection<ITEM, TAG>::output(TagcollConsumer<ITEM, TAG>& cons) const throw ()
{
	for (typename tagsets_t::const_iterator i = tagsets.begin(); i != tagsets.end(); i++)
		cons.consume(i->second, i->first);
}

// vim:set ts=4 sw=4:
