/*
 * Classes handling tag patches
 *
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma implementation

#include <tagcoll/Patches.h>

#include <stdio.h>

#include <tagcoll/stringf.h>

using namespace std;
using namespace Tagcoll;

template <class ITEM>
void PatchList<ITEM>::addAllEquals(const ITEM& item, char prefix, const OpSet<string>& tags) throw ()
{
	OpSet<string> patch;
	for (OpSet<string>::const_iterator t = tags.begin();
			t != tags.end(); t++)
		patch.insert(prefix + *t);
	if (patch.size() > 0)
		patches.insert(make_pair(item, patch));
}

template <class ITEM>
void PatchList<ITEM>::generate(const InputMerger<ITEM, std::string>& im1, const InputMerger<ITEM, std::string>& im2) throw ()
{
	typename map< ITEM, OpSet<string> >::const_iterator i1 = im1.coll.begin();
	typename map< ITEM, OpSet<string> >::const_iterator i2 = im2.coll.begin();

	while (i1 != im1.coll.end() || i2 != im2.coll.end())
	{
		if (i2 == im2.coll.end() || (i1 != im1.coll.end() && i1->first < i2->first))
		{
			// i1->first is only in im1: it has been deleted.  Insert all
			// tags with '-' sign
			//
			// Don't really add anything: since there are no info in the
			// target file about 'item', we can't say nothing about the
			// changes to his tags.  Warn, instead.
			//
			//addAllEquals(i1->first, '-', i1->second);
			fprintf(stderr, "Item \"%.*s\" exists only in the first collection: skipped.\n", PFSTR(i1->first));
			i1++;
		}
		else if (i1 == im1.coll.end() || (i2 != im2.coll.end() && i2->first < i1->first))
		{
			// i2->first is only in im2: it has been added.  Insert all
			// tags with '+' sign
			//
			// Don't really add anything: since there are no info in the
			// source file about 'item', we can't say nothing about the
			// changes to his tags.  Warn, instead.
			//
			//addAllEquals(i2->first, '+', i2->second);
			fprintf(stderr, "Item \"%.*s\" exists only in the second collection: skipped.\n", PFSTR(i2->first));
			i2++;
		}
		else
		{
			// the item is in both: compute the difference between the two
			// tag sets
			OpSet<string>::const_iterator t1 = i1->second.begin();
			OpSet<string>::const_iterator t2 = i2->second.begin();

			OpSet<string> patch;
			while (t1 != i1->second.end() || t2 != i2->second.end())
			{
				if (t2 == i2->second.end() || (t1 != i1->second.end() && *t1 < *t2))
				{
					// t1 is only in im1: it has been deleted.  Insert the tag with '-' sign
					patch.insert('-' + *t1);
					t1++;
				}
				else if (t1 == i1->second.end() || (t2 != i2->second.end() && *t2 < *t1))
				{
					// t2 is only in im2: it has been added.  Insert the tag with '+' sign
					patch.insert('+' + *t2);
					t2++;
				}
				else
				{
					// the tag is in both: no variation to register
					t1++;
					t2++;
				}
			}
			if (patch.size() > 0)
				patches.insert(make_pair(i1->first, patch));
			i1++;
			i2++;
		}
	}
}

template <class ITEM>
void PatchList<ITEM>::consume(const ITEM& item, const OpSet<string>& tags) throw ()
{
	patches.insert(make_pair(item, tags));
}

// Patch a tagged item: return the new set of tags
template <class ITEM>
OpSet<string> PatchList<ITEM>::patch(const ITEM& item, const OpSet<string>& tagset) const throw ()
{
	// Find the patch record for this item
	typename patch_t::const_iterator patch = patches.find(item);
	if (patch == patches.end())
		// If there are no patches, return the tagset unchanged
		return tagset;

	// There are patches: apply them:
	
	OpSet<string> res = tagset;
	for (OpSet<string>::const_iterator i = patch->second.begin();
			i != patch->second.end(); i++)
	{
		switch ((*i)[0])
		{
			case '+':
				res.insert(string((*i), 1, string::npos));
				break;
			case '-':
				res.erase(string((*i), 1, string::npos));
				break;
			default:
				fprintf(stderr, "Unsupported patch tag: %.*s\n", PFSTR(*i));
				break;
		}
	}
	return res;
}

// Output the patch list to a TagcollConsumer
template <class ITEM>
void PatchList<ITEM>::output(TagcollConsumer<ITEM, std::string>& consumer) const throw ()
{
	for (typename map< ITEM, OpSet<string> >::const_iterator i = patches.begin();
			i != patches.end(); i++)
		if (i->second.size() == 0)
			consumer.consume(i->first);
		else
			consumer.consume(i->first, i->second);
}

template <class ITEM>
void ApplyPatches<ITEM>::consume(const ITEM& item) throw ()
{
	OpSet<string> patched = patches.patch(item, OpSet<string>());
	if (patched.size())
		this->consumer->consume(item, patched);
	else
		this->consumer->consume(item);
}

template <class ITEM>
void ApplyPatches<ITEM>::consume(const ITEM& item, const OpSet<string>& tags) throw ()
{
	OpSet<string> patched = patches.patch(item, tags);
	if (patched.size())
		this->consumer->consume(item, patched);
	else
		this->consumer->consume(item);
}

// vim:set ts=4 sw=4:
