#ifndef TAGCOLL_INPUTMERGER_H
#define TAGCOLL_INPUTMERGER_H

/*
 * Merge tags of items appearing multiple times in a stream of tagged items
 * 
 * Copyright (C) 2003  Enrico Zini <enrico@debian.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#pragma interface

#include <tagcoll/TagcollConsumer.h>
#include <tagcoll/TagcollChange.h>

#include <map>

namespace Tagcoll
{

template<class ITEM>
class PatchList;

// Merge input values: if an item appears multiple times, merge his various
// tagsets in a single one
template<class T, class Tag>
class InputMerger : public TagcollConsumer<T, Tag>
{
protected:
	std::map< T, OpSet<Tag> > coll;
	
public:
	virtual ~InputMerger<T, Tag>() throw () {}

	virtual void consume(const T& item) throw ();
	virtual void consume(const T& item, const OpSet<Tag>& tags) throw ();

	// Return true if the collection contains `item'
	bool hasItem(const T& item) const throw () { return coll.find(item) != coll.end(); }

	// Get the tagset of item `item'.  Return an empty set if `item' does not exist
	OpSet<Tag> getTagsetForItem(const T& item) const throw ();

	// Send the merged data to a consumer
	void output(TagcollConsumer<T, Tag>& consumer) const throw ();

	// Apply a Change to the collection; return a reverse change that can be
	// reused to undo the operation
	TagcollChange<T, Tag> applyChange(const TagcollChange<T, Tag>& change) throw ();

	// Get the set of all tags in this collection
	OpSet<Tag> getAllTags() const throw ();

	// Get the set of all tags in this collection that appear in tagsets
	// containing `ts'
	OpSet<Tag> getCompanionTags(const OpSet<Tag>& ts) const throw ();

	// Get the related items at the given maximum distance
	OpSet<T> getRelatedItems(const T& item, int maxdistance = 1) const throw ();

	// Get the related items at the given maximum distance
	OpSet<T> getRelatedItems(const OpSet<Tag>& ts, int maxdistance = 1) const throw ();

	// Get the items whose tagset contains `ts'
	OpSet<T> getItemsContaining(const OpSet<Tag>& ts) const throw ();

	friend class PatchList<T>;
};

};

// vim:set ts=4 sw=4:
#endif
