// This code is (C) 1997 Francesco Chemolli <kinkie@comedia.it>
// It can be freely distributed and copied under the terms of the
// GNU General Public License, version 2 or later.
// This code comes with NO WARRANTY of any kind, either implicit or explicit.

// This module handles a GDBM-based (and in the future an NDBM one too)
// User Database. It's pretty much similar
// to Apache's DBM user database. Unluckily pike can't handle DBM databases
// yet, so you'll have to translate the database format if you want to use
// this as a drop-in replacement.
//
// The database is indexed on the usernames, its contents are:
// <crypted_password>[:home_dir[:login_shell]]
// (similar to a passwd entry line)
//
// Only the crypted password is necessary.
// To specify only password and login shell, specify a zero-length home dir
// (I can't see who could need this, but you never know...)

string cvs_version="$Id: GDBMuserdb.pike,v 1.7 1997/06/11 16:13:44 kinkie Exp $";

//#define GDBMAUTHDEBUG

#include <module.h>
inherit "roxenlib";
inherit "module";

#ifdef GDBMAUTHDEBUG
#define DEBUGLOG(X) perror("GDBMauth: "+X+"\n");
#else
#define DEBUGLOG(X) /**/
#endif

int att=0, succ=0, nouser=0, db_accesses=0, last_db_access=0;
object db=0;

/*
 * Utilities
 */
//temporary solution, until the patch I sent Hubbe finds its way to the
//official Pike distribution
string * get_db_indices(object db) {
	if (db->_indices)
		return indices (db);
	string *retval=({}), key;
	for (key=db->firstkey();key;key=db->nextkey(key))
		retval += ({key});
	return retval;
}

//gets a string in database format, and returns a 3-element
//array, containing crypt()ed password, home and login shell
array extract_fields (string authstring)
{
	if(!authstring||!sizeof(authstring))
		return 0;
	string *fields=authstring/":";
	array retval= ({fields[0],QUERY(defaulthome),QUERY(defaultshell)});
	if (sizeof(fields)>1 && sizeof(fields[1]))
		retval[1]=fields[1];
	if (sizeof(fields)>2 && sizeof(fields[2]))
		retval[2]=fields[2];
	return retval;
}

/*
 * Object management and configuration variables definitions
 */
void create() 
{
	defvar ("gdbmauthfile","/usr/local/etc/roxen_passwd.db","Passwords database",
			TYPE_FILE, "Where Roxen will look for the users database"
			);
	defvar ("gdbmauthcache",0,"Cache passwords", TYPE_FLAG,
			"This flag defines whether the module will cache the username/password "
			"entries"
			);
	defvar ("closedb",1,"Close the database if not used",TYPE_FLAG,
			"Setting this will save one filedescriptor without any significant "
			"performance loss."
			);
	defvar ("timer",60,"Database close timer", TYPE_INT,
			"The timer after which the database is closed",0,
			lambda(){return !QUERY(closedb);}
			);
	defvar ("defaultuid",geteuid(),"Default User ID", TYPE_INT,
			"User IDs don't have much meaning in this module's scope. However, "
			"some modules require an user ID to work correctly. This is the "
			"user ID which will be returned to all such requests."
			);
	defvar ("defaultgid", getegid(), "Default Group ID", TYPE_INT,
			"Same as User ID, only it refers rather to the group."
			);
	defvar ("defaultgecos", "", "Default Gecos", TYPE_STRING,
			"The default Gecos."
			);
	defvar ("defaulthome","/", "Default user Home Directory", TYPE_DIR, 
			"It is possible (but not mandatory) to specify an user's home "
			"directory in the passwords database. This is used if it's "
			"not provided. (<B>Not implemented yet</B>, it always uses this value)"
			);
	defvar ("defaultshell", "/bin/sh", "Default user Login Shell", TYPE_FILE,
			"Same as the default home, only referring to the user's login shell."
			" (<B>Not implemented yet</B>, it always uses this value)"
			);
}

void destruct() {
	if(objectp(db)) {
		db->close();
		::destruct(db);
		db=0;
	}
}

/*
 * DB management functions
 */
//this gets called only by call_outs, so we can avoid storing call_out_ids
//Also, I believe storing in a local variable the last time of an access
//to the database is more efficient than removing and resetting call_outs
//This leaves a degree of uncertainty on when the DB will be effectively
//closed, but it's below the value of the module variable "timer" for sure.
void close_db() {
	if( (time(1)-last_db_access) > QUERY(timer) ) {
		if(objectp(db)) //sanity check.
			db->close();
		db=0;
		DEBUGLOG("closing the database");
		return;
	}
	if(QUERY(closedb))
		call_out(close_db,QUERY(timer));
}

void open_db() {
	mixed err;
	last_db_access=time(1);
	db_accesses++; //I count DB accesses here, since this is called before each
	if(objectp(db)) //already open
		return;
	err=catch{
		db=Gdbm.gdbm(QUERY(gdbmauthfile),"rf");
	};
	if (err) {
		db=0;
		perror ("GDBMauth: Couldn't open database file "+QUERY(gdbmauthfile)+
			"	read-only! All authentication attempts will fail!\n");
		return;
	}
	DEBUGLOG("database successfully opened");
	if(QUERY(closedb))
		call_out(close_db,QUERY(timer));
}

/*
 * Module Callbacks
 */
string *userinfo (string u) {
	string *dbinfo;
	mixed err;
	DEBUGLOG ("userinfo ("+u+")");
	
	if (QUERY(gdbmauthcache))
		dbinfo=cache_lookup("gdbmauthentries",u);

	if (!dbinfo) {
		open_db();
		if (db)
			dbinfo=extract_fields(db[u]);
	}
	if (dbinfo) {
		if (QUERY(gdbmauthcache))
			cache_set("gdbmauthentries",u,dbinfo);
		array retval= ({ u, dbinfo[0], QUERY(defaultuid), QUERY(defaultgid),
				QUERY(defaultgecos), dbinfo[1], dbinfo[2]
				});
		DEBUGLOG(sprintf("Result: %O",retval)-"\n");
		return retval;
	}
	return 0;
}

string *userlist() {
	mixed err;

	DEBUGLOG ("userlist()");
	open_db();
	if (db)
		return get_db_indices(db);
	return ({});
}

string user_from_uid (int u) 
{
	return 0;
	//I have no idea of what default data I could return: uid doesn't have much
	//meaning in this context..
}

array|int auth (string *auth, object id)
{
	string u,p,*dbinfo;
	mixed err;

	att++;
	DEBUGLOG (sprintf("auth(%O)",auth)-"\n");

	sscanf (auth[1],"%s:%s",u,p);

	if (!p||!strlen(p)) {
		DEBUGLOG ("no password supplied by the user");
		return ({0, auth[1], -1});
	}

	if (QUERY(gdbmauthcache))
		dbinfo=cache_lookup("gdbmauthentries",u);

	if (!dbinfo) {
		open_db();

		if(!db) {
			DEBUGLOG ("Error in opening the database");
			return ({0, auth[1], -1});
		}
		dbinfo=extract_fields(db[u]);
		DEBUGLOG("DBinfo: "+(dbinfo?dbinfo*",":"0"));
		if (QUERY(gdbmauthcache)&&dbinfo[0]&&sizeof(dbinfo[0]))
			cache_set("gdbmauthentries",u,dbinfo);
	}

	// I suppose that the user's password is at least 1 character long
	if (!dbinfo||!sizeof(dbinfo[0])) {
		DEBUGLOG ("no such user");
		nouser++;
		return ({0,u,p});
	}

	if(!crypt (p,dbinfo[0])) {
		DEBUGLOG ("password check ("+dbinfo[0]+","+p+") failed");
		return ({0,u,p});
	}

	DEBUGLOG (u+" positively recognized");
	succ++;
	return ({1,u,0});
}

/*
 * Support Callbacks
 */
string status() {
	return "<H2>Security info</H2>"
			"Attempted authentications: "+att+"<BR>\n"
			"Failed: "+(att-succ+nouser)+" ("+nouser+" because of wrong username)"
			"<BR>\n"+
			db_accesses +" accesses to the database were required.<BR>\n"
			;
}

string|void check_variable (string name, mixed newvalue)
{
	switch (name) {
		case "timer":
			if (((int)newvalue)<=0) {
				set("timer",QUERY(timer));
				return "What? Have you lost your mind? How can I close the database"
					" before using it?";
			}
			return 0;
		default:
			return 0;
	}
	return 0; //should never reach here...
}

array register_module() {
	return ({
	MODULE_AUTH,
	"GDBM user database",
	"This module implements user authentication via a GDBM database.<p>\n"
	"The database is indexed on the usernames, and the contents are the "
	"crypt()-ed passwords.<BR>\n"
	"&copy; 1997 Francesco Chemolli, distributed freely under GPL license.",
	0,
	1
	});
};
