/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002 The Inti Development Team.
 *
 *  main.cc - Initialization and main processing loop wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "main.h"
#include "object.h"
#include "utf-string.h"
#include "gdk/events.h"
#include "glib/object.h"
#include "gtk/widget.h"

namespace Inti {

namespace Main {

struct SlotNode : public Node
{
	typedef void (*PF)(unsigned int);
	PF pf;

	unsigned int connect_id;

	SlotNode(const Slot *slot, PF func);
	~SlotNode();

	static void destroy(void *data);

	void block() {}

	void unblock() {}
	
	void disconnect();
};

} // namespace Main

} // namespace Inti

using namespace Inti;

void
Main::init(int *argc, char ***argv)
{
	gtk_init(argc, argv);
}
	
bool
Main::init_check(int *argc, char ***argv)
{
	return gtk_init_check(argc, argv);
}

namespace { // init_slot_callback

int init_slot_callback(void *data)
{
	Slot0<int> *slot = static_cast<Slot0<int>*>(data);
	return slot->call();
}

} // init_slot_callback

void
Main::init_add(const Slot0<int> *callback)
{
	gtk_init_add(&init_slot_callback, (void*)callback);
}	

PangoLanguage*
Main::default_language()
{
	return gtk_get_default_language();
}

bool 
Main::events_pending()
{
	return gtk_events_pending() != 0;
}

Pointer<Gtk::Widget>
Main::event_widget(Gdk::Event& event)
{
	GtkWidget *widget = gtk_get_event_widget(event.gdk_event());
	return widget ? G::Object::wrap<Gtk::Widget>(widget) : 0;
}

void 
Main::run()
{
	gtk_main();
}

int 
Main::level()
{
	return gtk_main_level();
}

void
Main::quit()
{
	gtk_main_quit();
}

bool 
Main::iterate(bool blocking)
{
	return gtk_main_iteration_do(blocking);
}

void 
Main::grab_add(Gtk::Widget& widget)
{
	gtk_grab_add(widget.gtk_widget());
}

Pointer<Gtk::Widget>
Main::grab_get_current()
{
	GtkWidget *widget = gtk_grab_get_current();
	return widget ? G::Object::wrap<Gtk::Widget>(widget) : 0;
}

void
Main::grab_remove(Gtk::Widget& widget)
{
	gtk_grab_remove(widget.gtk_widget());
}

/*  Main::SlotNode
 */

Main::SlotNode::SlotNode(const Slot *slot, PF func)
: Node(const_cast<Slot*>(slot)), pf(func), connect_id(0)
{
}

Main::SlotNode::~SlotNode()
{
}

void
Main::SlotNode::destroy(void *data)
{
	SlotNode *node = static_cast<SlotNode*>(data);
	node->unref();
}

void
Main::SlotNode::disconnect()
{
	(*pf)(connect_id);
}

/*  Main::QuitSignal
 */

Main::QuitSignal Main::quit_signal;

namespace { // quit_signal_callback

gboolean quit_signal_callback(gpointer data)
{
	Main::QuitSignal::SlotType *slot = dynamic_cast<Main::QuitSignal::SlotType*>(static_cast<Node*>(data)->slot());
	return slot->call();
}

} // quit_signal_callback

Connection
Main::QuitSignal::connect(const SlotType *slot, unsigned int main_level)
{
	g_return_val_if_fail(slot != 0, 0);
	SlotNode *node = new SlotNode(slot, &gtk_quit_remove);
	node->connect_id = gtk_quit_add_full(main_level, &quit_signal_callback, 0, node, (GtkDestroyNotify)&SlotNode::destroy);
	return node;
}
	
/*  Main::TimeoutSignal
 */

Main::TimeoutSignal Main::timeout_signal;

namespace { // timeout_signal_callback

gboolean timeout_signal_callback(gpointer data)
{
	Main::TimeoutSignal::SlotType *slot = dynamic_cast<Main::TimeoutSignal::SlotType*>(static_cast<Node*>(data)->slot());
	return slot->call();
}

} // timeout_signal_callback

Connection
Main::TimeoutSignal::connect(const SlotType *slot, unsigned int interval)
{
	g_return_val_if_fail(slot != 0, 0);
	SlotNode *node = new SlotNode(slot, &gtk_timeout_remove);
	node->connect_id = gtk_timeout_add_full(interval, &timeout_signal_callback, 0, node, (GtkDestroyNotify)&SlotNode::destroy);
	return node;
}

/*  Main::IdleSignal
 */

Main::IdleSignal Main::idle_signal;

namespace { // idle_signal_callback

gboolean idle_signal_callback(gpointer data)
{
	Main::IdleSignal::SlotType *slot = dynamic_cast<Main::IdleSignal::SlotType*>(static_cast<Node*>(data)->slot());
	return slot->call();
}

} // idle_signal_callback

Connection
Main::IdleSignal::connect(const SlotType* slot, int priority)
{
	g_return_val_if_fail(slot != 0, 0);
	SlotNode *node = new SlotNode(slot, &gtk_idle_remove);
	node->connect_id = gtk_idle_add_full(priority, &idle_signal_callback, 0, node, (GtkDestroyNotify)&SlotNode::destroy);
	return node;
}

/*  Main::KeySnooperSignal
 */
 
Main::KeySnooperSignal Main::key_snooper_signal;

Main::KeySnooperSignal::~KeySnooperSignal()
{
	int count = connection_list.size();
	while (count)
	{
		delete connection_list[count - 1];
		--count;
	}
}

namespace { // key_snooper_signal_callback

gint key_snooper_signal_callback(GtkWidget *widget, GdkEventKey *key, gpointer data)
{
	Main::KeySnooperSignal::SlotType *slot = dynamic_cast<Main::KeySnooperSignal::SlotType*>(static_cast<Node*>(data)->slot());
	Gdk::Event tmp_event((GdkEvent*)key);
	return slot->call( *(G::Object::wrap<Gtk::Widget>(widget)), *tmp_event.key());
}

} // key_snooper_signal_callback
	
Connection
Main::KeySnooperSignal::connect(const SlotType* slot)
{
	g_return_val_if_fail(slot != 0, 0);
	SlotNode *node = new SlotNode(slot, &gtk_key_snooper_remove);
	node->connect_id = gtk_key_snooper_install(&key_snooper_signal_callback, node);
	connection_list.push_back(node);
	return node;
}

/*  Main::InputSignal
 */
 
Main::InputSignal Main::input_signal;

namespace { // input_signal_callback

void input_signal_callback(gpointer data, int source, GdkInputCondition condition)
{
	Main::InputSignal::SlotType *slot = dynamic_cast<Main::InputSignal::SlotType*>(static_cast<Node*>(data)->slot());
	return slot->call(source, condition);
}

} // input_signal_callback

Connection
Main::InputSignal::connect(const SlotType* slot, int source, GdkInputCondition condition)
{
	g_return_val_if_fail(slot != 0, 0);
	SlotNode *node = new SlotNode(slot, &gtk_input_remove);
	node->connect_id = gtk_input_add_full(source, condition, &input_signal_callback, 0, node, (GtkDestroyNotify)&SlotNode::destroy);
	return node;
}

