/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  treesortable.cc - GtkTreeSortable C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "treesortable.h"
#include "private/treesortable_p.h"
#include "treemodel.h"
#include "../glib/object.h"

using namespace Inti;

/*  Gtk::TreeSortable
 */

Gtk::TreeSortable::~TreeSortable()
{
}
	
GtkTreeSortableIface*
Gtk::TreeSortable::gtk_tree_sortable_iface() const 
{ 
	return peek<GtkTreeSortableIface>(GTK_TYPE_TREE_SORTABLE); 
}

Gtk::TreeSortable::operator GtkTreeSortable* () const
{
	return this ? gtk_tree_sortable() : 0; 
}
	
bool 
Gtk::TreeSortable::get_sort_column_id(int *sort_column_id, SortType *order) const
{
	return gtk_tree_sortable_get_sort_column_id(gtk_tree_sortable(), sort_column_id, (GtkSortType*)order);
}

bool
Gtk::TreeSortable::has_default_sort_func() const
{
	return gtk_tree_sortable_has_default_sort_func(gtk_tree_sortable());
}

void
Gtk::TreeSortable::sort_column_changed()
{
	gtk_tree_sortable_sort_column_changed(gtk_tree_sortable());
}

void
Gtk::TreeSortable::set_sort_column_id(int sort_column_id, SortType order)
{
	gtk_tree_sortable_set_sort_column_id(gtk_tree_sortable(), sort_column_id, (GtkSortType)order);
}

namespace { // tree_iter_compare_slot

gint tree_iter_compare_slot(GtkTreeModel* model, GtkTreeIter *a, GtkTreeIter *b, gpointer user_data)
{
	Gtk::TreeSortable::TreeIterCompareSlot *slot = static_cast<Gtk::TreeSortable::TreeIterCompareSlot*>(user_data);
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(model));
	Gtk::TreeModel *tmp_model = dynamic_cast<Gtk::TreeModel*>(object);
	Gtk::TreeIter tmp_a(a);
	Gtk::TreeIter tmp_b(b);
	return slot->call(*tmp_model, &tmp_a, &tmp_b);
}

} // tree_iter_compare_slot

void
Gtk::TreeSortable::set_sort_func(int sort_column_id, const TreeIterCompareSlot *compare)
{
	gtk_tree_sortable_set_sort_func(gtk_tree_sortable(), sort_column_id, &tree_iter_compare_slot, (void*)compare, 0);
}

void
Gtk::TreeSortable::set_default_sort_func(const TreeIterCompareSlot *compare)
{
	gtk_tree_sortable_set_default_sort_func(gtk_tree_sortable(), &tree_iter_compare_slot, (void*)compare, 0);
}

/*  Gtk::TreeSortableIface
 */

void
Gtk::TreeSortableIface::init(GtkTreeSortableIface *g_iface)
{
	g_iface->sort_column_changed = &sort_column_changed_proxy;
	g_iface->get_sort_column_id = &get_sort_column_id_proxy;
	g_iface->set_sort_column_id = &set_sort_column_id_proxy;
	g_iface->set_sort_func = &set_sort_func_proxy;
	g_iface->set_default_sort_func = &set_default_sort_func_proxy;
	g_iface->has_default_sort_func = &has_default_sort_func_proxy;
}

gboolean
Gtk::TreeSortableIface::get_sort_column_id_proxy(GtkTreeSortable *sortable, gint *sort_column_id, GtkSortType *order)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(sortable));
	Gtk::TreeSortable *tmp_sortable = dynamic_cast<Gtk::TreeSortable*>(object);
	if (tmp_sortable)
		result = tmp_sortable->do_get_sort_column_id(sort_column_id, order);
	else
	{
		GtkTreeSortableIface *tmp_iface = GTK_TREE_SORTABLE_GET_IFACE(sortable);
		GtkTreeSortableIface *g_iface = static_cast<GtkTreeSortableIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->get_sort_column_id)
			result = g_iface->get_sort_column_id(sortable, sort_column_id, order);
	}
	return result;
}

void
Gtk::TreeSortableIface::set_sort_column_id_proxy(GtkTreeSortable *sortable, gint sort_column_id, GtkSortType order)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(sortable));
	Gtk::TreeSortable *tmp_sortable = dynamic_cast<Gtk::TreeSortable*>(object);
	if (tmp_sortable)
		tmp_sortable->do_set_sort_column_id(sort_column_id, order);
	else
	{
		GtkTreeSortableIface *tmp_iface = GTK_TREE_SORTABLE_GET_IFACE(sortable);
		GtkTreeSortableIface *g_iface = static_cast<GtkTreeSortableIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->set_sort_column_id)
			g_iface->set_sort_column_id(sortable, sort_column_id, order);
	}
}

void
Gtk::TreeSortableIface::set_sort_func_proxy(GtkTreeSortable *sortable, gint sort_column_id, GtkTreeIterCompareFunc func, gpointer data, GtkDestroyNotify destroy)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(sortable));
	Gtk::TreeSortable *tmp_sortable = dynamic_cast<Gtk::TreeSortable*>(object);
	if (tmp_sortable)
		tmp_sortable->do_set_sort_func(sort_column_id, func, data, destroy);
	else
	{
		GtkTreeSortableIface *tmp_iface = GTK_TREE_SORTABLE_GET_IFACE(sortable);
		GtkTreeSortableIface *g_iface = static_cast<GtkTreeSortableIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->set_sort_func)
			g_iface->set_sort_func(sortable, sort_column_id, func, data, destroy);
	}
}

void
Gtk::TreeSortableIface::set_default_sort_func_proxy(GtkTreeSortable *sortable, GtkTreeIterCompareFunc func, gpointer data, GtkDestroyNotify destroy)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(sortable));
	Gtk::TreeSortable *tmp_sortable = dynamic_cast<Gtk::TreeSortable*>(object);
	if (tmp_sortable)
		tmp_sortable->do_set_default_sort_func(func, data, destroy);
	else
	{
		GtkTreeSortableIface *tmp_iface = GTK_TREE_SORTABLE_GET_IFACE(sortable);
		GtkTreeSortableIface *g_iface = static_cast<GtkTreeSortableIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->set_default_sort_func)
			g_iface->set_default_sort_func(sortable, func, data, destroy);
	}
}

gboolean
Gtk::TreeSortableIface::has_default_sort_func_proxy(GtkTreeSortable *sortable)
{
	gboolean result = FALSE;
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(sortable));
	Gtk::TreeSortable *tmp_sortable = dynamic_cast<Gtk::TreeSortable*>(object);
	if (tmp_sortable)
		result = tmp_sortable->do_has_default_sort_func();
	else
	{
		GtkTreeSortableIface *tmp_iface = GTK_TREE_SORTABLE_GET_IFACE(sortable);
		GtkTreeSortableIface *g_iface = static_cast<GtkTreeSortableIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->has_default_sort_func)
			result = g_iface->has_default_sort_func(sortable);
	}
	return result;
}

void
Gtk::TreeSortableIface::sort_column_changed_proxy(GtkTreeSortable *sortable)
{
	G::Object *object = G::Object::pointer<G::Object>(G_OBJECT(sortable));
	Gtk::TreeSortable *tmp_sortable = dynamic_cast<Gtk::TreeSortable*>(object);
	if (tmp_sortable)
		tmp_sortable->on_sort_column_changed();
	else
	{
		GtkTreeSortableIface *tmp_iface = GTK_TREE_SORTABLE_GET_IFACE(sortable);
		GtkTreeSortableIface *g_iface = static_cast<GtkTreeSortableIface*>(g_type_interface_peek_parent(tmp_iface));
		if (g_iface->sort_column_changed)
			g_iface->sort_column_changed(sortable);
	}
}

/*  Overridable methods
 */

bool
Gtk::TreeSortable::do_get_sort_column_id(int *sort_column_id, GtkSortType *order)
{
	bool result = false;
	GtkTreeSortableIface *g_iface = peek_parent<GtkTreeSortableIface>(gtk_tree_sortable_iface());
	if (g_iface->get_sort_column_id)
		result = g_iface->get_sort_column_id(gtk_tree_sortable(), sort_column_id, order);
	return result;
}

void
Gtk::TreeSortable::do_set_sort_column_id(int sort_column_id, GtkSortType order)
{
	GtkTreeSortableIface *g_iface = peek_parent<GtkTreeSortableIface>(gtk_tree_sortable_iface());
	if (g_iface->set_sort_column_id)
		g_iface->set_sort_column_id(gtk_tree_sortable(), sort_column_id, order);
}

void
Gtk::TreeSortable::do_set_sort_func(int sort_column_id, GtkTreeIterCompareFunc func, void *data, GtkDestroyNotify destroy)
{
	GtkTreeSortableIface *g_iface = peek_parent<GtkTreeSortableIface>(gtk_tree_sortable_iface());
	if (g_iface->set_sort_func)
		g_iface->set_sort_func(gtk_tree_sortable(), sort_column_id, func, data, destroy);
}

void
Gtk::TreeSortable::do_set_default_sort_func(GtkTreeIterCompareFunc func, void *data, GtkDestroyNotify destroy)
{
	GtkTreeSortableIface *g_iface = peek_parent<GtkTreeSortableIface>(gtk_tree_sortable_iface());
	if (g_iface->set_default_sort_func)
		g_iface->set_default_sort_func(gtk_tree_sortable(), func, data, destroy);
}

bool
Gtk::TreeSortable::do_has_default_sort_func()
{
	bool result = false;
	GtkTreeSortableIface *g_iface = peek_parent<GtkTreeSortableIface>(gtk_tree_sortable_iface());
	if (g_iface->has_default_sort_func)
		result = g_iface->has_default_sort_func(gtk_tree_sortable());
	return result;
}

/*  Signal handlers
 */

void
Gtk::TreeSortable::on_sort_column_changed()
{
	GtkTreeSortableIface *g_iface = peek_parent<GtkTreeSortableIface>(gtk_tree_sortable_iface());
	if (g_iface->sort_column_changed)
		g_iface->sort_column_changed(gtk_tree_sortable());
}

/*  Signals
 */

const Gtk::TreeSortable::SortColumnChangedSignalType Gtk::TreeSortable::sort_column_changed_signal("sort_column_changed");
