/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2002-2003 The Inti Development Team.
 *
 *  clipboard.cc - GtkClipboard C++ wrapper implementation
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
#include "clipboard.h"
#include "selection.h"
#include "../gdk/display.h"
#include "../glib/object.h"

using namespace Inti;

Gtk::Clipboard::Clipboard(GtkClipboard *clipboard, bool reference)
: G::Object((GObject*)clipboard, reference)
{
}

Gtk::Clipboard::~Clipboard()
{
}
	
Gtk::Clipboard::operator GtkClipboard* () const
{ 
	return this ? gtk_clipboard() : 0; 
}

Gtk::Clipboard* 
Gtk::Clipboard::get(Gdk::Atom selection, const Gdk::Display *display)
{
	GdkDisplay *tmp_display = display ? display->gdk_display() : gdk_display_get_default();
	return G::Object::wrap<Clipboard>(gtk_clipboard_get_for_display(tmp_display, selection));
}

Gtk::Clipboard* 
Gtk::Clipboard::get_default(const Gdk::Display *display)
{
	return get(GDK_SELECTION_CLIPBOARD, display);
}

Gtk::Clipboard* 
Gtk::Clipboard::get_primary(const Gdk::Display *display)
{
	return get(GDK_SELECTION_PRIMARY, display);
}

Gdk::Display*
Gtk::Clipboard::get_display() const
{
	return G::Object::wrap<Gdk::Display>(gtk_clipboard_get_display(gtk_clipboard()));
}

namespace { // Clipboard slot callbacks
	
struct SlotType
{ 
	void *get; 
	void *clear; 
};

void received_slot_callback(GtkClipboard*, GtkSelectionData *selection_data, gpointer data)
{
	Gtk::Clipboard::ReceivedSlot *slot = static_cast<Gtk::Clipboard::ReceivedSlot*>(data);
	Gtk::SelectionData tmp_selection_data(selection_data);
	slot->call(tmp_selection_data);
}

void text_received_slot_callback(GtkClipboard*, const gchar *text, gpointer data)
{
	Gtk::Clipboard::TextReceivedSlot *slot = static_cast<Gtk::Clipboard::TextReceivedSlot*>(data);
	std::string tmp_text(text);
	slot->call(text);
}

void get_slot_callback(GtkClipboard*, GtkSelectionData *selection_data, guint info, gpointer data)
{
	Gtk::Clipboard::GetSlot *slot = static_cast<Gtk::Clipboard::GetSlot*>(((SlotType*)data)->get);
	Gtk::SelectionData tmp_selection_data(selection_data);
	slot->call(tmp_selection_data, info);
}

void clear_slot_callback(GtkClipboard*, gpointer data)
{
	SlotType *st = static_cast<SlotType*>(data);
	Gtk::Clipboard::ClearSlot *slot = static_cast<Gtk::Clipboard::ClearSlot*>(st->clear);
	if (slot)
		slot->call();
	delete st;
}

} // Clipboard slot callbacks

bool
Gtk::Clipboard::set(const std::vector<TargetEntry>& targets, const GetSlot *get, const ClearSlot *clear)
{
	g_return_val_if_fail(!targets.empty(), false);
	int count = targets.size();
	GtkTargetEntry *tmp_targets = new GtkTargetEntry[count];
	
	int i = 0;
	while (i < count)
	{
		tmp_targets[i] = *(targets[i].gtk_target_entry());
		++i;
	}
	
	SlotType *st = new SlotType;	
	st->get = (void*)get;
	st->clear = (void*)clear;
	bool result = gtk_clipboard_set_with_data(gtk_clipboard(), tmp_targets, count, &get_slot_callback, &clear_slot_callback, (void*)st);
	delete [] tmp_targets;
	return result;
}

void 
Gtk::Clipboard::clear()
{
	gtk_clipboard_clear(gtk_clipboard());
}

void 
Gtk::Clipboard::set_text(const String& text)
{
	gtk_clipboard_set_text(gtk_clipboard(), text.c_str(), text.size());
}

void 
Gtk::Clipboard::request_contents(Gdk::Atom target, const ReceivedSlot *received) const
{
	gtk_clipboard_request_contents(gtk_clipboard(), target, &received_slot_callback, (void*)received);
}

void 
Gtk::Clipboard::request_text(const TextReceivedSlot *received) const
{
	gtk_clipboard_request_text(gtk_clipboard(), &text_received_slot_callback, (void*)received);
}

Pointer<Gtk::SelectionData>
Gtk::Clipboard::wait_for_contents(Gdk::Atom target) const
{
	GtkSelectionData* data = gtk_clipboard_wait_for_contents(gtk_clipboard(), target);
	return data ? G::Boxed::wrap<SelectionData>(GTK_TYPE_SELECTION_DATA, data, false) : 0;
}

String
Gtk::Clipboard::wait_for_text() const
{
	return gtk_clipboard_wait_for_text(gtk_clipboard());
}

bool 
Gtk::Clipboard::wait_is_text_available() const
{
	return gtk_clipboard_wait_is_text_available(gtk_clipboard());
}

