/*  Inti: Integrated Foundation Classes
 *  Copyright (C) 2003 The Inti Development Team.
 *
 *  treadpool.cc - A GThreadPool C++ wrapper interface.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "threadpool.h"
#include "error.h"

using namespace Inti;

/*  G::ThreadPool
 */

namespace { // func_slot_callback

void task_slot_callback(gpointer data, gpointer /* user_data */)
{
	G::ThreadPool::TaskSlot *slot = static_cast<G::ThreadPool::TaskSlot*>(data);
	slot->call();
}

} // namespace

G::ThreadPool::ThreadPool(int max_threads, bool exclusive, G::Error *error)
: pool_(g_thread_pool_new(&task_slot_callback, 0, max_threads, exclusive, *error))
{
}

G::ThreadPool::~ThreadPool()
{
	while (task_list.size() > 0)
	{
		const_cast<TaskSlot*>(task_list.back())->unref();
		task_list.pop_back();
	}

	if (pool_)
	{
		g_thread_pool_free(pool_, TRUE, TRUE);
		pool_ = 0;
	}
}

G::ThreadPool::operator GThreadPool* () const
{
	return this ? g_thread_pool() : 0;
}

int
G::ThreadPool::get_max_threads() const
{
	return g_thread_pool_get_max_threads(pool_);
}

unsigned int
G::ThreadPool::get_num_threads() const
{
	return g_thread_pool_get_num_threads(pool_);
}

unsigned int
G::ThreadPool::unprocessed() const
{
	return g_thread_pool_unprocessed(pool_);
}

void
G::ThreadPool::push(const TaskSlot *task, G::Error *error)
{
	g_thread_pool_push(pool_, (void*)task, *error);
	task_list.push_back(task);
}

void
G::ThreadPool::set_max_threads(int max_threads, G::Error *error)
{
	g_thread_pool_set_max_threads(pool_, max_threads, *error);
}

void
G::ThreadPool::free(bool immediate, bool wait)
{
	if (pool_)
	{
		g_thread_pool_free(pool_, immediate, wait);
		pool_ = 0;
	}
}

unsigned int
G::ThreadPool::get_num_unused_threads()
{
	return g_thread_pool_get_num_unused_threads();
}

int
G::ThreadPool::get_max_unused_threads()
{
	return g_thread_pool_get_max_unused_threads();
}

void
G::ThreadPool::set_max_unused_threads(int max_threads)
{
         g_thread_pool_set_max_unused_threads(max_threads);
}

void
G::ThreadPool::stop_unused_threads()
{
	g_thread_pool_stop_unused_threads();
}

