/** Copyright  2003 by Jean-Hugues de Raigniac <jhraigniac@workingfrog.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.workingfrog.i18n.swing;

import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JToggleButton;

import org.workingfrog.i18n.util.LocaleEvent;
import org.workingfrog.i18n.util.LocaleListener;
import org.workingfrog.i18n.util.Translator;

/**
 * i18n version of javax.swing.JRadioButtonMenuItem
 *
 * @author Jean-Hugues de Raigniac
 */
public class I18NJRadioButtonMenuItem extends JRadioButtonMenuItem
    implements LocaleListener {

    /** bundle */
    private String bundle = "menu";

    /** i18n key */
    private String i18nKey = null;

    public I18NJRadioButtonMenuItem () {
        this(null, null, false);
    }

    /**
     * Override
     * javax.swing.JRadioButtonMenuItem.JRadioButtonMenuItem (java.lang.String).
     *
     * @param i18nKey i18n bundle key
     */
    public I18NJRadioButtonMenuItem (String i18nKey) {
        this(i18nKey, null, false);
    }

    public I18NJRadioButtonMenuItem (String i18nKey, Icon icon) {
        this(i18nKey, icon, false);
    }

    public I18NJRadioButtonMenuItem (String i18nKey, boolean selected) {
        this(i18nKey);
        setSelected(selected);
    }

    public I18NJRadioButtonMenuItem (String i18nKey, Icon icon, boolean selected) {
        super(null, icon);
        setText(i18nKey);
        setModel(new JToggleButton.ToggleButtonModel());
        setSelected(selected);
    }

    public void setText (String i18nKey) {
        if (this.i18nKey == null || this.i18nKey.equals("")) {
            this.i18nKey = i18nKey;
            Translator.checkKey(i18nKey, this);
        }

        if (i18nKey != null && ! i18nKey.equals("")) {
            super.setText(Translator.checkValue(i18nKey, this));
        } else {
            super.setText("");
        }
    }

    /**
     * Updates Action.NAME or JRadioButtonMenuItem label with
     * a localized version of i18nKey.
     *
     * @param event contains the new Locale
     */
    public void localeChanged (LocaleEvent event) {
        Action action = getAction();
        if (action != null) {
            if (action instanceof LocaleListener) {
                ((LocaleListener) action).localeChanged(event);
            }
        } else {
            setText(i18nKey);
        }
    }
}
