/******************************************************************************************************************************************
 ccombobox.c
******************************************************************************************************************************************/

#include "ccombobox.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CComboBoxItemFieldValueAPIListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// combobox item API listener handling
//-----------------------------------------------------------------------------------------------------------------------------------------
void CComboBoxItemFieldValueAPIListener::OnStateChange (CObject *inItemField)
{
	// retreive the item field value instance
	CItemFieldValue *inItemFieldValue = static_cast <CItemFieldValue *> (inItemField);

	// retreive the generic combo box item instance
	CComboBoxItem *inComboBoxItem = static_cast <CComboBoxItem *> (inItemFieldValue -> GetOwner());

	// set the combo box item its new item fields values and states
	if (inComboBoxItem != NULL) inComboBoxItem -> SetItemFieldValues (CItemFieldValues (inComboBoxItem -> GetItemFieldValues()));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_GENERIC_METACLASS (CComboBoxListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CComboBoxListener::CComboBoxListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CComboBoxListener::~CComboBoxListener ()
{ }

//-----------------------------------------------------------------------------------------------------------------------------------------
// OnChange combobox listener
//-----------------------------------------------------------------------------------------------------------------------------------------
void CComboBox::OnChange (GtkComboBox *, gpointer inData)
{
	// retreive the gtkol instance
	CWidget *inWidget = reinterpret_cast <CWidget *> (inData);

	// pointer check and notification to the listener if any
	if (inWidget != NULL && inWidget -> GetListener() != NULL)
		static_cast <CComboBoxListener *> (inWidget -> GetListener()) -> OnChange (inWidget);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CComboBoxItem);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CComboBoxItem::CComboBoxItem	 (CComboBox *inOwner, const CItemFieldValues &inFieldValues, const CObjectListener *inListener)
	      :CComponent    	 (NULL, inListener),
	       m_ItemFieldValues (inFieldValues),
	       m_GtkTreeIter	 ()
{
	// set the item's owner
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CComboBoxItem::CComboBoxItem	 (CComboBoxItem *inOwner, const CItemFieldValues &inFieldValues, const CObjectListener *inListener)
	      :CComponent    	 (NULL, inListener),
	       m_ItemFieldValues (inFieldValues),
	       m_GtkTreeIter	 ()
{
	// set the item's owner
	if (inOwner != NULL) SetOwner (inOwner);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CComboBoxItem::~CComboBoxItem ()
{
	// lock spine
	CComponent::Lock();

	// get the whole sub items as we cannot wait for the ccomponent destructor to do this job until this tree iter will be null
	// when the sub ones will try to remove their values from...
	CComponents inSubItems (GetSubComponents (__metaclass(CComboBoxItem)) - this);

	// delete each one : CComboBoxItem::~CComboBoxItem on each
	for (size_t i=inSubItems.GetLength(); i>0; i--) delete *inSubItems[i-1];

	// get the gtkol combo box owner of this instance
	CComboBox *inComboBox = static_cast <CComboBox *> (GetOwner (__metaclass(CComboBox)));

	// remove the item value from the gtk widget combo box
	if (inComboBox != NULL && inComboBox -> GetGtkWidget() != NULL)
		::gtk_tree_store_remove (GTK_TREE_STORE(::gtk_combo_box_get_model(GTK_COMBO_BOX(inComboBox->GetGtkWidget()))), 
					 &m_GtkTreeIter);

	// delete the handled combo box item instances values and their associated api listener
	for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--) delete *m_ItemFieldValues[i-1];

	// unlock spine
	CComponent::Unlock();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CComboBoxItem::OwnerMustBe () const
{
	return __metaclasses(CComboBox) + __metaclass(CComboBoxItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected children type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CComboBoxItem::ChildMustBe () const
{
	return __metaclasses(CComboBoxItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// owner affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CComboBoxItem::SetOwner (CComponent *inOwner, const SInt16 inIndex)
{
	// owner type check !
	if (!CheckSetOwner (inOwner)) return false;

	// get potential previous owner
	CComponent *inOldOwner = GetOwner ();

	// get the previous combo box owner
	CComboBox *inOldComboBox = inOldOwner != NULL ? inOldOwner -> ClassIs (__metaclass(CComboBox)) ? 
					static_cast <CComboBox *> (inOldOwner) : 
				 	static_cast <CComboBox *> (inOldOwner -> GetOwner (__metaclass(CComboBox))) : NULL;

	// get the new combo box owner
	CComboBox *inNewComboBox = inOwner != NULL ? inOwner -> ClassIs (__metaclass(CComboBox)) ? 
				 	static_cast <CComboBox *> (inOwner) : 
				 	static_cast <CComboBox *> (inOwner -> GetOwner (__metaclass(CComboBox))) : NULL;

	// if the old combo box owner and the new one differ
	if (inOldComboBox != NULL && inNewComboBox != NULL && inOldComboBox != inNewComboBox)
	{
		// get the respective models
		CMetaClasses inOldModel (inOldComboBox -> GetModel());
		CMetaClasses inNewModel (inNewComboBox -> GetModel());

		// remove any reference to a pack field specification
		for (size_t i=0; i<inOldModel.GetLength(); i++)
			if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePack), *inOldModel[i]))
				inOldModel.Delete (i--, 1);
		for (size_t i=0; i<inNewModel.GetLength(); i++)
			if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePack), *inNewModel[i]))
				inNewModel.Delete (i--, 1);

		// check their coherence
		if (inOldModel.GetLength() != inNewModel.GetLength()) return false; 
		for (size_t i=inNewModel.GetLength(); i>0; i--)
			if (*inNewModel[i-1] != *inOldModel[i-1]) return false;
	}

	// skelton call
	if (!CComponent::SetOwner (inOwner, inIndex)) return false;

	// if there were an old combo box widget for this item instance
	if (inOldComboBox != NULL)
	
		// remove the item from its old owner
		::gtk_tree_store_remove (GTK_TREE_STORE(::gtk_combo_box_get_model(GTK_COMBO_BOX(inOldComboBox->GetGtkWidget()))), 
					 &m_GtkTreeIter);

	// if the new owner is a combo box
	if (inOwner != NULL && inOwner -> ClassIs (__metaclass(CComboBox)))
	{
		// append requested
		if (inIndex < 0)

			// append this item as a new root of the combo box widget
			::gtk_tree_store_append (GTK_TREE_STORE(::gtk_combo_box_get_model(GTK_COMBO_BOX(inNewComboBox->GetGtkWidget()))), 
						 &m_GtkTreeIter, NULL);

		// insert requested (index range checked by gtk itself)
		else
			
			// insert this item as a new root of the combo box widget
			::gtk_tree_store_insert (GTK_TREE_STORE(::gtk_combo_box_get_model(GTK_COMBO_BOX(inNewComboBox->GetGtkWidget()))), 
						 &m_GtkTreeIter, NULL, inIndex);
	}
	// if the new owner is a combo box item
	else if (inOwner != NULL && inOwner -> ClassIs (__metaclass(CComboBoxItem)))
	{
		// append requested
		if (inIndex < 0)
		
			// append this item as a new child of the specified one
			::gtk_tree_store_append (GTK_TREE_STORE(::gtk_combo_box_get_model(GTK_COMBO_BOX(inNewComboBox->GetGtkWidget()))),
				 		 &m_GtkTreeIter, &(static_cast <CComboBoxItem *> (inOwner) -> m_GtkTreeIter));

		// insert requested
		else

			// insert this item as a new child of the specified one
			::gtk_tree_store_insert (GTK_TREE_STORE(::gtk_combo_box_get_model(GTK_COMBO_BOX(inNewComboBox->GetGtkWidget()))),
				 		 &m_GtkTreeIter, &(static_cast <CComboBoxItem *> (inOwner) -> m_GtkTreeIter), inIndex);
	}

	// item field values affectation
	if (!SetItemFieldValues (CItemFieldValues (m_ItemFieldValues))) return false;

	// if invoked, the gtk_tree_store_remove has removed the whole descendant hierarchy that has to be created again
	if (inOldOwner != NULL && inNewComboBox != NULL)
	{
		// potential sub components items of the current combo box item, do not consider this instance as the field values have 
		// already been affected
		CComponents inSubComponents (GetSubComponents (__metaclass(CComboBoxItem)) - this);

		// go through the sub components items
		for (size_t i=inSubComponents.GetLength(), j=0; i>0; i--, j++)
		{
			// the combo box item
			CComboBoxItem *inComboBoxItem = static_cast <CComboBoxItem *> (*inSubComponents[j]);

			// its owner
			inOwner = inComboBoxItem -> GetOwner ();

			// owner type analyse
			if (inOwner -> ClassIs (__metaclass(CComboBox)))
				::gtk_tree_store_append 
					(GTK_TREE_STORE(::gtk_combo_box_get_model(GTK_COMBO_BOX(inNewComboBox->GetGtkWidget()))), 
					&(inComboBoxItem->m_GtkTreeIter), NULL);
			else if (inOwner -> ClassIs (__metaclass(CComboBoxItem)))
				::gtk_tree_store_append 
					(GTK_TREE_STORE(::gtk_combo_box_get_model(GTK_COMBO_BOX(inNewComboBox->GetGtkWidget()))),
					&(inComboBoxItem->m_GtkTreeIter), &(static_cast <CComboBoxItem *> (inOwner) -> m_GtkTreeIter));

			// set the field values
			if (!inComboBoxItem -> SetItemFieldValues (CItemFieldValues (inComboBoxItem -> m_ItemFieldValues))) return false;
		}
	}

	// ok
	return true; 
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get the gtk tree iter associated to the gtkol tree item instance
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkTreeIter * CComboBoxItem::GetGtkTreeIter () const
{
	return const_cast <GtkTreeIter *> (&m_GtkTreeIter);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// get the gtkol tree view item of specified gtk tree iter if any
//-----------------------------------------------------------------------------------------------------------------------------------------
CComboBoxItem * CComboBoxItem::GetComboBoxItem (const GtkTreeIter *inGtkTreeIter)
{
	// pointer check
	if (inGtkTreeIter == NULL) return NULL;

	// get the whole tree view item instances
	CComponents inComponents (CComponent::GetComponents (__metaclass(CComboBoxItem)));

	// go through
	for (size_t i=inComponents.GetLength(), j=0; i>0; i--, j++)
		if (static_cast <CComboBoxItem *> (*inComponents[j]) -> m_GtkTreeIter.user_data == inGtkTreeIter->user_data)
			return static_cast <CComboBoxItem *> (*inComponents[j]);

	// not found
	return NULL;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// field values affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CComboBoxItem::SetItemFieldValues (const CItemFieldValues &inItemFieldValues)
{
	// get the combo box widget this item is under
	CComboBox *inComboBox = static_cast <CComboBox *> (GetOwner (__metaclass(CComboBox)));

	// delete the old values if any
	if (&inItemFieldValues != &m_ItemFieldValues)
		for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--) 
			if (*inItemFieldValues[i-1] != *m_ItemFieldValues[i-1]) 
				delete *m_ItemFieldValues[i-1];

	// pointers local copy
	m_ItemFieldValues = inItemFieldValues;

	// the combo box representation model
	CMetaClasses inModel;

	// preliminary check
	if (inComboBox != NULL)
	{
		// get the combo box representation model
		inModel = inComboBox -> GetModel(); 

		// remove any reference to a pack specification
		for (size_t i=0; i<inModel.GetLength(); i++)
			if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePack), *inModel[i]))
				inModel.Delete (i--, 1);

		// empty or part empty buffer list check
		if (m_ItemFieldValues.GetLength() < inModel.GetLength())
		{
			// go through the model and set default field value instances
			for (size_t i=m_ItemFieldValues.GetLength(); i<inModel.GetLength(); i++)
			{
				if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueString), *inModel[i]))
					m_ItemFieldValues += new CItemFieldValueString ();
				if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePixbuf), *inModel[i]))
					m_ItemFieldValues += new CItemFieldValuePixbuf (new CPixbuf("null"));
			}
		}
	}

	// go through the field values instances and set the friend protected attributes we are supposed to handle
	for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--)
	{ 
		// our combo box item handles the field value instance
		(*m_ItemFieldValues[i-1]) -> m_Owner = this;

		// set the item field value api listener so that when the programmer performs modifications on that field, the gtkol engine 
		// can handle gui modifications accordinaly
		if ((*m_ItemFieldValues[i-1]) -> m_Listener == NULL) 
			(*m_ItemFieldValues[i-1]) -> AssignListener (new CComboBoxItemFieldValueAPIListener());
	}
	
	// pointer check, memorization done anyway
	if (inComboBox == NULL) return true;

	// go through the model and set the associated values
	for (size_t i=inModel.GetLength(), j=0; i>0 && j<m_ItemFieldValues.GetLength(); i--, j++)
	{
		// get the field value of the incoming values
		CItemFieldValue *inItemFieldValue (*m_ItemFieldValues[j]);

		// model correspondance check
		if (!inItemFieldValue -> ClassIs (*inModel[j])) return false;

		// field type analyse, string field value
		if (inItemFieldValue -> ClassIs (__metaclass(CItemFieldValueString)))
		{
			// set the field values
			::gtk_tree_store_set (GTK_TREE_STORE(::gtk_combo_box_get_model(GTK_COMBO_BOX(inComboBox->GetGtkWidget()))),
				&m_GtkTreeIter, j, static_cast <CItemFieldValueString *> (inItemFieldValue) -> m_FieldValue.Get(), -1);
		}
		// pixbuf field value
		else if (inItemFieldValue -> ClassIs (__metaclass(CItemFieldValuePixbuf)))
		{
			// set the field values
			if (static_cast <CItemFieldValuePixbuf *> (inItemFieldValue) -> m_FieldValue != NULL)
			::gtk_tree_store_set (GTK_TREE_STORE(::gtk_combo_box_get_model(GTK_COMBO_BOX(inComboBox->GetGtkWidget()))),
			  &m_GtkTreeIter, j, static_cast <CItemFieldValuePixbuf *> (inItemFieldValue) -> m_FieldValue -> GetPixbuf(), -1);
		}
	}

	// ok
	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// field values access
//-----------------------------------------------------------------------------------------------------------------------------------------
CItemFieldValues CComboBoxItem::GetItemFieldValues () const
{
	return m_ItemFieldValues;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// indexed field value
//-----------------------------------------------------------------------------------------------------------------------------------------
CItemFieldValue & CComboBoxItem::operator [] (const size_t inIndex) const
{
	return **m_ItemFieldValues[inIndex];
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// item selection
//-----------------------------------------------------------------------------------------------------------------------------------------
void CComboBoxItem::Select ()
{
	// get the combobox owner
	CComboBox *inComboBox = static_cast <CComboBox *> (GetOwner (__metaclass(CComboBox)));

	// pointer check
	if (inComboBox == NULL || inComboBox -> GetGtkWidget() == NULL) return;

	// select this item
	::gtk_combo_box_set_active_iter (GTK_COMBO_BOX(inComboBox->GetGtkWidget()), &m_GtkTreeIter);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// item selected ?
//-----------------------------------------------------------------------------------------------------------------------------------------
bool CComboBoxItem::IsSelected () const
{
	// get the combobox owner
	CComboBox *inComboBox = static_cast <CComboBox *> (GetOwner (__metaclass(CComboBox)));

	// pointer check
	if (inComboBox == NULL || inComboBox -> GetGtkWidget() == NULL) return false;

	// get the active iter
	GtkTreeIter inGtkTreeIter; if (!::gtk_combo_box_get_active_iter (GTK_COMBO_BOX(inComboBox->GetGtkWidget()), &inGtkTreeIter)) 
		return false;

	// check and return
	return m_GtkTreeIter.user_data == inGtkTreeIter.user_data;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// combobox item serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CComboBoxItem::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// generic call first
	CComponent::Serialize (ioXMLElementNode, inMode);

	// request analyse
	switch (inMode)
	{
		// xml dump	
		case XML_WRITE :
		{
			// instanciate a new xml element
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_COMBOBOXITEM_ELEMENT);

			// modify the io xml node so that the overwritten definitions would continue under this new xml node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// instanciate the fields xml element under the current xml node
			newXMLElement = new CXMLElement (ioXMLElementNode, XML_COMBOBOXITEM_FIELDS_ELEMENT);

			// foreach of the fields, delegate the serialization process
			for (size_t i=m_ItemFieldValues.GetLength(), j=0; i>0; i--, j++)
			{
				// each of the field serializes under the current node
				CXMLElementNode *inXMLNode = newXMLElement -> GetXMLElementNode ();

				// serialize the field
				(*m_ItemFieldValues[j]) -> Serialize (inXMLNode, XML_WRITE);
			}
		}
		break;

		// xml load
		case XML_READ :
		{
			// get the node this instance should read from
			CXMLElementNode *inXMLNode = ::xml_node_get_child (ioXMLElementNode, XML_COMBOBOXITEM_ELEMENT);

			// check we got it
                        if (inXMLNode == NULL)
                                throw new CException (CString("CComboBoxItem::Serialize, specified xml node is not a \"") +
                                                              XML_COMBOBOXITEM_ELEMENT + CString("\" element one."), 
						      	      __exception(XMLPARSE));

			// modify the io xml node so that the overwritten definitions would continue under this new xml node
			ioXMLElementNode = inXMLNode;

			// get the fields xml child element node
			inXMLNode = ::xml_node_get_child (inXMLNode, XML_COMBOBOXITEM_FIELDS_ELEMENT);

			// check we got it
			if (inXMLNode != NULL)
			{
				// delete the handled combo box item instances values if any
				if (m_ItemFieldValues.GetLength() > 0) 
				{
					for (size_t i=m_ItemFieldValues.GetLength(); i>0; i--) delete *m_ItemFieldValues[i-1];
					m_ItemFieldValues.Delete (0, m_ItemFieldValues.GetLength());
				}

				// get the "fields" xml element children nodes
				CXMLElementNodes inXMLNodes (::xml_node_get_children (inXMLNode));

				// foreach child of the "children" element, launch the global serialization process
				for (size_t i=inXMLNodes.GetLength(), j=0; i>0; i--, j++)
				{
					// launch the instanciation process
					CSerialized *inSerialized = CSerialized::Instanciate (*inXMLNodes[j]) THROWABLE;

					// check the pointer type
					if (!inSerialized -> ClassIs (__metaclass(CItemFieldValue))) { delete inSerialized; continue; }

					// add the field value to the local buffer
					m_ItemFieldValues += static_cast <CItemFieldValue *> (inSerialized);
				}
			}

			// set the item field values
			SetItemFieldValues (CItemFieldValues (m_ItemFieldValues));
		}
		break;
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclass code resolution
//-----------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_DYNAMIC_METACLASS (CComboBox);

//-----------------------------------------------------------------------------------------------------------------------------------------
// constructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CComboBox::CComboBox (CContainer *inOwner, const CMetaClasses &inModel, const CComboBoxListener *inListener)
	  :CWidget   (inOwner, inListener),
	   m_Model   (inModel)
{
	// specified metaclasses model check in...
	for (size_t i=m_Model.GetLength(), j=0; i>0; i--, j++)
		if (!CMetaClass::MetaClassIs (__metaclass(CItemFieldValue), *m_Model[j]) || *m_Model[j] == __metaclass(CItemFieldValue))
			throw new CException ("CComboBox : the input model is not handled; the specified class type \"" + 
					      (*m_Model[j])->ClassName + "\" is not a handled CItemFieldValue derived one. Aborting...");

	// widget instanciation request
	if (inOwner != NULL) CWidget::CreateWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// destructor
//-----------------------------------------------------------------------------------------------------------------------------------------
CComboBox::~CComboBox ()
{
	// deletion coherence requested
	CWidget::DestroyWidget (this);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget instanciation
//-----------------------------------------------------------------------------------------------------------------------------------------
GtkWidget * CComboBox::PerformWidgetInstanciate ()
{
	// ok
	return ::gtk_combo_box_new();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// gtk widget initialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CComboBox::PerformWidgetInitialize ()
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return;

	// model affectation
	SetModel (CMetaClasses(m_Model));

	// static event association
	::g_signal_connect (G_OBJECT(inGtkWidget), "changed", G_CALLBACK(CComboBox::OnChange), this);

	// show the widget
	Show();
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// listener affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
const CMetaClass * CComboBox::ListenerMustBe () const
{
	return __metaclass(CComboBoxListener);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected owner type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CComboBox::OwnerMustBe () const
{
	return __metaclasses(CContainer);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// expected children type
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CComboBox::ChildMustBe () const
{
	return __metaclasses(CComboBoxItem);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// model affectation
//-----------------------------------------------------------------------------------------------------------------------------------------
Bool CComboBox::SetModel (const CMetaClasses &inModel)
{
	// model definition number check in...
	if (inModel.GetLength() == 0) return false;

	// model check in, all the specified metaclasses must derive the CItemFieldValue definition
	for (size_t i=m_Model.GetLength(), j=0; i>0; i--, j++)
		if (!CMetaClass::MetaClassIs (__metaclass(CItemFieldValue), *m_Model[j]) || *m_Model[j] == __metaclass(CItemFieldValue))
			return false;

	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// pointer check
	if (inGtkWidget == NULL) return false;

	// model input local copy
	m_Model = inModel;

	// GType buffer for the associated gtk tree store instance
	TBuffer <GType> outGTypeBuffer;

	// model input analyse
	for (size_t i=m_Model.GetLength(), j=0; i>0; i--, j++)
	{
		// string field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueString), *m_Model[j]))
		{
			outGTypeBuffer += G_TYPE_STRING;
		}
		// pixbuf field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePixbuf), *m_Model[j]))
		{
			outGTypeBuffer += GDK_TYPE_PIXBUF;
		}
	}

	// gtk tree store instanciation
	GtkTreeStore *inGtkTreeStore = ::gtk_tree_store_newv (outGTypeBuffer.GetLength(), outGTypeBuffer.Get());

	// gtk tree store affectation
	::gtk_combo_box_set_model (GTK_COMBO_BOX(inGtkWidget), GTK_TREE_MODEL(inGtkTreeStore));

	// delete local reference to the gtk tree store
	::g_object_unref (inGtkTreeStore);

	// delete potential old cell renderers
	::gtk_cell_layout_clear (GTK_CELL_LAYOUT(inGtkWidget));

	// create the tree view model associated renderers
	for (size_t i=m_Model.GetLength(), j=0; i>0; i--, j++)
	{
		// string field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValueString), *m_Model[j]))
		{
			// allocate the cell renderer
			GtkCellRenderer *inGtkCellRenderer = ::gtk_cell_renderer_text_new ();

			// pack the renderer into the layout
			::gtk_cell_layout_pack_start (GTK_CELL_LAYOUT(inGtkWidget), inGtkCellRenderer, FALSE);

			// set the cell layout attribute
			::gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT(inGtkWidget), inGtkCellRenderer, "text", j, NULL);
		}
		// pixbuf field value
		if (CMetaClass::MetaClassIs (__metaclass(CItemFieldValuePixbuf), *m_Model[j]))
		{
			// instanciate the cell renderer
			GtkCellRenderer *inGtkCellRenderer = ::gtk_cell_renderer_pixbuf_new ();

			// pack the renderer into the layout
			::gtk_cell_layout_pack_start (GTK_CELL_LAYOUT(inGtkWidget), inGtkCellRenderer, FALSE);

			// set the cell layout attribute
			::gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT(inGtkWidget), inGtkCellRenderer, "pixbuf", j, NULL);
		}
	}

	// ok
	return true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// model access
//-----------------------------------------------------------------------------------------------------------------------------------------
CMetaClasses CComboBox::GetModel () const
{
	// ok
	return m_Model;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// index selection
//-----------------------------------------------------------------------------------------------------------------------------------------
void CComboBox::Select (const SInt16 inIndex)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// set the property
	if (inGtkWidget != NULL) ::gtk_combo_box_set_active (GTK_COMBO_BOX(inGtkWidget), inIndex);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// item selection
//-----------------------------------------------------------------------------------------------------------------------------------------
void CComboBox::Select (const CComboBoxItem *inComboBoxItem)
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// set the property
	if (inGtkWidget != NULL && inComboBoxItem != NULL)
		::gtk_combo_box_set_active_iter (GTK_COMBO_BOX(inGtkWidget), inComboBoxItem -> GetGtkTreeIter());
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// selection index
//-----------------------------------------------------------------------------------------------------------------------------------------
SInt16 CComboBox::GetSelectionIndex () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// ok ?
	return inGtkWidget != NULL ? ::gtk_combo_box_get_active (GTK_COMBO_BOX(inGtkWidget)) : -1;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// selection item
//-----------------------------------------------------------------------------------------------------------------------------------------
CComboBoxItem *	CComboBox::GetSelectionItem () const
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	if (inGtkWidget == NULL) return NULL;

	// get the active iter
	GtkTreeIter inGtkTreeIter; if (!::gtk_combo_box_get_active_iter (GTK_COMBO_BOX(inGtkWidget), &inGtkTreeIter)) return NULL;

	// ok ?
	return CComboBoxItem::GetComboBoxItem (&inGtkTreeIter);
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// popup
//-----------------------------------------------------------------------------------------------------------------------------------------
void CComboBox::Popup ()
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	if (inGtkWidget == NULL) return;

	// popup
	::gtk_combo_box_popup (GTK_COMBO_BOX(inGtkWidget));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// popdown
//-----------------------------------------------------------------------------------------------------------------------------------------
void CComboBox::Popdown ()
{
	// get our gtk widget
	GtkWidget *inGtkWidget (GetGtkWidget());

	// check pointer
	if (inGtkWidget == NULL) return;
	
	// popdown
	::gtk_combo_box_popdown (GTK_COMBO_BOX(inGtkWidget));
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// serialization
//-----------------------------------------------------------------------------------------------------------------------------------------
void CComboBox::Serialize (CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE
{
	// request analyse
	switch (inMode)
	{
		// xml dump	
		case XML_WRITE :
		{
			// generic call first
			CWidget::Serialize (ioXMLElementNode, XML_WRITE);

			// create a new xml element node for this serialization level
			CXMLElement *newXMLElement = new CXMLElement (ioXMLElementNode, XML_COMBOBOX_ELEMENT);

			// write the selection attribute
			newXMLElement -> AddAttribute (XML_COMBOBOX_ATTR_SELECTION, CString((SInt32)GetSelectionIndex()));

			// modify the io xml element node so that the potential overloaded definitions will continue under the current node
			ioXMLElementNode = newXMLElement -> GetXMLElementNode ();

			// create a new xml element node for the ctreeview model serialization
			newXMLElement = new CXMLElement (ioXMLElementNode, XML_COMBOBOX_MODEL_ELEMENT);

			// serialize each model field
			for (size_t i=m_Model.GetLength(), j=0; i>0; i--, j++)
			{
				// instanciate a new xml element for the field model dump
				CXMLElement *newXMLModelElement = new CXMLElement (newXMLElement -> GetXMLElementNode(), 
										   XML_COMBOBOX_FIELDCLASS_ELEMENT);

				// append the attributes this instance is interested in
				newXMLModelElement -> AddAttribute (XML_COMBOBOX_FIELDCLASS_ATTR_NAME, 
								   (*m_Model[j])->ClassName);
				newXMLModelElement -> AddAttribute (XML_COMBOBOX_FIELDCLASS_ATTR_TAG, 
								    ::ClassTagToString((*m_Model[j])->ClassTag));
			}
		}
		break;

		// xml load
		case XML_READ :
		{
			// we first have to find the combo box xml node because the combo box model must be set before the owned items
			// are instanciated, so request the expected xml node
			CXMLElementNode *inXMLNode = ::xml_node_search (ioXMLElementNode, XML_COMBOBOX_ELEMENT);

			// check we got an expected ctreeview node
                        if (inXMLNode == NULL)
                                throw new CException (CString("CComboBox::Serialize, specified xml node is not a \"") +
                                                              XML_COMBOBOX_ELEMENT + CString("\" element one."), __exception(XMLPARSE));

			// get the selection if any
			SInt16 inSelection = ::xml_node_get_attribute (inXMLNode, XML_COMBOBOX_ATTR_SELECTION).GetValue().ToLong();

			// get the model xml node
			inXMLNode = ::xml_node_get_child (inXMLNode, XML_COMBOBOX_MODEL_ELEMENT);

			// check we got the expected xml node
                        if (inXMLNode == NULL)
                                throw new CException (CString("CComboBox::Serialize, specified xml node is not a \"") +
                                                              XML_COMBOBOX_MODEL_ELEMENT + CString("\" element one."), 
							      __exception(XMLPARSE));

			// get the combo box model
			m_Model = CMetaClasses(); for (size_t i=xml_node_get_children_number(inXMLNode), j=0; i>0; i--, j++)
			{
				// get the xml node
				CXMLElementNode *inChildXMLNode = ::xml_node_get_child (inXMLNode, j);

				// check it is the expected xml element node
				if (::xml_node_get_name(inChildXMLNode) != XML_COMBOBOX_FIELDCLASS_ELEMENT) continue;

				// get the associated metaclass
				const CMetaClass *inMetaClass = CMetaClass::GetMetaClass (::StringToClassTag (
					(::xml_node_get_attribute (inChildXMLNode, XML_COMBOBOX_FIELDCLASS_ATTR_TAG).GetValue())));

				// check we got the metaclass, add it to the input model list
				if (inMetaClass != NULL) m_Model += inMetaClass;
			}

			// set the tree view model from the serialization read in
			SetModel (m_Model);

			// the combo box model is set so we can call the generic definition that will indirectly instanciate the items
			CWidget::Serialize (ioXMLElementNode, XML_READ);

			// select the specified item if any
			Select (inSelection);
		}
		break;
	}	
	
}


