/*******************************************************************************************************************************************
 ctextview.h
 
 CClass					CClass
 |--CSerialized				|--CSerialized
    |--CMetaModule			   |--CMetaModule
       |--CObject			      |--CObjectListener
          |--CComponent				 |
   	     |--CControl			 |
	        |--CWidget			 |--CWidgetListener
	   	   |--CContainer		    |
		      |--CTextView		    |--CTextViewListener

 This class binds the GtkTextView widget. The member functions proposed here are just convenience ones; in order to use full 
 functionnalities of gtk on the text view and associated features, just think about inserting pure gtk functions calls in your main code
 if the proposed ones does not fullfil your needs...

 For now, the serialization - if called on such an instance - will just dump / load the text content of the view; there won't be any text 
 tag preservation; when reloaded, the potential children widgets of the view would be added at the end of the view.
*******************************************************************************************************************************************/

#ifndef __CTEXTVIEW_H__
#define __CTEXTVIEW_H__

#include "ccontainer.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// text view xml serialization constants
//-----------------------------------------------------------------------------------------------------------------------------------------
// <ctextview>
//    text of the view
// </ctextview>
//-----------------------------------------------------------------------------------------------------------------------------------------
static CString XML_TEXTVIEW_ELEMENT		("ctextview");

//------------------------------------------------------------------------------------------------------------------------------------------
// CTextViewListener class
//------------------------------------------------------------------------------------------------------------------------------------------
class CTextViewListener : public CWidgetListener
{
	// instanciation section
	public :

		CTextViewListener		();
		virtual ~CTextViewListener	() =0;

	// specific listener handling
	public :

		// called when the text buffer of the inSender text view has changed
		virtual void 			OnChange			(CObject *inSender)					{ }

		// called when the specified string has been inserted into the text buffer of the inSender text view at given offset
		virtual void			OnInsert			(CObject *inSender, const CString &inString,
										 const SInt32 inOffset)					{ }

		// called when the specified range has been deleted from the inSender text view
		virtual void			OnDelete 			(CObject *inSender, const SInt32 inStart, 
										 const SInt32 inEnd)					{ }

		// called when the inSender text view flips the modified bit (see SetModified / GetModified)
		virtual void			OnModifiedChanged		(CObject *inSender)					{ }

		
		// called on the inSender text view on copy to clipboard
		virtual void			OnCopyClipboard			(CObject *inSender)					{ }

		// called on the inSender text view on cut to clipboard
		virtual void			OnCutClipboard			(CObject *inSender)					{ }

		// called on the inSender text view on paste from clipboard
		virtual void			OnPasteClipboard		(CObject *inSender)					{ }


		// called when a tag has been added to the inSender text view associated text buffer tag array
		virtual void			OnTextTagTableAdded		(CObject *inSender, GtkTextTag *inGtkTextTag)		{ }

		// called when a tag has been removed from the inSender text view associated text buffer tag array
		virtual void			OnTextTagTableRemoved		(CObject *inSender, GtkTextTag *inGtkTextTag)		{ }

		// called when a tag has been changed on the inSender text view associated text buffer tag array
		virtual void			OnTextTagTableChanged		(CObject *inSender, GtkTextTag* inGtkTextTag)		{ }

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('_txw', CTextViewListener, CWidgetListener);

//------------------------------------------------------------------------------------------------------------------------------------------
// CTextView class
//------------------------------------------------------------------------------------------------------------------------------------------
class CTextView : public CContainer
{
	// instanciation section
	public :

		CTextView			(CContainer *inOwner=NULL, const CTextViewListener *inListener=NULL);
		CTextView			(CContainer *inOwner, const GtkTextBuffer *inGtkTextBuffer,
						 const CTextViewListener *inListener=NULL);
		virtual ~CTextView		();

	// protected widget requests handling
	protected :

		// gtk widget instanciation and initialization
		virtual GtkWidget *		PerformWidgetInstanciate	();
		virtual void			PerformWidgetInitialize		();

	// protected container requests handling
	protected :

		// gtk widget addon / removal
		virtual void			PerformContainerAdd		(CWidget *);
		virtual void			PerformContainerRemove		(CWidget *);

	// cobject redefinition
	protected :

		// listener affectation, must be derived from CTextViewListener
		virtual const CMetaClass *	ListenerMustBe			() const;

	// ccomponent redefinition
	public :

		// expected owner type : CContainer
		virtual CMetaClasses		OwnerMustBe			() const;

	// ccontainer redefinition
	public :

		// no restriction about children number
		virtual SInt16			GetGtkChildrenNumberLeft	() const;

	// text view specifics attributes access
	public :

		// get the buffer associated gtk table tags
		GtkTextTagTable *		GetGtkTextTable			() const;

		// get the associated gtk buffer
		GtkTextBuffer *			GetGtkTextBuffer		() const;

	// text view specifics
	public :

		// simple full text affectation
		virtual void			SetText				(const CString &inText);

		// text insertion at given offset (-1 to insert at the end of the text buffer)
		virtual void			InsertText			(const CString &inText, const SInt32 inOffset=0);

		// filtered text access : excludes undisplayed text (text marked with tags that set the invisibility attribute) if
		// inIncludeHiddenChars is false; does not include characters representing embedded images, so byte and character 
		// indexes into the returned string do not correspond to byte and character indexes into the buffer.
		virtual CString			GetText				(const SInt32 inStart=0, const SInt32 inEnd=-1,
										 const bool inIncludeHiddenChars=false) const;
		
		// full text access : excludes undisplayed text (text marked with tags that set the invisibility attribute) if
		// inIncludeHiddenChars is false; the returned string includes a 0xFFFC character whenever the buffer contains embedded
		// images, so byte and character indexes into the returned string do correspond to byte and character indexes into the
		// buffer; note that 0xFFFC can occur in normal text as well, so it is not a reliable indicator that a pixbuf or widget 
		// is in the buffer.
		virtual CString			GetSlice			(const SInt32 inStart=0, const SInt32 inEnd=-1,
										 const bool inIncludeHiddenChars=false) const;
	
		// apply a tag on the given text range (see the public attributes section below for commodity tags)
		virtual void			ApplyTag			(const GtkTextTag *inGtkTextTag, const SInt32 inStart=0,
										 const SInt32 inEnd=-1);

		// remove a tag from the given text range (see the public attributes section below for commodity tags)
		virtual void			RemoveTag			(const GtkTextTag *inGtkTextTag, const SInt32 inStart=0,
										 const SInt32 inEnd=-1);

		// indicates whether the buffer has been modified since the last call to SetModified (false); used for example to 
		// enable a "save" function in a text editor
		virtual void			SetModified			(const bool);
		virtual bool			GetModified			() const;

		// copy, cut, paste
		virtual void			Copy				() const;
		virtual void			Cut				();
		virtual void			Paste				();

	// cserialized redefinition
	public :

		// text view serialization
		virtual void			Serialize			(CXMLElementNode *&, const int) THROWABLE;
		
	// special operators on text view
	public :

		// string affectation
		CTextView &			operator =			(const CString &);
		CTextView &			operator +=			(const CString &);
		CTextView &			operator <<			(CFile &);
		CTextView &			operator >>			(CFile &);

	// protected section
	protected :

		// the text view tag table and the handled text buffer
		static GtkTextTagTable *	m_GtkTextTagTable;
		GtkTextBuffer *			m_GtkTextBuffer;

	// accesible public attributes : text view pre-instanciated tags (accessible only when one text view has been instanciated at 
	// least and attached to the gtkol hierarchy); those attributes are automatically added to the associated m_GtkTextTagTable : 
	// their names are the names of the attributes themself; of course this list is not restrictive and is available only for
	// commodity, you can add any more...
	public :

		static GtkTextTag *		m_GtkTextTagItalic;
		static GtkTextTag *		m_GtkTextTagUnderline;
		static GtkTextTag *		m_GtkTextTagDoubleUnderline;
		static GtkTextTag *		m_GtkTextTagStrikeThrough;

		static GtkTextTag *		m_GtkTextTagWeightUltraLight;
		static GtkTextTag *		m_GtkTextTagWeightLight;
		static GtkTextTag *		m_GtkTextTagWeightNormal;
		static GtkTextTag *		m_GtkTextTagWeightBold;
		static GtkTextTag *		m_GtkTextTagWeightUltraBold;
		static GtkTextTag *		m_GtkTextTagWeightHeavy;

		static GtkTextTag *		m_GtkTextTagScaleXXSmall;
		static GtkTextTag *		m_GtkTextTagScaleXSmall;
		static GtkTextTag *		m_GtkTextTagScaleSmall;
		static GtkTextTag *		m_GtkTextTagScaleMedium;
		static GtkTextTag *		m_GtkTextTagScaleLarge;
		static GtkTextTag *		m_GtkTextTagScaleXLarge;
		static GtkTextTag *		m_GtkTextTagScaleXXLarge;

		static GtkTextTag *		m_GtkTextTagJustificationLeft;
		static GtkTextTag *		m_GtkTextTagJustificationRight;
		static GtkTextTag *		m_GtkTextTagJustificationCenter;

		static GtkTextTag *		m_GtkTextTagNotEditable;

		static GtkTextTag *		m_GtkTextTagWrapWord;
		static GtkTextTag *		m_GtkTextTagWrapChar;
		static GtkTextTag *		m_GtkTextTagWrapNone;

		static GtkTextTag *		m_GtkTextTagSupScript;
		static GtkTextTag *		m_GtkTextTagSubScript;

		static GtkTextTag *		m_GtkTextTagFgDarkBlue;
		static GtkTextTag *		m_GtkTextTagFgBlue;
		static GtkTextTag *		m_GtkTextTagFgLightBlue;
		static GtkTextTag *		m_GtkTextTagFgDarkRed;
		static GtkTextTag *		m_GtkTextTagFgRed;
		static GtkTextTag *		m_GtkTextTagFgDarkGreen;
		static GtkTextTag *		m_GtkTextTagFgGreen;
		static GtkTextTag *		m_GtkTextTagFgLightGreen;
		static GtkTextTag *		m_GtkTextTagFgBlack;
		static GtkTextTag *		m_GtkTextTagFgWhite;

		static GtkTextTag *		m_GtkTextTagBgDarkBlue;
		static GtkTextTag *		m_GtkTextTagBgBlue;
		static GtkTextTag *		m_GtkTextTagBgLightBlue;
		static GtkTextTag *		m_GtkTextTagBgDarkRed;
		static GtkTextTag *		m_GtkTextTagBgRed;
		static GtkTextTag *		m_GtkTextTagBgDarkGreen;
		static GtkTextTag *		m_GtkTextTagBgGreen;
		static GtkTextTag *		m_GtkTextTagBgLightGreen;
		static GtkTextTag *		m_GtkTextTagBgBlack;
		static GtkTextTag *		m_GtkTextTagBgWhite;

		static GtkTextTag *		m_GtkTextTagMonospace;

	// listener section
	protected :

		static void        		OnChange			(GtkTextBuffer *, gpointer);
		static void        		OnInsert			(GtkTextBuffer *, GtkTextIter *, gchar *, gint, gpointer);
		static void			OnDelete			(GtkTextBuffer *, GtkTextIter *, GtkTextIter *, gpointer);
		static void			OnModifiedChanged		(GtkTextBuffer *, gpointer);

		static void			OnCopyClipboard			(GtkTextView *, gpointer);
		static void			OnCutClipboard			(GtkTextView *, gpointer);	
		static void			OnPasteClipboard		(GtkTextView *, gpointer);

		static void			OnTextTagTableAdded		(GtkTextTagTable *, GtkTextTag *, gpointer);
		static void			OnTextTagTableRemoved		(GtkTextTagTable *, GtkTextTag *, gpointer);
		static void			OnTextTagTableChanged		(GtkTextTagTable *, GtkTextTag *, gboolean, gpointer);

		// metaclass association
		SECTION_DYNAMIC_METACLASS;
};

// metaclass and class tag declarations
DECLARE_DYNAMIC_METACLASS ('txvw', CTextView, CContainer);

#endif
