/**
 * @file libgalago-gtk/galago-gtk-person-list.c
 *       Person list widget
 *
 * @Copyright (C) 2005-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-person-list.h>
#include <libgalago-gtk/galago-gdk-pixbuf.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <libgalago-gtk/galago-gtk-stock.h>
#include <gtk/gtk.h>

struct _GalagoGtkPersonListPriv
{
	GalagoPerson *person;

	gboolean show_offline;

	gboolean populate_lock;
};

enum
{
	COLUMN_DATA,
	COLUMN_ICON,
	COLUMN_TEXT,
	NUM_COLUMNS
};

enum
{
	SELECTION_CHANGED,
	PERSON_ACTIVATED,
	LAST_SIGNAL
};

static void galago_gtk_person_list_class_init(GalagoGtkPersonListClass *klass);
static void galago_gtk_person_list_init(GalagoGtkPersonList *list);
static void galago_gtk_person_list_finalize(GObject *obj);
static void galago_gtk_person_list_destroy(GtkObject *obj);
static void galago_gtk_person_list_row_activated(GtkTreeView *treeview,
												 GtkTreePath *path,
												 GtkTreeViewColumn *column);

static GtkTreeViewClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

GType
galago_gtk_person_list_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkPersonListClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_person_list_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkPersonList),
			0,
			(GInstanceInitFunc)galago_gtk_person_list_init
		};

		type = g_type_register_static(GTK_TYPE_TREE_VIEW,
									  "GalagoGtkPersonList", &info, 0);
	}

	return type;
}

static void
galago_gtk_person_list_class_init(GalagoGtkPersonListClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkTreeViewClass *treeview_class;

	galago_gtk_stock_init();

	parent_class = g_type_class_peek_parent(klass);

	gobject_class  = G_OBJECT_CLASS(klass);
	object_class   = GTK_OBJECT_CLASS(klass);
	treeview_class = GTK_TREE_VIEW_CLASS(klass);

	gobject_class->finalize = galago_gtk_person_list_finalize;

	object_class->destroy = galago_gtk_person_list_destroy;

	treeview_class->row_activated = galago_gtk_person_list_row_activated;

	signals[SELECTION_CHANGED] =
		g_signal_new("selection-changed",
					 G_TYPE_FROM_CLASS(gobject_class),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoGtkPersonListClass,
									 selection_changed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	signals[PERSON_ACTIVATED] =
		g_signal_new("person-activated",
					 G_TYPE_FROM_CLASS(gobject_class),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoGtkPersonListClass,
									 person_activated),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);
}

static void
selection_changed_cb(GtkTreeSelection *sel, GalagoGtkPersonList *list)
{
	GtkTreeModel *model;
	GtkTreeIter iter;
	GalagoPerson *person;

	if (!gtk_tree_selection_get_selected(sel, &model, &iter))
		return;

	gtk_tree_model_get(model, &iter,
					   COLUMN_DATA, &person,
					   -1);

	galago_gtk_person_list_set_person(list, person);

	g_signal_emit(list, signals[SELECTION_CHANGED], 0);
}

static void
populate_persons_list(GalagoGtkPersonList *list)
{
	GList *l;
	GtkListStore *model;

	if (list->priv->populate_lock)
		return;

	list->priv->populate_lock = TRUE;

	model = GTK_LIST_STORE(gtk_tree_view_get_model(GTK_TREE_VIEW(list)));

	gtk_list_store_clear(model);

	for (l = galago_get_people(GALAGO_REMOTE, TRUE);
		 l != NULL;
		 l = l->next)
	{
		GalagoPerson *person = (GalagoPerson *)l->data;
		GdkPixbuf *pixbuf;
		GtkTreeIter iter;

		if (!galago_person_has_accounts(person, TRUE) ||
			galago_person_get_display_name(person) == NULL)
		{
			continue;
		}

		if (!list->priv->show_offline)
		{
			GList *l2;
			gboolean offline = TRUE;

			for (l2 = galago_person_get_accounts(person, TRUE);
				 l2 != NULL;
				 l2 = l2->next)
			{
				GalagoAccount *account = (GalagoAccount *)l2->data;

				if (galago_account_is_connected(account))
				{
					offline = FALSE;
					break;
				}
			}
		}

		/* TODO: Make this a utility function, and be smart about gender. */
		pixbuf = gtk_widget_render_icon(GTK_WIDGET(list),
										GALAGO_GTK_STOCK_PERSON,
										GTK_ICON_SIZE_MENU, NULL);

		gtk_list_store_append(model, &iter);
		gtk_list_store_set(model, &iter,
						   COLUMN_DATA, person,
						   COLUMN_ICON, pixbuf,
						   COLUMN_TEXT,
						   galago_person_get_display_name(person),
						   -1);

		if (pixbuf != NULL)
			g_object_unref(G_OBJECT(pixbuf));
	}

	list->priv->populate_lock = FALSE;
}

static void
galago_gtk_person_list_init(GalagoGtkPersonList *list)
{
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *column;
	GtkListStore *model;
	GtkTreeSelection *sel;

	list->priv = g_new0(GalagoGtkPersonListPriv, 1);
	list->priv->show_offline = TRUE;

	model = gtk_list_store_new(NUM_COLUMNS, G_TYPE_POINTER,
							   GDK_TYPE_PIXBUF, G_TYPE_STRING);
	gtk_tree_view_set_model(GTK_TREE_VIEW(list), GTK_TREE_MODEL(model));
	gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(model),
										 COLUMN_TEXT, GTK_SORT_ASCENDING);
	g_object_unref(G_OBJECT(model));

	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(list), FALSE);
	gtk_tree_view_set_enable_search(GTK_TREE_VIEW(list), TRUE);
	gtk_tree_view_set_search_column(GTK_TREE_VIEW(list),
									COLUMN_TEXT);

	column = gtk_tree_view_column_new();
	gtk_tree_view_append_column(GTK_TREE_VIEW(list), column);

	renderer = gtk_cell_renderer_pixbuf_new();
	gtk_tree_view_column_pack_start(column, renderer, FALSE);
	gtk_tree_view_column_add_attribute(column, renderer,
									   "pixbuf", COLUMN_ICON);

	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_column_pack_start(column, renderer, TRUE);
	gtk_tree_view_column_add_attribute(column, renderer,
									   "text", COLUMN_TEXT);

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(list));
	g_signal_connect(G_OBJECT(sel), "changed",
					 G_CALLBACK(selection_changed_cb), list);

	populate_persons_list(list);
}

static void
galago_gtk_person_list_finalize(GObject *obj)
{
	GalagoGtkPersonList *list;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PERSON_LIST(obj));

	list = GALAGO_GTK_PERSON_LIST(obj);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_person_list_destroy(GtkObject *obj)
{
	GalagoGtkPersonList *list;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PERSON_LIST(obj));

	list = GALAGO_GTK_PERSON_LIST(obj);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

static void
galago_gtk_person_list_row_activated(GtkTreeView *treeview, GtkTreePath *path,
									 GtkTreeViewColumn *column)
{
	GalagoGtkPersonList *person_list;

	person_list = GALAGO_GTK_PERSON_LIST(treeview);

	g_signal_emit(person_list, signals[PERSON_ACTIVATED], 0,
				  galago_gtk_person_list_get_person(person_list));

	if (parent_class->row_activated != NULL)
		parent_class->row_activated(treeview, path, column);
}

GtkWidget *
galago_gtk_person_list_new(void)
{
	return GTK_WIDGET(g_object_new(GALAGO_GTK_TYPE_PERSON_LIST, NULL));
}

void
galago_gtk_person_list_set_person(GalagoGtkPersonList *list,
								  GalagoPerson *person)
{
	GtkTreeSelection *sel;

	g_return_if_fail(list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PERSON_LIST(list));

	if (list->priv->person == person)
		return;

	list->priv->person = person;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(list));

	if (person == NULL)
	{
		gtk_tree_selection_unselect_all(sel);
	}
	else
	{
		GtkTreeIter iter;
		GtkTreeModel *model;
		GalagoPerson *temp_person;
		gboolean valid;

		model = gtk_tree_view_get_model(GTK_TREE_VIEW(list));

		for (valid = gtk_tree_model_get_iter_first(model, &iter);
			 valid;
			 valid = gtk_tree_model_iter_next(model, &iter))
		{
			gtk_tree_model_get(model, &iter,
							   COLUMN_DATA, &temp_person,
							   -1);

			if (person == temp_person)
			{
				gtk_tree_selection_select_iter(sel, &iter);
				return;
			}
		}

		/* TODO: ASSERT */
	}
}

void
galago_gtk_person_list_set_show_offline(GalagoGtkPersonList *list,
										gboolean show_offline)
{
	g_return_if_fail(list != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PERSON_LIST(list));

	if (list->priv->show_offline == show_offline)
		return;

	list->priv->show_offline = show_offline;

	populate_persons_list(list);
}

GalagoPerson *
galago_gtk_person_list_get_person(const GalagoGtkPersonList *list)
{
	g_return_val_if_fail(list != NULL,                    NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_PERSON_LIST(list), NULL);

	return list->priv->person;
}

gboolean
galago_gtk_person_list_get_show_offline(const GalagoGtkPersonList *list)
{
	g_return_val_if_fail(list != NULL,                    FALSE);
	g_return_val_if_fail(GALAGO_GTK_IS_PERSON_LIST(list), FALSE);

	return list->priv->show_offline;
}
