/* 

                          Firewall Builder

                 Copyright (C) 2001 Vadim Zaliva, Vadim Kurland

  Author:  Vadim Zaliva lord@crocodile.org

  $Id: dns.hh,v 1.4 2001/12/19 12:54:00 lord Exp $


  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#ifndef __DNS_HH_FLAG__
#define __DNS_HH_FLAG__

#include <fwbuilder/libfwbuilder-config.h>

#include <netdb.h>
#include <arpa/nameser.h>

#ifdef DEBIAN
// Ddebian Pototo workaround
#undef __THROW
#define __THROW

extern "C" {
    extern int gethostbyname_r(__const char *__restrict __name,
                               struct hostent *__restrict __result_buf,
                               char *__restrict __buf, size_t __buflen,
                               struct hostent **__restrict __result,
                               int *__restrict __h_errnop);
    
    extern int gethostbyaddr_r(__const char *__restrict __addr, size_t __len,
                               int __type,
                               struct hostent *__restrict __result_buf,
                               char *__restrict __buf, size_t __buflen,
                               struct hostent **__restrict __result,
                               int *__restrict __h_errnop);
}
#endif

#include <resolv.h>

#include <fwbuilder/IPAddress.hh>
#include <fwbuilder/FWException.hh>
#include <fwbuilder/BackgroundOp.hh>
#include <fwbuilder/ThreadTools.hh>

#include <vector>
#include <map>
#include <queue>

#ifndef RES_DFLRETRY 
# define RES_DFLRETRY 1
#endif

namespace libfwbuilder
{

class HostEnt
{
    public:
    
    std::string           name    ;
    std::set<std::string> aliases ;
};

/**
 * This is abstract class
 */
class DNS : public BackgroundOp
{
    public:

    DNS::DNS();

    /**
     * Finds IP adddresses of the host with given host name.
     * This operation does not run in backgound.
     */
    static std::vector<IPAddress> getHostByName(const std::string &name) throw(FWException);

    /**
     * Find all host names of host with given IP.
     * This operation does not run in backgound.
     */
    static HostEnt getHostByAddr(const IPAddress &addr) throw(FWException);

    /**
     * Return name of host with given IP.
     */
    static HostEnt getHostByAddr(const IPAddress &addr, int retries_, int timeout_) throw(FWException);

    /**
     * Finds NS records for given domain
     */
    std::multimap<std::string, IPAddress> getNS(const std::string &domain, Logger *logger, int retries_=RES_DFLRETRY, int timeout_=RES_TIMEOUT) throw(FWException);

    /**
     * Attempts to fetch zone for given domain for
     * it's name server. If succeeded, finds all 'A'
     * records in it and returs hostname/IPAddress pairs.
     */
    std::map<std::string, std::set<IPAddress> > findA(const std::string &domain, Logger *logger,  int retries_=RES_DFLRETRY, int timeout_=RES_TIMEOUT) throw(FWException);

    /**
     * Attempts to fetch zone for given domain for
     * specific server. If succeeded, finds all 'A'
     * records in it and returs hostname/IPAddress pairs.
     */
    std::map<std::string, std::set<IPAddress> > findA(const std::string &domain, const IPAddress &ns, Logger *logger,  int retries_=RES_DFLRETRY, int timeout_=RES_TIMEOUT) throw(FWException);

    protected:

    static std::string getErrorMessage(int rcode);

    private:

    static const size_t RSP_BUF_SIZE;

};

class DNS_findA_query : public DNS
{
    public:

    explicit DNS_findA_query();

    DNS_findA_query(const std::string &domain_, const IPAddress &ns_, int retries_=RES_DFLRETRY, int timeout_=RES_TIMEOUT);

    void init(const std::string &domain_, const IPAddress &ns_, int retries_=RES_DFLRETRY, int timeout_=RES_TIMEOUT);
    
    virtual void run_impl(Logger *logger) throw(FWException);
    
    std::map<std::string, std::set<IPAddress> > getResult() { return result; }

    private:

    std::map<std::string, std::set<IPAddress> > result;    
    std::string domain;
    int retries;
    int timeout;
    IPAddress ns;
};

class DNS_getNS_query : public DNS
{
    public:

    DNS_getNS_query(const std::string &domain_, int retries_=RES_DFLRETRY, int timeout_=RES_TIMEOUT);
    
    virtual void run_impl(Logger *logger) throw(FWException);
    
    std::multimap<std::string, IPAddress> getResult() { return result; }
    
    private:
    
    std::multimap<std::string, IPAddress> result;    
    std::string domain;
    int retries;
    int timeout;
};

class DNS_bulkBackResolve_query : public DNS
{
    friend void *DNS_bulkBackResolve_Thread(void *);

    public:
    
    DNS_bulkBackResolve_query(std::set<IPAddress>, unsigned int nthreads,  
			      int retries_=RES_DFLRETRY,
			      int timeout_=RES_TIMEOUT);
    virtual ~DNS_bulkBackResolve_query();
    
    /**
     * Resolves all hosts in list. If resolving at least of one
     * of them failed - exception is thrown.
     * Even when exception is thrown, successfully resolved hosts
     * are returned.
     */
    virtual void run_impl(Logger *logger) throw(FWException);
    
    std::map<IPAddress, HostEnt> getResult() { return result; }
    std::set<IPAddress> getFailed() { return failed; }
    
    protected:
    
    std::map<IPAddress, HostEnt> result;
    Mutex failed_mutex;
    
    std::set<IPAddress> failed;
    Mutex result_mutex;
    
    std::queue<IPAddress> ips;
    Mutex queue_mutex;

    int retries;
    int timeout;
    
    private:

    Mutex running_mutex         ;
    Cond  running_cond          ;
    unsigned int runnning_count ;
    
    unsigned int nthreads;
};

}

#endif



