/* 

                          Firewall Builder

                 Copyright (C) 2001 Vadim Zaliva, Vadim Kurland

  Author:  Vadim Zaliva lord@crocodile.org

  $Id: FWObjectDatabase.cc,v 1.6 2001/12/25 18:58:25 lord Exp $

  This program is free software which we release under the GNU General Public
  License. You may redistribute and/or modify this program under the terms
  of that license as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  To get a copy of the GNU General Public License, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>

#include <fwbuilder/libfwbuilder-config.h>

#include <fwbuilder/FWObject.hh>
#include <fwbuilder/FWObjectDatabase.hh>

#include <fwbuilder/FWObject.hh>
#include <fwbuilder/Interval.hh>
#include <fwbuilder/ICMPService.hh>
#include <fwbuilder/IPService.hh>
#include <fwbuilder/TCPService.hh>
#include <fwbuilder/UDPService.hh>
#include <fwbuilder/CustomService.hh>
#include <fwbuilder/FWReference.hh>
#include <fwbuilder/FWObjectReference.hh>
#include <fwbuilder/FWServiceReference.hh>
#include <fwbuilder/FWIntervalReference.hh>
#include <fwbuilder/Host.hh>
#include <fwbuilder/Interface.hh>
#include <fwbuilder/InterfacePolicy.hh>
#include <fwbuilder/Group.hh>
#include <fwbuilder/Rule.hh>
#include <fwbuilder/RuleElement.hh>
#include <fwbuilder/RuleSet.hh>
#include <fwbuilder/FWOptions.hh>
#include <fwbuilder/Firewall.hh>
#include <fwbuilder/NAT.hh>
#include <fwbuilder/Policy.hh>
#include <fwbuilder/ObjectGroup.hh>
#include <fwbuilder/ServiceGroup.hh>
#include <fwbuilder/IntervalGroup.hh>
#include <fwbuilder/Network.hh>
#include <fwbuilder/XMLTools.hh>

using namespace std;
using namespace libfwbuilder;


FWObjectDatabase*  FWObjectDatabase::db        = NULL;
const char*        FWObjectDatabase::TYPENAME  = {"FWObjectDatabase"};
long               FWObjectDatabase::IDcounter = 0;
//Mutex FWObjectDatabase::IDmutex;
const string FWObjectDatabase::DTD_FILE_NAME  = "fwbuilder.dtd"    ;


FWObjectDatabase::FWObjectDatabase():FWObject()
{
    assert(db==NULL);
    db=this;

    IDcounter = time(NULL);
    data_file = "";

    setName(TYPENAME);

    std.rootId          = "root";

    std.AnyNetworkId    = "sysid0";
    std.AnyIPServiceId  = "sysid1";
    std.AnyIntervalId   = "sysid2";

    std.ObjectsId       = "stdid01";
    std.HostsId         = "stdid02";
    std.NetworksId      = "stdid03";
    std.ObjectGroupsId  = "stdid04";
    std.ServicesId      = "stdid05";
    std.IPServicesId    = "stdid06";
    std.ICMPServicesId  = "stdid07";
    std.UDPServicesId   = "stdid08";
    std.TCPServicesId   = "stdid09";
    std.ServiceGroupsId = "stdid10";
    std.CustomServicesId= "stdid13";

    std.TimeId          = "stdid11";
    std.FirewallsId     = "stdid12";

    setId( std.rootId );
    setDirty(false);
}

void FWObjectDatabase::setFileName(const string &filename)
{
    data_file = filename;
}

const string &FWObjectDatabase::getFileName() 
{ 
    return data_file; 
}

const string FWObjectDatabase::getFileDir() 
{ 
    string::size_type i = data_file.rfind('/');
    if (i==string::npos || i==0) 
	return "";
    else 
	return data_file.substr(0,i);
}

void FWObjectDatabase::load(const string &f, XMLTools::UpgradePredicate *upgrade) throw(FWException)
{
    if(f=="") 
        return;
    
    xmlDocPtr doc=XMLTools::loadFile(f, FWObjectDatabase::TYPENAME, FWObjectDatabase::DTD_FILE_NAME, upgrade);
    
    xmlNodePtr root = xmlDocGetRootElement(doc);
    
    if(!root || !root->name || strcmp(FROMXMLCAST(root->name), FWObjectDatabase::TYPENAME)!=SAME)
    {
	xmlFreeDoc(doc);
        throw FWException("Data file have invalid structure: "+f);
    }
    
    clearChildren();
    
    fromXML(root);
    setDirty(false,true);  // clear dirty flag for all objects, recursive oper.
    setFileName(f);
}

void FWObjectDatabase::saveXML(xmlDocPtr doc) throw(FWException)
{
    // create root node (with proper namespace)
    xmlNsPtr dtd = xmlNewNs(NULL, TOXMLCAST("http://www.fwbuilder.org/1.0/"), NULL);
    doc->children = xmlNewDocNode(doc, dtd, STRTOXMLCAST(getName()), NULL);

    toXML(xmlDocGetRootElement(doc));
    XMLTools::setDTD(doc, FWObjectDatabase::TYPENAME, FWObjectDatabase::DTD_FILE_NAME);
}

void FWObjectDatabase::saveFile(const string &filename) throw(FWException)
{
    xmlDocPtr doc = xmlNewDoc(TOXMLCAST("1.0"));
  
    doc->children = xmlNewDocNode(doc, NULL, STRTOXMLCAST(getName()), NULL);
    xmlNewNs(doc->children, TOXMLCAST("http://www.fwbuilder.org/1.0/"), NULL);

    toXML(xmlDocGetRootElement(doc));
    
    XMLTools::saveFile(doc, filename, FWObjectDatabase::TYPENAME, FWObjectDatabase::DTD_FILE_NAME);
    
    xmlFreeDoc(doc);  
    setDirty(false,true);  // clear dirty flag for all objects, recursive oper.
    
}

xmlNodePtr FWObjectDatabase::toXML(xmlNodePtr parent)
{
    FWObject *o;

    xmlNewProp(parent,NULL,NULL);

    xmlNewProp(parent, 
               TOXMLCAST("version") , 
               TOXMLCAST(LIBFWBUILDER_FORMAT_VERSION));

    string rootid = getId();

    xmlAttrPtr pr = xmlNewProp(parent, 
			       TOXMLCAST("id") , 
			       STRTOXMLCAST(rootid));

    xmlAddID(NULL, parent->doc, STRTOXMLCAST(rootid), pr);


    vector<FWObject*>::iterator j;
    for(j=begin(); j!=end(); ++j) 
        if ((o=(*j))!=NULL)   o->toXML(parent);
    
    return parent;
}

FWObject *FWObjectDatabase::createFromXML(xmlNodePtr data)
{
    FWObject *nobj;
    const char *n=FROMXMLCAST(data->name);
    if (!n) 
        return NULL;
    
    nobj=create(n);
    
    return(nobj);
}

string FWObjectDatabase::generateUniqueId()
{
    char res[20];

    //TODO: mutex could not be used since threads
    // library is not initialized.

//    IDmutex.lock();
    sprintf(res,"id%lX",IDcounter);
    IDcounter++;
//    IDmutex.unlock();
    
    return res;
}

FWObject *FWObjectDatabase::create(const string &type_name)
{
    const char *n=type_name.c_str();
    FWObject   *nobj;
    
    if(strcmp(Policy::TYPENAME,n)==SAME)    
    { nobj = new Policy();
    } else if(strcmp(InterfacePolicy::TYPENAME,n)==SAME)    
    { nobj = new InterfacePolicy();
    } else if(strcmp(NAT::TYPENAME,n)==SAME)    
    { nobj = new NAT();
    } else if(strcmp(PolicyRule::TYPENAME,n)==SAME)    
    { nobj = new PolicyRule();
    } else if(strcmp(NATRule::TYPENAME,n)==SAME)    
    { nobj = new NATRule();
    } else if (strcmp(RuleElementSrc::TYPENAME,n)==SAME)
    { nobj = new RuleElementSrc();
    } else if (strcmp(RuleElementDst::TYPENAME,n)==SAME)      
    { nobj = new RuleElementDst();
    } else if (strcmp(RuleElementSrv::TYPENAME,n)==SAME)
    { nobj = new RuleElementSrv();
    } else if (strcmp(RuleElementOSrc::TYPENAME,n)==SAME)
    { nobj = new RuleElementOSrc();
    } else if (strcmp(RuleElementODst::TYPENAME,n)==SAME)
    { nobj = new RuleElementODst();
    } else if (strcmp(RuleElementOSrv::TYPENAME,n)==SAME)
    { nobj = new RuleElementOSrv();
    } else if (strcmp(RuleElementTSrc::TYPENAME,n)==SAME)
    { nobj = new RuleElementTSrc();
    } else if (strcmp(RuleElementTDst::TYPENAME,n)==SAME)
    { nobj = new RuleElementTDst();
    } else if (strcmp(RuleElementTSrv::TYPENAME,n)==SAME)
    { nobj = new RuleElementTSrv();
    } else if (strcmp(RuleElementInterval::TYPENAME,n)==SAME)
    { nobj = new RuleElementInterval();
    } else if(strcmp(HostOptions::TYPENAME,n)==SAME)
    { nobj=new HostOptions();
    } else if(strcmp(FirewallOptions::TYPENAME,n)==SAME)
    { nobj=new FirewallOptions();
    } else if(strcmp(PolicyRuleOptions::TYPENAME,n)==SAME)
    { nobj=new PolicyRuleOptions();
    } else if(strcmp(NATRuleOptions::TYPENAME,n)==SAME)
    { nobj=new NATRuleOptions();
    } else if(strcmp(Host::TYPENAME,n)==SAME)
    { nobj=new Host();
    } else if(strcmp("AnyNetwork",n)==SAME)
    {
	nobj = new Network();
        nobj->setXMLName("AnyNetwork");
    } else if(strcmp("AnyIPService",n)==SAME)    
    { nobj=new IPService();
    nobj->setXMLName("AnyIPService");
    } else if(strcmp("AnyInterval",n)==SAME)    
    { nobj=new Interval();
    nobj->setXMLName("AnyInterval");
    } else if(strcmp(Firewall::TYPENAME,n)==SAME)    
    { nobj=new Firewall();
    } else if(strcmp(Network::TYPENAME,n)==SAME)    
    { nobj=new Network();
    } else if(strcmp(Interface::TYPENAME,n)==SAME)    
    { nobj=new Interface();
    } else if(strcmp(FWObjectReference::TYPENAME,n)==SAME)    
    { nobj=new FWObjectReference();
    } else if(strcmp(FWServiceReference::TYPENAME,n)==SAME)    
    { nobj=new FWServiceReference();
    } else if(strcmp(FWIntervalReference::TYPENAME,n)==SAME)    
    { nobj=new FWIntervalReference();
    } else if(strcmp(TCPService::TYPENAME,n)==SAME)    
    { nobj=new TCPService();
    } else if(strcmp(UDPService::TYPENAME,n)==SAME)    
    { nobj=new UDPService();
    } else if(strcmp(ICMPService::TYPENAME,n)==SAME)    
    { nobj=new ICMPService();
    } else if(strcmp(IPService::TYPENAME,n)==SAME)    
    { nobj=new IPService();
    } else if(strcmp(CustomService::TYPENAME,n)==SAME)    
    { nobj=new CustomService();
    } else if(strcmp(Interval::TYPENAME,n)==SAME)    
    { nobj=new Interval();
    } else if(strcmp(ObjectGroup::TYPENAME,n)==SAME)
    { nobj = new ObjectGroup();   
    } else if(strcmp(ServiceGroup::TYPENAME,n)==SAME)
    { nobj = new ServiceGroup();  
    } else if(strcmp(IntervalGroup::TYPENAME,n)==SAME)
    { nobj = new IntervalGroup();
    } else {   
        //TODO: throw exception
        nobj = NULL;
    }
    return nobj;
}
