/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "types.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>
#include "log.h"
#include "rmd160.h"
#include "tt.h"


/* --- Find by pointer --- */


int tt_has_child(TT *parent, TT *child)
{
  TT *tt;
  
  for (tt = tt_get_first_child(parent); tt; tt = tt_get_next(tt))
    if (tt == child) return(1);
  
  return(0);
}


/* --- Find exact match --- */


TT *tt_find_first_sibling(TT *tt, void *data, u32 len)
{
  for (tt = tt_get_first_sibling(tt); tt; tt = tt_get_next(tt))
    if (!tt_memcmp(tt, data, len)) break;

  return(tt);
}


TT *tt_find_last_sibling(TT *tt, void *data, u32 len)
{
  for (tt = tt_get_last_sibling(tt); tt; tt = tt_get_prev(tt))
    if (!tt_memcmp(tt, data, len)) break;

  return(tt);
}


TT *tt_find_next(TT *tt, void *data, u32 len)
{
  for (tt = tt_get_next(tt); tt; tt = tt_get_next(tt))
    if (!tt_memcmp(tt, data, len)) break;

  return(tt);
}


TT *tt_find_prev(TT *tt, void *data, u32 len)
{
  for (tt = tt_get_prev(tt); tt; tt = tt_get_prev(tt))
    if (!tt_memcmp(tt, data, len)) break;

  return(tt);
}


TT *tt_find_first_child(TT *tt, void *data, u32 len)
{
  tt = tt_get_first_child(tt);
  if (!tt) return(0);
  
  return(tt_find_first_sibling(tt, data, len));
}


TT *tt_find_last_child(TT *tt, void *data, u32 len)
{
  tt = tt_get_last_child(tt);
  if (!tt) return(0);
  
  return(tt_find_last_sibling(tt, data, len));
}


/* --- Find regexp match --- */


TT *tt_match_first_sibling(TT *tt, const char *regex)
{
  regex_t preg;
  int r;

  r = regcomp(&preg, regex, REG_EXTENDED | REG_NOSUB);
  if (r) return(0);

  for (tt = tt_get_first_sibling(tt); tt; tt = tt_get_next(tt))
    if (!tt_regcmp_precomp(tt, &preg)) break;
  
  regfree(&preg);
  return(tt);
}


TT *tt_match_last_sibling(TT *tt, const char *regex)
{
  regex_t preg;
  int r;

  r = regcomp(&preg, regex, REG_EXTENDED | REG_NOSUB);
  if (r) return(0);

  for (tt = tt_get_last_sibling(tt); tt; tt = tt_get_prev(tt))
    if (!tt_regcmp_precomp(tt, &preg)) break;
  
  regfree(&preg);
  return(tt);
}


TT *tt_match_next(TT *tt, const char *regex)
{
  regex_t preg;
  int r;

  r = regcomp(&preg, regex, REG_EXTENDED | REG_NOSUB);
  if (r) return(0);

  for (tt = tt_get_next(tt); tt; tt = tt_get_next(tt))
    if (!tt_regcmp_precomp(tt, &preg)) break;
  
  regfree(&preg);
  return(tt);
}


TT *tt_match_prev(TT *tt, const char *regex)
{
  regex_t preg;
  int r;

  r = regcomp(&preg, regex, REG_EXTENDED | REG_NOSUB);
  if (r) return(0);

  for (tt = tt_get_prev(tt); tt; tt = tt_get_prev(tt))
    if (!tt_regcmp_precomp(tt, &preg)) break;
  
  regfree(&preg);
  return(tt);
}


TT *tt_match_first_child(TT *tt, const char *regex)
{
  regex_t preg;
  int r;

  r = regcomp(&preg, regex, REG_EXTENDED | REG_NOSUB);
  if (r) return(0);

  for (tt = tt_get_first_child(tt); tt; tt = tt_get_next(tt))
    if (!tt_regcmp_precomp(tt, &preg)) break;
  
  regfree(&preg);
  return(tt);
}


TT *tt_match_last_child(TT *tt, const char *regex)
{
  regex_t preg;
  int r;

  r = regcomp(&preg, regex, REG_EXTENDED | REG_NOSUB);
  if (r) return(0);

  for (tt = tt_get_last_child(tt); tt; tt = tt_get_prev(tt))
    if (!tt_regcmp_precomp(tt, &preg)) break;
  
  regfree(&preg);
  return(tt);
}
