/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include "flux.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>


void _log_put_f(FILE *f, char *message, int level, unsigned long options)
{
  if (options & LOG_WITH_NAME && _log_opt.service_name)
    fprintf(f, "%s", _log_opt.service_name);

  if (options & LOG_WITH_PID) fprintf(f, "[%d]", (int) getpid());

  if (options & (LOG_WITH_NAME | LOG_WITH_PID))
  {
    if (level == LOG_INFO) fprintf(f, ": ");
    else fprintf(f, " ");
  }
  
  switch (level)
  {
    case LOG_EMERG:
      fprintf(f, "Emergency: ");
      break;
    case LOG_ALERT:
      fprintf(f, "Alert: ");
      break;
    case LOG_CRIT:
      fprintf(f, "Critical: ");
      break;
    case LOG_ERR:
      fprintf(f, "Error: ");
      break;
    case LOG_WARNING:
      fprintf(f, "Warning: ");
      break;
    case LOG_DEBUG:
      fprintf(f, "Debug: ");
      break;
    case LOG_NOTICE:
      fprintf(f, "Notice: ");
      break;
    default: /* No header for LOG_INFO */
      break;
  }

  fprintf(f, "%s\n", message);
  fflush(f);
  return;
}


void log_put(const char *format, ...)
{
  va_list args;

  if (!_log_opt_initialized)
    log_init(0, LOG_TYPE_GENERIC, LOG_TO_STDERR | LOG_TO_STDOUT, LOG_INFO);
  
  va_start(args, format);
  log_vput_opt(_log_opt.level, log_opt_get(), format, args);
  va_end(args);
  
  return;
}


void log_put_opt(int level, unsigned long options, const char *format, ...)
{
  va_list args;
  
  if (!_log_opt_initialized)
    log_init(0, LOG_TYPE_GENERIC, LOG_TO_STDERR | LOG_TO_STDOUT, LOG_INFO);
  
  va_start(args, format);
  if (!options) options = log_opt_get();
  log_vput_opt(level, options, format, args);
  va_end(args);
  
  return;
}


void log_vput_opt(int level, unsigned long options, const char *format, 
                  va_list args)
{
  char *str;

  if (!_log_opt_initialized)
    log_init(0, LOG_TYPE_GENERIC, LOG_TO_STDERR | LOG_TO_STDOUT, LOG_INFO);
  
  vasprintf(&str, format, args);

  /* FIXME: Escape any lingering % characters. */
  if (options & LOG_TO_SYSLOG) syslog(level, str);

  switch (level)
  {
    case LOG_EMERG:
    case LOG_ALERT:
    case LOG_CRIT:
    case LOG_ERR:
    case LOG_WARNING:
    case LOG_DEBUG:
      if (options & LOG_TO_STDERR) _log_put_f(stderr, str, level, options);
      break;
    
    default:
      if (options & LOG_TO_STDOUT) _log_put_f(stdout, str, level, options);
      break;
  }
  
  free(str);
  return;
}


void log_put_abort_do(const char *func, const char *message)
{
  char *message_abort, *info;

  asprintf(&info, "Aborting in %s. ", func);
  message_abort = malloc(strlen(message) + strlen(info) + 1);
  strcpy(message_abort, info);
  strcat(message_abort, message);

  if (!_log_opt_initialized)
    log_init(0, LOG_TYPE_GENERIC, LOG_TO_STDERR | LOG_TO_STDOUT, LOG_INFO);

  log_put_opt(LOG_CRIT, log_opt_get(), message_abort);
  
  free(info);
  closelog();
  abort();
}
