/** \file
 * Defines the interface for the Factory class.
 *
 * \author Martin F. Krafft <krafft@ailab.ch>
 * \date $Date: 2004/05/16 17:06:30 $
 * \version $Revision: 1.15 $
 */

#ifndef __INCLUDE_FACTORY_FACTORY_H__
#define __INCLUDE_FACTORY_FACTORY_H__

#include <factory++/builderbase.hpp>
#include <factory++/builderregistry.hpp>
#include <factory++/request.hpp>

#include <memory>

namespace Factory
{
  template <typename _Type, typename _Key>
    class Factory
    {
    private:
      typedef _Builder_base<_Type> const _Builder;
      typedef BuilderRegistry<_Key, _Type> _Registry;

    public:
      typedef _Type base_type;
      typedef _Key key_type;

      template <typename _SubType>
        static inline _Type*
        create(ConstructionRequest<_Key, _SubType> const __REQ)
        {
          if (__REQ.has_constructor())
            return create(__REQ.key(), __REQ.constructor());
          else
            return create(__REQ.key());
        }

      static inline _Type*
      create(_Key const& __KEY)
      {
        _Builder* __b = _S_registry().get_builder(__KEY);
        return (__b) ? __b->build() : 0;
      }

      static inline _Type*
      create(_Key const& __KEY, _Type const& __THAT)
      {
        _Builder* __b = _S_registry().get_builder(__KEY);
        return (__b) ? __b->build(__THAT) : 0;
      }

      template <typename _SubType>
        static inline _Type*
        create(_Key const& __KEY, Constructor<_SubType>& __constructor)
        {
          _Builder* __b = _S_registry().get_builder(__KEY);
          return (__b) ? __b->custom_build(__constructor) : 0;
        }

      template <typename _SubType>
        static inline _Type*
        create(_Key const& __KEY, Constructor<_SubType> const& __CONSTRUCTOR)
        {
          _Builder* __b = _S_registry().get_builder(__KEY);
          return (__b) ? __b->custom_build(__CONSTRUCTOR) : 0;
        }

      template <typename _SubType>
        static inline void
        destroy(DestructionRequest<_Key, _SubType> const __REQ,
                _Type* const __OBJ)
        {
          if (__REQ.has_destructor())
            destroy(__REQ.key(), __OBJ, __REQ.destructor());
          else
            destroy(__REQ.key(), __OBJ);
        }

      static inline void
      destroy(_Key const& __KEY, _Type* const __OBJ)
      {
        _Builder* __b = _S_registry().get_builder(__KEY);
        if (__b) __b->teardown(__OBJ);
      }
      
      template <typename _SubType>
        static inline void
        destroy(_Key const& __KEY, _SubType* const __OBJ,
                Destructor<_SubType>& __destructor)
        {
          _Builder* __b = _S_registry().get_builder(__KEY);
          if (__b) __b->custom_teardown(__OBJ, __destructor);
        }
      
      template <typename _SubType>
        static inline void
        destroy(_Key const& __KEY, _SubType* const __OBJ,
                Destructor<_SubType> const& __DESTRUCTOR)
        {
          _Builder* __b = _S_registry().get_builder(__KEY);
          if (__b) __b->custom_teardown(__OBJ, __DESTRUCTOR);
        }
      
      static inline ConstructionRequest<_Key, Constructor<_Type> const>
      make_construction_request(_Key const& __KEY)
      {
        return ConstructionRequest<_Key, Constructor<_Type> const>(__KEY);
      }

      template <typename _Constructor>
        static inline ConstructionRequest<_Key, _Constructor>
        make_construction_request(_Key const& __KEY,
                                  _Constructor& __constructor)
        {
          return ConstructionRequest<_Key, _Constructor>(__KEY, __constructor);
        }

      template <typename _Constructor>
        static inline ConstructionRequest<_Key, _Constructor const>
        make_construction_request(_Key const& __KEY,
                                  _Constructor const& __CONSTRUCTOR)
        {
          return ConstructionRequest<_Key, _Constructor const>(__KEY, __CONSTRUCTOR);
        }

      static inline DestructionRequest<_Key, Destructor<_Type> const>
      make_destruction_request(_Key const& __KEY)
      {
        return DestructionRequest<_Key, Destructor<_Type> const>(__KEY);
      }

      template <typename _Destructor>
        static inline DestructionRequest<_Key, _Destructor>
        make_destruction_request(_Key const& __KEY,
                                 _Destructor& __destructor)
        {
          return DestructionRequest<_Key, _Destructor>(__KEY, __destructor);
        }

      template <typename _Destructor>
        static inline DestructionRequest<_Key, _Destructor const>
        make_destruction_request(_Key const& __KEY,
                                 _Destructor const& __CONSTRUCTOR)
        {
          return DestructionRequest<_Key, _Destructor const>(__KEY, __CONSTRUCTOR);
        }

      static inline bool
      register_builder(_Key const& __KEY, _Builder& __BUILDER)
        { return _S_registry().register_builder(__KEY, &__BUILDER); }

      static inline bool
      unregister_builder(_Key const& __KEY)
        { return _S_registry().unregister_builder(__KEY); }

      static inline bool
      has_builder(_Key const& __KEY)
        { return _S_registry().has_builder(__KEY); }

    private:

      static _Registry& _S_registry()
      {
        static _Registry __registry;
        return __registry;
      }

    };
  
}; // namespace Factory

#endif // include guard

/* COPYRIGHT --
 *
 * This file is part of libfactory++, a C++ template factory framework.
 * libfactory++ is (c) 2004 Martin F. Krafft <krafft@ailab.ch>
 * and distributed under the terms of the Artistic Licence.
 * See the ./COPYING file in the source tree root for more information.
 *
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES
 * OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
