/** \file
 * Defines the interface for the Builder class.
 *
 * \author Martin F. Krafft <krafft@ailab.ch>
 * \date $Date: 2004/06/03 18:14:03 $
 * \version $Revision: 1.22 $
 */

#ifndef __INCLUDE_FACTORY_BUILDER_H__
#define __INCLUDE_FACTORY_BUILDER_H__

#include <factory++/allocbase.hpp>
#include <factory++/builderbase.hpp>
#include <factory++/factory.hpp>

#include <memory>

namespace Factory
{
   /**
      Builder objects abstract the interface for creation of
      objects. They do not do the allocation - that is delegated to an
      Allocator.
   */
  template <typename _BaseType, typename _SubType = _BaseType,
            typename _Alloc = std::allocator<_SubType> >
    class Builder :
      public _Builder_base<_BaseType>,
      public _Alloc_base<_SubType, _Alloc>
    {
    protected:
      typedef _Builder_base<_BaseType> _Builder_base;

    public:
      
      /**
         Creates a Builder capable of creating _SubType objects.
      */
      inline
      Builder(_Alloc const& __ALLOC = _Alloc())
        : _Builder_base<_BaseType>(), _Alloc_base<_SubType, _Alloc>(__ALLOC) {}
      virtual inline
      ~Builder() {}

      /**
         Returns a new _BaseType (possibly NULL), which the caller
         takes ownership of.
      */
      inline _BaseType*
      build() const
      {
        return this->custom_build(_SM_subtype_constructor());
      }

       /**
          Creates a new _BaseType object, copying it from __THAT using
          the copy ctor.
       */
      inline _BaseType*
      build(_BaseType const& __THAT) const
      {
        return this->custom_build(_SM_subtype_constructor(),
            *reinterpret_cast<_SubType const*>(&__THAT));
      }


      /**
       */
      template <typename _Type>
        inline _BaseType*
        build(Constructor<_Type>& __constructor) const
        {
          return this->custom_build(__constructor);
        }

      /**
       */
      template <typename _Type>
        inline _BaseType*
        build(Constructor<_Type> const& __CONSTRUCTOR) const
        {
          return this->custom_build(__CONSTRUCTOR);
        }


      /**
         Destroys __OBJ. After this call __OBJ is no longer a valid
         object.
      */
      virtual inline void
      teardown(_BaseType* const __OBJ) const
      {
        this->custom_teardown(reinterpret_cast<_SubType*>(__OBJ),
            _SM_subtype_destructor());
      }

       /**
          Destructs __OBJ using __destructor.
        */
      template <typename _Type>
        inline void
        teardown(_BaseType* const __OBJ, Destructor<_Type>& __destructor) const
        {
          return this->custom_teardown(__OBJ, __destructor);
        }

       /**
          Destructs __OBJ using __DESTRUCTOR (note the constness of
          __DESTRUCTOR).
        */
      template <typename _Type>
        inline void
        teardown(_BaseType* const __OBJ,
              Destructor<_Type> const& __DESTRUCTOR) const
        {
          return this->custom_teardown(__OBJ, __DESTRUCTOR);
        }

    protected:
      /**
         Returns a shared Contructor&lt;_SubType&gt; instance.
       */
      typedef Constructor<_SubType> _SubTypeConstructor;
      static _SubTypeConstructor& _SM_subtype_constructor()
      {
        static _SubTypeConstructor ret;
        return ret;
      }

      /**
         Returns a shared Destructor&lt;_SubType&gt; instance.
       */
      typedef Destructor<_SubType> _SubTypeDestructor;
      static _SubTypeDestructor& _SM_subtype_destructor()
      {
        static _SubTypeDestructor ret;
        return ret;
      }

      /**
         Allocates memory for 1 _BaseType object. The caller owns the
         returned pointer.
       */
      virtual _BaseType*
      _M_allocate() const
      {
        return this->get_allocator().allocate(1);
      }

      /**
         Deallocates __P. After calling this function __P is no longer
         a valid object.
       */
      virtual void
      _M_deallocate(_BaseType* const __P) const
      {
        return this->get_allocator().deallocate(reinterpret_cast<_SubType*>(__P), 1);
      }
    };

}; // namespace Factory

#endif // include guard

/* COPYRIGHT --
 *
 * This file is part of libfactory++, a C++ template factory framework.
 * libfactory++ is (c) 2004 Martin F. Krafft <krafft@ailab.ch>
 * and distributed under the terms of the Artistic Licence.
 * See the ./COPYING file in the source tree root for more information.
 * 
 * THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES
 * OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
