/* 
 * $Id: ctkwidget.c,v 1.63 2000/07/25 00:36:46 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <glib.h>
#include <arr.h>

#include "ctk.h"
#include "ctkcolor.h"
#include "ctksize.h"

gboolean ctk_widget_receive_focus(CtkWidget* widget, CdkEventFocus* data, gpointer udata)
{
	widget->inversed = TRUE;
	ctk_draw_mark_changed(widget);
	return TRUE;
}

gboolean ctk_widget_lose_focus(CtkWidget* widget, CdkEventFocus* data, gpointer udata)
{
	widget->inversed = FALSE;
	ctk_draw_mark_changed(widget);
	return TRUE;
}

/* Initialize a widget Struct */
void ctk_widget_init(CtkWidget* widget)
{
	if (!widget)
	{
		ctk_close();
		g_error("ctk_widget_init: CtkWidget *widget == NULL!");
	}

	ctk_object_init(&widget->object);

	/* We are the widget */
	CTK_OBJECT(widget)->type = CtkTypeWidget;

	/* Give all widgets a GNode at initialization */
	widget->node = g_node_new(widget);
	
	/* Position and Size on Screen */
	widget->width = 0;
	widget->height = 0;
	widget->row = 0;
	widget->col = 0;		
	widget->clip_width = 0;
	widget->clip_height = 0;
	widget->clip_row = 0;
	widget->clip_col = 0;
	
	widget->destroyed = FALSE;		

	widget->orig_width = 0;
	widget->orig_height = 0;
	
	widget->min_width = 0;
	widget->min_height = 0;
	
	/* Colors */
	widget->main_col            = ctk_calculate_palette(CTK_COLOR_LIGHT_GRAY,CTK_COLOR_BLACK);
	widget->inverse_col         = ctk_calculate_palette(CTK_COLOR_BLACK,CTK_COLOR_LIGHT_GRAY);
	widget->shadow_col          = ctk_calculate_palette(CTK_COLOR_BLACK,CTK_COLOR_BLACK);
	widget->title_col           = ctk_calculate_palette(CTK_COLOR_LIGHT_GRAY,CTK_COLOR_BLACK);
	widget->selected_col        = ctk_calculate_palette(CTK_COLOR_LIGHT_GRAY,CTK_COLOR_BLACK);
	widget->inv_selected_col    = ctk_calculate_palette(CTK_COLOR_LIGHT_GRAY,CTK_COLOR_BLACK);
	widget->insensitive_col     = ctk_calculate_palette(CTK_COLOR_LIGHT_GRAY,CTK_COLOR_BLACK);

	/* Expand, Fill, and Padding */

	widget->xexpand = FALSE;
	widget->yexpand = FALSE;
	widget->xfill = FALSE;
	widget->yfill = FALSE;
	widget->xpadding = 0;
	widget->ypadding = 0;
	
	/* Changed flag */
	widget->changed = FALSE;

	/* Sensitivity Flag */
	widget->sensitive = FALSE;

	/* Inversed. (ie, if a button is clicked it will be displayed in the
	 * assigned inverse color.  CTK needs to know when a widget is
	 * inversed in order to draw it in this color. */
	widget->inversed = FALSE;
	widget->selected = FALSE;
	
	/* Show Status */
	widget->show = FALSE;

	/* For now - to save rewriting every widget use a compatibility layer */
	widget->set_min_size  = &ctk_widget_min_size; //&ctk_deprecated_set_min_size;
	widget->set_real_size = &ctk_widget_real_size; //&ctk_deprecated_set_real_size;

	/* Initialize Signals */
	ctk_signal_new("button_press_event",   CtkTypeWidget);
	ctk_signal_new("button_release_event", CtkTypeWidget);

	// FIXME: do real dragging some other time
	ctk_signal_new("ctk_drag", CtkTypeWidget);
	
	ctk_signal_new("focus_in_event",  CtkTypeWidget);	
	ctk_signal_new("focus_out_event", CtkTypeWidget);
	
	ctk_signal_new("delete_event",  CtkTypeWidget);
	ctk_signal_new("destroy_event", CtkTypeWidget);
	
	ctk_signal_new("enter_notify_event", CtkTypeWidget);
	ctk_signal_new("leave_notify_event", CtkTypeWidget);
	
	ctk_signal_new("key_press_event",   CtkTypeWidget);
	ctk_signal_new("key_release_event", CtkTypeWidget);

	ctk_signal_connect(CTK_OBJECT(widget), "focus_in_event", 
			   CTK_SIGNAL_FUNC(&ctk_widget_receive_focus), NULL);
	ctk_signal_connect(CTK_OBJECT(widget), "focus_out_event", 
			   CTK_SIGNAL_FUNC(&ctk_widget_lose_focus),    NULL);
}

/* Set a Widget to be shown */
void ctk_widget_show(CtkWidget* widget)
{
	CtkObject *parent = NULL;
	
	if (!widget) 
	{
		ctk_close();
		g_error("ctk_widget_show: CtkWidget *widget == NULL!");
	}

	if (widget->node && widget->node->parent)
	    parent = CTK_OBJECT(widget->node->parent->data);

	widget->show = TRUE;
	
	if (CTK_OBJECT(widget)->type == CtkTypeWindow)
	{
	      ctk_window_raise(CTK_WINDOW(widget));
	}
	
	ctk_size_mark_changed(widget);
}

/* Set a Widget to be hidden */
void ctk_widget_hide(CtkWidget* widget)
{
	if (!widget)
	{
		ctk_close();
		g_error("ctk_widget_hide: CtkWidget *widget == NULL!");
	}

	widget->show = FALSE;	

	if (CTK_OBJECT(widget)->type == CtkTypeWindow)
	{
	      if (CTK_WIDGET(ctk_window_get_focus_window()) == widget)
	      {
		    ctk_lose_focus();
		    stack_pop(widget->node);
		    ctk_give_focus();
	      }
	      else
	      {
		    stack_pop(widget->node);
	      }
	}
		
	ctk_size_mark_changed(widget);
}

gboolean ctk_unlink_object(CtkObject* object)
{
	/* Remove focus from the window to this object */
	CtkWidget* widget;
	CtkWindow* window;
	
	widget = CTK_WIDGET(object);
	window = ctk_window_container(widget);
	
	if (window && window->focus_widget == widget)
	{
		window->focus_widget = CTK_WIDGET(window);
	}
	
	/* Hide the widget - also pops it if a window */
	ctk_widget_hide(widget);
	
	/* Remove mouse persistant variables */
	ctk_main_mouse_remove_stale_global(object);
	
	return TRUE;
}

gboolean ctk_destructive_release(CtkObject* object)
{
      /* Unlink the object - has to be done on each node */
      ctk_unlink_object(object);
      
      /* Remove all signals */
      ctk_signal_destroy_all_signals(CTK_WIDGET(object));
      
      /* Destroy the object data.  Should not free
       * the actual structure though. */
      if (object->destroy_func)
	    object->destroy_func(object);
      
	/* Frees the node data which is the widget structure. */
      g_free(object);
      
      return FALSE;
}

/* Node - Destroy Function */
gboolean node_destroy_widget(GNode* node, gpointer data)
{
	CtkObject *object;

	object = CTK_OBJECT(node->data);
	
	if (object)
	{
		ctk_destructive_release(object);
	}
	
	node->data = NULL;

	return FALSE;
}

/* Destroy a widget */
void ctk_widget_destroy_real(CtkWidget* widget)
{
	if (!widget)
	    return;

	/* Destroy all functions here */
	
	if (!widget->node)
	{
		ctk_destructive_release(CTK_OBJECT(widget));
	}
	else
	{
		GNode* node = widget->node;
		
		g_node_traverse(node,
				G_POST_ORDER,
				G_TRAVERSE_ALL,
				-1,
				node_destroy_widget,
				NULL);
				
		g_node_destroy(node);
	}
}

static const gchar* caller_file;
static gint         caller_line;
static const gchar* caller_func;

void ctk_record_caller(const gchar* file, gint line, const gchar* func)
{
	caller_file = file;
	caller_line = line;
	caller_func = func;
}

/* Destroy a widget */
void ctk_widget_destroy_helper(CtkWidget* widget)
{	
	if (!widget)
	    return;
	
	ctk_assert_lie(widget->destroyed == FALSE, 
		"Attempt to destroy an already destroyed widget",
		caller_file, caller_line, caller_func);
		
	widget->destroyed = TRUE;
	
	/* Unlink the widget from the object tree
	 * We do this now so that we have a tree cut out of everything else
	 * that we can stick on the garbage queue.
	 */
	if (widget->node->parent)
	{
	      CtkContainer* container;
	      
	      ctk_assert(CTK_CHECK_TYPE(widget->node->parent->data, CtkTypeContainer),
	      	"Contained by a non-container!");
	      
	      container = CTK_CONTAINER(widget->node->parent->data);
	      ctk_container_remove(container, widget);
	}
	
	/* Remove all other simple references to this object */
	ctk_unlink_object(CTK_OBJECT(widget));

	/* Add widget to the garbage queue */
	ctk_main_garbage_add(widget);
}

gboolean node_widget_show_all(GNode* node, gpointer data)
{
	CtkWidget* widget = CTK_WIDGET(node->data);
	
	ctk_widget_show(widget);

	return FALSE;
}

/* Show All */
void ctk_widget_show_all(CtkWidget* widget)
{
	if (!widget)
	    return;
	
	g_node_traverse(widget->node,
			G_POST_ORDER,
			G_TRAVERSE_ALL,
			-1,
			node_widget_show_all,
			NULL);
}

/* Set Widget position */
void ctk_widget_set_uposition(CtkWidget* widget, gint x, gint y)
{
	if (!widget)
	    return;

	widget->col = x;
	widget->row = y;
}

void ctk_widget_set_usize(CtkWidget *widget, gint width, gint height)
{
	if (!widget)
	    return;

	if (width)
	{
		widget->orig_width = width;
		widget->width      = width;
	}
	
	if (height)
	{
		widget->orig_height = height;
		widget->height      = height;
	}
}

gint node_set_sensitive(GNode* node, gpointer data)
{
	CtkWindow* window;
	CtkWidget* widget;
	
	widget = CTK_WIDGET(node->data);
	
	/* FIXME FIXME FIXME: we really should have two flags: interactable & sensitive */
	if (CTK_CHECK_TYPE(widget, CtkTypeLabel))
		return FALSE;
	if (CTK_CHECK_TYPE(widget, CtkTypeBox))
		return FALSE;
	if (CTK_CHECK_TYPE(widget, CtkTypeTable))
		return FALSE;
	if (CTK_CHECK_TYPE(widget, CtkTypeFrame))
		return FALSE;
	if (CTK_CHECK_TYPE(widget, CtkTypeSeparator))
		return FALSE;
	if (CTK_CHECK_TYPE(widget, CtkTypeSeparator))
		return FALSE;
	if (CTK_CHECK_TYPE(widget, CtkTypeViewport))
		return FALSE;
	if (CTK_CHECK_TYPE(widget, CtkTypeTableChild))
		return FALSE;
		
	if (!*(gboolean *)data)
	{
		window = ctk_window_container(widget);
		if (window && window->focus_widget == widget)
		{
			ctk_window_set_focus(window, CTK_WIDGET(window));
		}
	}
	
	widget->sensitive = *(gboolean *)data;
	
	return FALSE;
}

void ctk_widget_set_sensitive(CtkWidget *widget, gboolean sensitive)
{
	if (widget->node)
	{
		g_node_traverse(widget->node,
				G_LEVEL_ORDER,
				G_TRAVERSE_ALL,
				-1,
				node_set_sensitive,
		        &sensitive);
	}
	
	widget->sensitive = sensitive;
	
	ctk_draw_mark_changed(widget);
}

void ctk_widget_min_size(CtkWidget* widget)
{
	widget->min_width  = widget->orig_width;
	widget->min_height = widget->orig_height;
}

void ctk_widget_real_size(CtkWidget* widget)
{
}
