/* 
 * $Id: ctkspinbutton.c,v 1.20 2000/07/24 19:28:58 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>

#include "ctk.h"
#include "ctkcolor.h"
#include "ctkutil.h"

gboolean ctk_spin_button_keypress(CtkWidget* widget, CdkEventKey* event, gpointer data);
gboolean ctk_spin_button_post_keypress(CtkWidget* widget, CdkEventKey* event, gpointer data);
gboolean ctk_spin_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data);
gpointer ctk_spin_button_destroy(CtkObject *object);

/* Initialize the spin button Widget */
void ctk_spin_button_init(CtkSpinButton* spinbutton)
{
	ctk_entry_init(&spinbutton->entry);
	
	spinbutton->climb_rate = 1;
	spinbutton->digits = 0;
	
	ctk_signal_new("changed",CtkTypeSpinButton);

	((CtkObject *)spinbutton)->type = CtkTypeSpinButton;
	((CtkWidget *)spinbutton)->width = 15;
	((CtkWidget *)spinbutton)->orig_width = 15;
	((CtkWidget *)spinbutton)->height = 1;
	((CtkWidget *)spinbutton)->orig_height = 1;
	((CtkWidget *)spinbutton)->main_col = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_BLACK);
	((CtkWidget *)spinbutton)->title_col = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_BLUE);

	((CtkObject *)spinbutton)->destroy_func = ctk_spin_button_destroy;

	ctk_signal_connect(CTK_OBJECT(spinbutton), "button_press_event",
			   CTK_SIGNAL_FUNC(&ctk_spin_button_press), NULL);
	ctk_signal_connect(CTK_OBJECT(spinbutton), "key_press_event",
			   CTK_SIGNAL_FUNC(&ctk_spin_button_keypress), NULL);
	ctk_signal_connect_after(CTK_OBJECT(spinbutton), "key_press_event",
			   CTK_SIGNAL_FUNC(&ctk_spin_button_post_keypress), NULL);
}

/* Create a Spin Button Widget */
CtkWidget* ctk_spin_button_new(CtkAdjustment* adjustment, 
	gfloat climb_rate, guint digits)
{
	CtkSpinButton *spinbutton;
	
	spinbutton = g_malloc(sizeof(CtkSpinButton));
	
	ctk_spin_button_init(spinbutton);

	spinbutton->adjustment = adjustment;
/*	spinbutton->adjustment->upper = 10;
	spinbutton->adjustment->lower = 0;
	spinbutton->adjustment->value = 0;
*/
	spinbutton->adjustment->step_increment = climb_rate;
	
	spinbutton->digits = digits;
	ctk_spin_button_value_to_text(spinbutton);

	return ((CtkWidget *)spinbutton);
}

/* Change Value into Text */
void ctk_spin_button_value_to_text(CtkSpinButton* spinbutton)
{
	CtkEditable *editable;
	
	if (!spinbutton || !spinbutton->adjustment)
	    return;
	
	editable = CTK_EDITABLE(spinbutton);

	if (editable->text)
	    free(editable->text);
	
	editable->text = g_strdup(ctk_util_ftos(
		spinbutton->adjustment->value, spinbutton->digits));
	
	if (editable->text)
	    editable->text_start = editable->text;
	   
	ctk_size_mark_changed(CTK_WIDGET(spinbutton));
}

/* Handle up arrow event */
void ctk_spin_button_handle_up(CtkSpinButton* spinbutton)
{
	CtkAdjustment* adjustment;
            
	if (!spinbutton)
	    return;
	
	adjustment = spinbutton->adjustment;
	adjustment->value += adjustment->step_increment;
	
	if (adjustment->value > adjustment->upper)
	    adjustment->value = adjustment->upper;
	
	ctk_spin_button_value_to_text(spinbutton);
	ctk_draw_mark_changed(CTK_WIDGET(spinbutton));
}

/* Handle down arrow event */
void ctk_spin_button_handle_down(CtkSpinButton* spinbutton)
{
	CtkAdjustment* adjustment;
	
	if (!spinbutton)
	    return;	
	
	adjustment = spinbutton->adjustment;
	adjustment->value -= adjustment->step_increment;
	
	if (adjustment->value < adjustment->lower)
	    adjustment->value = adjustment->lower;

	ctk_spin_button_value_to_text(spinbutton);
	ctk_draw_mark_changed(CTK_WIDGET(spinbutton));
}

/* Scrolled Window mouse handle events */
gboolean ctk_spin_button_press(CtkWidget* widget, CdkEventButton* event, gpointer data)
{
    if (event->x == widget->col + widget->width - 2)
    { /* Click up */
	  ctk_spin_button_handle_up(CTK_SPIN_BUTTON(widget));
	  return TRUE;
    }
    
    if (event->x == widget->col + widget->width - 1)
    { /* Click Down */
    	ctk_spin_button_handle_down(CTK_SPIN_BUTTON(widget));
    	return TRUE;
    }
    
    return FALSE;
}

/* Set Text */
void ctk_spin_button_set_text(CtkSpinButton *spinbutton, const gchar *text)
{
	CtkWidget *widget;
	CtkEditable *editable;

	if (!spinbutton)
	    return;

	if (!text)
	    return;

	editable = CTK_EDITABLE(spinbutton);
	
	if (editable->text)
	    free(editable->text);

	editable->text = g_strdup(text);
	editable->text_start = editable->text;
	editable->curpos = 0;
	ctk_adjustment_set_value(spinbutton->adjustment, atof(text));

	widget = (CtkWidget *)editable;

	ctk_size_mark_changed(widget);
}

/* Set visibility */
void ctk_spin_button_set_visibility(CtkSpinButton* spinbutton, gboolean visible)
{
	if (!spinbutton)
	    return;

	spinbutton->entry.editable.visible = visible;
	ctk_spin_button_value_to_text(spinbutton);
	ctk_size_mark_changed(CTK_WIDGET(spinbutton));
}

/* Set SpinButton digits */
void ctk_spin_button_set_digits(CtkSpinButton* spin_button, guint digits)
{
	if (!spin_button)
	    return;
	
	spin_button->digits = digits;
	ctk_spin_button_value_to_text(spin_button);
	ctk_size_mark_changed(CTK_WIDGET(spin_button));
}

/* Set Spin Button Value */
void ctk_spin_button_set_value(CtkSpinButton* spin_button, gfloat value)
{
	if (!spin_button)
	    return;

	ctk_adjustment_set_value(spin_button->adjustment,value);
	ctk_spin_button_value_to_text(spin_button);
	
	ctk_size_mark_changed(CTK_WIDGET(spin_button));
}

/* Get spin button adjustment */
CtkAdjustment* ctk_spin_button_get_adjustment(CtkSpinButton *spin_button)
{
       	if (!spin_button)
	    return NULL;

	return spin_button->adjustment;
}

/* Set spin button adjustment */
void ctk_spin_button_set_adjustment(CtkSpinButton* spin_button,
				   CtkAdjustment* adjustment)
{
	ctk_assert(spin_button, "Null spin button passed!");
	ctk_assert(spin_button->adjustment == adjustment,
		"Attempt to reset the adjustment will cause borked memory!");
	
	if (spin_button->adjustment)
	    g_free(spin_button->adjustment);
	
	spin_button->adjustment = adjustment;	
	ctk_spin_button_value_to_text(spin_button);
	
	ctk_size_mark_changed(CTK_WIDGET(spin_button));
}

/* Get value as float */
gfloat ctk_spin_button_get_value_as_float(CtkSpinButton* spin_button)
{
	if (!spin_button)
	    return -1;
	
	if (!spin_button->adjustment)
	    return -1;
	
	return spin_button->adjustment->value;
}

/* Destroy Data */
gpointer ctk_spin_button_destroy(CtkObject* object)
{
	CtkSpinButton* spinbutton;
	
	spinbutton = CTK_SPIN_BUTTON(object);
	
	if (spinbutton->adjustment)
	    g_free(spinbutton->adjustment);
	
	ctk_editable_destroy(CTK_EDITABLE(object));

	return NULL;
}

gboolean ctk_spin_button_keypress(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
	switch (event->keyval)
	{
	case AK_ARROW_UP:
		ctk_spin_button_handle_up(CTK_SPIN_BUTTON(widget));
		return TRUE;
	case AK_ARROW_DOWN:
		ctk_spin_button_handle_down(CTK_SPIN_BUTTON(widget));
		return TRUE;
	}
	
	return FALSE;
}

gboolean ctk_spin_button_post_keypress(CtkWidget* widget, CdkEventKey* event, gpointer data)
{
	CtkEditable*   editable;
	CtkSpinButton* spinbutton;
	
	editable   = CTK_EDITABLE(widget);
	spinbutton = CTK_SPIN_BUTTON(widget);
	
	if (editable->text)
		ctk_adjustment_set_value(spinbutton->adjustment, 
			atof(editable->text));
	
	return TRUE;
}
