/* 
 * $Id: ctklabel.c,v 1.22 2000/08/07 23:19:18 terpstra Exp $
 *
 * CTK - Console Toolkit
 *
 * Copyright (C) 1998-2000 Stormix Technologies Inc.
 *
 * License: LGPL
 *
 * Authors: Kevin Lindsay, Wesley Terpstra
 *  
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation; either
 *    version 2 of the License, or (at your option) any later version.
 *    
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *    
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <ctype.h>
#include <string.h>
#include <stdarg.h>

#include "ctk.h"
#include "ctkcolor.h"
#include "ctkutil.h"

gboolean ctk_label_wrap_hack_report_vertible;

gpointer ctk_label_destroy(CtkObject *object);

/* Initialize the Label */
void ctk_label_init(CtkLabel *label)
{
	CtkWidget* widget;
	
	ctk_misc_init(&label->misc);
	CTK_OBJECT(label)->type = CtkTypeLabel;

	widget  = CTK_WIDGET(label);
	
	label->text = NULL;
	label->text_split = NULL;
	label->wrap = FALSE;
	label->first = TRUE;

	((CtkObject *)label)->destroy_func = ctk_label_destroy;

	widget->width       = 0;
	widget->orig_width  = 0;
	widget->height      = 0;
	widget->orig_height = 0;
	
	widget->main_col = ctk_calculate_palette(CTK_COLOR_WHITE,CTK_COLOR_BLUE);
	widget->inverse_col = ctk_calculate_palette(CTK_COLOR_RED,CTK_COLOR_BLUE);

	widget->set_min_size = &ctk_label_min_size;
}

/* Create a Label Widget */
CtkWidget* ctk_label_new(const gchar* text)
{
	CtkLabel* label;

	label = g_malloc(sizeof(CtkLabel));

	ctk_label_init(label);
	ctk_label_set_text(label, text);

	return CTK_WIDGET(label);
}

CtkWidget* ctk_label_newf(const gchar* format, ...)
{
	gchar*     tmp;
	va_list    ap;
	CtkWidget* out;
	
	va_start(ap, format);
	tmp = g_strdup_vprintf(format, ap);
	out = ctk_label_new(tmp);
	g_free(tmp);
	va_end(ap);
	
	return out;
}

/* Set the text in a label */
void ctk_label_set_text(CtkLabel* label, const gchar* text)
{
	CtkWidget* widget = CTK_WIDGET(label);

	if (!text)
	    return;

	if (label->text)
	    free(label->text);

	label->text = g_strdup(text);
	ctk_size_mark_changed(widget);
}

/* Set Line Wrap */
void ctk_label_set_line_wrap(CtkLabel* label, gboolean wrap)
{
	if (!label)
	    return;
	
	label->wrap = wrap;
	ctk_size_mark_changed(CTK_WIDGET(label));
}

/* Set Justify */
void ctk_label_set_justify(CtkLabel *label, CtkJustification jtype)
{
	switch (jtype)
	{
	case CTK_JUSTIFY_LEFT:
		ctk_misc_set_alignment(CTK_MISC(label), 
			0, CTK_MISC(label)->yalign);
		break;
	case CTK_JUSTIFY_RIGHT:
		ctk_misc_set_alignment(CTK_MISC(label), 
			1, CTK_MISC(label)->yalign);
		break;
	case CTK_JUSTIFY_CENTER:
		ctk_misc_set_alignment(CTK_MISC(label), 
			0.5, CTK_MISC(label)->yalign);
		break;
	case CTK_JUSTIFY_FILL:
		// FIXME
		break;
	}
}

/* Destroy label data */
gpointer ctk_label_destroy(CtkObject* object)
{
      CtkLabel* label = CTK_LABEL(object);
      
      if (label->text)
	    g_free(label->text);
      if (label->text_split)
	    g_strfreev(label->text_split);
      
      return NULL;
}

void ctk_label_min_size(CtkWidget* widget)
{
      CtkLabel* label = CTK_LABEL(widget);
      gint      i;
      
      if (!label->text)
      {
      	widget->min_width  = 1;
      	widget->min_height = 1;
      	return;
      }

      if (label->wrap)
      {
	    GSList* cuts;
	    GSList* node;
	    gchar*  scan;
	    gchar*  start;
	    gchar*  lastgood;
	    gint    delta;
	    gint    width;
	    gint    count;

	    if (ctk_label_wrap_hack_report_vertible)
	    {
		  /* Here we scan through and split stuff up based on width */

		  /* Fail if we were set to 0x0 (invisible) */
		  if (widget->width < widget->min_width)
			return;

		  if (label->text_split)
			g_strfreev(label->text_split);

		  cuts = NULL;
		  count = 0;
		  width = widget->width - CTK_MISC(widget)->xpad*2;
		  
		  for (start = label->text; *start; start = lastgood)
		  {
			lastgood = start;

			for (scan = start; *scan; scan++)
			{
			      delta = (int)scan - (int)start;

			      if (delta >= width) 
				    break;

			      if (isspace(*scan)) 
				    lastgood = scan;
			      if (*scan == '\n')
				    break;
			}

			if (!*scan || *scan == '\n')
			{
			      lastgood = scan;
			}
			else if (lastgood == start)
			{    /* Couldn't cut on nice place */
			      lastgood = start + width;
			}

			delta = (int)lastgood - (int)start;
			cuts = g_slist_prepend(cuts, g_strndup(start, delta));
			count++;

			while (*lastgood && isspace(*lastgood)) lastgood++;
		  }

		  label->text_split = (gchar**)malloc(sizeof(gchar*) * (count+1));
		  label->text_split[count] = NULL;
		  if (count == 0)
		  	count++;
		  widget->min_height = count;

		  for (node = cuts; node; node = node->next)
		  {
			label->text_split[--count] = node->data;
		  }

		  g_slist_free(cuts);
	    }
	    else
	    {
		  /* Here we just report <max word length>x1 */
		  widget->min_width = 0;

		  for (start = label->text; *start; start = scan)
		  {
			for (scan = start; *scan; scan++)
			{
			      if (isspace(*scan)) break;
			}
			
			if (*scan) scan++;
			delta = (int)scan - (int)start;
			if (delta > widget->min_width) 
			      widget->min_width = delta;
		  }
		  
		  /* Always eat one col */
		  if (widget->min_width == 0)
		  	widget->min_width++;

		  widget->min_width += CTK_MISC(widget)->xpad*2;

		  widget->min_height = 1 + CTK_MISC(widget)->ypad*2;
	    }
      }
      else
      {
	    if (label->text_split)
		  g_strfreev(label->text_split);

	    label->text_split = g_strsplit(label->text, "\n", 0);
	    
	    widget->min_width  = 0;
	    widget->min_height = 0;

	    for (i = 0; label->text_split[i]; i++)
	    {
		  gint len = strlen(label->text_split[i]);

		  if (len > widget->min_width)
			widget->min_width = len;
	    }
	    
	    /* Always eat up one row at least */
	    if (i == 0)
	    	i++;
	    
	    /* Always eat up one col at least */
	    if (widget->min_width == 0)
	    	widget->min_width++;

	    widget->min_height = i + CTK_MISC(widget)->ypad*2;
	    widget->min_width += CTK_MISC(widget)->xpad*2;
      }
}
