#line 2 "share.c"
/*-
 * C-SaCzech
 * Copyright (c) 1996-2002 Jaromir Dolecek <dolecek@ics.muni.cz>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Jaromir Dolecek
 *	for the CSacek project.
 * 4. The name of Jaromir Dolecek may not be used to endorse or promote
 *    products derived from this software without specific prior written
 *    permission.
 *
 * THIS SOFTWARE IS PROVIDED BY JAROMIR DOLECEK ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL JAROMIR DOLECEK BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/* $Id: share.c,v 1.247.2.1 2002/02/17 12:31:25 dolecek Exp $ */

#include "csacek.h"
#include "csa_version.h"

/* number of various functions of general importance */

/* this is global */
const csa_conf_t csa_cfg_def = {
	CSA_CFG_ENGINEON | CSA_CFG_COMPRESS | CSA_CFG_CHANGEURL
		| CSA_CFG_IMPLICITWORK
#ifdef CSA_WANT_DECODEQUERY
		| CSA_CFG_RECODEINPUT
#endif
	,
	CSA_DEFSOURCE, CSA_DEFAULT_PARTNAME, CSA_TEMPLATEDIR,
	NULL, CSA_DEFAULT_IGNOREPREFIX,
};

/* variables used by code; the ones beginning with ! are not mandatory */
/* mutation-dependant code can use this array for implementing */
/* csa_md_getvalueof() */
static const char * const csa_needvars[] = {
	/* PATH_INFO & SCRIPT_NAME first, they are used in csa_http_error() */
	"PATH_INFO",		"SCRIPT_NAME",
	"SERVER_NAME",		"SERVER_PORT",		"SERVER_PROTOCOL",
	"!REMOTE_HOST",		"REMOTE_ADDR",		"PATH_TRANSLATED",
	"REQUEST_METHOD",	"!QUERY_STRING",
	"!SERVER_SOFTWARE",	"!SERVER_URL",		"!CLIENT_PROTOCOL",
	NULL
};

/*
 * Global list of CSacek servers. This is shared by all configurations -
 * keeping it per directory or per virtual server just doesn't make sense
 * in this case.
 */
csa_slist_t *csacek_servers = NULL;

/* local constants */

/* size of one output buffer - value has been carefully adjusted */
/* to minimize memory overhead - Apache's alloc functions are not */
/* very good when man needs lot's of memory in big pieces */
#define X_CSA_OUT_BUFLEN	1512
#define X_CSA_OUT_BUFLEN_CHUNK	8150

/* space which is left in the buffer for chunk size + \r\n - maximum
 * 4 chars are currently needed for chunked buflen */
#define X_CSA_OUT_CHUNKED_SIZESPACE 4 + 2

#define X_CSA_NO_ARGS		-1

/* local functions */
static csa_item_t *x_item_lookup __P((csa_item_t *, const char *));
static int x_compare_Part __P((csa_params_t *p, const char *value));
static int x_compare_Domain __P((csa_params_t *p, const char *value));
static int x_compare_Charset __P((csa_params_t *p, const char *value));
static void x_set_outnames __P((csa_params_t *));
static int x_process_vars __P((csa_params_t *, const csa_conf_t *,
					const char **));
static void x_add_2buf __P((struct pool *, csa_String_b *, const char *, int));
static void x_finish_body __P((csa_params_t *));
#ifdef CSA_WANT_COMPRESSION
static void x_init_compression __P((csa_params_t *));
static csa_compress_t x_guess_compression __P((csa_params_t *p));
#define CSA_COMPRESS_NAME(x)	\
	((x) == CSA_C_GZIP ? "gzip" : \
	 ((x) == CSA_C_DEFLATE ? "deflate" : \
	  ((x) == CSA_C_COMPRESS ? "compress" : NULL)))
#endif

/* 
 * tries to find an item with the key ``key''
 */
static csa_item_t *
x_item_lookup(item_list, key)
  csa_item_t *item_list;                       
  const char *key;
{
	csa_item_t *item, *retval=NULL;
	size_t len;
	
	if (item_list == NULL || key == NULL || *key == '\0') return NULL;

	len = strlen(key);
	item = item_list;
	for(;item; item = item->prev)
	{
		if ( item->key.len == len 
		     && CSA_UPPER(item->key.value[0]) == CSA_UPPER(key[0])
		     && CSA_UPPER(item->key.value[len-1])==CSA_UPPER(key[len-1])
		     && strncasecmp(item->key.value, key, len) == 0 )
		{
			retval = item;
			break;
		}
		
	}

	return retval;
}

/*
 * sets p->charset & p->lampacharset accordingly to value of
 * p->outcharset or to zero csa_Strings if p->outcharset is invalid
 */
static void
x_set_outnames(p)
   csa_params_t *p;
{
	const char *tmpname;

	/* set CSacek name of output charset */
	tmpname = cstools_name(p->outcharset, CSTOOLS_TRUENAME);
	if (!tmpname) tmpname = "";
	csa_fillstring(&p->charset, tmpname, -1, -1);

	/* set Lampa name of output charset */
	tmpname = cstools_name(p->outcharset, CSTOOLS_LAMPANAME);
	if (!tmpname) tmpname = "";
	csa_fillstring(&p->lampacharset, tmpname, -1, -1);
}

#ifdef CSA_WANT_COMPRESSION
/*
 * Returns best possible compression client can handle (as specified
 * in Accept-Encoding header send by the client).
 */
static csa_compress_t
x_guess_compression(p)
  csa_params_t *p;
{
	const csa_String *accept_encoding;

	accept_encoding = csa_getheaderin(p, "Accept-Encoding");
	if (accept_encoding) {
		if (csa_strcasestr(accept_encoding->value, "deflate"))
			return CSA_C_DEFLATE;
		else if (csa_strcasestr(accept_encoding->value, "gzip"))
			return CSA_C_GZIP;
		else
		  if (csa_strcasestr(accept_encoding->value, "compress"))
			return CSA_C_COMPRESS;
		else /* nope */
			return CSA_C_NONE;
	}

	/* don't use any compression */
	return CSA_C_NONE;
}

/* 
 * initialize compression if needed
 */
static void
x_init_compression(p)
  csa_params_t *p;
{
	int errored=0;

	/* call compression-dependant init routine */
	switch(p->ua_compress) {
	case CSA_C_COMPRESS:
		errored = csa_init_compress(p);
		break;
	case CSA_C_GZIP:
		errored = csa_init_gzip(p);
		break;
	case CSA_C_DEFLATE:
		errored = csa_init_deflate(p);
		break;
	default:
		/* invalid compression, don't update p->compress */
		errored = 1;
	}

	if (!errored)
		p->compress = p->ua_compress;

	CSA_SET(p->flags, CSA_FL_C_INITIALIZED);
}
#endif /* CSA_WANT_COMPRESSION */

/*
 * adds the string to specified buffer, allocating more space and
 * updating buffer structure if needed
 */
static void
x_add_2buf(wpool, buf, str, len)
  struct pool *wpool;
  csa_String_b *buf;
  const char *str;
  int len;
{
	size_t llen = (len < 0) ? strlen(str) : (size_t) len;

	if ((buf->maxlen - buf->len) < llen) {
		const char *oldval = buf->value;
		size_t newlen = buf->maxlen;
		newlen += (llen > buf->maxlen) ? llen + 1 : buf->maxlen;
		buf->value = (char *)ap_palloc(wpool, (int) newlen);
		buf->maxlen = newlen;
		memcpy(buf->value, oldval, buf->len);
	}

	/* LINTED */
	memcpy(&(buf->value[buf->len]), str, llen);
	buf->len += llen;
}

/*****************************************************************/
/*		global init/clearing procedures	     		 */

/*
 * initialize structure to contain default values
 * each mutation can use cookie to pass mutation-dependant data
 * note that all old content of (*p) is lost
 */
/* ARGSUSED */
int 
csa_init_params(p, req_pool, cookie, cfg, dbg)
  csa_params_t *p;
  struct pool *req_pool;
  void *cookie;
  const csa_conf_t *cfg;
  FILE *dbg;
{
	char *temp;
	const char *partp, *outname;
	int retval;
#ifdef CSA_WANT_INFO
	const csa_String *str;
#endif

	/* if cfg is NULL, set it so it points to default configuration */
	if (!cfg) cfg = &csa_cfg_def;

	/* first clear passed csa_params_t structure */
	memset((void *)p, '\0', sizeof(csa_params_t));

	/* initialize pools and cookie */
	p->m_cookie = cookie;
	p->pool_req = req_pool ? req_pool : ap_make_sub_pool(req_pool);
	p->pool_tmp = ap_make_sub_pool(p->pool_req);

#ifdef CSA_DEBUG
	/* set debug log file pointer */
	p->dbg = dbg;

	csa_debug(p->dbg, "csa_init_params(): called");
#endif

	p->incharset  = CSTOOLS_UNKNOWN;
	p->outcharset = CSTOOLS_UNKNOWN;

	/* create default bar */
	p->Bar = csa_bar_create(p->pool_req);

	/* invalidate conversion table */
	p->mp.source = CSTOOLS_UNKNOWN;
	p->mp.target = CSTOOLS_UNKNOWN;

	p->available_in = -1; /* invalidate it */

	/* p->changeurl is set accordingly to cfg->changeurl */

	/* initialize flags for processiong of CSacek "parts" */
	CSA_SET(p->flags_parts, CSA_VALID_MASK|CSA_VALIDOTHER_MASK);
	
	/* used protocol is HTTP/1.0 initially */
	p->protocol = p->req_protocol = 10;

	/* needed in x_process_vars() */
	if (CSA_ISSET(cfg->flags, CSA_CFG_RECODEINPUT))
		CSA_SET(p->flags, CSA_FL_RECODEINPUT);

	if (CSA_ISSET(cfg->flags, CSA_CFG_TESTJS))
		CSA_SET(p->flags, CSA_FL_TESTJS);

	/* Ensure temp pool is cleared at least once after startup */
	CSA_SET(p->flags, CSA_FL_CLRTMPPOOL);

	/* this has to be set before x_process_vars() is called in order
	 * for csa_decodequery() to know to which charset it should
	 * recode GET data
	 */
	p->incharset = cstools_whichcode(cfg->DefaultCharset, 0);

	/* set up needed variables */
	retval = x_process_vars(p, cfg, &outname);
	if (retval != CSA_OK) return retval;

	/* create list of incoming headers */
	csa_set_headersin(p);                 

	/* use chunked encoding on output if possible, so we won't need to
	 * buffer all the data in memory in order to send Content-Length;
	 * all HTTP/1.1 client support it, but we can't use it if client
	 * is enterested in a part of document only */
	/* XXX IIS doesn't like chunked output for https request - before
	 * the real reason is found out, don't use chunked encoding
	 * when handling https request */
	if (p->protocol >= 11 && !csa_getheaderin(p, "Range")) {
#ifdef CSA_MUTACE_ISAPI
		if (!CSA_ISSET(p->flags, CSA_FL_ISHTTPS))
			CSA_SET(p->flags, CSA_FL_OUT_CONT|CSA_FL_OUT_CHUNKED);
#else
		/* Apache 1.3 core handles chunking, so we just pass the
		 * data without chunking in that case */
		CSA_SET(p->flags, CSA_FL_OUT_CONT);
#ifndef CSA_USE_APACHE13_API
		CSA_SET(p->flags, CSA_FL_OUT_CHUNKED);
#endif /* Apache 1.3 */
#endif /* ISAPI */
	}

	/* apply rest of configuration (some bits are already set before
	 * x_process_vars() is called)
	 */
	p->dd = cfg->DefaultPartname;
	if (CSA_ISSET(p->flags, CSA_FL_PART_IS_DEF)) {
		temp = (char *)ap_palloc(p->pool_req, (int) strlen(p->dd) + 2);
		sprintf(temp, ".%s", p->dd);
		csa_fillstring(&p->part, temp, -1, -1);
	}
	if (CSA_ISSET(cfg->flags, CSA_CFG_CHANGEURL))
		CSA_SET(p->flags, CSA_FL_CHANGEURL);

	/* run BarDef is it's defined */
	if (cfg->BarDef) {
		size_t len = 7 + strlen(cfg->BarDef);
		temp = (char *) ap_palloc(p->pool_req, (int)len + 1);
		sprintf(temp, "BARDEF %s", cfg->BarDef);
		csa_run_cmd(p, temp, len, 0);
	}

	/* string in cfg->IgnorePrefix is always ignored when checking
	 * if CSacek is in URI */
	p->ignoreprefix = csa_createstring(p->pool_req, cfg->IgnorePrefix);

	p->csacek_servers = csa_slist_init(p->pool_req);

	/* ALWAYS add server on which we process the request */
	csa_slist_add(p->csacek_servers,
		csa_getvar(p, "SERVER_NAME")->value,
		atoi(csa_getvar(p, "SERVER_PORT")->value));

#ifdef CSA_WANT_INFO
	/* if QUERY_STRING was "csacek_info", generate a HTML page */
	/* with list of compiled-in defaults */
	str = csa_getvar(p, "QUERY_STRING");
	if (str && str->len == 11
	    && strncasecmp(str->value, "csacek_info", 11) == 0)
		return csa_info(p, cfg);
#endif /* CSA_WANT_INFO */

	/* ensure input code is good */
	if (p->incharset == CSTOOLS_UNKNOWN)
	{
		csa_http_error(p, "Bad compiled-in default",
			"Bad default source code name.");
		return HTTP_INTERNAL_SERVER_ERROR;
	}

	/* Check output encoding; take care of special "charset names", too  */
	if (p->outcharset == CSTOOLS_UNKNOWN) {
	    if (*outname == '\0') {
		/* don't redirect client, guess it's charset and normally */
		/* process the request */
		csa_toguess(p, 0);

		/* set outnames accordingly to new p->outcharset */
		x_set_outnames(p);
	    }
	    else if ( strcasecmp(outname, "GUESS") == 0
		    || strcmp(outname, "__CHARSET__") == 0
		    || ( (partp = csa_strcasestr(outname, "__PART__"))
			    && partp[8] == '\0') )
	    {
		csa_toguess(p, 1);
		return HTTP_MOVED_TEMPORARILY;
	    }
	    else if (strncasecmp(outname, "whichcode", 9) == 0) {
		char *fname;
		size_t len;
		len = strlen(cfg->TemplateDir)+1+strlen(CSA_WHICHCODETEMPLATE);
		fname = (char *)ap_palloc(p->pool_req, (int) len + 1);
		sprintf(fname, "%s/%s", cfg->TemplateDir,CSA_WHICHCODETEMPLATE);

		p->outcharset = CSTOOLS_ASCII;
		return csa_md_call_whichcode(p, fname);
	    }
	    else {
		char *foo = ap_palloc(p->pool_req, 100 + (int)strlen(outname));
		sprintf(foo, "Specified output code name (%s) is invalid.",
			outname);
		csa_http_error(p, "Bad output code name", foo);
		return HTTP_INTERNAL_SERVER_ERROR;
	    }
	}

#ifdef CSA_WANT_COMPRESSION
	if (CSA_ISSET(cfg->flags, CSA_CFG_COMPRESS)) {
		const csa_String *ua;

		p->ua_compress = x_guess_compression(p);

		/*
		 * Workaround for bug in MSIE 4.0 - it crashes if data are sent
		 * chunked and compressed; it's better to switch off compression
		 * than chunking.
		 */
		if (p->ua_compress != CSA_C_NONE
		    && CSA_ISSET(p->flags, CSA_FL_OUT_CHUNKED)
		    && (ua = csa_getheaderin(p, "User-Agent"))
		    && strstr(ua->value, "MSIE") != NULL)
		{
			p->ua_compress = CSA_C_NONE;
		}
	}
#endif

	/*
	 * Initialize recoding engine.
	 */
	cstools_init(&p->mp, p->incharset, p->outcharset);

	return CSA_OK;
}

/* 
 * send output to client
 */
int 
csa_output(p)
  csa_params_t *p;
{
#ifdef CSA_DEBUG
	csa_debug(p->dbg, "csa_output: start");
#endif

	x_finish_body(p);

	if (CSA_SEND_HEADERS(p)) {
		/* send headers */
		csa_send_headersout(p);

		/* send separator */
		csa_md_send_separator(p);
	}

	/* send body */
	if (!CSA_ISSET(p->flags, CSA_FL_HEADERS_ONLY)) {
		if (CSA_ISSET(p->flags, CSA_FL_OUT_CONT)) {
			/* send all remaining data */
			csa_flush_output(p);
#ifndef CSA_USE_APACHE13_API
			/* if chunking, send the final EOF mark */
			if (CSA_ISSET(p->flags, CSA_FL_OUT_CHUNKED))
				csa_md_send_output(p, "0\r\n\r\n", 5);
#endif
		}
		else
			csa_send_body(p);
	}

#ifdef CSA_DEBUG
	csa_debug(p->dbg, "csa_output: end");
#endif

	return 0;
}

/*
 * saves all headers sent by client into internal CSacek structure
 */
int
csa_set_headersin(p)
  csa_params_t *p;
{
	return csa_md_set_headersin(p);
}

/*
 * get value of all needed variables and save their values into internal
 * CSacek structure
 */
static int 
x_process_vars(p, cfg, outnamep)
  csa_params_t *p;
  const csa_conf_t *cfg;
  const char **outnamep;
{
	char *temp, *charsetname, *dotpartname;
	const char *csa_dir, *csa_suffix, *ctemp;
	const csa_String *script_name, *str, *rm, *qs;
	int need_it, path_not_set=0, i;

#ifdef CSA_DEBUG
	csa_debug(p->dbg, "process_vars: called");
#endif

	for(i=0; csa_needvars[i]; i++)
	{
		const char *var;

		var = csa_needvars[i];
		if (*var == '!') { var++; need_it = 0;}
		else need_it = 1;
		ctemp = csa_md_getvalueof(p, var);
		if (ctemp) csa_setvar(p, var, ctemp, 0);
		else if (strncasecmp(var, "PATH_", 5) == 0) path_not_set = 1;
		else if (need_it) {
		    csa_http_error(p, "Needed variable not set", var);
		    return CSA_FATAL;
		}
	}

	/* unset QUERY_STRING if it's null */
	qs = csa_getvar(p, "QUERY_STRING");
	if (qs && qs->len == 0) {
		csa_unsetvar(p, "QUERY_STRING");
		qs = NULL;
	}

	/* find out if SSL is used - we must use https:// instead of http:// */
	/* in Location headers (and anywhere else as well) */
	ctemp = csa_md_getvalueof(p, "HTTPS");
	if (ctemp && strncasecmp(ctemp, "on", 2) == 0)
		CSA_SET(p->flags, CSA_FL_ISHTTPS);

	/* REMOTE_HOST could be not set if DNS lookups are off - set it */
	/* to REMOTE_ADDR in that case */
	str = csa_getvar(p, "REMOTE_HOST");
	if (!str) {
		str = csa_getvar(p, "REMOTE_ADDR");
		csa_setvar(p, "REMOTE_HOST", str->value, 0);
	}

	/* Netscape server doesn't set SERVER_NAME for virtual servers  */
	/* correctly - it sets it to "real" name of server; right one  */
	/* can be extracted from SERVER_URL though */
	str = csa_getvar(p, "SERVER_URL");
	if (str && str->value) {
		csa_url_t *tempurl = csa_parse_url(p, str->value);
		csa_setvar(p, "SERVER_NAME", tempurl->server, CSA_I_COPYVALUE);
	}

	/* Get request protocol. Use CLIENT_PROTOCOL if it's
	 * available - for example Zeus Web server sets 
	 * SERVER_PROTOCOL to protocol supported by server and not
	 * the one used by client */
	if ((str = csa_getvar(p, "CLIENT_PROTOCOL")) == NULL)
		str = csa_getvar(p, "SERVER_PROTOCOL");
	temp = strchr(str->value, '/');
	ctemp = (temp) ? temp + 1 : str->value;
	p->protocol = p->req_protocol = (int) (atof(ctemp) * 10);

	/* needed for further processing */
	script_name = csa_getvar(p, "SCRIPT_NAME");

#ifdef CSA_MUTACE_XCGI
	/* Netscape-Enterprise/2.01 for NT sets SCRIPT_NAME and PATH_INFO
	 * badly, if extra path behind script name ends with '/'; i.e
	 * if URI is /script_name/path_info/, server parses the URI incorrectly
	 * as /script_name/ and /path_info */
	str = csa_getvar(p, "SERVER_SOFTWARE");
	if (str && strstr(str->value, "Netscape-Enterprise") == str->value )
	{
		char *temps = strrchr(script_name->value, '/');
		if (temps && *(temps+1) == '\0') {
			const csa_String *path_info;
			char *foo;
			int len;

			/* correct SCRIPT_NAME */
			len = script_name->len - 1; /* skip slash */
			foo = ap_pstrndup(p->pool_req, script_name->value, len);
			csa_setvar(p, "SCRIPT_NAME", foo, 0);
			script_name = csa_getvar(p, "SCRIPT_NAME");

			/* correct PATH_INFO */
			path_info = csa_getvar(p, "PATH_INFO");
			if (path_info) {
				len = path_info->len + 1; /* add slash */
				foo = ap_palloc(p->pool_req, len+1);
				sprintf(foo, "%s/", path_info->value); /*safe*/
				csa_setvar(p, "PATH_INFO", foo, 0);
			}
		}
	}
#endif /* CSA_MUTACE_(FAST)CGI */
			
	/* get CSacek dir - i.e. the part before "/toXXX.YYY" */
	ctemp = strrchr(script_name->value, '/');
	csa_dir = ctemp ? ap_pstrndup(p->pool_req,
			script_name->value, ctemp - script_name->value) : "";
	p->csacek_dir = csa_createstring(p->pool_req, csa_dir);
	
	if (!ctemp) ctemp = script_name->value;    /* no slash in name */
	else ctemp++;

	csa_suffix = "";

	/* parse SCRIPT_NAME for charset and partname */
	if (csa_parse_sn(p->pool_req, ctemp, &charsetname, &dotpartname, NULL,
		NULL, NULL))
	{
		/* cut unsuable CSacek script suffix, if there is any */
		if (dotpartname) {
			const char *ignsuff;
			ignsuff = csa_has_suffix(dotpartname,
				CSA_IGNORE_SUFFIXES, CSA_SEP);
			if (ignsuff) {
				csa_suffix = ap_pstrdup(p->pool_req, ignsuff);
				/* cut the suffix off dotpartname */
				dotpartname[ignsuff - dotpartname] = '\0';
			}
		}

		/* process partname */
		if (!dotpartname || dotpartname[0] == '\0')
		    CSA_SET(p->flags, CSA_FL_PART_IS_DEF|CSA_FL_PART_IS_EMPTY);

		p->outcharset = cstools_whichcode(charsetname, 0);
	}
	else {
		dotpartname = NULL;
		charsetname = NULL;
	}

	/* remember the suffix for processing in csa_unparse_url() */
	if (*csa_suffix) p->csa_suffix = csa_suffix;
	/* remember name of output encoding as specified in URL for
	 * later processing in csa_init_params() */
	*outnamep = (charsetname) ? charsetname : "";

	/* if input recoding is on, recode QUERY_STRING into server encoding */
	if (CSA_ISSET(cfg->flags, CSA_CFG_RECODEINPUT)) {
		if (qs && qs->len) {
			csa_String res;
			char *swp = ap_pstrndup(p->pool_tmp, qs->value,
				(int) qs->len + 1);
			csa_decodequery(&res, p, swp, qs->len);
			swp[res.len] = '\0';
			csa_setvar(p, "QUERY_STRING", res.value,
				CSA_I_COPYVALUE);
		}
	}
	
	/* lookup a few often needed variables */
	p->csacek = csa_getvar(p, "SCRIPT_NAME");

	csa_fillstring(&p->part, (dotpartname) ? dotpartname : "", -1, -1);

	/* set output names accordingly to p->outcharset */
	x_set_outnames(p);

	/* find out if client uses request method HEAD */
	rm = csa_getvar(p, "REQUEST_METHOD");
	if (rm && strcasecmp("HEAD", rm->value) == 0)
		CSA_SET(p->flags, CSA_FL_HEADERS_ONLY);

	/* now, redirect client to URL ending with '/', if PATH_INFO has */
	/* not been set */
	if (path_not_set) {
		char *msg;
		csa_setheaderout(p, "Status", "301 Moved Permanently", 0);
		temp = csa_construct_url(p, NULL, "/");
		msg = csa_alloca(strlen(temp) + 100, p->pool_tmp);
		sprintf(msg,"Use following URL instead: <A HREF=\"%s\">%s</A>.",
			temp, temp);
		csa_setheaderout(p, "Location", temp, CSA_I_COPYVALUE);
		return HTTP_MOVED_PERMANENTLY;
	}
		
	return CSA_OK;
}

/*
 * flushes compression buffers (if any) and sets Content-Length to correct value
 */
static void
x_finish_body(p)
  csa_params_t *p;
{
#ifdef CSA_DEBUG
	csa_debug(p->dbg, "finish_body: called");
#endif

	/* add remains in buffer to output */
	if (p->body_buf && p->body_buf->len > 0)
		csa_add_subs_output(p, p->body_buf, p->body_buf->len, 1);

	/* automatically write BAR on the end of HTML document */
	if (CSA_ISSET(p->flags, CSA_FL_ISHTML) && csa_bar_printalways(p->Bar))
		csa_run_cmd(p, "BAR", 3, 0);

	/* handle Range header sent by client */
	/* note - we don't need to flush compression buffers - compression */
	/* is disabled when client sends Range header, because we need to  */
	/* process the data later */
	if (p->range) {
	    csa_range_fixup(p->range, p->content_length);
	    if (p->range[0]) { /* still some valid ranges after fixup */
		const csa_String *ct = csa_getheaderout(p, "Content-Type");
		char *border=NULL, *content_range;
		size_t border_len=0, all_content_length=p->content_length;
		size_t cr_len, sindex, from, to, where;
		csa_queue_t *oneitem, *all = p->output;

#ifdef CSA_DEBUG
		csa_debug(p->dbg, "finish_body: processing Ranges");
#endif

		csa_setheaderout(p, "Status", "206 Partial Content",
			CSA_I_OVERWRITE);

		/* empty output as if nothing has been processed */
		p->output = NULL;
		p->content_length = 0;

		if (p->range[1]) { /* more then one valid range */
			char *newct;
			border = (char *) csa_alloca(sizeof("C-SaCzech")
					+ 1 + sizeof(CSA_VERSION) + 1
					+ 2*CSA_GETMAXNUMCOUNT(int) + 1 + 1,
					p->pool_tmp);
			border_len = sprintf(border, "C-SaCzech_%s_%d_%d",
					CSA_VERSION,
					p->outcharset,
					(int) time(NULL)); /* safe */
			newct = (char *) ap_palloc(p->pool_req,
				40 + (int)border_len + 1);
			sprintf(newct, "multipart/byteranges; boundary=%s",
				border); /* safe */
			csa_setheaderout(p, "Content-Type", newct,
				CSA_I_OVERWRITE);
		}

		/* Content-Range is in form "bytes NUM-NUM/NUM", so it's */
		/* safe to pre-allocate space for resulting string */
		content_range = (char *) csa_alloca(
			6 + 3*CSA_GETMAXNUMCOUNT(int) + 1,
			p->pool_tmp);

		for(sindex=0; p->range[sindex]; sindex++) {
			if (border) {
				csa_add_output(p, border, border_len, 0);
				csa_add_output(p, "\r\n", 2, 0);
				csa_add_output(p, ct->value, ct->len, 0);
				csa_add_output(p, "\r\n", 2, 0);
			}
	
			cr_len = sprintf(content_range, "bytes %d-%d/%u",
				p->range[sindex]->from, p->range[sindex]->to,
				(unsigned int)all_content_length);
			
			if (border) {
				csa_add_output(p, "Content-Range", 13, 0);
				csa_add_output(p, content_range, cr_len, 0);
				csa_add_output(p, "\r\n\r\n", 4, 0);
			}
			else {
				csa_setheaderout(p, "Content-Range",
					content_range, CSA_I_OVERWRITE);
			}

			oneitem = all;
			from = p->range[sindex]->from;
			to = p->range[sindex]->to;
			where = 0;
			/* find buffer with start of input client wants */
			while(oneitem && where < from) {
				if (where + oneitem->len > to)
					break;
				else {
					where += oneitem->len;
					oneitem = oneitem->prev;
				}
			}
			for(; oneitem && where < to; oneitem = oneitem->prev) {
				size_t cnt = to - where;
				size_t to_read = (cnt > oneitem->len)
					? oneitem->len : cnt;
				csa_add_output(p, oneitem->value,
					to_read, 0);
				where += to_read;
			}
		} /* for(;;) */
	    } /* if any range remains after range fixup */
	} /* if (p->range) */
#ifdef CSA_WANT_COMPRESSION
	else if (CSA_ISSET(p->flags, CSA_FL_C_INITIALIZED)) {
		/* flush compress structures */
		if (p->compress == CSA_C_COMPRESS)
			csa_done_compress(p);
		else if (p->compress == CSA_C_GZIP)
			csa_done_gzip(p);
		else if (p->compress == CSA_C_DEFLATE)
			csa_done_deflate(p);
	}
#endif /* CSA_WANT_COMPRESSION */
}

/*
 * send output headers
 * caller is responsible for ensuring the headers actually SHOULD be
 * sent (i.e. the protocol is HTTP/1.0 or higher and the headers
 * haven't been sent already)
 */
int
csa_send_headersout(p)
  csa_params_t *p;
{
	csa_item_t *ho; /* headers out */
	const csa_String *status;

#ifdef CSA_DEBUG
	csa_debug(p->dbg, "csa_send_headersout: called");
#endif

#ifdef CSA_MUTACE_XCGI
	/* some older mod_html crashes when CGI script doesn't return
	 * Content-Type -- maybe only when the script sends unusual Status,
	 * such as "304 Nod Modified" */
	if (!csa_getheaderout(p, "Content-Type")
	    && csa_getheaderin(p, "Charset-To")) {
		csa_setheaderout(p, "Content-Type", "text/plain", 0);
	}
#endif

#ifdef CSA_WANT_COMPRESSION
	/* provide Content-Encoding header if we used compression
	 * add also appropriate value to Vary header to please proxies; add 
	 * the Accept-Encoding value into it only when actually _using_ the
	 * compression, since if proxy caches uncompressed version, we want
	 * it to use the cached version even for clients supporting
	 * compression (i.e. better cache than compress) */
	if (p->compress != CSA_C_NONE) {
		csa_setheaderout(p, "Content-Encoding",
			CSA_COMPRESS_NAME(p->compress), CSA_I_HEADERCAT);
		csa_setheaderout(p, "Vary", "Accept-Encoding",
			CSA_I_HEADERCAT);
	}
#endif

	/* if not sending the output continually or sending the headers only,
	 * create Content-Length header and add it to output headers */
	if (!CSA_ISSET(p->flags, CSA_FL_OUT_CONT)) {
		char cl[CSA_GETMAXNUMCOUNT(p->content_length)+1];
		sprintf(cl, "%u", (unsigned int) p->content_length);
		csa_setheaderout(p, "Content-Length", cl,
			CSA_I_OVERWRITE|CSA_I_COPYVALUE);
	}

	/* send proper Transfer-Encoding header if needed */
	if (CSA_ISSET(p->flags, CSA_FL_OUT_CHUNKED)) {
		csa_setheaderout(p, "Transfer-Encoding", "chunked",
			CSA_I_OVERWRITE);
	}

	/*
	 * If we guessed something by the value of Accept-Charset,
	 * Accept-Charset or User-Agent, provide appropriate Vary header.
	 */
	if (CSA_ISSET(p->flags, CSA_FL_VARY_AC))
		csa_setheaderout(p, "Vary", "Accept-Charset", CSA_I_HEADERCAT);
	if (CSA_ISSET(p->flags, CSA_FL_VARY_AL))
		csa_setheaderout(p, "Vary", "Accept-Language", CSA_I_HEADERCAT);
	if (CSA_ISSET(p->flags, CSA_FL_VARY_UA))
		csa_setheaderout(p, "Vary", "User-Agent", CSA_I_HEADERCAT);
	
	/* Status header has to go first */
	status = csa_getheaderout(p, "Status");
	if (status) {
		csa_md_send_header(p, "Status", status->value);
#ifdef CSA_DEBUG
		csa_debug(p->dbg, "send_headersout: Status: ``%s''",
			status->value);
#endif
		csa_unsetheaderout(p, "Status");
	}

	/* send output headers - no need to keep order in which they */
	/* have been added to the list */
	ho = p->headersout;
	for(;ho; ho = ho->prev) {
		csa_md_send_header(p, ho->key.value, ho->value.value);
#ifdef CSA_DEBUG
		csa_debug(p->dbg, "send_headersout: sending header ``%s: %s''",
			ho->key.value, ho->value.value);
#endif
	}

	CSA_SET(p->flags, CSA_FL_HEADERS_SENT);
	return 0;
}

/*
 * called when some alloc function fails
 */
void
csa_alloc_fail()
{
	/* call MD handler */
	csa_md_alloc_fail();

#if defined(CSA_DEBUG) && !defined(__MSWIN__)
	raise(SIGSTOP);
#endif
	exit(CSA_FATAL);
}

/*
 * flush output buffer and send the data to client
 * caller is responsible for making sure client supports chunked encoding
 * i.e. that it's HTTP/1.1+ client
 */
void
csa_flush_output(p)
  csa_params_t *p;
{
#ifndef CSA_USE_APACHE13_API
	char chunk_len[7];
	size_t len;
#endif

#ifdef CSA_DEBUG
	csa_debug(p->dbg, "csa_flush_output: called");
#endif

	if (!CSA_ISSET(p->flags, CSA_FL_HEADERS_SENT)) {
		csa_send_headersout(p);
		csa_md_send_separator(p);
	}

	/* don't send any body data if client used HEAD request method */
	if (CSA_ISSET(p->flags, CSA_FL_HEADERS_ONLY)) return;

	/* don't do anything if there are no data to be sent yet */
	if (!p->output) return;

	/* if not chunking, just send the contents of buffer and return */
	if (!CSA_ISSET(p->flags, CSA_FL_OUT_CHUNKED)) {
		if (p->output->len) {
			csa_md_send_output(p, p->output->value, p->output->len);
			p->output->len = 0;
		}
		return;
	}

#ifndef CSA_USE_APACHE13_API
	/* following would not be ever needed for Apache 1.3 */

	/* don't do anything if there is nothing to be sent */
	if (p->output->len <= 6) return;

	/* insert chunk length on the beginning of the chunk */
	len = sprintf(chunk_len, "%x\r\n", p->output->len - 6);
	memcpy(p->output->value, chunk_len, len);

	/* if chunk size string is < 6 (i.e. the chunk len expressed
	 * in hex was less than 4), we have to move the contents of buffer
	 * to eliminate the extra space - we can't help with prepending
	 * 0's on the beginning of the chunk length, HTTP/1.1 explicitly
	 * states the chunk size has to be expressed as hex number
	 * not starting with 0 */
	if (len < 6) {
		memmove(p->output->value + len, p->output->value + 6,
			p->output->len - 6);
		p->output->len -= (6 - len);
	}
	 
	/* add extra \r\n to end the chunk */
	p->output->value[p->output->len++] = '\r';
	p->output->value[p->output->len++] = '\n';

	/* send the data to client */
	csa_md_send_output(p, p->output->value, p->output->len);

	/* null len, but leave place for chunk length & \r\n */
	p->output->len = X_CSA_OUT_CHUNKED_SIZESPACE;
#endif /* !Apache 1.3 */
}

/*
 * adds (possibly compressed) string into output
 */
int
csa_add_output(p, src, len, flags)
  csa_params_t *p;
  const char *src;
  size_t len;
  int flags;
{
	csa_queue_t *item;
	size_t buflen;
	size_t copylen;

	if (CSA_ISSET(flags, CSA_OUT_STR))
		len = strlen((const char *)src);
	
	if (len == 0) return 0; /* nothing to do, exit */

#ifdef CSA_DEBUG
	if (CSA_ISSET(flags, CSA_OUT_RAW))
		csa_debug(p->dbg, "csa_add_output: len=%d, <binary data>",len);
	else
        	csa_debug(p->dbg, "csa_add_output: len=%d, %.*s", len,len,src);
#endif

#ifdef CSA_WANT_COMPRESSION
	if (p->compress != CSA_C_NONE && !CSA_ISSET(flags, CSA_OUT_RAW)) {
		/*
		 * If compression engine is not yet initialized, initialize
		 * it now.
		 */
		if (!CSA_ISSET(p->flags, CSA_FL_C_INITIALIZED))
			x_init_compression(p);

		/* Note: csa_add_output() is recursively called by  */
		/* compression routines with CSA_RAW_INPUT flag set */
		if (p->compress == CSA_C_COMPRESS)
			return csa_compress(p, src, (int) len);
		else if (p->compress==CSA_C_DEFLATE || p->compress==CSA_C_GZIP)
			return csa_deflate(p, src, len);
	}
#endif /* CSA_WANT_COMPRESSION */
		
	if (CSA_ISSET(p->flags, CSA_FL_OUT_CONT))
		buflen = X_CSA_OUT_BUFLEN_CHUNK;
	else
		buflen = X_CSA_OUT_BUFLEN;
	p->content_length += len;
	item = p->output;
	while(len >0)
	{
		/* allocate new buffer, if it's needed */
		if (!p->output || p->output->len == p->output->maxlen)
		{
			if (p->output && CSA_ISSET(p->flags, CSA_FL_OUT_CONT)) {
				csa_flush_output(p);
				continue;
			}

			item = (csa_queue_t *)
				ap_palloc(p->pool_req, sizeof(csa_queue_t));

			item->maxlen = buflen;
			item->value = ap_palloc(p->pool_req, (int)item->maxlen);
			item->len = 0;

			/* insert buffer into output queue */
			item->next = NULL;
			item->prev = p->output; 
			if (p->output) p->output->next = item;
			p->output = item;

#ifndef CSA_USE_APACHE13_API
			if (CSA_ISSET(p->flags, CSA_FL_OUT_CHUNKED)) {
				/* leave place for ending \r\n */
				item->maxlen -= 2;
				/* leave place for chunk length & \r\n */
				item->len = X_CSA_OUT_CHUNKED_SIZESPACE; 
			}
#endif /* !Apache 1.3 */
		}

		copylen = item->maxlen - item->len;
		if (len < copylen) copylen = len;
		memcpy(&(item->value[item->len]), src, copylen);
		item->len += copylen;

		len -= copylen;
		src += copylen;
	}

	return 0;
}

/*
 * flushes output queue to real output
 */
int
csa_send_body(p)
  csa_params_t *p;
{
	csa_queue_t	*q;
#ifdef CSA_DEBUG
	csa_debug(p->dbg, "send_body: called");
#endif

	/* rewind to the start of queue */
	q = p->output;
	while(q && q->prev) q = q->prev;

	/* send all the data */
	for(;q ; q = q->next)
		csa_md_send_output(p, q->value, q->len);

	return 0;
}

/*
 * adds an item to a list; see csacek.h for definition of CSA_I_* flags
 */
int
csa_setitem(p, item_listp, key, value, flags)
  csa_params_t *p;
  csa_item_t **item_listp;
  const char *key, *value;
  int flags;
{
	csa_item_t *item=NULL;
	struct pool *wpool;

	/* sanity check */
	if (item_listp == NULL || key == NULL || *key == '\0' 
		|| value == NULL )
		return 1;

	if (CSA_ISSET(flags, CSA_I_TMP)) wpool = p->pool_tmp;
	else wpool = p->pool_req;

	/* remove old value of the key, if present */
	if (flags & CSA_I_OVERWRITE)
		csa_unsetitem(item_listp, key);

	if ((flags & (CSA_I_IFNOTSET|CSA_I_HEADERCAT))
	    && (item = x_item_lookup(*item_listp, key)) )
	{
		if (CSA_ISSET(flags, CSA_I_HEADERCAT)) {
			char *chp;
			int len = item->value.len + 2 + strlen(value) + 1;

			chp = (char *) ap_palloc(wpool, len);
			sprintf(chp, "%s, %s", item->value.value, value);
			csa_fillstring(&(item->value), chp, len, -1);
			return 0;
		}
		else if (CSA_ISSET(flags, CSA_I_IFNOTSET))
			/* item set already, exit */
			return -1;
	}

	item = (csa_item_t *) ap_pcalloc(wpool, sizeof(*item));

	if (CSA_ISSET(flags, CSA_I_COPYKEY))
		key = ap_pstrdup(wpool, key);
	csa_fillstring(&(item->key), key, -1,-1);

	if (CSA_ISSET(flags, CSA_I_COPYVALUE))
		value = ap_pstrdup(wpool, value);
	csa_fillstring(&(item->value), value, -1, -1);
		
	/* insert item into the list */
	item->prev	= *item_listp;
	if (*item_listp) (*item_listp)->next = item;
	*item_listp	= item;

	return 0;
}

/*
 * returns value of item with name ``key'' or NULL if the is no
 * such item in the list
 */
const csa_String *
csa_getitem(item_list, key)
  csa_item_t *item_list;
  const char *key;
{
	csa_item_t *var;

	var = x_item_lookup(item_list, key);
	if (var) return (const csa_String *) &(var->value);
	else return NULL;
}

/*
 * removes item from the list
 */
void
csa_unsetitem(item_list, key)
  csa_item_t **item_list;
  const char *key;
{
	csa_item_t *item;

	if (item_list == NULL || *item_list == NULL)
		return;

	/* remove it from two-way chained list */
	for(;;) {
		item = x_item_lookup(*item_list, key);
		if (!item) break;
		if (item->next) item->next->prev = item->prev;
		if (item->prev) item->prev->next = item->next;
		/* if removed item is first on the list, update */
		/* list pointer so it would point to the next item */
		if (item == *item_list) *item_list = item->prev;
	}

	return;
}

/*
 * special hack used when we won't recode data returned by request and
 * the response contained Content-Length headers or it's been chunked;
 * procedure avoids caching of the response data within CSacek and just
 * forwards the data as they are to client
 * This MUST be called right AFTER the output headers are processed
 * Note: it's safe to call this routine several times, just as input data
 * become available; special care is taken to send the output headers
 * just once before first chunk of data
 */
void
csa_direct_forward(p)
  csa_params_t *p;
{
	size_t to_read, num_read;
	char buf[8196];

#ifdef CSA_DEBUG
	csa_debug(p->dbg, "csa_direct_forward: called");
#endif

	/* if the headers haven't been sent yet, send them now */ 
	if (CSA_SEND_HEADERS(p)) {
		if (CSA_ISSET(p->flags, CSA_FL_CHUNKED_RESP)) {
			/* set flag indicating chunked output -
			 * csa_send_headersout() will DTRT */
			CSA_SET(p->flags, CSA_FL_OUT_CHUNKED);
		}
		else if (p->available_in > 0) {
			/* just set the content_length to proper value -
			 * csa_send_headersout() will add correct
			 * Content-Length header */
			p->content_length = p->available_in;
		}

		csa_send_headersout(p);
		csa_md_send_separator(p);
	}

	/* suck all available input and forward it to client */
	to_read = p->available_in;
	for(;;) {
		num_read = csa_md_read_response(p, buf,
			(sizeof(buf)<to_read) ? sizeof(buf) : to_read);
		if (num_read == 0) break;
		csa_md_send_output(p, buf, (size_t) num_read);
		to_read -= num_read;
	}
} /* csa_direct_forward */

#if defined(CSA_MUTACE_APACHE) || defined(CSA_MUTACE_ISAPI)
/*
 * Reads list of csacek servers and adds them into array.
 */
void
csa_add_servers(wpool, list, value, len)
  struct pool *wpool;
  csa_slist_t *list;
  char *value;
  size_t len;
{
	int port;
	size_t spanned;
	const char *name;

	value[len] = 0;
	name = strtok(value, ",");
	for(; name != NULL; name = strtok(NULL, ",")) {
		/* skip potential leading whitespace */
		name += strspn(name, " \t\r\n");

		/* get server name & optional port */
		if ((spanned = strcspn(name, " \t\r\n:")) == 0)
			continue;

		if (name[spanned] == ':')
			port = atoi(&name[spanned+1]);
		else if (name[spanned] != '\0')
			port = atoi(&name[spanned]);
		else
			port = 0;

		name = ap_pstrndup(wpool, name, (int)spanned);

		csa_slist_add(list, name, port);
	}
}
#endif /* APACHE or ISAPI */

#if defined(CSA_MUTACE_XCGI) || defined(CSA_MUTACE_ISAPI)
/*
 * this routine is called by ISAPI & (Fast)CGI csa_md_call_whichcode()
 * function - it opens and processes whichcode template file given
 * as parameter
 * returns CSA_OK if succesful 
 */
int
csa_process_whichcode_file(csa_params_t *p, const char *filename)
{
	FILE *fp;
	const char *alt_filename;

	fp = fopen(filename, "r");
	if (!fp) {
	    alt_filename = strrchr(filename, '/');
	    if (!alt_filename) alt_filename = strrchr(filename, '\\');
	    if (!alt_filename || !(fp = fopen(alt_filename, "r"))) {
		csa_http_error(p,
		    "Unable to open template file for whichcode", filename);
		return CSA_FATAL;
	    }
	}
	csa_set_fio(p, &(p->resp), 0, fp, fileno(fp));
	ap_note_cleanups_for_file(p->pool_req, fp);

	csa_setheaderout(p, "Status", "200 OK", 0);
	csa_setheaderout(p, "Content-Type", "text/html", 0);

	/* set flags accordingly */
	CSA_SET(p->flags, CSA_FL_ISHTML|CSA_FL_CONVERT);
	CSA_UNSET(p->flags, CSA_FL_NOEXECCMDS);

	/* parse the template through CSacek */
	csa_process_body(p);

	return CSA_OK;
}
#endif /* CSA_MUTACE_{CGI|FASTCGI|ISAPI} */

/*
 * Switch input code set to new one, if the new is different.
 * Return 1 if we switched to new charset, 0 if no change was made,
 * -1 in case of error.
 */
int
csa_switch_incharset(p, new_incharset)
	csa_params_t *p;
	cstools_t new_incharset;
{
	if (new_incharset == CSTOOLS_UNKNOWN)
		return (-1);

	if (p->incharset == new_incharset)
		return (0);

	/*
	 * Some software (for example PHP 4.0b3) use iso-8859-1
	 * when they really mean "I don't know which charset it's in" -
	 * so just ignore the charset if it's iso-8859-1.
	 */
	if (new_incharset == CSTOOLS_ISOLatin1)
		return (0);

	p->incharset = new_incharset;
	(void) cstools_init(&p->mp, p->incharset, p->outcharset);

	return (1);
}

/****************************************************************/
/*		implementation of CSacek commands		*/

/*
 * compare_func for PART
 */
static int
x_compare_Part(p, value)
  csa_params_t *p;
  const char *value;
{
	size_t len = strlen(value);

	/* don't forget to not count leading dot in p->part */
	if (len > p->part.len - 1) len = p->part.len - 1;

	return (len > 0
		&& strncasecmp(value, p->part.value + 1, len) == 0);
		/* don't forget to skip leading dot in p->part */
}

/*
 * compare_func for DOMAIN
 */
static int
x_compare_Domain(p, value)
  csa_params_t *p;
  const char *value;
{
	const csa_String *addr = p->tmp_strp[0];
	const csa_String *host = p->tmp_strp[1];
	int retval;

	retval = ( (host->len > 0
			&& csa_has_suffix(host->value, value, 0))
		   || (addr->len > 0  
			&& strncasecmp(value, addr->value, addr->len) == 0) );

	return retval;
}

/*
 * compare_func for CHARSET
 */
static int
x_compare_Charset(p, value)
  csa_params_t *p;
  const char *value;
{
	return (cstools_whichcode(value, 0) == p->outcharset);
}

/*
 * function which implements commands PART, DOMAIN and CHARSET
 * special parameters ALL and OTHER are recognized and treated specially
 * returns 0 if everything is okay, non-zero otherwise
 */
int
csa_DocParts(p, prm)
  csa_params_t *p;
  void *prm;
{
	/* LINTED */ /* lint: conversion of pointer to 'int' may lose bits */
	int cmd = (int) prm;
	int flag;
	int (*compare_func) __P((csa_params_t *p, const char *value));
	const csa_arg_t *item;
	const char *compstr, *put_out, *key, *value;
	const csa_String *stra[2];
	int some_args=0;

	/* cmd dependant initialization */
	if (cmd == CSA_PART) {
		flag = CSA_VALID_PART;
		compare_func = x_compare_Part;
	}
	else if (cmd == CSA_DOMAIN) {
		flag = CSA_VALID_DOMAIN;
		compare_func = x_compare_Domain;
	
		stra[0] = csa_getvar(p, "REMOTE_ADDR");
		stra[1] = csa_getvar(p, "REMOTE_HOST");
		if (stra[0] == NULL || stra[1] == NULL) return 1;

		if (atoi(stra[1]->value)) {
			const char *name;
			name = csa_gethostbyaddr(p->pool_req, stra[1]->value);
			if (name) {
			    csa_setvar(p, "REMOTE_HOST", name, 0);
			    stra[1] = csa_getvar(p, "REMOTE_HOST");
			}
		}

		p->tmp_strp = stra;
	}
	else if (cmd == CSA_CHARSET) {
		flag = CSA_VALID_CHARSET;
		compare_func = x_compare_Charset;
	}
	else
		return 1;

	put_out = NULL;
	while ((item=csa_arg_take(p)) != NULL) {
		key = csa_arg_getkey(item);
		value = csa_arg_getvalue(item);
		some_args = 1;		/* mark some args were passed */

		/* set values for comparison */
		if (key) compstr = key;
		else compstr = value;

		/* the comparison itself */
		if (strncasecmp(compstr, "ALL", 3) == 0)
		{
			if (key) put_out = value;
			else {
				CSA_SET(p->flags_parts, flag);
				CSA_SET(p->flags_parts, CSA_VALID_2OTHER(flag));
			}
			break;
		}
		else if ((CSA_ISSET(p->flags_parts, CSA_VALID_2OTHER(flag))
			 	&& strncasecmp(compstr, "OTHER", 5) == 0)
			 || compare_func(p, compstr) == 1)
		{
			if (key) put_out = value;
			else {
				CSA_SET(p->flags_parts, flag);
				CSA_UNSET(p->flags_parts,
					CSA_VALID_2OTHER(flag));
			}
			break;
		}
		else {
			/*
			 * invalid name => if the input was NOT in form
			 * key=value, unset validating flag
			 */
			if (!key) CSA_UNSET(p->flags_parts, flag);
		}
	} /* while */

	/* add text to output */
	if (put_out) {
		csa_String_b str;
		str.len   = strlen(put_out);
		str.value = ap_pstrndup(p->pool_tmp, put_out, (int)str.len);
		(void) csa_add_subs_output(p, &str, str.len, 1);
	}

	/* cmd dependant finish code */
	if (cmd == CSA_PART && !some_args) {
		/* compatibility with old CSacek series 1.3.x; PART w/o  */
		/* arguments succeeds if the name of output PART is same */
		/* as the default one */

		if (CSA_ISSET(p->flags, CSA_FL_PART_IS_DEF) ||
		    (p->part.len && strcasecmp(p->part.value+1, p->dd) == 0))
		{
			CSA_SET(p->flags_parts, flag);
			CSA_UNSET(p->flags_parts, CSA_VALID_2OTHER(flag));
			some_args = 1;
		}
		else {
			CSA_UNSET(p->flags_parts, flag);
			CSA_SET(p->flags_parts, CSA_VALID_2OTHER(flag));
		}
	}

	return (some_args == 0);
}

/*
 * implementation of META command; if the header set by META tag is
 * Content-Type and contains optional ``charset'', sets CSacek input
 * charset value accordingly and returns 0 (success), otherwise returns 1
 */
/* ARGSUSED */
int
csa_Meta(p, prm)
  csa_params_t *p;
  void *prm;
{
	int retval=1, isct=0;
	const csa_arg_t *arg;
	const char *key, *value;
	csa_String_b buff, *buf = &buff;
	char q;

	CSA_FILLSTRING(buf, (char *)csa_alloca(100, p->pool_tmp), 0, 100);

	x_add_2buf(p->pool_tmp, buf, "<", 1);
	x_add_2buf(p->pool_tmp, buf, csa_yy_getcmdname(p->yy), -1);

	while((arg = csa_arg_take(p)) != NULL) {
		key   = csa_arg_getkey(arg);
		value = csa_arg_getvalue(arg);

		x_add_2buf(p->pool_tmp, buf, " ", 1);
		if (!key) {
			x_add_2buf(p->pool_tmp, buf, value, -1);
			continue;
		}

		if (strcasecmp(key, "HTTP-EQUIV") == 0
		    && strcasecmp(value, "Content-Type") == 0) {
			isct = 1;
		} else if (strcasecmp(key, "CONTENT") == 0) {
			const char *foop = strchr(value, ';');
			const char *argvalue = NULL;
			char *newvalue;
			size_t len;

			if (foop && (foop = csa_strcasestr(foop, "charset="))) {
				foop += 8; /* skip "charset=" */

				len = strcspn(foop, " \r\t\n");

				if (csa_switch_incharset(p,
					cstools_whichcode(foop, len)) > 0) {
#ifdef CSA_DEBUG
					csa_debug(p->dbg, "csa_Meta: incharset switched to %.*s",
						len, foop);
#endif
				}

				argvalue = cstools_name(p->outcharset,
					CSTOOLS_MIMENAME);

				/*
				 * Make new value as concatenation of:
				 * 1. everything up to and including charset=
				 * 2. name of the output charset
				 */
				len = foop - value + strlen(argvalue) + 10;
				newvalue = csa_alloca(len, p->pool_tmp);
				sprintf(newvalue, "%.*s%s",
					foop - value, value, argvalue); /*safe*/
				value = newvalue;
				
				retval = 0;
			}
		}

		x_add_2buf(p->pool_tmp, buf, key, -1);
		x_add_2buf(p->pool_tmp, buf, "=", 1);
		if ((q = csa_arg_getflags(arg)))
			x_add_2buf(p->pool_tmp, buf, &q, 1);
		x_add_2buf(p->pool_tmp, buf, value, -1);
		if (q)
			x_add_2buf(p->pool_tmp, buf, &q, 1);
	}

	if (retval == 0) {
		x_add_2buf(p->pool_tmp, buf, ">", 1);
		csa_add_recode_output(p, buf->value, buf->len, NULL);
	}

	return (retval);
}

/*
 * this takes care of special HTML tags (aka "commands"), which use
 * URLs; it plain copies all the parameters, but adds CSacek in front
 * of URL when HREF or SRC parameter is specified
 */
int
csa_ChangeURL(p, prm)
  csa_params_t *p;
  void *prm;
{
	/* LINTED */
	int cmd = (int) prm, rewrite_url;
	const csa_arg_t *arg;
	csa_url_t *urlt;
	const char *key, *value;
	csa_String_b buff, *buf = &buff;
	char quot;

	CSA_FILLSTRING(buf, (char *)csa_alloca(100, p->pool_tmp), 0, 100);

	x_add_2buf(p->pool_tmp, buf, "<", 1);
	x_add_2buf(p->pool_tmp, buf, csa_yy_getcmdname(p->yy), -1);

	for(;;) {
		arg = csa_arg_take(p);
		if (!arg) break;

		key   = csa_arg_getkey(arg);
		value = csa_arg_getvalue(arg);		
		quot  = (char) csa_arg_getflags(arg);

		x_add_2buf(p->pool_tmp, buf, " ", 1);
		if (!key) {
			/* no key, not interesting for me, just add value */
			/* to output and process next item */
			x_add_2buf(p->pool_tmp, buf, value, -1);
			continue;
		}

		if ( (cmd == CSA_HREF && strcasecmp(key, "HREF") == 0)
			|| (cmd == CSA_SRC && strcasecmp(key, "SRC") == 0)
			|| (cmd==CSA_ACTION && strcasecmp(key, "ACTION") == 0))
		{
			value = csa_subs_string(p,
					ap_pstrdup(p->pool_tmp, value));
			urlt  = csa_parse_url(p, value);

			rewrite_url = urlt->can_rewrite;
			/* if URI contained only CSacek part, append URI
			 * of current document */
			if (urlt->csacek && (!urlt->uri || !*urlt->uri)) {
				urlt->uri = csa_getvar(p, "PATH_INFO")->value;
				rewrite_url = 1;
				if (!urlt->qs) {
					const csa_String *qs;
					qs = csa_getvar(p, "QUERY_STRING");
					if (qs && *qs->value)
						urlt->qs = qs->value;
				}
			}
			/* if active part is not the default one and URI
			 * contained CSacek, rewrite URI so that it would
			 * contain correct part */
			if (urlt->csacek
			    && !CSA_ISSET(p->flags, CSA_FL_PART_IS_DEF))
				rewrite_url = 1;

			if (rewrite_url && CSA_IS_CSACEK_SERVER(p, urlt))
			{
			   value = csa_unparse_url(p, urlt, p->csacek->value);
			}
		}

		/* construct key="value" string on output; use quotes
		 * only when they were specified in the source */
		x_add_2buf(p->pool_tmp, buf, key, -1);
		x_add_2buf(p->pool_tmp, buf, "=", 1);
		if (quot)
			x_add_2buf(p->pool_tmp, buf, &quot, 1);
		x_add_2buf(p->pool_tmp, buf, value, -1);
		if (quot)
			x_add_2buf(p->pool_tmp, buf, &quot, 1);
	}

	x_add_2buf(p->pool_tmp, buf, ">", 1);
	csa_add_recode_output(p, buf->value, buf->len, NULL);

	return 0;
}

/*
 * this command serves as general interface to internal CSacek structures
 */
int
csa_Set(p, prm)
  csa_params_t *p;
  void *prm;
{
	/* LINTED */
	int cmd = (int) prm;
	int flag, isyes, setifyes;
	const csa_arg_t *arg;
	const char *key, *value;

	if (cmd == CSA_EXECCMDS) {
		CSA_UNSET(p->flags, CSA_FL_NOEXECCMDS);
		return 0;
	}
	else if (cmd == CSA_NOEXECCMDS) {
		CSA_SET(p->flags, CSA_FL_NOEXECCMDS);
		return 0;
	}

	for(;;) {
		arg = csa_arg_take(p);
		if (!arg) break;

		key   = csa_arg_getkey(arg);
		value = csa_arg_getvalue(arg);		

		if (strcasecmp(key, "EXECCMDS") == 0)
			flag = CSA_FL_NOEXECCMDS, setifyes = 0;
		else if (strcasecmp(key, "CHANGEURL") == 0)
			flag = CSA_FL_CHANGEURL, setifyes = 1;
		else
			/* invalid parameter, quit */
			return -1;

		isyes = (strcasecmp(value, "YES") == 0) ? 1 : 0;
		if (isyes == setifyes)
			CSA_SET(p->flags, flag);
		else
			CSA_UNSET(p->flags, flag);

	}

	return 0;
}

/*
 * adds server[s] to the list of CSacek servers
 */
/* ARGSUSED */
int
csa_csacekServers(p, prm)
  csa_params_t *p;
  void *prm;
{
	const csa_arg_t *arg;
	const char *name, *dd;
	int port=0;

	/*
	 * The parameters are either "name port" (old deprecated csacekServer)
	 * or "name[:port] [name[:port] ...]"
	 */
	while ((arg = csa_arg_take(p)) != NULL) {
		name = csa_arg_getvalue(arg);
		port = 0;

		/*
		 * Get server port - we first look at next parameter,
		 * whether it's a number (to be compatible with deprecated
		 * csacekServer command); if it isn't a number, we look
		 * if name contains ':' and take the port off it if it does.
		 */
		arg = csa_arg_peek(p);
		if (arg && (port = atoi(csa_arg_getvalue(arg))))
			(void) csa_arg_take(p);
		else  {
			if ((dd = strchr(name, ':')) != NULL) {
				port = atoi(&dd[1]);
				name = ap_pstrndup(p->pool_req, name,
					dd - name);
			} else
				name = ap_pstrdup(p->pool_req, name);
		}

		/* add server into list of CSacek servers */
		csa_slist_add(p->csacek_servers, name, port);
	}

	return 0;
}

/*
 * yet another work-around for MSIE - if the document is sent in iso-8859-2
 * and is displayed under windoze (which use windows-1250 natively),
 * it uses bad (non-czech) fonts, if the font name doesn't explicitely
 * specify to use CE variant - apparently, it's handled correctly if
 * the document encoding is windows-1250, *sniff*
 */
/* ARGSUSED */
int
csa_Font(p, prm)
  csa_params_t *p;
  void *prm;
{
	const csa_arg_t *arg;
	csa_String_b buff, *buf = &buff;
	const char *key, *value, *beg, *com;
	char *new, *hlp;

	/* don't do anything if source encoding is not windows-1250 */
	/* or target encoding is widows-1250 */
	if (p->incharset != CSTOOLS_CP1250 || p->outcharset == CSTOOLS_CP1250)
		return 1;

	CSA_FILLSTRING(buf, (char *)csa_alloca(100, p->pool_tmp), 0, 100);
	x_add_2buf(p->pool_tmp, buf, "<FONT", 5);
	for(;;) {
		arg = csa_arg_take(p);
		if (!arg) break;

		key   = csa_arg_getkey(arg);
		value = csa_arg_getvalue(arg);		

		/* key is EVER set - this cmd accepts CMD_P_EQUATION only */
		x_add_2buf(p->pool_tmp, buf, " ", 1);
		x_add_2buf(p->pool_tmp, buf, key, -1);
		x_add_2buf(p->pool_tmp, buf, "=\"", 2);
		
		if (strcasecmp(key, "FACE") == 0) {
			/* for every font name which doesn't end with CE, */
			/* prepend the same name with " CE" appended */
			beg = value;
			for(;;) {
				com = strchr(beg, ',');
				new = (!com) ? ap_pstrdup(p->pool_tmp, beg)
					: ap_pstrndup(p->pool_tmp,beg,com-beg);

				/* cut trailing spaces */
				hlp = strchr(new, '\0'); /* find EOS */
				while((--hlp > new) && isspace((unsigned char)*hlp));
				hlp[1] = '\0';

				/* find out if the string ends with CE */
				if (hlp - 2 > new
					&& strcasecmp(&hlp[-1], "CE") != 0)
				{
					/* got ya, prepend "$fontname CE," */
					x_add_2buf(p->pool_tmp, buf,
						new, -1);
					x_add_2buf(p->pool_tmp, buf, " CE,", 4);
				}
				x_add_2buf(p->pool_tmp, buf, new, -1);
				if (com) {
					x_add_2buf(p->pool_tmp, buf, ",", 1);
					beg = com + 1;
				}
				else break;
			}
		}
		else {
			/* just plain copy the value to output */
			x_add_2buf(p->pool_tmp, buf, value, -1);
		}
		x_add_2buf(p->pool_tmp, buf, "\"", 1);
	}
	x_add_2buf(p->pool_tmp, buf, ">", 1);
	csa_add_recode_output(p, buf->value, buf->len, NULL);
			
	return 0;
}

/*
 * Implementation of XML "command" - if XML tag has attribute "encoding",
 * set it to correct value (our output encoding).
 */
/* ARGSUSED */
int
csa_Xml(p, prm)
  csa_params_t *p;
  void *prm;
{
	const csa_arg_t *arg;
	const char *key, *value, *begintag, *endtag;
	char quot;
	int seen_encoding=0;
	csa_String_b buff, *buf = &buff;

	if (!csa_yy_gettags(p, &begintag, &endtag))
		return 1;

	CSA_FILLSTRING(buf, (char *)csa_alloca(100, p->pool_tmp), 0, 100);
	x_add_2buf(p->pool_tmp, buf, begintag, -1);
	/*
	 * Do NOT put any space between begin tag (<? in this case) and
	 * command name (xml) - it has to be together as <?xml !
	 */
	x_add_2buf(p->pool_tmp, buf, csa_yy_getcmdname(p->yy), -1);

	while((arg = csa_arg_take(p)) != NULL) {
		key   = csa_arg_getkey(arg);
		value = csa_arg_getvalue(arg);
		quot = (char) csa_arg_getflags(arg);

		/*
		 * If the attribute is not encoding, just pass it
		 * unchanged.
		 */
		if (strcasecmp(key, "encoding") == 0) {
			seen_encoding = 1;
			if (csa_switch_incharset(p,cstools_whichcode(value,0))){
#ifdef CSA_DEBUG
				csa_debug(p->dbg,
				  "csa_Xml: incharset switched to %s", value);
#endif
			}

			/* for encoding, pass the name of output code set */
			value = cstools_name(p->outcharset, CSTOOLS_MIMENAME);
		}

		x_add_2buf(p->pool_tmp, buf, " ", 1);
		x_add_2buf(p->pool_tmp, buf, key, -1);
		x_add_2buf(p->pool_tmp, buf, "=", 1);
		if (quot) x_add_2buf(p->pool_tmp, buf, &quot, 1);
		x_add_2buf(p->pool_tmp, buf, value, -1);
		if (quot) x_add_2buf(p->pool_tmp, buf, &quot, 1);
	}

	/*
	 * If we haven't seen "encoding" attribute, return "error",
	 * so that the original command string would be written to output.
	 * No need to use the string generated by CSacek, even through it
	 * should be equivalent to the original string in this case.
	 */
	if (!seen_encoding)
		return 1;

	x_add_2buf(p->pool_tmp, buf, endtag, -1);
	csa_add_recode_output(p, buf->value, buf->len, NULL);
	return 0;
}

/*
 * Implementation of MYCHARSET command. Stripping of '=' is done
 * by parser (see cmds.y, rule CSA_MYCHARSET), so we deal with just
 * one parameter - charset name.
 */
/* ARGSUSED */
int
csa_MyCharset(p, prm)
  csa_params_t *p;
  void *prm;
{
	cstools_t code;
	const csa_arg_t *arg;
	
	arg = csa_arg_take(p);
	if (!arg) 
		return 1;

	code = cstools_whichcode(csa_arg_getvalue(arg), 0);
		    
	if (code == CSTOOLS_UNKNOWN)
		return 1;

	if (code != p->incharset) {
		p->incharset = code;
		cstools_init(&p->mp, p->incharset, p->outcharset);
#ifdef CSA_DEBUG
		csa_debug(p->dbg,
			"csa_MyCharset: incharset switched to %s",
			csa_arg_getvalue(arg));
#endif
	}

	return 0;
}

/*
 * Called upon finding <SCRIPT> or </SCRIPT> tags.
 */
int
csa_Script(p, prm)
  csa_params_t *p;
  void *prm;
{
	if (prm) {
		/* <SCRIPT> */
		CSA_SET(p->flags, CSA_FL_NOHTML);

		/* call csa_ChangeURL() to update URL in src="" accordingly */
		return csa_ChangeURL(p, prm);
	} else {
		/* </SCRIPT> */
		CSA_UNSET(p->flags, CSA_FL_NOHTML);

		/* return "failure", so that the command is passed to output */
		return 1;
	}
}
