/*- vim:sw=4:sts=4
 * In-memory representation of an MPH data file containing FCH data, as
 * generated by the utility cmph.  I don't want external files, so this MPH
 * data file (representation of the minimal perfect hash function) is stored
 * in these structures and compiled in.
 *
 * In addition to the hash function, which maps keys to bucket numbers,
 * actual data is included too.  This data the 32 bit hash value (to
 * identify hits and misses) and zero or more bytes of user defined data.
 */

/* Complete information on what hash algorithm to use, plus optional extra
 * data depending on the algorithm. */
struct hash_state {
    int hashfunc;		/* which hash function to use */
    int seed;			/* for jenkins */
};

struct fch_data {
    unsigned int m, b;		// m=number of buckets
    unsigned int g_size;	// 16 or 32
    unsigned int p1, p2;	// magic parameters
    struct hash_state h1, h2;
    unsigned short g[];		// 16 or 32 bit entries
};
// b = ceil(c * m / log(m) / log(2) + 1)
// p1 = ceil(0.6 * m)
// p2 = ceil(0.3 * b)

struct bdz_data {
    unsigned int
	m,		/* edges (words) count */
	n,		/* vertex count */
	r,		/* partition vertex count */
	k,		/* kth index in ranktable = log2(n=3r)/epsilon */
	b,		/* number of bits of k */
	ranktablesize,	/* number of entries in ranktable, n/k+1 */
	rt_item_size;	/* bytes per item in ranktable */
    unsigned char *ranktable;	/* items in high...low byte order */
    struct hash_state hl;
    unsigned char g[];
};

struct hash_info {
    const void *algo_data;		// data to calculate the bucket nr
    const unsigned char *index;		// array of hash_bucket
    const unsigned char *data;
    int offset_size;			// bytes per hash_bucket.data_offset
    int hash_size;			// bytes per hash_bucket.hash_value
    enum { HASH_FCH, HASH_BDZ } method;
};


/* in hash-cmph.c */
const unsigned char *hash_search(const struct hash_info *hi,
    const char *key, int keylen, int *datalen);

/* in hash-algorithms.c */
unsigned int compute_hash(const struct hash_state *state,
    const unsigned char *key, int keylen, unsigned int *vector);

/* in hash-bdz.c */
unsigned int bdz_search(const struct hash_info *hi, const unsigned char *key,
    int keylen, unsigned int *hash_value);

/* in hash-fch.c */
unsigned int fch_search(const struct hash_info *hi, const unsigned char *key,
    int keylen, unsigned int *hash_value);

#include <cmph_types.h>	// CMPH_HASH_xxx constants

/* Should you not have the above include file, just comment out the #include
 * directive.  The following ENUM should be the same as in that include file.
 */
#ifndef __CMPH_TYPES_H__
typedef enum { CMPH_HASH_DJB2, CMPH_HASH_FNV, CMPH_HASH_JENKINS, 
	       CMPH_HASH_SDBM, CMPH_HASH_COUNT } CMPH_HASH;
#endif

