/* Copyright (C) 2000/2001 sgop@users.sourceforge.net
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#include <ctype.h>
#include <stdio.h>
#include <sys/time.h>
#include <time.h>
#include <sys/socket.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>

#include <gtk/gtk.h>

#include "lopster.h"
#include "callbacks.h"
#include "connection.h"
#include "handler.h"
#include "global.h"
#include "search.h"
#include "transfer.h"
#include "scheme.h"
#include "resume.h"
#include "string_list.h"
#include "support.h"
#include "whois.h"

char *get_search_string(char *str)
{
  static char result[1024];
  char *pos;
  char *old_pos;
  char save;
  char *last;
  char *str2;

  str2 = l_strdup(str);
  pos = str2;

  last = strrchr(str2, '.');
  *result = 0;
  while (1) {
    old_pos = pos;
    while (isalnum(*pos))
      pos++;
    save = *pos;
    if (pos > old_pos + 1) {
      *pos = 0;
      if (*result)
	strcat(result, " ");
      strcat(result, old_pos);
    }
    if (last && (pos >= last))
      break;
    if (save == 0)
      break;
    pos++;
  }
  l_free(str2);
  return result;
}


resume_t *resume_new()
{
  resume_t *resume;

  resume = (resume_t *) l_malloc(sizeof(resume_t));
  resume->filename = NULL;
  resume->dirname = NULL;
  resume->size = 0;
  resume->local_size = -1;
  resume->search = NULL;
  resume->status = R_ACTIVE;
  resume->socket = NULL;
  resume->user = NULL;
  resume->org_file = NULL;
  resume->search_string = NULL;
  resume->online = 0;
  resume->downloads = NULL;
  resume->last_access = global.current_time;
  resume->ctree = NULL;
  resume->node = NULL;
  resume->is_dcc = 0;

  global.incomplete = g_list_append(global.incomplete, resume);

  return resume;
}

void resume_destroy(resume_t * resume)
{
  if (!resume)
    return;

  if (resume->filename)
    l_free(resume->filename);
  if (resume->dirname)
    l_free(resume->dirname);
  if (resume->search)
    resume_remove_search(resume);
#ifdef RESUME_DEBUG
  if (resume->socket) {
    g_warning("resume_destroy(): resume has socket pointer");
  }
#endif
  if (resume->user)
    l_free(resume->user);
  if (resume->org_file)
    l_free(resume->org_file);
  if (resume->search_string)
    l_free(resume->search_string);
  g_list_free(resume->downloads);
  global.incomplete = g_list_remove(global.incomplete, resume);
  resume_save();
  l_free(resume);
}

void resume_load(char *fname, int import)
{
  FILE *fd;
  resume_t *resume;
  char line[2048];
  char *filename;
  char *size;
  char *dirname;
  char *user;
  char *org_file;
  char *search;
  char *last_access;
  char *short_file;
  char* status;
  int have_incomplete;

  //  printf("reading [%s][%s]\n", fname, global.incomplete_path);
  if (!global.incomplete_path || !directory_exists(global.incomplete_path))
  {
    if (import) {
      client_message(_("Error"),
		     _
		     ("Could not import incomplete list: no valid incomplete folder"));
      return;
    }
    have_incomplete = 0;
  } else
    have_incomplete = 1;

  if ((fd = fopen(fname, "r")) == NULL)
    return;

  while (fgets(line, 2048 - 1, fd)) {
    filename = arg(line, 0);
    short_file = extract_filename(filename);
    size = arg(NULL, 0);
    dirname = arg(NULL, 0);
    user = arg(NULL, 0);
    org_file = arg(NULL, 0);
    search = arg(NULL, 0);
    last_access = arg(NULL, 0);
    status = arg(NULL, 0);

    if (!filename || !size)
      continue;

    if (resume_search_short_file(short_file))
      continue;
    resume = resume_new();
    if (have_incomplete && import)
      resume->filename =
	  l_strdup_printf("%s/%s", global.incomplete_path, short_file);
    else
      resume->filename = l_strdup(filename);

    resume->size = strtoul(size, NULL, 10);
    resume->local_size = -1;

    // backward compatible
    if (dirname && *dirname)
      resume->dirname = l_strdup(dirname);
    else
      resume->dirname = NULL;

    // backward compatible
    if (user && *user)
      resume->user = l_strdup(user);
    else
      resume->user = NULL;

    // backward compatible
    if (org_file && *org_file)
      resume->org_file = l_strdup(org_file);
    else
      resume->org_file = NULL;
    
    // backward compatible
    if (search && *search)
      resume->search_string = l_strdup(search);
    else
      resume->search_string = NULL;

    // backward compatible
    if (last_access && *last_access)
      resume->last_access = strtoul(last_access, NULL, 10);
    else
      resume->last_access = global.current_time;

    if (status) resume->status = atoi(status);
    
    resume_show(resume, 2);
  }
  fclose(fd);
  if (import)
    resume_save();
}

// function patched by saturn_de
void resume_save()
{
  GList *dlist;
  resume_t *resume;
  char *fname;
  char *fname_new;
  FILE *fd;

  fname = l_strdup_printf("%s/incomplete.list", global.options.config_dir);
  fname_new =
      l_strdup_printf("%s/incomplete.list.new", global.options.config_dir);

  if ((fd = fopen(fname_new, "w")) == NULL) {
    g_warning(_("Could not write [%s]\n"), fname);
    l_free(fname);
    l_free(fname_new);
    return;
  }

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t *) (dlist->data);
    if (resume->status == R_FINISHED) continue;
    if (resume->is_dcc) continue;
    fprintf(fd, "\"%s\" %lu \"%s\" \"%s\" \"%s\" \"%s\" %lu %d\n",
	    resume->filename, resume->size,
	    (resume->dirname) ? (resume->dirname) : "",
	    (resume->user) ? (resume->user) : "",
	    (resume->org_file) ? (resume->org_file) : "",
	    (resume->search_string) ? (resume->search_string) : "",
	    resume->last_access, resume->status);
  }

  if (!ferror(fd))
    rename(fname_new, fname);
  else {
    g_warning(_("Could not write [%s]\n"), fname);
  }

  fclose(fd);
  l_free(fname);
  l_free(fname_new);
}

void resume_insert_file(resume_t * resume, file_t * file)
{
  socket_t *socket;
  transfer_t *transfer;

  if (resume->user && resume->org_file) {
    // dont insert file, if it is the original one
    if (!l_strcasecmp(resume->user, file->user) &&
	!l_strcasecmp(resume->org_file, file->longname))
      return;
  }
  socket = download_create(file, 0, resume);
  if (!socket) return;

  transfer = (transfer_t *) (socket->data);
  
  if (!transfer_in_progress(transfer) && (socket->timeout < 0) &&
      (transfer->status != S_QUEUED) && (transfer->status != S_FINISHED) &&
      !string_list_search(LIST_NODOWNLOAD, transfer->user_info->user)) {
    transfer_status_set(socket, S_INACTIVE);
  }
}

resume_t *resume_search_short_file(char *filename)
{
  GList *dlist;
  resume_t *resume;
  char *fname;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t *) (dlist->data);
    fname = extract_filename(resume->filename);
    if (!strcmp(fname, filename))
      return resume;
  }
  return NULL;
}

resume_t *resume_search_long_file(char *filename)
{
  GList *dlist;
  resume_t *resume;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t *) (dlist->data);
    if (!strcmp(resume->filename, filename))
      return resume;
  }
  return NULL;
}

resume_t *resume_search_file(file_t * file)
{
  GList *dlist;
  resume_t *resume;
  char *pos;
  char *pos2;

  pos2 = extract_filename(file->longname);
  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t *) (dlist->data);
    pos = extract_filename(resume->filename);
    if (!strcmp(pos2, pos) && (file->size == resume->size))
      return resume;
  }
  return NULL;
}

GList *resume_search_size(unsigned long size)
{
  GList *dlist;
  resume_t *resume;
  GList* result = NULL;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = dlist->data;
    if (size == resume->size)
      result = g_list_append(result, resume);
  }
  return result;
}

void resume_create_search(resume_t * resume)
{
  search_t *search;
  search_pattern_t *pattern;
  char *pos;

  search = search_new();
  search->resume = 1;
  search->link = resume;

  pos = l_strdup(extract_filename(resume->filename));

  pattern = search_pattern_new();
  if (resume->search_string && *(resume->search_string))
    pattern->include = l_strdup(resume->search_string);
  else
    pattern->include = l_strdup(get_search_string(pos));
  pattern->name = l_strdup(pattern->include);
  l_free(pos);
  pattern->exclude = l_strdup("");
  pattern->max_results = 100;
  pattern->destination = DEST_NAPSTER;

  pattern->size_lo = pattern->size_hi = resume->size;
  pattern->media_type = l_strdup("any");

  search->pattern = pattern;
  resume->search = search;
  search_queue(search);
}

void resume_search(resume_t * resume, int ping)
{
  if (!resume) return;
  if (resume->status != R_ACTIVE) return;

  if (resume->user && ping) {
    speed_request(resume->user, SPEED_RESUME);
  }
  resume_remove_search(resume);
  resume_create_search(resume);
}

void resume_search_all(int force)
{
  resume_t *resume;
  GList *users = NULL;
  GList *dlist;

  if (global.status.connection < 2) return;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t *) (dlist->data);
    if (!resume->socket) {
      if (resume->user && (!is_string_in_list(users, resume->user))) {
	users = g_list_append(users, resume->user);
      }
      if (force || global.options.auto_search) resume_search(resume, 0);
    }
  }

  for (dlist = users; dlist; dlist = dlist->next) {
    speed_request(dlist->data, SPEED_RESUME);
  }
  g_list_free(users);
}

void resume_remove_search(resume_t * resume)
{

  if (!resume->search)
    return;

  // remove files from resume
  if (resume->search)
    search_remove(resume->search);
  resume->search = NULL;
  
  resume_update(resume);
}

void resume_show(resume_t * resume, int where)
{
  GtkCTree *ctree;
  char *pos;
  GtkCTreeNode *node;
  GList* dlist;
  socket_t* socket;

  if (!resume) return;
  
  if (!where) ctree = resume_belongs_to(resume);
  else if (where == 1) ctree = GTK_CTREE(lookup_widget(global.win, "ctree4"));
  else ctree = GTK_CTREE(lookup_widget(global.win, "ctree5"));
  if (!ctree) return;
  
  if (resume->ctree) {
    if (resume->ctree != ctree) resume_hide(resume);
    else return;
  }
  pos = extract_filename(resume->filename);

  strcpy(tstr[0], pos);
  tstr[1][0] = 0;
  tstr[2][0] = 0;
  tstr[3][0] = 0;
  tstr[4][0] = 0;
  tstr[5][0] = 0;
  tstr[6][0] = 0;
  tstr[7][0] = 0;

  node = gtk_ctree_insert_node(ctree, NULL, NULL, list, 5,
			       NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(ctree, node, resume);
  resume->ctree = ctree;
  resume->node = node;

  for (dlist = resume->downloads; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    download_show(socket);
  }

  resume_update(resume);
}

// intern function
int resume_update_transfer(resume_t * resume)
{
  GtkCTree *ctree;
  GtkCTreeNode *node;
  char str[200];
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  int sec;
  transfer_t *transfer;
  socket_t *socket;

  if (!resume)
    return 1;
  socket = resume->socket;
  if (!socket)
    return 1;
  transfer = (transfer_t *) (socket->data);

  ctree = resume->ctree;
  node = resume->node;
  if (!ctree || !node) return 1;

  // user
  if (transfer->user_info->user)
    strcpy(str, transfer->user_info->user);
  else
    strcpy(str, _("Not known yet"));
  gtk_ctree_node_set_text(ctree, node, 2, str);

  // filesize
  sprintf(str, "%ld/%ld", transfer->progress, transfer->size);
  gtk_ctree_node_set_text(ctree, node, 1, str);

  // status
  if ((transfer->status < 0) || (transfer->status >= S_NUMBER)) {
    g_warning("transfer_update: invalid status %d", transfer->status);
    return 1;
  }
  if (transfer->status == S_WAITING)
    sprintf(str, _("Waiting %d secs"), (socket->max_cnt - socket->cnt));
  else
    strcpy(str, status_names(transfer->status));
  gtk_ctree_node_set_text(ctree, node, 3, str);

  // speed
  gtk_ctree_node_set_text(ctree, node, 4, LineSpeed(transfer->user_info->linespeed));

  resume->local_size = transfer->progress;
  // progress
  if ((transfer->status == S_DOWNLOADING) ||
      (transfer->status == S_FINISHED) ||
      (transfer->status == S_UPLOADING)) {

    if (transfer->size > 0) {
      gtk_ctree_node_get_pixmap(ctree, node, 5, &pixmap, &bitmap);

      pixmap =
	  transfer_draw_progress(GTK_CLIST(ctree), pixmap,
				 (double) transfer->progress /
				 (double) transfer->size,
				 GTK_CLIST(ctree)->column[5].width,
				 transfer->status != S_FINISHED);
      gtk_ctree_node_set_pixmap(ctree, node, 5, pixmap, NULL);
    } else {
      gtk_ctree_node_set_text(ctree, node, 5, "");
    }

    print_speed(str, (int) transfer->rate, 1);
    detect_speed_pixs(transfer->rate, &pixmap, &bitmap);
    gtk_ctree_node_set_pixtext(ctree, node, 6, str, 5, pixmap, bitmap);

    if ((transfer->status == S_FINISHED) || (transfer->rate > 0)) {
      if (transfer->status == S_FINISHED)
	sec = transfer->end_time - transfer->start_time;
      else
	sec =
	    (int) ((transfer->size - transfer->progress) / transfer->rate);
      transfer->timeleft = sec;

      print_time_average(str, sec);
    } else {
      sprintf(str, _("stalled"));
    }
    gtk_ctree_node_set_text(ctree, node, 7, str);
  } else {
    gtk_ctree_node_set_text(ctree, node, 6, "");
    gtk_ctree_node_set_text(ctree, node, 7, "");
  }

  return 1;
}

// intern function
int resume_update_resume(resume_t * resume)
{
  GtkCTree *ctree;
  GtkCTreeNode *node;
  char text[1024];
  struct stat st;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;

  if (!resume)
    return 1;

  ctree = resume->ctree;
  node = resume->node;
  if (!ctree || !node) {
    g_warning(_("Resume not shown"));
    return 1;
  }

  if (resume->user) {
    gtk_ctree_node_set_text(ctree, node, 2, resume->user);
    if (resume->online)
      gtk_ctree_node_set_pixtext(ctree, node, 2, resume->user, 5,
				 global.pix.user1, global.pix.user1b);
    else
      gtk_ctree_node_set_pixtext(ctree, node, 2, resume->user, 5,
				 global.pix.user2, global.pix.user2b);
  } else {
    gtk_ctree_node_set_text(ctree, node, 2, _("? Unknown ?"));
  }

  if (resume->status == R_FINISHED) {
    // this is needed, because the file maybe isnt moved to the download folder yet
    // force lopster to show it complete rather than not existent
    resume->local_size = resume->size;
  } else if (stat(resume->filename, &st) < 0) {
    resume->local_size = -1;
  } else {
    resume->local_size = st.st_size;
  }
  sprintf(text, "%ld", resume->size);
  gtk_ctree_node_set_text(ctree, node, 1, text);

  if ((resume->size > 0) && (resume->local_size >= 0)) {
    gtk_ctree_node_get_pixmap(ctree, node, 5, &pixmap, &bitmap);

    pixmap =
	transfer_draw_progress(GTK_CLIST(ctree), pixmap,
			       (double) resume->local_size /
			       (double) resume->size,
			       GTK_CLIST(ctree)->column[5].width,
			       resume->status != R_FINISHED);
    gtk_ctree_node_set_pixmap(ctree, node, 5, pixmap, NULL);
  } else {
    gtk_ctree_node_set_text(ctree, node, 5, _("not existent"));
  }

  if (resume->status == R_FINISHED) {
    sprintf(text, _("Finished"));
  } else if (resume->status == R_QUEUED) {
    sprintf(text, _("Deactivated"));
  } else if (!resume->search) {
    sprintf(text, _("Not Searched"));
  } else if (resume->search->status == SEARCH_QUEUED) {
    sprintf(text, _("Search queued"));
  } else if (resume->search->status == SEARCH_ACTIVE) {
    sprintf(text, _("Search in progress"));
  } else {
    sprintf(text, _("%d results"), g_list_length(resume->search->results));
  }
  gtk_ctree_node_set_text(ctree, node, 3, text);
  gtk_ctree_node_set_text(ctree, node, 4, "");

  return 1;
}

void resume_update_resume2(resume_t * resume)
{
  GtkCTree *ctree = resume->ctree;
  GtkCTreeNode *node = resume->node;
  char text[1024];
  struct stat st;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *bitmap = NULL;
  time_t tim;

  if (!resume) return;

  if (resume->user) {
    if (resume->online)
      gtk_ctree_node_set_pixtext(ctree, node, 2, resume->user, 5,
				 global.pix.user1, global.pix.user1b);
    else
      gtk_ctree_node_set_pixtext(ctree, node, 2, resume->user, 5,
				 global.pix.user2, global.pix.user2b);
  } else {
    gtk_ctree_node_set_text(ctree, node, 2, _("? Unknown ?"));
  }

  if (stat(resume->filename, &st) < 0) {
    resume->local_size = -1;
    sprintf(text, "0/%ld", resume->size);
  } else {
    resume->local_size = st.st_size;
    sprintf(text, "%ld/%ld", resume->local_size, resume->size);
  }
  gtk_ctree_node_set_text(ctree, node, 1, text);

  if ((resume->size > 0) && (resume->local_size >= 0)) {
    gtk_ctree_node_get_pixmap(ctree, node, 3, &pixmap, &bitmap);

    pixmap =
      transfer_draw_progress(GTK_CLIST(ctree), pixmap,
			       (double) resume->local_size /
			       (double) resume->size,
			       GTK_CLIST(ctree)->column[3].width,
			       (resume->status != R_FINISHED));
    gtk_ctree_node_set_pixmap(ctree, node, 3, pixmap, NULL);
  } else {
    gtk_ctree_node_set_text(ctree, node, 3, _("not existent"));
  }

  if (resume->status == R_FINISHED) {
    sprintf(text, _("Finished"));
  } else if (resume->status == R_QUEUED) {
    sprintf(text, _("Deactivated"));
  } else if (!resume->search) {
    sprintf(text, _("Not Searched"));
  } else if (resume->search->status == SEARCH_QUEUED) {
    sprintf(text, _("Search queued"));
  } else if (resume->search->status == SEARCH_ACTIVE) {
    sprintf(text, _("Search in progress"));
  } else {
    sprintf(text, _("%d results"), g_list_length(resume->search->results));
  }
  gtk_ctree_node_set_text(ctree, node, 4, text);

  tim = resume_time_of_death(resume);
  if (tim == 0)
    strcpy(text, _("Never"));
  else if (tim == 1)
    strcpy(text, _("Now"));
  else {
    strcpy(text, ctime(&tim));
    text[strlen(text) - 1] = 0;
  }
  gtk_ctree_node_set_text(ctree, node, 5, text);
  return;
}

void resume_update(resume_t * resume)
{
  GtkCTree *ctree1;
  GtkCTree *ctree2;

  if (!resume)
    return;
  if (!resume->ctree || !resume->node) {
#ifdef RESUME_DEBUG
    g_warning("resume_update(): resume not shown\n");
#endif
    return;
  }

  if (resume_belongs_to(resume) != resume->ctree) {
    resume_show(resume, 0);
    return;
  }

  ctree1 = GTK_CTREE(lookup_widget(global.win, "ctree4"));
  ctree2 = GTK_CTREE(lookup_widget(global.win, "ctree5"));
  if (resume->ctree == ctree1) {
    if (resume->socket)
      resume_update_transfer(resume);
    else
      resume_update_resume(resume);
  } else if (resume->ctree == ctree2) {
#ifdef RESUME_DEBUG
    if (resume->socket)
      g_warning("resume_update(): resume in wrong tab");
    else
#endif
      resume_update_resume2(resume);
#ifdef RESUME_DEBUG
  } else {
    g_warning("resume_update(): resume not correctly shown");
#endif
  }
}

void resume_cancel(resume_t * resume)
{
  GList *dlist;
  socket_t *socket;
  transfer_t *transfer;

  if (!resume) return;

#ifdef RESUME_DEBUG
  printf("stopping r:%p\n", resume);
#endif
  // first remove resume link from unactive transfers
  for (dlist = resume->downloads; dlist; dlist = dlist->next) {
    socket = dlist->data;
    if (!socket) return;
    transfer = socket->data;
    if (!transfer) return;
    if ((transfer->status == S_QUEUED) || (transfer->status == S_INACTIVE) ||
	transfer_in_progress(transfer) || (socket->timeout != -1)) {
#ifdef RESUME_DEBUG
      printf("cancel t:%p\n", transfer);
#endif
      socket_end(socket, &(SocketStatus[S_CANCELED]));
    }
  }
}

void resume_remove(resume_t * resume)
{
  GList *dlist;
  socket_t *socket;

  if (!resume)
    return;

  //  resume_cancel(resume);

  // removing from visual list
  resume_hide(resume);

  if (resume->status != R_FINISHED) {
    //    printf("delete = %s\n", resume->filename);
    unlink(resume->filename);
  }

  for (dlist = resume->downloads; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    socket_destroy(socket, &(SocketStatus[S_DELETE]));
  }
  resume_destroy(resume);
}

void resume_freeze(resume_t * resume)
{
  if (resume->status != R_ACTIVE)
    return;
  resume_cancel(resume);
  resume->status = R_QUEUED;
  resume_update(resume);
}

void resume_thaw(resume_t * resume)
{
  if (resume->status != R_QUEUED)
    return;
  resume->status = R_ACTIVE;
  resume_update(resume);
}

void resume_dont_search(resume_t* resume) {
  if (!resume->user || !resume->user[0]) return;

  if (resume->search_string) l_free(resume->search_string);
  resume->search_string = l_strdup("__DONT_SEARCH__");
  resume_remove_search(resume);
}

void resume_allow_search(resume_t* resume) {
  if (resume->search_string) l_free(resume->search_string);
  resume->search_string = NULL;
}

int resume_timer(gpointer data ATTR_UNUSED)
{
  resume_search_all(0);

  if (global.resume_timer >= 0) gtk_timeout_remove(global.resume_timer);
  global.resume_timer =
    gtk_timeout_add(global.options.resume_timeout * 60 * 1000, resume_timer,  NULL);
  
  return 1;
}

int resume_has_potentials(resume_t* resume) {
  socket_t *socket;
  transfer_t *transfer;
  GList *dlist;
  int res = 0;

  if (!resume) return 0;

  for (dlist = resume->downloads; dlist; dlist = dlist->next) {
    socket = (socket_t *) (dlist->data);
    if (socket->timeout != -1) res |= 1;
    transfer = (transfer_t *) (socket->data);
    if ((transfer->status == S_INACTIVE) && global.options.auto_resume)
      res |= 2;
    if (transfer->status == S_QUEUED)
      res |= 4;
  }
  
  return res;
}

GtkCTree* resume_belongs_to(resume_t * resume)
{
  GtkCTree* ctree[2];

  ctree[0] = GTK_CTREE(lookup_widget(global.win, "ctree5"));
  ctree[1] = GTK_CTREE(lookup_widget(global.win, "ctree4"));

  if (!resume)
    return NULL;
  if (resume->socket)
    return ctree[1];
  if (resume->status == R_FINISHED)
    return ctree[1];
  if (resume->status == R_QUEUED)
    return ctree[0];

  if (resume_has_potentials(resume)) return ctree[1];
  else if (global.options.dl_autoremove & REMOVE_D_ABORTED) return ctree[0];
  else if (resume->ctree == ctree[0]) return ctree[0];
  else return ctree[1];
}

time_t resume_time_of_death(resume_t* resume) {
  time_t value;
  unsigned long lsize;

  if (resume->local_size >= 0) lsize = resume->local_size;
  else lsize = 0;

  if ((double)(lsize)*100. < 
      (double)(global.options.resume_abort[0])*(double)(resume->size))
    return 1;  // now
  if (!global.options.resume_abort[1]) return 0;

  if (resume->size <= 0) return 0;

  value = (time_t)global.options.resume_abort[2]*24*60*60 +
    (time_t)((double)global.options.resume_abort[3]*24*60*60*(double)(resume->local_size)/(double)resume->size);
  if (resume->last_access + value <= global.current_time) {
    return 1;
  } else {
    return resume->last_access + value;
  }
  return 0;
}

void resume_remove_outdated() {
  GList* dlist;
  resume_t *resume;
  GList* result = NULL;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = dlist->data;
    if (resume->status == R_FINISHED) {
      if (global.options.dl_autoremove & REMOVE_D_FINISHED)
	result = g_list_prepend(result, resume);;
    } else if (!resume->socket && !resume_has_potentials(resume) &&
	       (resume_time_of_death(resume) == 1))
      result = g_list_prepend(result, resume);;
  }
  for (dlist = result; dlist; dlist = dlist->next) {
    resume = dlist->data;
    resume_remove(resume);
  }
  g_list_free(result);
}

void resume_hide(resume_t * resume)
{
  if (!resume->ctree || !resume->node) {
#ifdef RESUME_DEBUG
    printf("resume_hide(): resume not shown\n");
#endif
    return;
  }
  gtk_ctree_remove_node(resume->ctree, resume->node);
  resume->ctree = NULL;
  resume->node = NULL;
}

void resume_user_online(char *user, int link)
{
  GList *dlist;
  resume_t *resume;
  file_t *new_file;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t *) (dlist->data);
    if (!resume->user)
      continue;
    if (l_strcasecmp(resume->user, user))
      continue;

    resume->online = 1;

    if (!resume->org_file) continue;
    //    if (resume->search->queued) continue;

    new_file = (file_t *) l_malloc(sizeof(file_t));
    new_file->winname = l_strdup(resume->org_file);
    new_file->md5 = l_strdup("UNKNOWN");
    new_file->size = resume->size;
    new_file->bitrate = 128;
    new_file->frequency = 44100;
    new_file->duration = 0;
    new_file->user = l_strdup(resume->user);
    new_file->ip = 0;
    new_file->linespeed = link;
    new_file->longname = l_strdup(new_file->winname);
    convert_to_unix(new_file->longname);
    new_file->shortname = l_strdup(extract_short_name(new_file->longname));
    new_file->mime_type = get_mimetype(new_file->shortname);
    new_file->flags = 0;
    new_file->local = 0;	//remote file
    new_file->ping = 0;
    if (resume->search) {
      resume_insert_file(resume, new_file);
      if (!search_find_file(resume->search, new_file))
	resume->search->results = g_list_append(resume->search->results, new_file);
      else destroy_file_row(new_file);
    } else {
      resume_insert_file(resume, new_file);
      destroy_file_row(new_file);
    }
    resume_update(resume);
  }
}

void resume_user_offline(char *user)
{
  GList *dlist;
  resume_t *resume;

  for (dlist = global.incomplete; dlist; dlist = dlist->next) {
    resume = (resume_t *) (dlist->data);
    if (!resume->user)
      continue;
    if (l_strcasecmp(resume->user, user))
      continue;

    resume->online = 0;
    resume_update(resume);
  }
}

GtkWidget* create_abort_win (void)
{
  GtkWidget *abort_win;
  GtkWidget *frame324;
  GtkWidget *vbox192;
  GtkWidget *vbox193;
  GtkWidget *hbox687;
  GtkWidget *label1012;
  GtkWidget *hscale1;
  GtkWidget *hbox686;
  GtkWidget *checkbutton69;
  GtkObject *spinbutton63_adj;
  GtkWidget *spinbutton63;
  GtkWidget *label1017;
  GtkWidget *entry122;
  GtkWidget *label1018;
  GtkObject *spinbutton64_adj;
  GtkWidget *spinbutton64;
  GtkWidget *label1016;
  GtkWidget *hseparator36;
  GtkWidget *frame325;
  GtkWidget *hbox685;
  GtkWidget *button318;
  GtkWidget *button319;
  GtkObject *adj;

  abort_win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_object_set_data (GTK_OBJECT (abort_win), "abort_win", abort_win);
  gtk_window_set_title (GTK_WINDOW (abort_win), _("Resume Abort"));
  gtk_window_set_default_size (GTK_WINDOW (abort_win), 520, -1);

  frame324 = gtk_frame_new (NULL);
  gtk_widget_show (frame324);
  gtk_container_add (GTK_CONTAINER (abort_win), frame324);
  gtk_container_set_border_width (GTK_CONTAINER (frame324), 5);

  vbox192 = gtk_vbox_new (FALSE, 5);
  gtk_widget_show (vbox192);
  gtk_container_add (GTK_CONTAINER (frame324), vbox192);
  gtk_container_set_border_width (GTK_CONTAINER (vbox192), 5);

  vbox193 = gtk_vbox_new (FALSE, 6);
  gtk_widget_show (vbox193);
  gtk_box_pack_start (GTK_BOX (vbox192), vbox193, TRUE, TRUE, 0);

  hbox687 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox687);
  gtk_box_pack_start (GTK_BOX (vbox193), hbox687, FALSE, FALSE, 0);

  label1012 = gtk_label_new (_("Keep incomplete files only if progress was at least"));
  gtk_widget_show (label1012);
  gtk_box_pack_start (GTK_BOX (hbox687), label1012, FALSE, FALSE, 0);

  adj = gtk_adjustment_new ((double)global.options.resume_abort[0]/(double)100,
			    0, 1, 0, 0, 0);
  gtk_object_set_data (GTK_OBJECT (abort_win), "adj1", adj);
  hscale1 = gtk_hscale_new (GTK_ADJUSTMENT (adj));
  gtk_widget_show (hscale1);
  gtk_box_pack_start (GTK_BOX (vbox193), hscale1, FALSE, FALSE, 0);
  gtk_scale_set_value_pos (GTK_SCALE (hscale1), GTK_POS_RIGHT);
  gtk_scale_set_digits (GTK_SCALE (hscale1), 2);

  hbox686 = gtk_hbox_new (FALSE, 5);
  gtk_widget_show (hbox686);
  gtk_box_pack_start (GTK_BOX (vbox193), hbox686, FALSE, FALSE, 0);

  checkbutton69 = gtk_check_button_new_with_label (_("Keep incomplete files at most"));
  if (global.options.resume_abort[1]) 
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbutton69), TRUE);
  gtk_widget_show (checkbutton69);
  gtk_object_set_data (GTK_OBJECT (abort_win), "check", checkbutton69);
  gtk_box_pack_start (GTK_BOX (hbox686), checkbutton69, FALSE, FALSE, 0);

  spinbutton63_adj = gtk_adjustment_new (global.options.resume_abort[2],
					 0, 1000, 1, 10, 10);
  gtk_object_set_data (GTK_OBJECT (abort_win), "adj2", spinbutton63_adj);
  spinbutton63 = gtk_spin_button_new (GTK_ADJUSTMENT (spinbutton63_adj), 1, 0);
  gtk_widget_show (spinbutton63);
  gtk_box_pack_start (GTK_BOX (hbox686), spinbutton63, TRUE, TRUE, 0);

  label1017 = gtk_label_new ("+");
  gtk_widget_show (label1017);
  gtk_box_pack_start (GTK_BOX (hbox686), label1017, FALSE, FALSE, 0);

  entry122 = gtk_entry_new ();
  gtk_widget_show (entry122);
  gtk_box_pack_start (GTK_BOX (hbox686), entry122, FALSE, FALSE, 0);
  gtk_widget_set_usize (entry122, 69, -2);
  gtk_entry_set_editable (GTK_ENTRY (entry122), FALSE);
  gtk_entry_set_text (GTK_ENTRY (entry122), _("progress"));

  label1018 = gtk_label_new ("x");
  gtk_widget_show (label1018);
  gtk_box_pack_start (GTK_BOX (hbox686), label1018, FALSE, FALSE, 0);

  spinbutton64_adj = gtk_adjustment_new (global.options.resume_abort[3],
					 0, 1000, 1, 10, 10);
  gtk_object_set_data (GTK_OBJECT (abort_win), "adj3", spinbutton64_adj);
  spinbutton64 = gtk_spin_button_new (GTK_ADJUSTMENT (spinbutton64_adj), 1, 0);
  gtk_widget_show (spinbutton64);
  gtk_box_pack_start (GTK_BOX (hbox686), spinbutton64, TRUE, TRUE, 0);

  label1016 = gtk_label_new (_("days since last success"));
  gtk_widget_show (label1016);
  gtk_box_pack_start (GTK_BOX (hbox686), label1016, FALSE, FALSE, 0);

  hseparator36 = gtk_hseparator_new ();
  gtk_widget_show (hseparator36);
  gtk_box_pack_start (GTK_BOX (vbox192), hseparator36, FALSE, FALSE, 0);

  frame325 = gtk_frame_new (NULL);
  gtk_widget_show (frame325);
  gtk_box_pack_start (GTK_BOX (vbox192), frame325, FALSE, FALSE, 0);
  gtk_widget_set_usize (frame325, -2, 41);
  gtk_frame_set_shadow_type (GTK_FRAME (frame325), GTK_SHADOW_IN);

  hbox685 = gtk_hbox_new (TRUE, 5);
  gtk_widget_show (hbox685);
  gtk_container_add (GTK_CONTAINER (frame325), hbox685);
  gtk_container_set_border_width (GTK_CONTAINER (hbox685), 5);

  button318 = gtk_button_new_with_label (_("Ok"));
  gtk_widget_show (button318);
  gtk_box_pack_start (GTK_BOX (hbox685), button318, TRUE, TRUE, 0);

  button319 = gtk_button_new_with_label (_("Cancel"));
  gtk_widget_show (button319);
  gtk_box_pack_start (GTK_BOX (hbox685), button319, TRUE, TRUE, 0);

  gtk_signal_connect (GTK_OBJECT (abort_win), "destroy",
                      GTK_SIGNAL_FUNC (gtk_widget_destroy),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (button318), "clicked",
                      GTK_SIGNAL_FUNC (on_button318_clicked),
                      (gpointer)abort_win);
  gtk_signal_connect (GTK_OBJECT (button319), "clicked",
                      GTK_SIGNAL_FUNC (on_button319_clicked),
                      (gpointer)abort_win);

  return abort_win;
}

